def share_dataset(source_project, source_datasets_name, current_project_key):
    """
    Shares a list of datasets from the source project with the specified target project.

    Parameters:
    source_project (dataikuapi.dss.project.DSSProject): The source project object where the datasets are located.
    source_datasets_name (list of str): A list of dataset names to be shared with the target project.
    current_project_key (str): The project key of the current project where the datasets will be shared.

    Behavior:
    - If the dataset is already shared with the target project, no changes are made.
    - If the dataset is not yet shared with the target project, it will be added to the sharing rules.
    - If the dataset does not exist in the list of exposed objects, it will be added with the appropriate rules.

    Returns:
    None
    """
    # Get the settings of the source project
    project_settings = source_project.get_settings()
    exposed_objects = project_settings.get_raw()["exposedObjects"]["objects"]
    
    # Iterate over each dataset to be shared
    for dataset_name in source_datasets_name:
        # Find the dataset in the exposed objects
        dataset_found = False
        for obj in exposed_objects:
            if obj["type"] == "DATASET" and obj["localName"] == dataset_name:
                dataset_found = True
                print(f'Dataset {dataset_name} found!')
                # Update the rules for the current dataset
                if "rules" not in obj:
                    obj["rules"] = []
                # Add the current project key if it's not already shared
                if not any(rule["targetProject"] == current_project_key for rule in obj["rules"]):
                    obj["rules"].append({"targetProject": current_project_key, "appearOnFlow": True})
                    print(f'Dataset shared to Project {current_project_key}!')
                break
        
        # If dataset is not found, add it to the exposed objects
        if not dataset_found:
            exposed_objects.append({
                "type": "DATASET",
                "localName": dataset_name,
                "quickSharingEnabled": True,
                "rules": [{"targetProject": current_project_key, "appearOnFlow": True}]
            })
            print(f'Dataset {dataset_name} found!')
            print(f'Dataset shared to Project {current_project_key}!')
    # Save the updated settings
    project_settings.save()

    
def unshare_dataset(source_project, source_datasets_name, current_project_key):
    """
    Unshares a list of datasets from the source project with the specified target project.

    Parameters:
    source_project (dataikuapi.dss.project.DSSProject): The source project object where the datasets are located.
    source_datasets_name (list of str): A list of dataset names to be unshared with the target project.
    current_project_key (str): The project key of the current project where the datasets will be unshared.

    Behavior:
    - Removes the sharing rule for the specified target project from the dataset's rules.
    - If no rules remain after removing the target project, the dataset is completely removed from the exposed objects.

    Returns:
    None
    """
    # Get the settings of the source project
    project_settings = source_project.get_settings()
    exposed_objects = project_settings.get_raw()["exposedObjects"]["objects"]
    
    # Iterate over each dataset to be unshared
    for dataset_name in source_datasets_name:
        # Find the dataset in the exposed objects
        for obj in exposed_objects:
            if obj["type"] == "DATASET" and obj["localName"] == dataset_name:
                # Remove the rule for the current project key
                obj["rules"] = [rule for rule in obj["rules"] if rule["targetProject"] != current_project_key]
                
                # If no rules are left, remove the dataset from the exposed objects
                if not obj["rules"]:
                    exposed_objects.remove(obj)
                break
    
    # Save the updated settings
    project_settings.save()