import numpy as np
import geopy.distance

from ..creation.polygons_creation import compute_polygon_envelope
from .inversion import reverse_geo_point_coordinates


def get_geo_points_collection_center(geo_points_collection):
    """
    geo points formats :
    - (longitude, latitude)
    - [longitude, latitude]
    """
    longitudes = []
    latitudes = []
    for geo_point in geo_points_collection:
        longitudes.append( geo_point[0] )
        latitudes.append( geo_point[1] )

    geo_points_center = [np.mean( longitudes ), np.mean( latitudes )]
    return geo_points_center


def get_geodesic_distance(geo_point_1, geo_point_2, reverse_coordinates):
    """
    geo_point_1, geo_point_2 formats :
    - (latitude, longitude)
    - [latitude, longitude]

    --> distance in km
    """
    try:
        if reverse_coordinates:
            distance = geopy.distance.distance( reverse_geo_point_coordinates( geo_point_1 ),
                                                reverse_geo_point_coordinates( geo_point_2 ) ).km
        else:
            distance = geopy.distance.distance( geo_point_1, geo_point_2 ).km
        return distance

    except:
        return None
    pass


def get_shapely_polygon_length(shapely_polygon):
    """
    shapely_polygon : shapely.geometry.polygon.Polygon
    """
    return shapely_polygon.exterior.coords.xy[0]


def get_polygon_envelope_metadata(polygon):
    longitudes, latitudes = compute_polygon_envelope(polygon)
    min_lat = np.min(latitudes)
    min_lon = np.min(longitudes)
    max_lat = np.max(latitudes)
    max_lon = np.max(longitudes)
    envelope_data = {"min_lat":min_lat,
                     "min_lon":min_lon,
                     "max_lat":max_lat,
                     "max_lon":max_lon}

    envelope_data_str = "({},{},{},{})".format(min_lat,
                                               min_lon,
                                               max_lat,
                                               max_lon)
    # get length of the envelope edges :
    min_lon_min_lat_corner = [min_lon, min_lat]
    max_lon_min_lat_corner = [max_lon, min_lat]
    min_lon_max_lat_corner = [min_lon, max_lat]
    envelope_edges_lengths = [get_geodesic_distance(min_lon_min_lat_corner, max_lon_min_lat_corner, True),
                              get_geodesic_distance(min_lon_min_lat_corner, min_lon_max_lat_corner, True)]

    # get length of the envelope as the longest edge
    envelope_length = max(envelope_edges_lengths)

    # get width of the envelope as the shortest edge
    envelope_width = min(envelope_edges_lengths)
    enveloppe_metadata = {
        "envelope_data": envelope_data,
        "envelope_data_str": envelope_data_str,
        "envelope_length": envelope_length,
        "envelope_width": envelope_width
                         }
    return enveloppe_metadata


def merge_list_multipolygon_geo_points(list_multipolygon):
    merged_geo_points = []
    for list_polygon in list_multipolygon:
        merged_geo_points += list_polygon
    return merged_geo_points


def sort_neighbor_geo_points_indexes_by_geodesic_distance(reference_geo_point, neighbor_geo_points, reverse_coordinates):
    neighbors_distances = [get_geodesic_distance( reference_geo_point, neighbor_geo_point, reverse_coordinates )
                           for neighbor_geo_point in neighbor_geo_points]
    sorted_neighbor_geo_points_indexes = np.argsort( neighbors_distances )
    return sorted_neighbor_geo_points_indexes
