import dataiku
import dataikuapi
import pandas as pd
import numpy as np
from flask import request
import json

from dss_project_utils import enrich_with_dictionary
from dku_utils import read_pickle_from_dss_folder, get_managed_folder_id_with_folder_name
from real_estate_pricing.geographic_handling.formating.points import convert_from_list_geo_point_to_geojson
from real_estate_pricing.geographic_handling.feature_engineering.information_extraction import get_geodesic_distance
from real_estate_pricing.services.utils import (load_features_from_parameters,
                                                geocode_address, enrich_address_with_city_name)
from real_estate_pricing.services.census_lookup import lookup_census_features
from real_estate_pricing.services.stations_lookup import lookup_stations_features
from real_estate_pricing.services.time_features_lookup import (lookup_district_time_features,
                                                               lookup_polygon_time_features,
                                                               lookup_subway_station_time_features)

from real_estate_pricing.flow.constants import CITY_CENTER, WINDOW_SIZES


project_key = dataiku.get_custom_variables()["projectKey"]
variables = dataiku.api_client().get_project(project_key).get_variables()

USE_API_NODE_FOR_PREDICTIONS = False # <-- SWITCH IT TO "True" if you deployed the API service
API_NODE_URL = variables["standard"]["api_node_url"] # <-- OR REPLACE BY YOUR URL STRING HERE
API_SERVICE = "real_estate_pricing"
if USE_API_NODE_FOR_PREDICTIONS:
    client = dataikuapi.APINodeClient(API_NODE_URL, API_SERVICE)

else:
    property_pricing_model = dataiku.Model("GxEFXycR")
    property_pricing_predictor = property_pricing_model.get_predictor()
    # Backend data loading :
    ## Census polygons :
    census_folder_id = get_managed_folder_id_with_folder_name(project_key, "census_data_indexing")
    census_years = read_pickle_from_dss_folder("census_years.p", census_folder_id)
    last_census_year = np.max(census_years)
    census_polygons_indexer = read_pickle_from_dss_folder("census_data_indexer_year_{}.p".format(last_census_year), census_folder_id)
    census_metadata = read_pickle_from_dss_folder("dict_census_metadata.p", census_folder_id)

    ## Subway stations : 
    subway_stations_folder_id = get_managed_folder_id_with_folder_name(project_key, "subway_stations_indexing")
    subway_stations_indexer = read_pickle_from_dss_folder("subway_stations_indexer.p", subway_stations_folder_id)
    stations_metadata = read_pickle_from_dss_folder("dict_subway_stations_metadata.p", subway_stations_folder_id)

    ## Time features : 
    time_features_folder_id = get_managed_folder_id_with_folder_name(project_key, "time_features_indexing")
    census_district_features = read_pickle_from_dss_folder("dict_census_district_last_features.p", time_features_folder_id)
    census_polygon_features = read_pickle_from_dss_folder("dict_census_polygon_last_features.p", time_features_folder_id)
    subway_stations_features = read_pickle_from_dss_folder("dict_closest_station_last_features.p", time_features_folder_id)

    
@app.route("/send_app_features/", methods=["POST"])
def send_app_features():
    front_parameters = request.get_json(force=True)
    print("Front parameters : {}".format(front_parameters))
    current_address = front_parameters["current_address"]
    current_address = enrich_address_with_city_name(current_address)
    print("address to geocode : '{}'".format(current_address))
    
    # Features loading :
    print("Loading features ...")
    features = load_features_from_parameters(front_parameters)
    print("Loaded features are : '{}'".format(features))
    
    if USE_API_NODE_FOR_PREDICTIONS:
        API_ENDPOINT = "wrapup"
        print("Doing API call...")
        print ("features issued", current_address,"apartment" , features["number_of_rooms"] ,features["living_surface"] , features["number_of_lots"])
        query_result = client.run_function(API_ENDPOINT,
                                           address=current_address,
                                           property_type="apartment",
                                           number_of_rooms=features["number_of_rooms"],
                                           living_surface=features["living_surface"],
                                           number_of_lots=features["number_of_lots"])
        print("API call done!")
        geocoding_response = query_result.get("response")["geocoding_response"]
        property_price_prediction = int(query_result.get("response")["property_price"])
        
    
    else:
        
        geocoding_response = geocode_address(current_address, 0)
        print(geocoding_response)

        address_latitude = float(geocoding_response["address_latitude"])
        address_longitude = float(geocoding_response["address_longitude"])
        address_geo_point = [address_latitude, address_longitude]
        print("Address geo point is : '{}'".format(address_geo_point))
        
        features["average_rooms_surface"] = features["living_surface"] / features["number_of_rooms"]
        # Distance from city center : 
        features["distance_from_city_center"] = get_geodesic_distance(address_geo_point, CITY_CENTER, False)

        # Census features :
        print("Retrieving census features ...")
        census_features = lookup_census_features(address_geo_point, census_metadata, census_polygons_indexer)
        census_district = census_features["census_district"]
        census_polygon_name = census_features["census_polygon_name"]
        features = enrich_with_dictionary(features, census_features, True)

        # Station features :
        print("Retrieving station features ...")
        station_features = lookup_stations_features(address_geo_point, stations_metadata, subway_stations_indexer)
        closest_station_name = station_features["closest_station_name"]
        features = enrich_with_dictionary(features, station_features, True)

        # Time features :
        print("Retrieving time features ...")
        district_time_features = lookup_district_time_features(census_district, census_district_features)
        polygon_time_features = lookup_polygon_time_features(census_polygon_name, census_polygon_features)
        station_time_features = lookup_subway_station_time_features(closest_station_name, subway_stations_features)

        for time_features in [district_time_features, polygon_time_features, station_time_features]:
            features = enrich_with_dictionary(features, time_features, True)

        for window_size in WINDOW_SIZES:
            polygon_xx_days_square_meter_price = features["census_polygon_name_{}_days_living_surface_square_meter_price_avg".format(window_size)]
            features["polygon_{}_days_price_estimate".format(window_size)] = features["living_surface"] * polygon_xx_days_square_meter_price

            district_xx_days_square_meter_price = features["census_district_{}_days_living_surface_square_meter_price_avg".format(window_size)]
            features["district_{}_days_price_estimate".format(window_size)] = features["living_surface"] * district_xx_days_square_meter_price

            closest_station_xx_days_square_meter_price = features["closest_station_{}_days_living_surface_square_meter_price_avg".format(window_size)]
            features["closest_station_{}_days_price_estimate".format(window_size)] = features["living_surface"] * closest_station_xx_days_square_meter_price

        # Property price prediction :

        print("Predicting property price ...")
        property_price_prediction = int(property_pricing_predictor.predict(pd.DataFrame(features, index=[0]))["prediction"][0])
    
    property_price_prediction_message = " {:,} €".format(property_price_prediction)
    print("Property price predicted : '{}'".format(property_price_prediction))
    print("Features gathered : '{}'".format(features))
    response = {"status": 200,
                "geocoding_response": geocoding_response,
                "property_price_prediction": property_price_prediction_message
               }
    print(f"BACKEND_LOG_MESSAGE | Sending app features to the front : {response}")
    return json.dumps(response)
