Data preparation and feature engineering steps should be done in a separate project to match the expected format of the input datasets. Respecting the expected data model when loading new data helps ensure the results are reliable and easy to interpret.

# Input datasets format
This Solution requires that the database assumes [OMOP CDM v5.3](https://ohdsi.github.io/CommonDataModel/cdm53.html) or a later version. 

## Converting OMOP to DSS datatype 💡 
We recommend that users conduct data preparation and feature engineering steps in a separate project to match the expected format of the input datasets. In most situations, date and datetime values require parsing via a prepared recipe on DSS. Please review [DSS documentation](https://doc.dataiku.com/dss/latest/preparation/dates.html#parsing-dates) on handling dates. However, the Solution pipeline involves an automatic datatype conversion as the following: 

| Datatype | Acceptable DSS datatype at source project | Result DSS datatype after ingestion |
|----------|-------------------------------------------|-------------------------------------|
| Integer  | int, bigint                               | int                                 |
| Float    | float, double                             | float                               |
| Date  | dateonly                                      | dateonly                                |
| Datetime     | datetimenotz                                      | datetimenotz                                |




This Solution requires a minimum subset of OMOP CDM tables:
 - person
 - observation_period
 - visit_occurrence
 - condition_occurrence
 - drug_exposure
 - death
 - location
 - condition_era
 - concept
 - concept_ancestor
 
The following schema is compatible with Dataiku DSS:  

<br>
## Standardized Clinical Data
### person

| Dataset name | Description                                                                                                                                                                                | Granularity      | Mandatory |   |
|--------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------|-----------|---|
| person       | This table serves as the central identity management for all Persons in the database. It contains records that uniquely identify each person or patient, and some demographic information. | 1 row = 1 person | yes       |   |
|                    |                                                                                                                                                                                                                                                                                     |                              |           |   |

| Name                        | Description                                                                                                                                                                                                                | Datatype | Mandatory |   |
|-----------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| person_id                   | It is assumed that every person with a different unique identifier is in fact a different person and should be treated independently.                                                                                      | integer  | Yes       |   |
| gender_concept_id           | This field is meant to capture the biological sex at birth of the Person. This field should not be used to study gender identity issues.                                                                                   | integer  | Yes       |   |
| year_of_birth               | Compute age using year_of_birth.                                                                                                                                                                                           | integer  | Yes       |   |
| month_of_birth              |                                                                                                                                                                                                                            | integer  | No       |   |
| day_of_birth                |                                                                                                                                                                                                                            | integer  | No       |   |
| birth_datetime              |                                                                                                                                                                                                                            | datetimenotz | Yes       |   |
| race_concept_id             | This field captures race or ethnic background of the person.                                                                                                                                                               | integer  | Yes       |   |
| ethnicity_concept_id        | This field captures Ethnicity as defined by the Office of Management and Budget (OMB) of the US Government: it distinguishes only between “Hispanic” and “Not Hispanic”. Races and ethnic backgrounds are not stored here. | integer  | Yes       |   |
| location_id                 | The location refers to the physical address of the person. This field should capture the last known location of the person.                                                                                                | integer  | No       |   |
| provider_id                 | The Provider refers to the last known primary care provider (General Practitioner).                                                                                                                                        | integer  | No       |   |
| care_site_id                | The Care Site refers to where the Provider typically provides the primary care.                                                                                                                                            | integer  | No       |   |
| person_source_value         | Use this field to link back to persons in the source data. This is typically used for error checking of ETL logic.                                                                                                         | string   | No       |   |
| gender_source_value         | This field is used to store the biological sex of the person from the source data. It is not intended for use in standard analytics but for reference only.                                                                | string   | No       |   |
| gender_source_concept_id    | Due to the small number of options, this tends to be zero.                                                                                                                                                                 | integer  | No       |   |
| race_source_value           | This field is used to store the race of the person from the source data. It is not intended for use in standard analytics but for reference only.                                                                          | string   | No       |   |
| race_source_concept_id      | Due to the small number of options, this tends to be zero.                                                                                                                                                                 | integer  | No       |   |
| ethnicity_source_value      | This field is used to store the ethnicity of the person from the source data. It is not intended for use in standard analytics but for reference only.                                                                     | string   | No       |   |
| ethnicity_source_concept_id | Due to the small number of options, this tends to be zero.                                                                                                                                                                 | integer  | No       |   |

### Observation period

| Dataset name       | Description                                                                                                                                                                                                                                                                         | Granularity                  | Mandatory |   |
|--------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------------------|-----------|---|
| observation_period | This table contains records which define spans of time during which two conditions are expected to hold: (i) Clinical Events that happened to the Person are recorded in the Event tables, and (ii) absence of records indicate such Events did not occur during this span of time. | 1 row = 1 observation period | yes       |   |
|                    |                                                                                                                                                                                                                                                                                     |                              |           |   |

| Name                          | Description                                                                                                                                                                                          | Datatype | Mandatory |   |
|-------------------------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| observation_period_id         | A Person can have multiple discrete Observation Periods which are identified by the Observation_Period_Id.                                                                                           | integer  | Yes       |   |
| person_id                     | The Person ID of the PERSON record for which the Observation Period is recorded.                                                                                                                     | integer  | Yes       |   |
| observation_period_start_date | Use this date to determine the start date of the Observation Period.                                                                                                                                 | dateonly     | Yes       |   |
| observation_period_end_date   | Use this date to determine the end date of the period for which we can assume that all events for a Person are recorded.                                                                             | dateonly     | Yes       |   |
| period_type_concept_id        | This field can be used to determine the provenance of the Observation Period as in whether the period was determined from an insurance enrollment file, EHR healthcare encounters, or other sources. | integer  | Yes       |   |


### Visit occurrence

| Dataset name     | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                | Granularity                | Mandatory |   |
|------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------------|-----------|---|
| visit_occurrence | This table contains Events where Persons engage with the healthcare system for a duration of time. They are often also called “Encounters”. Visits are defined by a configuration of circumstances under which they occur, such as (i) whether the patient comes to a healthcare institution, the other way around, or the interaction is remote, (ii) whether and what kind of trained medical staff is delivering the service during the Visit, and (iii) whether the Visit is transient or for a longer period involving a stay in bed. | 1 row = 1 visit occurrence | yes       |   |
|                  |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            |                            |           |   |

| Name                          | Description                                                                                                                                                                                                                                                                                                                                     | Datatype | Mandatory |   |
|-------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| visit_occurrence_id           | Use this to identify unique interactions between a person and the health care system. This identifier links across the other CDM event tables to associate events with a visit.                                                                                                                                                                 | integer  | Yes       |   |
| person_id                     |                                                                                                                                                                                                                                                                                                                                                 | integer  | Yes       |   |
| visit_concept_id              | This field contains a concept id representing the kind of visit, like inpatient or outpatient. All concepts in this field should be standard and belong to the Visit domain.                                                                                                                                                                    | integer  | Yes       |   |
| visit_start_date              | For inpatient visits, the start date is typically the admission date. For outpatient visits the start date and end date will be the same.                                                                                                                                                                                                       | dateonly     | Yes       |   |
| visit_start_datetime              |                                                                                                                                                                                                                                                                                                                                                 | datetimenotz     | No       |   |
| visit_end_date                | For inpatient visits the end date is typically the discharge date.                                                                                                                                                                                                                                                                              | dateonly     | Yes       |   |
| visit_end_datetime                |                                                                                                                                                                                                                                                                                                                                                 | datetimenotz     | No       |   |
| visit_type_concept_id         | Use this field to understand the provenance of the visit record, or where the record comes from.                                                                                                                                                                                                                                                | Integer  | Yes       |   |
| provider_id                   | There will only be one provider per visit record and the ETL document should clearly state how they were chosen (attending, admitting, etc.). If there are multiple providers associated with a visit in the source, this can be reflected in the event tables (CONDITION_OCCURRENCE, PROCEDURE_OCCURRENCE, etc.) or in the VISIT_DETAIL table. | integer  | No       |   |
| care_site_id                  | This field provides information about the Care Site where the Visit took place.                                                                                                                                                                                                                                                                 | integer  | No       |   |
| visit_source_value            | This field houses the verbatim value from the source data representing the kind of visit that took place (inpatient, outpatient, emergency, etc.)                                                                                                                                                                                               | string   | No       |   |
| visit_source_concept_id       |                                                                                                                                                                                                                                                                                                                                                 | integer  | No       |   |
| admitting_source_concept_id   | Use this field to determine where the patient was admitted from. This concept is part of the visit domain and can indicate if a patient was admitted to the hospital from a long-term care facility, for example.                                                                                                                               | integer  | No       |   |
| admitting_source_value        |                                                                                                                                                                                                                                                                                                                                                 | string   | No       |   |
| discharge_to_concept_id       | Use this field to determine where the patient was discharged to after a visit. This concept is part of the visit domain and can indicate if a patient was discharged to home or sent to a long-term care facility, for example.                                                                                                                 | integer  | No       |   |
| discharge_to_source_value     |                                                                                                                                                                                                                                                                                                                                                 | string   | No       |   |
| preceding_visit_occurrence_id | Use this field to find the visit that occurred for the person prior to the given visit. There could be a few days or a few years in between.                                                                                                                                                                                                    | integer  | No       |   |
|                               |                                                                                                                                                                                                                                                                                                                                                 |          |           |   |

### Visit detail

| Dataset name | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | Granularity     | Mandatory |   |
|--------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-----------------|-----------|---|
| visit_detail | The VISIT_DETAIL table is an optional table used to represents details of each record in the parent VISIT_OCCURRENCE table. A good example of this would be the movement between units in a hospital during an inpatient stay or claim lines associated with a one insurance claim. For every record in the VISIT_OCCURRENCE table there may be 0 or more records in the VISIT_DETAIL table with a 1:n relationship where n may be 0. The VISIT_DETAIL table is structurally very similar to VISIT_OCCURRENCE table and belongs to the visit domain. | 1 row = 1 visit | no       |   |
|              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      |                 |           |   |

| Name                           | Description                                                                                                                                                                                                                                                                                                                                                                                   | Datatype    | Mandatory |   |
|--------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------|-----------|---|
| visit_detail_id                | Use this to identify unique interactions between a person and the health care system. This identifier links across the other CDM event tables to associate events with a visit detail.                                                                                                                                                                                                        | integer     | Yes       |   |
| person_id                      |                                                                                                                                                                                                                                                                                                                                                                                               | integer     | Yes       |   |
| visit_detail_concept_id        | This field contains a concept id representing the kind of visit detail, like inpatient or outpatient. All concepts in this field should be standard and belong to the Visit domain.                                                                                                                                                                                                           | integer     | Yes       |   |
| visit_detail_start_date        | This is the date of the start of the encounter. This may or may not be equal to the date of the Visit the Visit Detail is associated with.                                                                                                                                                                                                                                                    | dateonly        | Yes       |   |
| visit_detail_start_datetime    |                                                                                                                                                                                                                                                                                                                                                                                               | datetimenotz    | No       |   |
| visit_detail_end_date          | This the end date of the patient-provider interaction.                                                                                                                                                                                                                                                                                                                                        | dateonly        | Yes       |   |
| visit_detail_end_datetime      |                                                                                                                                                                                                                                                                                                                                                                                               | datetimenotz    | No       |   |
| visit_detail_type_concept_id   | Use this field to understand the provenance of the visit detail record, or where the record comes from.                                                                                                                                                                                                                                                                                       | integer     | Yes       |   |
| provider_id                    | There will only be one provider per visit record and the ETL document should clearly state how they were chosen (attending, admitting, etc.). This is a typical reason for leveraging the VISIT_DETAIL table as even though each VISIT_DETAIL record can only have one provider, there is no limit to the number of VISIT_DETAIL records that can be associated to a VISIT_OCCURRENCE record. | integer     | No       |   |
| care_site_id                   | This field provides information about the Care Site where the Visit Detail took place.                                                                                                                                                                                                                                                                                                        | integer     | No       |   |
| visit_detail_source_value      | This field houses the verbatim value from the source data representing the kind of visit detail that took place (inpatient, outpatient, emergency, etc.)                                                                                                                                                                                                                                      | string | No       |   |
| visit_detail_source_concept_id |                                                                                                                                                                                                                                                                                                                                                                                               | Integer     | No       |   |
| admitting_source_value         |                                                                                                                                                                                                                                                                                                                                                                                               | string | No       |   |
| admitting_source_concept_id    | Use this field to determine where the patient was admitted from. This concept is part of the visit domain and can indicate if a patient was admitted to the hospital from a long-term care facility, for example.                                                                                                                                                                             | Integer     | No       |   |
| discharge_to_source_value      |                                                                                                                                                                                                                                                                                                                                                                                               | string | No       |   |
| discharge_to_concept_id        | Use this field to determine where the patient was discharged to after a visit detail record. This concept is part of the visit domain and can indicate if a patient was discharged to home or sent to a long-term care facility, for example.                                                                                                                                                 | integer     | No       |   |
| preceding_visit_detail_id      | Use this field to find the visit detail that occurred for the person prior to the given visit detail record. There could be a few days or a few years in between.                                                                                                                                                                                                                             | integer     | No       |   |
| visit_detail_parent_id         | Use this field to find the visit detail that subsumes the given visit detail record. This is used in the case that a visit detail record needs to be nested beyond the VISIT_OCCURRENCE/VISIT_DETAIL relationship.                                                                                                                                                                            | integer     | No       |   |
| visit_occurrence_id            | Use this field to link the VISIT_DETAIL record to its VISIT_OCCURRENCE.                                                                                                                                                                                                                                                                                                                       | integer     | Yes       |   |


### Condition occurrence

| Dataset name         | Description                                                                                                                                                                                                                 | Granularity                    | Mandatory |   |
|----------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|--------------------------------|-----------|---|
| condition_occurrence | This table contains records of Events of a Person suggesting the presence of a disease or medical condition stated as a diagnosis, a sign, or a symptom, which is either observed by a Provider or reported by the patient. | 1 row = 1 condition occurrence | yes       |   |
|                      |                                                                                                                                                                                                                             |                                |           |   |

| Name                          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | Datatype | Mandatory |   |
|-------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| condition_occurrence_id       | The unique key given to a condition record for a person. Refer to the ETL for how duplicate conditions during the same visit were handled.                                                                                                                                                                                                                                                                                                                                                                                        | integer  | Yes       |   |
| person_id                     | The PERSON_ID of the PERSON for whom the condition is recorded.                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | integer  | Yes       |   |
| condition_concept_id          | The CONDITION_CONCEPT_ID field is recommended for primary use in analyses, and must be used for network studies. This is the standard concept mapped from the source value which represents a condition                                                                                                                                                                                                                                                                                                                           | integer  | Yes       |   |
| condition_start_date          | Use this date to determine the start date of the condition                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | dateonly     | Yes       |   |
| condition_start_datetime          |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | datetimenotz     | No       |   |
| condition_end_date            | Use this date to determine the end date of the condition                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | dateonly     | Yes       |   |
| condition_end_datetime            |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | datetimenotz     | No       |   |
| condition_type_concept_id     | This field can be used to determine the provenance of the Condition record, as in whether the condition was from an EHR system, insurance claim, registry, or other sources.                                                                                                                                                                                                                                                                                                                                                      | integer  | Yes       |   |
| condition_status_concept_id   | This concept represents the point during the visit the diagnosis was given (admitting diagnosis, final diagnosis), whether the diagnosis was determined due to laboratory findings, if the diagnosis was exclusionary, or if it was a preliminary diagnosis, among others.                                                                                                                                                                                                                                                        | integer  | No       |   |
| stop_reason                   | The Stop Reason indicates why a Condition is no longer valid with respect to the purpose within the source data. Note that a Stop Reason does not necessarily imply that the condition is no longer occurring.                                                                                                                                                                                                                                                                                                                    | string   | No       |   |
| provider_id                   | The provider associated with condition record, e.g. the provider who made the diagnosis or the provider who recorded the symptom.                                                                                                                                                                                                                                                                                                                                                                                                 | integer  | No       |   |
| visit_occurrence_id           | The visit during which the condition occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | integer  | No       |   |
| visit_detail_id               | The VISIT_DETAIL record during which the condition occurred. For example, if the person was in the ICU at the time of the diagnosis the VISIT_OCCURRENCE record would reflect the overall hospital stay and the VISIT_DETAIL record would reflect the ICU stay during the hospital visit.                                                                                                                                                                                                                                         | integer  | No       |   |
| condition_source_value        | This field houses the verbatim value from the source data representing the condition that occurred. For example, this could be an ICD10 or Read code.                                                                                                                                                                                                                                                                                                                                                                             | string   | No       |   |
| condition_source_concept_id   | This is the concept representing the condition source value and may not necessarily be standard. This field is discouraged from use in analysis because it is not required to contain Standard Concepts that are used across the OHDSI community, and should only be used when Standard Concepts do not adequately represent the source detail for the Condition necessary for a given analytic use case. Consider using CONDITION_CONCEPT_ID instead to enable standardized analytics that can be consistent across the network. | integer  | No       |   |
| condition_status_source_value | This field houses the verbatim value from the source data representing the condition status.                                                                                                                                                                                                                                                                                                                                                                                                                                      | string   | No       |   |


### Drug exposure

| Dataset name  | Description                                                                                                                                                                                                                                                                                                                                                                                                                                              | Granularity             | Mandatory |   |
|---------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------------|-----------|---|
| drug_exposure | This table captures records about the exposure to a Drug ingested or otherwise introduced into the body. A Drug is a biochemical substance formulated in such a way that when administered to a Person it will exert a certain biochemical effect on the metabolism. Drugs include prescription and over-the-counter medicines, vaccines, and large-molecule biologic therapies. Radiological devices ingested or applied locally do not count as Drugs. | 1 row = 1 drug exposure | yes       |   |
|               |                                                                                                                                                                                                                                                                                                                                                                                                                                                          |                         |           |   |

| Name                     | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | Datatype | Mandatory |   |
|--------------------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| drug_exposure_id         | The unique key given to records of drug dispensings or administrations for a person. Refer to the ETL for how duplicate drugs during the same visit were handled.                                                                                                                                                                                                                                                                                                                                                                                                               | integer  | Yes       |   |
| person_id                | The PERSON_ID of the PERSON for whom the drug dispensing or administration is recorded. This may be a system generated code.                                                                                                                                                                                                                                                                                                                                                                                                                                                    | integer  | Yes       |   |
| drug_concept_id          | The DRUG_CONCEPT_ID field is recommended for primary use in analyses, and must be used for network studies. This is the standard concept mapped from the source concept id which represents a drug product or molecule otherwise introduced to the body. The drug concepts can have a varying degree of information about drug strength and dose. This information is relevant in the context of quantity and administration information in the subsequent fields plus strength information from the DRUG_STRENGTH table, provided as part of the standard vocabulary download. | integer  | Yes       |   |
| drug_exposure_start_date | Use this date to determine the start date of the drug record.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | dateonly     | Yes       |   |
| drug_exposure_start_datetime |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | datetimenotz     | No       |   |
| drug_exposure_end_date   | The DRUG_EXPOSURE_END_DATE denotes the day the drug exposure ended for the patient.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | dateonly     | Yes       |   |
| drug_exposure_end_datetime   |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | datetimenotz     | No       |   |
| verbatim_end_date        | This is the end date of the drug exposure as it appears in the source data, if it is given                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | dateonly     | No       |   |
| drug_type_concept_id     | You can use the TYPE_CONCEPT_ID to delineate between prescriptions written vs. prescriptions dispensed vs. medication history vs. patient-reported exposure, etc.                                                                                                                                                                                                                                                                                                                                                                                                               | integer  | Yes       |   |
| stop_reason              | The reason a person stopped a medication as it is represented in the source. Reasons include regimen completed, changed, removed, etc. This field will be retired in v6.0.                                                                                                                                                                                                                                                                                                                                                                                                      | string   | No       |   |
| refills                  | This is only filled in when the record is coming from a prescription written this field is meant to represent intended refills at time of the prescription.                                                                                                                                                                                                                                                                                                                                                                                                                     | integer  | No       |   |
| quantity                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | float    | No       |   |
| days_supply              | The number of days of supply of the medication as recorded in the original prescription or dispensing record. Days supply can differ from actual drug duration (i.e. prescribed days supply vs actual exposure).                                                                                                                                                                                                                                                                                                                                                                | integer  | No       |   |
| sig                      | This is the verbatim instruction for the drug as written by the provider.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | string   | No       |   |
| route_concept_id         |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | integer  | No       |   |
| lot_number               |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | string   | No       |   |
| provider_id              | The Provider associated with drug record, e.g. the provider who wrote the prescription or the provider who administered the drug.                                                                                                                                                                                                                                                                                                                                                                                                                                               | integer  | No       |   |
| visit_occurrence_id      | The Visit during which the drug was prescribed, administered or dispensed.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | integer  | No       |   |
| visit_detail_id          | The VISIT_DETAIL record during which the drug exposure occurred. For example, if the person was in the ICU at the time of the drug administration the VISIT_OCCURRENCE record would reflect the overall hospital stay and the VISIT_DETAIL record would reflect the ICU stay during the hospital visit.                                                                                                                                                                                                                                                                         | integer  | No       |   |
| drug_source_value        | This field houses the verbatim value from the source data representing the drug exposure that occurred. For example, this could be an NDC or Gemscript code.                                                                                                                                                                                                                                                                                                                                                                                                                    | string   | No       |   |
| drug_source_concept_id   | This is the concept representing the drug source value and may not necessarily be standard. This field is discouraged from use in analysis because it is not required to contain Standard Concepts that are used across the OHDSI community, and should only be used when Standard Concepts do not adequately represent the source detail for the Drug necessary for a given analytic use case. Consider using DRUG_CONCEPT_ID instead to enable standardized analytics that can be consistent across the network.                                                              | integer  | No       |   |
| route_source_value       | This field houses the verbatim value from the source data representing the drug route.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | string   | No       |   |
| dose_unit_source_value   | This field houses the verbatim value from the source data representing the dose unit of the drug given.                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | string   | No       |   |


### Procedure occurrence

| Dataset name         | Description                                                                                                                                                          | Granularity         | Mandatory |   |
|----------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------|---------------------|-----------|---|
| procedure_occurrence | This table contains records of activities or processes ordered by, or carried out by, a healthcare provider on the patient with a diagnostic or therapeutic purpose. | 1 row = 1 procedure | no        |   |
|                      |                                                                                                                                                                      |                     |           |   |

| Name                        | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | Datatype | Mandatory |   |
|-----------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| procedure_occurrence_id     | The unique key given to a procedure record for a person. Refer to the ETL for how duplicate procedures during the same visit were handled.                                                                                                                                                                                                                                                                                                                                                                                        | integer  | Yes       |   |
| person_id                   | The PERSON_ID of the PERSON for whom the procedure is recorded. This may be a system generated code.                                                                                                                                                                                                                                                                                                                                                                                                                              | integer  | Yes       |   |
| procedure_concept_id        | The PROCEDURE_CONCEPT_ID field is recommended for primary use in analyses, and must be used for network studies. This is the standard concept mapped from the source value which represents a procedure                                                                                                                                                                                                                                                                                                                           | integer  | Yes       |   |
| procedure_date              | Use this date to determine the date the procedure occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                                                       | dateonly     | Yes       |   |
| procedure_datetime              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                   | datetimenotz     | No       |   |
| procedure_type_concept_id   | This field can be used to determine the provenance of the Procedure record, as in whether the procedure was from an EHR system, insurance claim, registry, or other sources.                                                                                                                                                                                                                                                                                                                                                      | integer  | Yes       |   |
| modifier_concept_id         | The modifiers are intended to give additional information about the procedure but as of now the vocabulary is under review.                                                                                                                                                                                                                                                                                                                                                                                                       | integer  | No       |   |
| quantity                    | If the quantity value is omitted, a single procedure is assumed.                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | integer  | No       |   |
| provider_id                 | The provider associated with the procedure record, e.g. the provider who performed the Procedure.                                                                                                                                                                                                                                                                                                                                                                                                                                 | integer  | No       |   |
| visit_occurrence_id         | The visit during which the procedure occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    | integer  | No       |   |
| visit_detail_id             | The VISIT_DETAIL record during which the Procedure occurred. For example, if the Person was in the ICU at the time of the Procedure the VISIT_OCCURRENCE record would reflect the overall hospital stay and the VISIT_DETAIL record would reflect the ICU stay during the hospital visit.                                                                                                                                                                                                                                         | integer  | No       |   |
| procedure_source_value      | This field houses the verbatim value from the source data representing the procedure that occurred. For example, this could be an CPT4 or OPCS4 code.                                                                                                                                                                                                                                                                                                                                                                             | string   | No       |   |
| procedure_source_concept_id | This is the concept representing the procedure source value and may not necessarily be standard. This field is discouraged from use in analysis because it is not required to contain Standard Concepts that are used across the OHDSI community, and should only be used when Standard Concepts do not adequately represent the source detail for the Procedure necessary for a given analytic use case. Consider using PROCEDURE_CONCEPT_ID instead to enable standardized analytics that can be consistent across the network. | integer  | No       |   |
| modifier_source_value       | This field houses the verbatim value from the source data representing the modifier code for the procedure that occurred.                                                                                                                                                                                                                                                                                                                                                                                                         | string   | No       |   |


### Device exposure

| Dataset name    | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | Granularity      | Mandatory |   |
|-----------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------|-----------|---|
| device_exposure | The Device domain captures information about a person’s exposure to a foreign physical object or instrument which is used for diagnostic or therapeutic purposes through a mechanism beyond chemical action. Devices include implantable objects (e.g. pacemakers, stents, artificial joints), medical equipment and supplies (e.g. bandages, crutches, syringes), other instruments used in medical procedures (e.g. sutures, defibrillators) and material used in clinical care (e.g. adhesives, body material, dental material, surgical material). | 1 row = 1 device | no        |   |
|                 |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |                  |           |   |

| Name                           | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | Datatype | Mandatory |   |
|--------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| device_exposure_id     | The unique key given to records a person’s exposure to a foreign physical object or instrument.                                                                                                                                                                                                                                                                                                                                                                                     | integer  | Yes       |   |
| person_id                      |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | integer  | Yes       |   |
| device_concept_id              | The DEVICE_CONCEPT_ID field is recommended for primary use in analyses, and must be used for network studies. This is the standard concept mapped from the source concept id which represents a foreign object or instrument the person was exposed to.                                                                                                                                                                                                                                                                  | integer  | Yes       |   |
| device_exposure_start_date     | Use this date to determine the start date of the device record.                                                                                                                                                                                                                                                                                                                                                                                                                                                          | dateonly     | Yes       |   |
| device_exposure_start_datetime |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | datetimenotz     | No       |   |
| device_exposure_end_date       | The DEVICE_EXPOSURE_END_DATE denotes the day the device exposure ended for the patient, if given.                                                                                                                                                                                                                                                                                                                                                                                                                        | dateonly     | No       |   |
| device_exposure_end_datetime   |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | datetimenotz | No       |   |
| device_type_concept_id         | You can use the TYPE_CONCEPT_ID to denote the provenance of the record, as in whether the record is from administrative claims or EHR.                                                                                                                                                                                                                                                                                                                                                                                   | integer  | Yes       |   |
| unique_device_id               | This is the Unique Device Identification number for devices regulated by the FDA, if given.                                                                                                                                                                                                                                                                                                                                                                                                                              | date     | No       |   |
| quantity                       |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          | integer  | No       |   |
| provider_id                    | The Provider associated with device record, e.g. the provider who wrote the prescription or the provider who implanted the device.                                                                                                                                                                                                                                                                                                                                                                                       | integer  | No       |   |
| visit_occurrence_id            | The Visit during which the device was prescribed or given.                                                                                                                                                                                                                                                                                                                                                                                                                                                               | integer  | No       |   |
| visit_detail_id                | The Visit Detail during which the device was prescribed or given.                                                                                                                                                                                                                                                                                                                                                                                                                                                        | integer  | No       |   |
| device_source_value            | This field houses the verbatim value from the source data representing the device exposure that occurred. For example, this could be an NDC or Gemscript code.                                                                                                                                                                                                                                                                                                                                                           | string   | No       |   |
| device_source_concept_id       | This is the concept representing the device source value and may not necessarily be standard. This field is discouraged from use in analysis because it is not required to contain Standard Concepts that are used across the OHDSI community, and should only be used when Standard Concepts do not adequately represent the source detail for the Device necessary for a given analytic use case. Consider using DEVICE_CONCEPT_ID instead to enable standardized analytics that can be consistent across the network. | integer  | No       |   |


### Measure

| Dataset name | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | Granularity       | Mandatory |   |
|--------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------|-----------|---|
| measurement  | The MEASUREMENT table contains records of Measurements, i.e. structured values (numerical or categorical) obtained through systematic and standardized examination or testing of a Person or Person’s sample. The MEASUREMENT table contains both orders and results of such Measurements as laboratory tests, vital signs, quantitative findings from pathology reports, etc. Measurements are stored as attribute value pairs, with the attribute as the Measurement Concept and the value representing the result. The value can be a Concept (stored in VALUE_AS_CONCEPT), or a numerical value (VALUE_AS_NUMBER) with a Unit (UNIT_CONCEPT_ID). The Procedure for obtaining the sample is housed in the PROCEDURE_OCCURRENCE table, though it is unnecessary to create a PROCEDURE_OCCURRENCE record for each measurement if one does not exist in the source data. Measurements differ from Observations in that they require a standardized test or some other activity to generate a quantitative or qualitative result. If there is no result, it is assumed that the lab test was conducted but the result was not captured. | 1 row = 1 measure | no        |   |
|              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        |                   |           |   |

| Name                          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | Datatype | Mandatory |   |
|-------------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| measurement_id                | The unique key given to a Measurement record for a Person. Refer to the ETL for how duplicate Measurements during the same Visit were handled.                                                                                                                                                                                                                                                                                                                                                                                          | integer  | Yes       |   |
| person_id                     | The PERSON_ID of the Person for whom the Measurement is recorded. This may be a system generated code.                                                                                                                                                                                                                                                                                                                                                                                                                                  | integer  | Yes       |   |
| measurement_concept_id        | The MEASUREMENT_CONCEPT_ID field is recommended for primary use in analyses, and must be used for network studies. This is the standard concept mapped from the source value which represents a measurement.                                                                                                                                                                                                                                                                                                                            | integer  | Yes       |   |
| measurement_date              | Use this date to determine the date of the measurement.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | dateonly     | Yes       |   |
| measurement_datetime          |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | datetimenotz     | No       |   |
| measurement_time              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | string   | No       |   |
| measurement_type_concept_id   | This field can be used to determine the provenance of the Measurement record, as in whether the measurement was from an EHR system, insurance claim, registry, or other sources.                                                                                                                                                                                                                                                                                                                                                        | integer  | Yes       |   |
| operator_concept_id           | The meaning of Concept 4172703 for ‘=’ is identical to omission of a OPERATOR_CONCEPT_ID value. Since the use of this field is rare, it’s important when devising analyses to not to forget testing for the content of this field for values different from =.                                                                                                                                                                                                                                                                          | integer  | No       |   |
| value_as_number               | This is the numerical value of the Result of the Measurement, if available. Note that measurements such as blood pressures will be split into their component parts i.e. one record for systolic, one record for diastolic.                                                                                                                                                                                                                                                                                                             | float    | No       |   |
| value_as_concept_id           | If the raw data gives a categorial result for measurements those values are captured and mapped to standard concepts in the ‘Meas Value’ domain.                                                                                                                                                                                                                                                                                                                                                                                        | integer  | No       |   |
| unit_concept_id               | At present, there isn’t a prescribed unit for individual measurements, such as Hemoglobin A1C, meaning it’s not obligatory to express these measurements as a percentage. UNIT_SOURCE_VALUES should be linked to a Standard Concept within the Unit domain that most accurately reflects the unit provided in the source data.                                                                                                                                                                                                          | integer  | No       |   |
| range_low                     | Ranges have the same unit as the VALUE_AS_NUMBER. These ranges are provided by the source and should remain NULL if not given.                                                                                                                                                                                                                                                                                                                                                                                                          | float    | No       |   |
| range_high                    | Ranges have the same unit as the VALUE_AS_NUMBER. These ranges are provided by the source and should remain NULL if not given.                                                                                                                                                                                                                                                                                                                                                                                                          | float    | No       |   |
| provider_id                   | The provider associated with measurement record, e.g. the provider who ordered the test or the provider who recorded the result.                                                                                                                                                                                                                                                                                                                                                                                                        | integer  | No       |   |
| visit_occurrence_id           | The visit during which the Measurement occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | integer  | No       |   |
| visit_detail_id               | The VISIT_DETAIL record during which the Measurement occurred. For example, if the Person was in the ICU at the time the VISIT_OCCURRENCE record would reflect the overall hospital stay and the VISIT_DETAIL record would reflect the ICU stay during the hospital visit.                                                                                                                                                                                                                                                              | integer  | No       |   |
| measurement_source_value      | This field contains the exact value from the source data that represents the measurement that occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                 | string   | No       |   |
| measurement_source_concept_id | This is the concept representing the MEASUREMENT_SOURCE_VALUE and may not necessarily be standard. This field is discouraged from use in analysis because it is not required to contain Standard Concepts that are used across the OHDSI community, and should only be used when Standard Concepts do not adequately represent the source detail for the Measurement necessary for a given analytic use case. Consider using MEASUREMENT_CONCEPT_ID instead to enable standardized analytics that can be consistent across the network. | integer  | No       |   |
| unit_source_value             | This field contains the exact value from the source data that represents the unit of measurement used.                                                                                                                                                                                                                                                                                                                                                                                                                                  | string   | No       |   |
| value_source_value            | This field houses the verbatim result value of the Measurement from the source data .                                                                                                                                                                                                                                                                                                                                                                                                                                                   | string   | No       |   |


### Observation

| Dataset name | Description                                                                                                                                                                                                                                                                             | Granularity           | Mandatory |   |
|--------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-----------------------|-----------|---|
| observation  | The OBSERVATION table captures clinical facts about a Person obtained in the context of examination, questioning or a procedure. Any data that cannot be represented by any other domains, such as social and lifestyle facts, medical history, family history, etc. are recorded here. | 1 row = 1 observation | no        |   |
|              |                                                                                                                                                                                                                                                                                         |                       |           |   |

| Name                          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                      | Datatype | Mandatory |   |
|-------------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| observation_id                | The unique key given to an Observation record for a Person. Refer to the ETL for how duplicate Observations during the same Visit were handled.                                                                                                                                                                                                                                                                                                                                                                                                  | integer  | Yes       |   |
| person_id                     | The PERSON_ID of the Person for whom the Observation is recorded. This may be a system generated code.                                                                                                                                                                                                                                                                                                                                                                                                                                           | integer  | Yes       |   |
| observation_concept_id        | The OBSERVATION_CONCEPT_ID field is recommended for primary use in analyses, and must be used for network studies.                                                                                                                                                                                                                                                                                                                                                                                                                               | integer  | Yes       |   |
| observation_date              | The date of when the Observation was obtained. Depending on what the Observation represents this could be the date of a lab test, the date of a survey, or the date a patient’s family history was taken.                                                                                                                                                                                                                                                                                                                                        | dateonly     | Yes       |   |
| observation_datetime          |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  | datetimenotz     | No       |   |
| observation_type_concept_id   | This field can be used to determine the provenance of the Observation record, as in whether the measurement was from an EHR system, insurance claim, registry, or other sources.                                                                                                                                                                                                                                                                                                                                                                 | integer  | Yes       |   |
| value_as_number               | This is the numerical value of the Result of the Observation, if applicable and available. It is not expected that all Observations will have numeric results, rather, this field is here to house values should they exist.                                                                                                                                                                                                                                                                                                                     | float    | No       |   |
| value_as_string               | This is the categorical value of the Result of the Observation, if applicable and available.                                                                                                                                                                                                                                                                                                                                                                                                                                                     | string   | No       |   |
| value_as_concept_id           | It is possible that some records destined for the Observation table have two clinical ideas represented in one source code. This is common with ICD10 codes that describe a family history of some Condition, for example. In OMOP the Vocabulary breaks these two clinical ideas into two codes; one becomes the OBSERVATION_CONCEPT_ID and the other becomes the VALUE_AS_CONCEPT_ID. It is important when using the Observation table to keep this possibility in mind and to examine the VALUE_AS_CONCEPT_ID field for relevant information. | Integer  | No       |   |
| qualifier_concept_id          | This field contains all attributes specifying the clinical fact further, such as as degrees, severities, drug-drug interaction alerts etc.                                                                                                                                                                                                                                                                                                                                                                                                       | integer  | No       |   |
| unit_concept_id               | There is currently no recommended unit for individual observation concepts. UNIT_SOURCE_VALUES should be mapped to a Standard Concept in the Unit domain that best represents the unit as given in the source data.                                                                                                                                                                                                                                                                                                                              | integer  | No       |   |
| provider_id                   | The provider associated with the observation record, e.g. the provider who ordered the test or the provider who recorded the result.                                                                                                                                                                                                                                                                                                                                                                                                             | integer  | No       |   |
| visit_occurrence_id           | The visit during which the Observation occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 | integer  | No       |   |
| visit_detail_id               | The VISIT_DETAIL record during which the Observation occurred. For example, if the Person was in the ICU at the time the VISIT_OCCURRENCE record would reflect the overall hospital stay and the VISIT_DETAIL record would reflect the ICU stay during the hospital visit.                                                                                                                                                                                                                                                                       | integer  | No       |   |
| observation_source_value      | This field houses the verbatim value from the source data representing the Observation that occurred. For example, this could be an ICD10 or Read code.                                                                                                                                                                                                                                                                                                                                                                                          | string   | No       |   |
| observation_source_concept_id | This is the concept representing the OBSERVATION_SOURCE_VALUE and may not necessarily be standard. This field is discouraged from use in analysis because it is not required to contain Standard Concepts that are used across the OHDSI community, and should only be used when Standard Concepts do not adequately represent the source detail for the Observation necessary for a given analytic use case. Consider using OBSERVATION_CONCEPT_ID instead to enable standardized analytics that can be consistent across the network.          | integer  | No       |   |
| unit_source_value             | This field houses the verbatim value from the source data representing the unit of the Observation that occurred.                                                                                                                                                                                                                                                                                                                                                                                                                                | string   | No       |   |
| qualifier_source_value        | This field houses the verbatim value from the source data representing the qualifier of the Observation that occurred.                                                                                                                                                                                                                                                                                                                                                                                                                           | string   | No       |   |


### Death

| Dataset name | Description                                                                                                                                                                                                                                                                                     | Granularity     | Mandatory |   |
|--------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-----------------|-----------|---|
| death        | The death domain contains the clinical event for how and when a Person dies. A person can have up to one record if the source system contains evidence about the Death, such as: Condition in an administrative claim, status of enrollment into a health plan, or explicit record in EHR data. | 1 row = 1 death | yes       |   |
|              |                                                                                                                                                                                                                                                                                                 |                 |           |   |

| Name                    | Description                                                                                                                                                                                                                                           | Datatype | Mandatory |   |
|-------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| person_id               |                                                                                                                                                                                                                                                       | integer  | Yes       |   |
| death_date              | The date the person was deceased.                                                                                                                                                                                                                     | dateonly     | Yes       |   |
| death_datetime          |                                                                                                                                                                                                                                                       | datetimenotz     | No       |   |
| death_type_concept_id   | This is the provenance of the death record, i.e., where it came from. It is possible that an administrative claims database would source death information from a government file so do not assume the Death Type is the same as the Visit Type, etc. | integer  | No       |   |
| cause_concept_id        | This is the Standard Concept representing the Person’s cause of death, if available.                                                                                                                                                                  | integer  | No       |   |
| cause_source_value      |                                                                                                                                                                                                                                                       | string   | No       |   |
| cause_source_concept_id |                                                                                                                                                                                                                                                       | integer  | No       |   |

### Note

| Dataset name | Description                                                                                                                                                                                                                                     | Granularity    | Mandatory |   |
|--------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------|-----------|---|
| note         | The NOTE table captures unstructured information that was recorded by a provider about a patient in free text (in ASCII, or preferably in UTF8 format) notes on a given date. The type of note_text is CLOB or varchar(MAX) depending on RDBMS. | 1 row = 1 note | no        |   |
|              |                                                                                                                                                                                                                                                 |                |           |   |

| Name                  | Description                                                                                           | Datatype | Mandatory |   |
|-----------------------|-------------------------------------------------------------------------------------------------------|----------|-----------|---|
| note_id               | A unique identifier for each note.                                                                    | integer  | Yes       |   |
| person_id             |                                                                                                       | integer  | Yes       |   |
| note_date             | The date the note was recorded.                                                                       | dateonly     | Yes       |   |
| note_datetime         |                                                                                                       | datetimenotz     | No       |   |
| note_type_concept_id  | The provenance of the note. Most likely this will be EHR.                                             | integer  | Yes       |   |
| note_class_concept_id | A Standard Concept Id representing the HL7 LOINC Document Type Vocabulary classification of the note. | integer  | Yes       |   |
| note_title            | The title of the note.                                                                                | string   | No       |   |
| note_text             | The content of the note.                                                                              | string   | Yes       |   |
| encoding_concept_id   | This is the Concept representing the character encoding type.                                         | integer  | Yes       |   |
| language_concept_id   | The language of the note.                                                                             | integer  | Yes       |   |
| provider_id           | The Provider who wrote the note.                                                                      | integer  | No       |   |
| visit_occurrence_id   | The Visit during which the note was written.                                                          | integer  | No       |   |
| visit_detail_id       | The Visit Detail during which the note was written.                                                   | integer  | No       |   |
| note_source_value     |                                                                                                       | string   | No       |   |


### Note NLP

| Dataset name | Description                                                                                                              | Granularity    | Mandatory |   |
|--------------|--------------------------------------------------------------------------------------------------------------------------|----------------|-----------|---|
| note_nlp     | The NOTE_NLP table encodes all output of NLP on clinical notes. Each row represents a single extracted term from a note. | 1 row = 1 note | no        |   |
|              |                                                                                                                          |                |           |   |

| Name                       | Description                                                              | Datatype | Mandatory |   |
|----------------------------|--------------------------------------------------------------------------|----------|-----------|---|
| note_nlp_id                | A unique identifier for the NLP record.                                  | integer  | Yes       |   |
| note_id                    | This is the NOTE_ID for the NOTE record the NLP record is associated to. | integer  | Yes       |   |
| section_concept_id         |                                                                          | integer  | No       |   |
| snippet                    | A small window of text surrounding the term                              | string   | No       |   |
| “offset”                   | Character offset of the extracted term in the input note                 | string   | No       |   |
| lexical_variant            | Raw text extracted from the NLP tool.                                    | string   | Yes       |   |
| note_nlp_concept_id        |                                                                          | integer  | No       |   |
| note_nlp_source_concept_id |                                                                          | integer  | No       |   |
| nlp_system                 |                                                                          | string   | No       |   |
| nlp_date                   | The date of the note processing.                                         | dateonly     | Yes       |   |
| nlp_datetime               | The date and time of the note processing.                                | datetimenotz     | No       |   |
| term_exists                |                                                                          | string   | No       |   |
| term_temporal              |                                                                          | string   | No       |   |
| term_modifiers             |                                                                          | string   | No       |   |


### Specimen

| Dataset name | Description                                                                            | Granularity         | Mandatory |   |
|--------------|----------------------------------------------------------------------------------------|---------------------|-----------|---|
| specimen     | The specimen domain contains the records identifying biological samples from a person. | 1 row = 1 speciment | no        |   |
|              |                                                                                        |                     |           |   |

| Name                        | Description                                                     | Datatype | Mandatory |   |
|-----------------------------|-----------------------------------------------------------------|----------|-----------|---|
| specimen_id                 | Unique identifier for each specimen.                            | integer  | Yes       |   |
| person_id                   | The person from whom the specimen is collected.                 | integer  | Yes       |   |
| specimen_concept_id         |                                                                 | integer  | Yes       |   |
| specimen_type_concept_id    |                                                                 | integer  | Yes       |   |
| specimen_date               | The date the specimen was collected.                            | dateonly     | Yes       |   |
| specimen_datetime           |                                                                 | datetimenotz     | No       |   |
| quantity                    | The amount of specimen collected from the person.               | float    | No       |   |
| unit_concept_id             | The unit for the quantity of the specimen.                      | integer  | No       |   |
| anatomic_site_concept_id    | This is the site on the body where the specimen is from.        | integer  | No       |   |
| disease_status_concept_id   |                                                                 | integer  | No       |   |
| specimen_source_id          | This is the identifier for the specimen from the source system. | string   | No       |   |
| specimen_source_value       |                                                                 | string   | No       |   |
| unit_source_value           |                                                                 | string   | No       |   |
| anatomic_site_source_value  |                                                                 | string   | No       |   |
| disease_status_source_value |                                                                 | string   | No       |   |


### Fact relationship

| Dataset name      | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | Granularity                 | Mandatory |   |
|-------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-----------------------------|-----------|---|
| fact_relationship | The FACT_RELATIONSHIP table contains records about the relationships between facts stored as records in any table of the CDM. Relationships can be defined between facts from the same domain, or different domains. Examples of Fact Relationships include: Person relationships (parent-child), care site relationships (hierarchical organizational structure of facilities within a health system), indication relationship (between drug exposures and associated conditions), usage relationships (of devices during the course of an associated procedure), or facts derived from one another (measurements derived from an associated specimen). | 1 row = 1 fact relationship | no        |   |
|                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |                             |           |   |

| Name                    | Description | Datatype | Mandatory |   |
|-------------------------|-------------|----------|-----------|---|
| domain_concept_id_1     |             | integer  | Yes       |   |
| fact_id_1               |             | integer  | Yes       |   |
| domain_concept_id_2     |             | integer  | Yes       |   |
| fact_id_2               |             | integer  | Yes       |   |
| relationship_concept_id |             | integer  | Yes       |   |

<br>
## Standardized Health System
### Location

| Dataset name | Description                                                                                                                | Granularity        | Mandatory |   |
|--------------|----------------------------------------------------------------------------------------------------------------------------|--------------------|-----------|---|
| location     | The LOCATION table represents a generic way to capture physical location or address information of Persons and Care Sites. | 1 row = 1 location | yes       |   |
|              |                                                                                                                            |                    |           |   |

| Name                  | Description                                | Datatype | Mandatory |   |
|-----------------------|--------------------------------------------|----------|-----------|---|
| location_id           | The unique key given to a unique Location. | integer  | Yes       |   |
| address_1             | This is the first line of the address.     | string   | No       |   |
| address_2             | This is the second line of the address     | string   | No       |   |
| city                  |                                            | string   | No       |   |
| state                 |                                            | string   | Yes       |   |
| zip                   |                                            | string   | No       |   |
| county                |                                            | string   | Yes       |   |
| location_source_value |                                            | string   | No       |   |


### Care site

| Dataset name | Description                                                                                                                                                                                    | Granularity    | Mandatory |   |
|--------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------|-----------|---|
| care_site    | The CARE_SITE table contains a list of uniquely identified institutional (physical or organizational) units where healthcare delivery is practiced (offices, wards, hospitals, clinics, etc.). | 1 row = 1 site | no        |   |
|              |                                                                                                                                                                                                |                |           |   |

| Name                          | Description                                                                                                                                                                                                       | Datatype | Mandatory |   |
|-------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| care_site_id                  |                                                                                                                                                                                                                   | integer  | Yes       |   |
| care_site_name                | The name of the care_site as it appears in the source data                                                                                                                                                        | string   | No       |   |
| place_of_service_concept_id   | This is a high-level way of characterizing a Care Site. Typically, however, Care Sites can provide care in multiple settings (inpatient, outpatient, etc.) and this granularity should be reflected in the visit. | integer  | No       |   |
| location_id                   | The location_id from the LOCATION table representing the physical location of the care_site.                                                                                                                      | integer  | No       |   |
| care_site_source_value        | The identifier of the care_site as it appears in the source data. This could be an identifier separate from the name of the care_site.                                                                            | string   | No       |   |
| place_of_service_source_value |                                                                                                                                                                                                                   | string   | No       |   |


### Provider

| Dataset name | Description                                                                                                                                                                                                                                 | Granularity        | Mandatory |   |
|--------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|--------------------|-----------|---|
| provider     | The PROVIDER table contains a list of uniquely identified healthcare providers; duplication is not allowed. These are individuals providing hands-on healthcare to patients, such as physicians, nurses, midwives, physical therapists etc. | 1 row = 1 provider | no        |   |
|              |                                                                                                                                                                                                                                             |                    |           |   |

| Name                        | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                | Datatype | Mandatory |   |
|-----------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| provider_id                 | It is assumed that every provider with a different unique identifier is in fact a different person and should be treated independently.                                                                                                                                                                                                                                                                                                                                    | integer  | Yes       |   |
| provider_name               | This field contains information that describes a healthcare provider.                                                                                                                                                                                                                                                                                                                                                                                                      | string   | No       |   |
| npi                         | This is the National Provider Number issued to health care providers in the US by the Centers for Medicare and Medicaid Services (CMS).                                                                                                                                                                                                                                                                                                                                    | string   | No       |   |
| dea                         | This is the identifier issued by the DEA, a US federal agency, that allows a provider to write prescriptions for controlled substances.                                                                                                                                                                                                                                                                                                                                    | string   | No       |   |
| specialty_concept_id        | This field either represents the most common specialty that occurs in the data or the most specific concept that represents all specialties listed, should the provider have more than one. This includes physician specialties such as internal medicine, emergency medicine, etc. and allied health professionals such as nurses, midwives, and pharmacists.                                                                                                             | integer  | No       |   |
| care_site_id                | This is the CARE_SITE_ID for the location that the provider primarily practices in.                                                                                                                                                                                                                                                                                                                                                                                        | integer  | No       |   |
| year_of_birth               |                                                                                                                                                                                                                                                                                                                                                                                                                                                                            | integer  | No       |   |
| gender_concept_id           | This field represents the recorded gender of the provider in the source data.                                                                                                                                                                                                                                                                                                                                                                                              | integer  | No       |   |
| provider_source_value       | Use this field to link back to providers in the source data. This is typically used for error checking of ETL logic.                                                                                                                                                                                                                                                                                                                                                       | string   | No       |   |
| specialty_source_value      | This refers to the specific type of healthcare provider or field of expertise listed in the source data, encompassing physician specialties like internal medicine, emergency medicine, etc., as well as allied health professionals such as nurses, midwives, and pharmacists. It covers medical specialties like surgery, internal medicine, and radiology, while other services like prosthetics, acupuncture, and physical therapy fall under the domain of “Service.” | string   | No       |   |
| specialty_source_concept_id | This is often zero as many sites use proprietary codes to store physician speciality.                                                                                                                                                                                                                                                                                                                                                                                      | integer  | No       |   |
| gender_source_value         | This is provider’s gender as it appears in the source data.                                                                                                                                                                                                                                                                                                                                                                                                                | string   | No       |   |
| gender_source_concept_id    | This is often zero as many sites use proprietary codes to store provider gender.                                                                                                                                                                                                                                                                                                                                                                                           | integer  | No       |   |

<br>
## Standardized Health Ecomonics
### Payer plan period

| Dataset name      | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | Granularity          | Mandatory |   |
|-------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------|-----------|---|
| payer_plan_period | The PAYER_PLAN_PERIOD table captures details of the period of time that a Person is continuously enrolled under a specific health Plan benefit structure from a given Payer. Each Person receiving healthcare is typically covered by a health benefit plan, which pays for (fully or partially), or directly provides, the care. These benefit plans are provided by payers, such as health insurances or state or government agencies. In each plan the details of the health benefits are defined for the Person or her family, and the health benefit Plan might change over time typically with increasing utilization (reaching certain cost thresholds such as deductibles), plan availability and purchasing choices of the Person. The unique combinations of Payer organizations, health benefit Plans and time periods in which they are valid for a Person are recorded in this table. | 1 row = 1 payer plan | no        |   |
|                   |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                      |           |   |

| Name                          | Description                                                                                                                                           | Datatype | Mandatory |   |
|-------------------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| payer_plan_period_id          | A unique identifier for each unique combination of a Person, Payer, Plan, and Period of time.                                                         | integer  | Yes       |   |
| person_id                     | The Person covered by the Plan.                                                                                                                       | integer  | Yes       |   |
| payer_plan_period_start_date  | Start date of Plan coverage.                                                                                                                          | dateonly     | Yes       |   |
| payer_plan_period_end_date    | End date of Plan coverage.                                                                                                                            | dateonly     | Yes       |   |
| payer_concept_id              | This field represents the organization who reimburses the provider which administers care to the Person.                                              | integer  | No       |   |
| payer_source_value            | This is the Payer as it appears in the source data.                                                                                                   | string   | No       |   |
| payer_source_concept_id       |                                                                                                                                                       | integer  | No       |   |
| plan_concept_id               | This field represents the specific health benefit Plan the Person is enrolled in.                                                                     | integer  | No       |   |
| plan_source_value             | This is the health benefit Plan of the Person as it appears in the source data.                                                                       | string   | No       |   |
| plan_source_concept_id        |                                                                                                                                                       | integer  | No       |   |
| sponsor_concept_id            | This field represents the sponsor of the Plan who finances the Plan. This includes self-insured, small group health plan and large group health plan. | integer  | No       |   |
| sponsor_source_value          | The Plan sponsor as it appears in the source data.                                                                                                    | string   | No       |   |
| sponsor_source_concept_id     |                                                                                                                                                       | integer  | No       |   |
| family_source_value           | The common identifier for all people (often a family) that covered by the same policy.                                                                | string   | No       |   |
| stop_reason_concept_id        | This field represents the reason the Person left the Plan, if known.                                                                                  | integer  | No       |   |
| stop_reason_source_value      | The Plan stop reason as it appears in the source data.                                                                                                | string   | No       |   |
| stop_reason_source_concept_id |                                                                                                                                                       | integer  | No       |   |


### Cost

| Dataset name | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         | Granularity    | Mandatory |   |
|--------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------|-----------|---|
| cost         | The COST table captures records containing the cost of any medical event recorded in one of the OMOP clinical event tables such as DRUG_EXPOSURE, PROCEDURE_OCCURRENCE, VISIT_OCCURRENCE, VISIT_DETAIL, DEVICE_OCCURRENCE, OBSERVATION or MEASUREMENT.  Each record in the cost table account for the amount of money transacted for the clinical event. So, the COST table may be used to represent both receivables (charges) and payments (paid), each transaction type represented by its COST_CONCEPT_ID. The COST_TYPE_CONCEPT_ID field will use concepts in the Standardized Vocabularies to designate the source (provenance) of the cost data. A reference to the health plan information in the PAYER_PLAN_PERIOD table is stored in the record for information used for the adjudication system to determine the persons benefit for the clinical event. | 1 row = 1 cost | no        |   |
|              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     |                |           |   |

| Name                      | Description                                                                                                 | Datatype | Mandatory |   |
|---------------------------|-------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| cost_id                   |                                                                                                             | integer  | Yes       |   |
| cost_event_id             |                                                                                                             | integer  | Yes       |   |
| cost_domain_id            |                                                                                                             | string   | Yes       |   |
| cost_type_concept_id      |                                                                                                             | integer  | Yes       |   |
| currency_concept_id       |                                                                                                             | integer  | No       |   |
| total_charge              |                                                                                                             | float    | No       |   |
| total_cost                |                                                                                                             | float    | No       |   |
| total_paid                |                                                                                                             | float    | No       |   |
| paid_by_payer             |                                                                                                             | float    | No       |   |
| paid_by_patient           |                                                                                                             | float    | No       |   |
| paid_patient_copay        |                                                                                                             | float    | No       |   |
| paid_patient_coinsurance  |                                                                                                             | float    | No       |   |
| paid_patient_deductible   |                                                                                                             | float    | No       |   |
| paid_by_primary           |                                                                                                             | float    | No       |   |
| paid_ingredient_cost      |                                                                                                             | float    | No       |   |
| paid_dispensing_fee       |                                                                                                             | float    | No       |   |
| payer_plan_period_id      |                                                                                                             | integer  | No       |   |
| amount_allowed            |                                                                                                             | float    | No       |   |
| revenue_code_concept_id   |                                                                                                             | integer  | No       |   |
| revenue_code_source_value | Revenue codes are a method to charge for a class of procedures and conditions in the U.S. hospital system.  | string   | No       |   |
| drg_concept_id            |                                                                                                             | integer  | No       |   |
| drg_source_value          | Diagnosis Related Groups are US codes used to classify hospital cases into one of approximately 500 groups. | string   | No       |   |

<br>
## Standardized Derived Elements
### Drug era

| Dataset name | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                              | Granularity        | Mandatory |   |
|--------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|--------------------|-----------|---|
| drug_era     | A Drug Era is defined as a span of time when the Person is assumed to be exposed to a particular active ingredient. A Drug Era is not the same as a Drug Exposure: Exposures are individual records corresponding to the source when Drug was delivered to the Person, while successive periods of Drug Exposures are combined under certain rules to produce continuous Drug Eras. Every record in the DRUG_EXPOSURE table should be part of a drug era based on the dates of exposure. | 1 row = 1 drug era | no        |   |
|              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          |                    |           |   |

| Name                | Description                                                                                                                                                       | Datatype | Mandatory |   |
|---------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| drug_era_id         |                                                                                                                                                                   | integer  | Yes       |   |
| person_id           |                                                                                                                                                                   | integer  | Yes       |   |
| drug_concept_id     | The drug_concept_id should conform to the concept class ‘ingredient’ as the drug_era is an era of time where a person is exposed to a particular drug ingredient. | integer  | Yes       |   |
| drug_era_start_date |                                                                                                                                                                   | dateonly     | Yes       |   |
| drug_era_end_date   |                                                                                                                                                                   | dateonly     | Yes       |   |
| drug_exposure_count | The count of grouped DRUG_EXPOSURE records that were included in the DRUG_ERA row                                                                                 | integer  | No       |   |
| gap_days            |                                                                                                                                                                   | integer  | No       |   |


### Dose era

| Dataset name | Description                                                                                                                          | Granularity        | Mandatory |   |
|--------------|--------------------------------------------------------------------------------------------------------------------------------------|--------------------|-----------|---|
| dose_era     | A Dose Era is defined as a span of time when the Person is assumed to be exposed to a constant dose of a specific active ingredient. | 1 row = 1 dose era | no        |   |
|              |                                                                                                                                      |                    |           |   |

| Name                | Description                                                                                         | Datatype | Mandatory |   |
|---------------------|-----------------------------------------------------------------------------------------------------|----------|-----------|---|
| dose_era_id         |                                                                                                     | integer  | Yes       |   |
| person_id           |                                                                                                     | integer  | Yes       |   |
| drug_concept_id     | The Concept Id representing the specific drug ingredient.                                           | integer  | Yes       |   |
| unit_concept_id     | The Concept Id representing the unit of the specific drug ingredient.                               | integer  | Yes       |   |
| dose_value          | The numeric value of the dosage of the drug_ingredient.                                             | float    | Yes       |   |
| dose_era_start_date | The date the Person started on the specific dosage, with at least 31 days since any prior exposure. | dateonly     | Yes       |   |
| dose_era_end_date   |                                                                                                     | dateonly     | Yes       |   |


### Condition era

| Dataset name  | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | Granularity             | Mandatory |   |
|---------------|---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------------|-----------|---|
| condition_era | A Condition Era is defined as a span of time when the Person is assumed to have a given condition. Similar to Drug Eras, Condition Eras are chronological periods of Condition Occurrence and every Condition Occurrence record should be part of a Condition Era. Combining individual Condition Occurrences into a single Condition Era serves two purposes:  It allows aggregation of chronic conditions that require frequent ongoing care, instead of treating each Condition Occurrence as an independent event. It allows aggregation of multiple, closely timed doctor visits for the same Condition to avoid double-counting the Condition Occurrences. For example, consider a Person who visits her Primary Care Physician (PCP) and who is referred to a specialist. At a later time, the Person visits the specialist, who confirms the PCP’s original diagnosis and provides the appropriate treatment to resolve the condition. These two independent doctor visits should be aggregated into one Condition Era. | 1 row = 1 condition era | yes       |   |
|               |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |           |   |

| Name                       | Description                                                                                                                                                                                                                                                            | Datatype | Mandatory |   |
|----------------------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| condition_era_id           |                                                                                                                                                                                                                                                                        | integer  | Yes       |   |
| person_id                  |                                                                                                                                                                                                                                                                        | integer  | Yes       |   |
| condition_concept_id       | The Concept Id representing the Condition.                                                                                                                                                                                                                             | integer  | Yes       |   |
| condition_era_start_date   | The start date for the Condition Era constructed from the individual instances of Condition Occurrences. It is the start date of the very first chronologically recorded instance of the condition with at least 31 days since any prior record of the same Condition. | dateonly     | Yes       |   |
| condition_era_end_date     | The end date for the Condition Era constructed from the individual instances of Condition Occurrences. It is the end date of the final continuously recorded instance of the Condition.                                                                                | dateonly     | Yes       |   |
| condition_occurrence_count | The number of individual Condition Occurrences used to construct the condition era.                                                                                                                                                                                    | integer  | No       |   |

<br>
## Standardized Vocabulary
### Concept

| Dataset name | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                             | Granularity       | Mandatory |   |
|--------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------|-----------|---|
| concept      | The Standardized Vocabularies contains records, or Concepts, that uniquely identify each fundamental unit of meaning used to express clinical information in all domain tables of the CDM. Concepts are derived from vocabularies, which represent clinical information across a domain (e.g. conditions, drugs, procedures) through the use of codes and associated descriptions. Some Concepts are designated Standard Concepts, meaning these Concepts can be used as normative expressions of a clinical entity within the OMOP Common Data Model and standardized analytics. Each Standard Concept belongs to one Domain, which defines the location where the Concept would be expected to occur within the data tables of the CDM. Concepts can represent broad categories (‘Cardiovascular disease’), detailed clinical elements (‘Myocardial infarction of the anterolateral wall’), or modifying characteristics and attributes that define Concepts at various levels of detail (severity of a disease, associated morphology, etc.). Records in the Standardized Vocabularies tables are derived from national or international vocabularies such as SNOMED-CT, RxNorm, and LOINC, or custom OMOP Concepts defined to cover various aspects of observational data analysis. | 1 row = 1 concept | yes       |   |
|              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         |                   |           |   |

| Name             | Description                                                                                                                                                                                                                                                        | Datatype | Mandatory |   |
|------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| concept_id       | A unique identifier for each Concept across all domains.                                                                                                                                                                                                           | integer  | Yes       |   |
| concept_name     | An unambiguous, meaningful and descriptive name for the Concept.                                                                                                                                                                                                   | string   | Yes       |   |
| domain_id        | A foreign key to the DOMAIN table the Concept belongs to.                                                                                                                                                                                                          | string   | Yes       |   |
| vocabulary_id    | A foreign key to the VOCABULARY table indicating from which source the Concept has been adapted.                                                                                                                                                                   | string   | Yes       |   |
| concept_class_id | The attribute or concept class of the Concept. Examples are ‘Clinical Drug’, ‘Ingredient’, ‘Clinical Finding’ etc.                                                                                                                                                 | string   | Yes       |   |
| standard_concept | This flag determines where a Concept is a Standard Concept, i.e. is used in the data, a Classification Concept, or a non-standard Source Concept. The allowable values are ‘S’ (Standard Concept) and ‘C’ (Classification Concept), otherwise the content is NULL. | string   | No       |   |
| concept_code     | The concept code represents the identifier of the Concept in the source vocabulary, such as SNOMED-CT concept IDs, RxNorm RXCUIs etc. Note that concept codes are not unique across vocabularies.                                                                  | string   | Yes       |   |
| valid_start_date | The date when the Concept was first recorded. The default value is 1-Jan-1970, meaning, the Concept has no (known) date of inception.                                                                                                                              | dateonly     | Yes       |   |
| valid_end_date   | The date when the Concept became invalid because it was deleted or superseded (updated) by a new concept. The default value is 31-Dec-2099, meaning, the Concept is valid until it becomes deprecated.                                                             | dateonly     | Yes       |   |
| invalid_reason   | Reason the Concept was invalidated. Possible values are D (deleted), U (replaced with an update) or NULL when valid_end_date has the default value.                                                                                                                | string   | No       |   |


### Vocabulary

| Dataset name | Description                                                                                                                                                                                                                                                                    | Granularity          | Mandatory |   |
|--------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------|-----------|---|
| vocabulary   | The VOCABULARY table includes a list of the Vocabularies integrated from various sources or created de novo in OMOP CDM. This reference table contains a single record for each Vocabulary and includes a descriptive name and other associated attributes for the Vocabulary. | 1 row = 1 vocabulary | no        |   |
|              |                                                                                                                                                                                                                                                                                |                      |           |   |

| Name                  | Description                                                                                                                                                  | Datatype | Mandatory |   |
|-----------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| vocabulary_id         | A unique identifier for each Vocabulary, such as ICD9CM, SNOMED, Visit.                                                                                      | string   | Yes       |   |
| vocabulary_name       | The name describing the vocabulary, for example, International Classification of Diseases, Ninth Revision, Clinical Modification, Volume 1 and 2 (NCHS) etc. | string   | Yes       |   |
| vocabulary_reference  | External reference to documentation or available download of the about the vocabulary.                                                                       | string   | Yes       |   |
| vocabulary_version    | Version of the Vocabulary as indicated in the source.                                                                                                        | string   | No       |   |
| vocabulary_concept_id | A Concept that represents the Vocabulary the VOCABULARY record belongs to.                                                                                   | integer  | Yes       |   |


### Domain

| Dataset name | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | Granularity      | Mandatory |   |
|--------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------|-----------|---|
| domain       | The DOMAIN table includes a list of OMOP-defined Domains to which the Concepts of the Standardized Vocabularies can belong. A Domain represents a clinical definition whereby we assign matching Concepts for the standardized fields in the CDM tables. For example, the Condition Domain contains Concepts that describe a patient condition, and these Concepts can only be used in the condition_concept_id field of the CONDITION_OCCURRENCE and CONDITION_ERA tables. This reference table is populated with a single record for each Domain, including a Domain ID and a descriptive name for every Domain. | 1 row = 1 domain | no        |   |
|              |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                  |           |   |

| Name              | Description                                                                 | Datatype | Mandatory |   |
|-------------------|-----------------------------------------------------------------------------|----------|-----------|---|
| domain_id         | A unique key for each domain.                                               | string   | Yes       |   |
| domain_name       | The name describing the Domain, e.g. Condition, Procedure, Measurement etc. | string   | Yes       |   |
| domain_concept_id | A Concept representing the Domain Concept the DOMAIN record belongs to.     | integer  | Yes       |   |


### Concept class

| Dataset name  | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                     | Granularity             | Mandatory |   |
|---------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------------|-----------|---|
| concept_class | The CONCEPT_CLASS table includes semantic categories that reference the source structure of each Vocabulary. Concept Classes represent so-called horizontal (e.g. MedDRA, RxNorm) or vertical levels (e.g. SNOMED) of the vocabulary structure. Vocabularies without any Concept Classes, such as HCPCS, use the vocabulary_id as the Concept Class. This reference table is populated with a single record for each Concept Class, which includes a Concept Class ID and a fully specified Concept Class name. | 1 row = 1 concept class | no        |   |
|               |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 |                         |           |   |

| Name                     | Description                                                                    | Datatype | Mandatory |   |
|--------------------------|--------------------------------------------------------------------------------|----------|-----------|---|
| concept_class_id         | A unique key for each class.                                                   | string   | Yes       |   |
| concept_class_name       | The name describing the Concept Class, e.g. Clinical Finding, Ingredient, etc. | string   | Yes       |   |
| concept_class_concept_id | A Concept that represents the Concept Class.                                   | integer  | Yes       |   |


### Concept relationship

| Dataset name         | Description                                                                                                                                                                                                                                                                                                                                                                                                       | Granularity                    | Mandatory |   |
|----------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|--------------------------------|-----------|---|
| concept_relationship | The CONCEPT_RELATIONSHIP table contains records that define relationships between any two Concepts and the nature or type of the relationship. This table captures various types of relationships, including hierarchical, associative, and other semantic connections, enabling comprehensive analysis and interpretation of clinical concepts. Every kind of relationship is defined in the RELATIONSHIP table. | 1 row = 1 concept relationship | no        |   |
|                      |                                                                                                                                                                                                                                                                                                                                                                                                                   |                                |           |   |

| Name             | Description                                                                                                          | Datatype | Mandatory |   |
|------------------|----------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| concept_id_1     |                                                                                                                      | integer  | Yes       |   |
| concept_id_2     |                                                                                                                      | integer  | Yes       |   |
| relationship_id  | The relationship between CONCEPT_ID_1 and CONCEPT_ID_2. Please see the Vocabulary Conventions. for more information. | string   | Yes       |   |
| valid_start_date | The date when the relationship is first recorded.                                                                    | date     | Yes       |   |
| valid_end_date   | The date when the relationship is invalidated.                                                                       | date     | Yes       |   |
| invalid_reason   | Reason the relationship was invalidated. Possible values are ‘D’ (deleted), ‘U’ (updated) or NULL.                   | string   | No       |   |


### Relationship

| Dataset name | Description                                                                                                                                                     | Granularity            | Mandatory |   |
|--------------|-----------------------------------------------------------------------------------------------------------------------------------------------------------------|------------------------|-----------|---|
| relationship | The RELATIONSHIP table provides a reference list of all types of relationships that can be used to associate any two concepts in the CONCEPT_RELATIONSHP table. | 1 row = 1 relationship | no        |   |
|              |                                                                                                                                                                 |                        |           |   |

| Name                    | Description                                                                                                                                                          | Datatype | Mandatory |   |
|-------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| relationship_id         | The type of relationship captured by the relationship record.                                                                                                        | string   | Yes       |   |
| relationship_name       |                                                                                                                                                                      | string   | Yes       |   |
| is_hierarchical         | Defines whether a relationship defines concepts into classes or hierarchies. Values are 1 for hierarchical relationship or 0 if not.                                 | string   | Yes       |   |
| defines_ancestry        | Defines whether a hierarchical relationship contributes to the concept_ancestor table. These are subsets of the hierarchical relationships. Valid values are 1 or 0. | string   | Yes       |   |
| reverse_relationship_id | The identifier for the relationship used to define the reverse relationship between two concepts.                                                                    | string   | Yes       |   |
| relationship_concept_id | A foreign key that refers to an identifier in the CONCEPT table for the unique relationship concept.                                                                 | integer  | Yes       |   |


### Concept synonym

| Dataset name    | Description                                                                               | Granularity               | Mandatory |   |
|-----------------|-------------------------------------------------------------------------------------------|---------------------------|-----------|---|
| concept_synonym | The CONCEPT_SYNONYM table is used to store alternate names and descriptions for Concepts. | 1 row = 1 concept synonym | no        |   |
|                 |                                                                                           |                           |           |   |

| Name                 | Description | Datatype | Mandatory |   |
|----------------------|-------------|----------|-----------|---|
| concept_id           |             | integer  | Yes       |   |
| concept_synonym_name |             | string   | Yes       |   |
| language_concept_id  |             | integer  | Yes       |   |


### Concept Ancestor

| Dataset name     | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        | Granularity                | Mandatory |   |
|------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------------------------|-----------|---|
| concept_ancestor | The CONCEPT_ANCESTOR table is designed to simplify observational analysis by providing the complete hierarchical relationships between Concepts. Only direct parent-child relationships between Concepts are stored in the CONCEPT_RELATIONSHIP table. To determine higher level ancestry connections, all individual direct relationships would have to be navigated at analysis time. The CONCEPT_ANCESTOR table includes records for all parent-child relationships, as well as grandparent-grandchild relationships and those of any other level of lineage. Using the CONCEPT_ANCESTOR table allows for querying for all descendants of a hierarchical concept. For example, drug ingredients and drug products are all descendants of a drug class ancestor.  This table is entirely derived from the CONCEPT, CONCEPT_RELATIONSHIP and RELATIONSHIP tables. | 1 row = 1 concept ancestor | yes       |   |
|                  |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                    |                            |           |   |

| Name                     | Description                                                                                                                                                          | Datatype | Mandatory |   |
|--------------------------|----------------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| ancestor_concept_id      | The Concept Id for the higher-level concept that forms the ancestor in the relationship.                                                                             | integer  | Yes       |   |
| descendant_concept_id    | The Concept Id for the lower-level concept that forms the descendant in the relationship.                                                                            | integer  | Yes       |   |
| min_levels_of_separation | The minimum separation in number of levels of hierarchy between ancestor and descendant concepts. This is an attribute that is used to simplify hierarchic analysis. | integer  | Yes       |   |
| max_levels_of_separation | The maximum separation in number of levels of hierarchy between ancestor and descendant concepts. This is an attribute that is used to simplify hierarchic analysis. | integer  | Yes       |   |


### Source to concept map

| Dataset name          | Description                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               | Granularity                 | Mandatory |   |
|-----------------------|-------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-----------------------------|-----------|---|
| source_to_concept_map | The source to concept map table is a legacy data structure within the OMOP Common Data Model, recommended for use in ETL processes to maintain local source codes which are not available as Concepts in the Standardized Vocabularies, and to establish mappings for each source code into a Standard Concept as target_concept_ids that can be used to populate the Common Data Model tables. The SOURCE_TO_CONCEPT_MAP table is no longer populated with content within the Standardized Vocabularies published to the OMOP community. | 1 row = 1 source to concept | no        |   |
|                       |                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                           |                             |           |   |

| Name                    | Description                                                                                                                                                  | Datatype | Mandatory |   |
|-------------------------|--------------------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| source_code             | The source code being translated into a Standard Concept.                                                                                                    | string   | Yes       |   |
| source_concept_id       | A foreign key to the Source Concept that is being translated into a Standard Concept.                                                                        | integer  | Yes       |   |
| source_vocabulary_id    | A foreign key to the VOCABULARY table defining the vocabulary of the source code that is being translated to a Standard Concept.                             | string   | Yes       |   |
| source_code_description | An optional description for the source code. This is included as a convenience to compare the description of the source code to the name of the concept.     | string   | No       |   |
| target_concept_id       | The target Concept to which the source code is being mapped.                                                                                                 | integer  | Yes       |   |
| target_vocabulary_id    | The Vocabulary of the target Concept.                                                                                                                        | string   | Yes       |   |
| valid_start_date        | The date when the mapping instance was first recorded.                                                                                                       | date     | Yes       |   |
| valid_end_date          | The date when the mapping instance became invalid because it was deleted or superseded (updated) by a new relationship. Default value is 31-Dec-2099.        | date     | Yes       |   |
| invalid_reason          | Reason the mapping instance was invalidated. Possible values are D (deleted), U (replaced with an update) or NULL when valid_end_date has the default value. | string   | No       |   |


### Drug strength

| Dataset name  | Description                                                                                                                                                                                                                                                                  | Granularity             | Mandatory |   |
|---------------|------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------|-------------------------|-----------|---|
| drug_strength | The DRUG_STRENGTH table contains structured content about the amount or concentration and associated units of a specific ingredient contained within a particular drug product. This table is supplemental information to support standardized analysis of drug utilization. | 1 row = 1 drug strength | no        |   |
|               |                                                                                                                                                                                                                                                                              |                         |           |   |

| Name                        | Description                                                                                                                                         | Datatype | Mandatory |   |
|-----------------------------|-----------------------------------------------------------------------------------------------------------------------------------------------------|----------|-----------|---|
| drug_concept_id             | The Concept representing the Branded Drug or Clinical Drug Product.                                                                                 | integer  | Yes       |   |
| ingredient_concept_id       | The Concept representing the active ingredient contained within the drug product.                                                                   | integer  | Yes       |   |
| amount_value                | The numeric value or the amount of active ingredient contained within the drug product.                                                             | float    | No       |   |
| amount_unit_concept_id      | The Concept representing the Unit of measure for the amount of active ingredient contained within the drug product.                                 | integer  | No       |   |
| numerator_value             | The concentration of the active ingredient contained within the drug product.                                                                       | float    | No       |   |
| numerator_unit_concept_id   | The Concept representing the Unit of measure for the concentration of active ingredient.                                                            | integer  | No       |   |
| denominator_value           | The amount of total liquid (or other divisible product, such as ointment, gel, spray, etc.).                                                        | float    | No       |   |
| denominator_unit_concept_id | The Concept representing the denominator unit for the concentration of active ingredient.                                                           | integer  | No       |   |
| box_size                    | The number of units of Clinical Branded Drug or Quantified Clinical or Branded Drug contained in a box as dispensed to the patient.                 | integer  | No       |   |
| valid_start_date            | The date when the Concept was first recorded. The default value is 1-Jan-1970.                                                                      | dateonly     | Yes       |   |
| valid_end_date              | The date when then Concept became invalid.                                                                                                          | dateonly     | Yes       |   |
| invalid_reason              | Reason the concept was invalidated. Possible values are D (deleted), U (replaced with an update) or NULL when valid_end_date has the default value. | string   | No       |   |

