import { useSelection } from "./use-selection";
import { ref, computed, watch, onMounted } from "vue";
import { API } from "../Api";
import Plotly from "plotly.js-dist-min";
import { exportFile, Notify, Loading } from "quasar";

export function useData() {
  const {
    selected,
    diseases,
    diseasePrevalence,
    stateOptions,
    countiesOptions,
    stateCounties,
    tableTitle,
    updateState,
    updateDisease,
    updatePrevalence,
    updateCounty,
  } = useSelection();

  const choropleth = ref(null);
  const scatter = ref(null);

  const selectedTheme = ref(null);

  const themes = ref([]);

  const prelevanceBoundries = ref({
    min: 0,
    max: 0,
  });

  const selectedPoints = ref({
    type: "county",
    ids: [],
  });

  const plotWidth = ref(600);

  const data = ref({
    counties_data: {},
    tracts_data: {},
    geometries: {},
  });

  const tableColumns = computed(() => {
    if (themes.value && themes.value.length > 0) {
      return [
        {
          name: "state_name",
          label: "State",
          align: "left",
          field: "state_name",
          sortable: true,
        },
        {
          name: "county_name",
          label: "County",
          align: "left",
          field: "county_name",
          sortable: true,
        },
        { name: "tract_code", label: "FIPS", field: "tract_code" },
        {
          name: "population",
          label: "Population",
          field: "population",
          sortable: true,
        },
        {
          name: `disease_percent`,
          label: `${selected.value.disease.label} Prevalence`,
          field: "disease_percent",
          sortable: true,
        },
        {
          name: "cluster_labels",
          label: "Social Vulnerability Cluster",
          field: "cluster_labels",
        },
        // { name : `${selected.value.disease.value}_SHAP_value`, label: `Top SV Importance Factor for ${selected.value.disease.label} Prevalence Prediction`, field: `${selected.value.disease.value}_SHAP_value`},
      ].concat(
        [...themes.value].map((theme, index) => {
          return {
            name: theme,
            label: theme,
            field: theme,
            sortable: true,
          };
        })
      );
    }
    return [];
  });

  function filterTractsData(tracts_data, selectedPoints_) {
    if (selectedPoints_.ids.length > 0) {
      const mask =
        selectedPoints_.type === "tract"
          ? tracts_data.FIPS.map((item) => {
              return selectedPoints_.ids.includes(item);
            })
          : tracts_data.FIPS.map((item, index) => {
              return selectedPoints_.ids.includes(
                tracts_data.State_code[index] + tracts_data.County_code[index]
              );
            });
      return Object.fromEntries(
        Object.entries(tracts_data).map((item) => {
          return [item[0], [...item[1]].filter((_, i) => mask[i])];
        })
      );
    }
    return tracts_data;
  }

  const tableData = computed(() => {
    // Add filtering on selected points before populating rows
    const initialPagination = {
      sortBy: "disease_percent",
      descending: true,
      page: 1,
      rowsPerPage: 50,
    };

    if (
      data.value.tracts_data &&
      data.value.tracts_data.State_code &&
      themes.value &&
      themes.value.length > 0
    ) {
      const dataFilteredPrevalence = filterFromPrevalence(
        diseasePrevalence.value,
        data.value
      );

      const dataFiltered = filterTractsData(
        dataFilteredPrevalence.tracts_data,
        selectedPoints.value
      );
      // const shapColum = `${selected.value.disease.value}_SHAP_value`;
      const rows = [...dataFiltered.State_code].map((state, index) => {
        const countyName = stateCounties.value[state]
          ? stateCounties.value[state].counties[dataFiltered.County_code[index]]
            ? stateCounties.value[state].counties[
                dataFiltered.County_code[index]
              ].county_name
            : ""
          : "";
        const result = {
          state_name: (stateCounties.value[state] || { state_name: "" })
            .state_name,
          county_name: countyName,
          tract_code: dataFiltered.FIPS[index],
          population: dataFiltered.population[index],
          disease_percent: dataFiltered.disease_percent[index],
          cluster_labels: dataFiltered.cluster_labels[index],
        };
        [...themes.value].forEach((element) => {
          result[element] = dataFiltered[element][index];
        });
        return result;
      });

      return {
        rows: rows,
        columns: tableColumns.value,
        rowKey: "tract_code",
        initialPagination: initialPagination,
      };
    }
    return {
      rows: [],
      columns: [],
      rowKey: "",
      initialPagination: initialPagination,
    };
  });

  function wrapCsvValue(val, formatFn, row) {
    let formatted = formatFn !== void 0 ? formatFn(val, row) : val;

    formatted =
      formatted === void 0 || formatted === null ? "" : String(formatted);

    formatted = formatted.split('"').join('""');
    /**
     * Excel accepts \n and \r in strings, but some other CSV parsers do not
     * Uncomment the next two lines to escape new lines
     */
    // .split('\n').join('\\n')
    // .split('\r').join('\\r')

    return `"${formatted}"`;
  }

  function tableToCsv(columns, rows) {
    const content = [columns.map((col) => wrapCsvValue(col.label))]
      .concat(
        rows.map((row) =>
          columns
            .map((col) =>
              wrapCsvValue(
                typeof col.field === "function"
                  ? col.field(row)
                  : row[col.field === void 0 ? col.name : col.field],
                col.format,
                row
              )
            )
            .join(",")
        )
      )
      .join("\r\n");

    const status = exportFile("table-export.csv", content, "text/csv");

    if (status !== true) {
      Notify.create({
        message: "Browser denied file download...",
        color: "negative",
        icon: "warning",
      });
    }
  }

  function exportTable() {
    tableToCsv(tableData.value.columns, tableData.value.rows);
  }

  function customGraphData(data_, stateCounties_) {
    if (data_.counties_data == null) {
      if (data_.tracts_data) {
        const states = data_.tracts_data.State_code;
        if (states) {
          const counties = data_.tracts_data.County_code;
          const tracts = data_.tracts_data.FIPS;
          const result = states.map((state, index) => {
            const countyName = stateCounties_[state]
              ? stateCounties_[state].counties[counties[index]]
                ? stateCounties_[state].counties[counties[index]].county_name
                : ""
              : "";
            return {
              state: {
                state_code: state,
                state_name: (stateCounties_[state] || { state_name: "" })
                  .state_name,
              },
              county: { county_code: counties[index], county_name: countyName },
              tract: { tract_code: tracts[index] },
            };
          });
          return result;
        }
      }
    } else {
      if (data_.counties_data) {
        const states = data_.counties_data.State_code;
        if (states) {
          const counties = data_.counties_data.County_code;
          const result = states.map((state, index) => {
            const countyName = stateCounties_[state]
              ? stateCounties_[state].counties[counties[index]]
                ? stateCounties_[state].counties[counties[index]].county_name
                : ""
              : "";
            return {
              state: {
                state_code: state,
                state_name: (stateCounties_[state] || { state_name: "" })
                  .state_name,
              },
              county: { county_code: counties[index], county_name: countyName },
            };
          });
          return result;
        }
      }
    }
    return [];
  }

  function linspace(start, stop, n) {
    const step = (stop - start) / Math.max(n - 1, 1);
    return Array.from({ length: n }, (_, i) => start + i * step);
  }

  function generateColorScale(colors, naColor) {
    const colorArray = [];
    colorArray.push([0, naColor]);
    const spacedArr = linspace(0.01, 1, colors.length);
    spacedArr.forEach((val, index) => {
      colorArray.push([val, colors[index]]);
    });
    return colorArray;
  }

  const colorScale = generateColorScale(
    [
      "rgb(255,245,240)",
      "rgb(254,224,210)",
      "rgb(252,187,161)",
      "rgb(252,146,114)",
      "rgb(251,106,74)",
      "rgb(239,59,44)",
      "rgb(203,24,29)",
      "rgb(165,15,21)",
      // "rgb(103,0,13)",
    ],
    "rgb(211,211,211)"
  );

  function get_choropleth_data(data_, selected_, stateCounties_, plotWidth_) {
    let hoverTitle =
      data_.counties_data == null ? "Tracts view" : "Counties view";
    let tractHoverValue =
      data_.counties_data == null
        ? "<b>FIPS</b>: %{customdata.tract.tract_code}<br>"
        : "";
    const geo_all = {
      scope: "usa",
    };
    const geo_fips = {
      visible: false,
      fitbounds: "locations",
      showsubunits: true,
      subunitscolor: "black",
    };

    const geo_state = {
      fitbounds: "locations",
      visible: false,
    };

    const geo =
      data_.counties_data == null
        ? geo_fips
        : selected_.state.value === "all"
        ? geo_all
        : geo_state;

    const colors =
      data_.counties_data == null
        ? data_.tracts_data.disease_percent
        : data_.counties_data.disease_percent;

    let zmin = colors ? Math.min(colors) : 0;
    let zmax = colors ? Math.max(colors) : 0;

    const locations =
      data_.counties_data == null
        ? data_.tracts_data.FIPS
        : data_.counties_data.FIPS;

    const customdata = customGraphData(data_, stateCounties_);

    // ADD Null values
    if (data_ && data_.geometries && data_.geometries.features) {
      data_.geometries.features.forEach((value) => {
        if (locations && value.id && !locations.includes(value.id)) {
          locations.push(value.id);
          colors.push(-1);
          if (customdata) {
            customdata.push({
              state: { state_name: "Data not found" },
              county: { county_name: "Data not found" },
              tract: { tract_code: "Data not found" },
            });
          }
        }
      });
    }

    const isEmpty = colors && colors.every((value) => value == -1);

    if (isEmpty) {
      zmin = 0;
      zmax = 1;
    }

    const layout = {
      width: plotWidth_,
      height: 520,
      autosize: true,
      autocolorscale: false,

      geo: geo,

      title: {
        text: `${selected_.disease.label} Prevalence Heatmap`,
        font: {
          family: "SourceSansPro",
          size: 14,
        },
        xref: "paper",
        x: 0,
      },
      margin: {
        r: 0,
      },
      hovermode: "closest",
    };

    const data = [
      {
        type: "choropleth",
        showscale: !isEmpty,
        colorscale: colorScale,
        zmin: zmin,
        zmax: zmax,
        geojson: data_.geometries,
        locations: locations,
        z: colors,
        colorbar: {},
        marker: {
          // to define
        },
        // hoverinfo: "none",
        // hovertemplate: null,
        customdata: customdata,
        hovertemplate:
          `<b>${hoverTitle}</b><br><br>` +
          "<b>Health Measure prevalence</b>: %{z}<br>" +
          "<b>State</b>: %{customdata.state.state_name}<br>" +
          "<b>County</b>: %{customdata.county.county_name}<br>" +
          tractHoverValue +
          "<extra></extra>",
      },
    ];

    // IF DATA IS NULL RETURN AN EMPTY MAP
    layout.geo = data[0].z && data[0].z.length > 0 ? layout.geo : geo_all;

    return { layout: layout, data: data };
  }

  function get_scatter_data(
    data_,
    theme,
    stateCounties_,
    selected_,
    plotWidth_
  ) {
    const viewType = data_.counties_data == null ? "Tract" : "County";
    const hoverTitle = `${viewType} view`;
    let tractHoverValue =
      data_.counties_data == null
        ? "<b>FIPS</b>: %{customdata.tract.tract_code}<br>"
        : "";
    const size =
      (data_.counties_data == null
        ? data_.tracts_data.population_density
        : data_.counties_data.population_density) || [];
    const data = [
      {
        type: "scatter",
        mode: "markers",
        marker: {
          sizemode: "area",
          sizemin: 4,
          autocolorscale: false,
          colorscale: colorScale,
          color:
            data_.counties_data == null
              ? data_.tracts_data.disease_percent
              : data_.counties_data.disease_percent,
          // TODO : Add size calculation to have nice bubble plots
          size: size,
          sizeref: (2.0 * Math.max(...size)) / 60 ** 2,
          colorbar: {},
        },
        x:
          data_.counties_data == null
            ? data_.tracts_data.disease_percentile
            : data_.counties_data.disease_percentile,
        y:
          data_.counties_data == null
            ? data_.tracts_data[theme]
            : data_.counties_data[theme],
        customdata: customGraphData(data_, stateCounties_),
        hovertemplate:
          `<b>${hoverTitle}</b><br><br>` +
          "<b>Health Measure percentile</b>: %{x}<br>" +
          `<b>${theme + " Percentile"}</b>: %{y}<br>` +
          "<b>Health Measure prevalence</b>: %{marker.color:,}<br>" +
          "<b>Population Density</b>: %{marker.size:,}<br>" +
          "<b>State</b>: %{customdata.state.state_name}<br>" +
          "<b>County</b>: %{customdata.county.county_name}<br>" +
          tractHoverValue +
          "<extra></extra>",
      },
    ];

    const layout = {
      width: plotWidth_,
      height: 520,
      title: {
        text: `${viewType} Social Factors by Health Measure Prevalence Rankings`,
        font: {
          family: "SourceSansPro",
          size: 14,
        },
        xref: "paper",
        x: 0,
      },
      xaxis: {
        autorange: true,
        showgrid: true,
        zeroline: false,
        position: 0,
        title: {
          text: `${selected_.disease.label} Prevalence Percentile`,
          font: {
            family: "SourceSansPro",
            size: 14,
            color: "#7f7f7f",
          },
        },
      },
      yaxis: {
        autorange: true,
        showgrid: true,
        position: 0,
        zeroline: false,
        title: {
          text: `${theme}`,
          font: {
            family: "SourceSansPro",
            size: 14,
            color: "#7f7f7f",
          },
        },
      },
    };

    return { layout: layout, data: data };
  }

  function updateTheme(value) {
    selectedTheme.value = value;
    const data_ = get_scatter_data(
      data.value,
      value,
      stateCounties.value,
      selected.value,
      plotWidth.value
    );
    Plotly.react(scatter.value, data_.data, data_.layout);
  }

  function onResize(size) {
    try {
      setTimeout(() => {
        if (size.width > 1253 && choropleth.value && scatter.value) {
          plotWidth.value = (size.width * 0.9 - 46) / 2;
          var update = { width: (size.width * 0.9 - 46) / 2, height: 520 };
          Plotly.relayout(choropleth.value, update);
          Plotly.relayout(scatter.value, update);
        } else {
          plotWidth.value = 600;
        }
      }, 500);
    } catch (error) {
      console.error(error);
    }
  }

  watch(selected, (newValue) => {
    const meta = {
      state: newValue.state.value,
      county: newValue.county.value,
      disease: {
        value: newValue.disease.value,
        isTract: newValue.disease.isTract,
      },
    };
    if (meta.disease.value != null) {
      Loading.show({
        message: "Loading data ...",
        boxClass: "bg-grey-2 text-grey-9",
        spinnerColor: "primary",
      });
      API.getData(meta)
        .then((response) => {
          if (response && response.tracts_data) {
            Object.assign(data.value, response);
            const percents =
              response.counties_data == null
                ? response.tracts_data
                  ? response.tracts_data.disease_percent
                  : []
                : response.counties_data
                ? response.counties_data.disease_percent
                : [];
            const prevalences = {
              min: percents.length > 0 ? Math.min(...percents) : 0,
              max: percents.length > 0 ? Math.max(...percents) : 0,
            };
            prelevanceBoundries.value = prevalences;
            diseasePrevalence.value = prevalences;
            const data_ = get_choropleth_data(
              response,
              selected.value,
              stateCounties.value,
              plotWidth.value
            );

            const data_scatter = get_scatter_data(
              response,
              selectedTheme.value,
              stateCounties.value,
              selected.value,
              plotWidth.value
            );
            Plotly.react(choropleth.value, data_.data, data_.layout);
            Plotly.react(scatter.value, data_scatter.data, data_scatter.layout);
            Loading.hide();
          }
        })
        .catch((err) => {
          console.log(err);
          Loading.hide();
        });
    }
  });

  function filterItemfilterFromPrevalence(diseasePrevalence, item) {
    const indexToKeep = [];
    item.disease_percent.filter((elem, index, array) => {
      if (elem >= diseasePrevalence.min && elem <= diseasePrevalence.max) {
        indexToKeep.push(index);
      }
    });
    return Object.fromEntries(
      Object.entries(item).map((value) => {
        return [value[0], indexToKeep.map((idx) => value[1][idx])];
      })
    );
  }

  function filterFromPrevalence(diseasePrevalence, data) {
    let dataFiltered = { ...data };
    if (data && data.counties_data) {
      dataFiltered = {
        ...dataFiltered,
        counties_data: filterItemfilterFromPrevalence(
          diseasePrevalence,
          data.counties_data
        ),
      };
    }
    if (data && data.tracts_data) {
      dataFiltered = {
        ...dataFiltered,
        tracts_data: filterItemfilterFromPrevalence(
          diseasePrevalence,
          data.tracts_data
        ),
      };
    }
    return dataFiltered;
  }

  watch(diseasePrevalence, (newValue) => {
    // filter the data
    const dataFiltered = filterFromPrevalence(
      diseasePrevalence.value,
      data.value
    );

    const data_ = get_choropleth_data(
      dataFiltered,
      selected.value,
      stateCounties.value,
      plotWidth.value
    );

    const data_scatter = get_scatter_data(
      dataFiltered,
      selectedTheme.value,
      stateCounties.value,
      selected.value,
      plotWidth.value
    );
    Plotly.react(choropleth.value, data_.data, data_.layout);
    Plotly.react(scatter.value, data_scatter.data, data_scatter.layout);
  });

  onMounted(() => {
    API.getThemes().then((response) => {
      themes.value = response;
      selectedTheme.value = response[0];
    });
    const data_ = get_choropleth_data(
      data.value,
      selected.value,
      stateCounties.value,
      plotWidth.value
    );
    const data_scatter = get_scatter_data(
      data.value,
      selectedTheme.value,
      stateCounties.value,
      selected.value,
      plotWidth.value
    );
    const config = { responsive: true };
    Plotly.newPlot(choropleth.value, data_.data, data_.layout, config);
    // Attach hover event
    Plotly.newPlot(
      scatter.value,
      data_scatter.data,
      data_scatter.layout,
      config
    );

    // Attach on select event
    scatter.value.on("plotly_selected", function (eventData) {
      const ids_ = [];
      let type_ = "county";

      eventData.points.forEach((value) => {
        if (value && value.customdata) {
          if (value.customdata.tract) {
            ids_.push(value.customdata.tract.tract_code);
            type_ = "tract";
          } else {
            if (value.customdata.state && value.customdata.county) {
              ids_.push(
                value.customdata.state.state_code +
                  value.customdata.county.county_code
              );
            }
          }
        }
      });

      selectedPoints.value = {
        type: type_,
        ids: ids_,
      };
    });
  });

  return {
    choropleth,
    scatter,
    selected,
    diseases,
    stateOptions,
    countiesOptions,
    tableTitle,
    updateState,
    updateDisease,
    updateCounty,
    updatePrevalence,
    diseasePrevalence,
    prelevanceBoundries,
    data,
    updateTheme,
    themes,
    selectedTheme,
    tableData,
    exportTable,
    onResize,
  };
}
