import { ref, onMounted, computed } from "vue";
import { API } from "../Api";

const selected = ref({
  state: { value: "all", label: "ALL" },
  county: { value: "all", label: "ALL" },
  disease: { value: null, label: null, isTract: null },
});

const diseasePrevalence = ref({
  min: 0,
  max: 0,
});

export function useSelection() {
  const stateCounties = ref({});
  const diseases = ref([]);
  const tableTitle = ref("");

  const tableCountyTitle = "County Level Data";
  const tableTractTitle = "Tract Level Data";

  const stateOptions = computed(() => {
    if (stateCounties.value) {
      const selection = Object.values(stateCounties.value)
        .map((item) => ({ label: item.state_name, value: item.state_code }))
        .sort((a, b) => a.label.localeCompare(b.label));

      return [{ value: "all", label: "ALL" }].concat(selection);
    }
    return [];
  });

  const countiesOptions = computed(() => {
    if (
      selected.value.state.value != "all" &&
      stateCounties.value &&
      stateCounties.value[selected.value.state.value]
    ) {
      const selection = Object.values(
        Object.values(stateCounties.value[selected.value.state.value].counties)
      )
        .map((item) => ({ label: item.county_name, value: item.county_code }))
        .sort((a, b) => a.label.localeCompare(b.label));
      return [{ value: "all", label: "ALL" }].concat(selection);
    }
    return [];
  });

  function updateState(newState) {
    const newSelected = {
      ...selected.value,
      state: newState,
      county: { value: "all", label: "ALL" },
    };
    selected.value = newSelected;
  }

  function updateDisease(newDisease) {
    const newSelected = {
      ...selected.value,
      disease: newDisease,
      county: newDisease.isTract
        ? selected.value.county
        : { value: "all", label: "ALL" },
    };
    selected.value = newSelected;
    tableTitle.value = newDisease.isTract ? tableTractTitle : tableCountyTitle;
  }

  function updateCounty(newCounty) {
    const newSelected = { ...selected.value, county: newCounty };
    selected.value = newSelected;
  }

  function updatePrevalence(newPrevalence) {
    diseasePrevalence.value = newPrevalence;
  }

  onMounted(() => {
    API.getStateCounties()
      .then((response) => {
        stateCounties.value = response;
      })
      .catch((err) => {
        console.log(err);
      });
    API.getDiseases()
      .then((response) => {
        const result = response.sort((a, b) => a.value.localeCompare(b.value));
        diseases.value = result;
        const newSelected = { ...selected.value, disease: diseases.value[0] };
        selected.value = newSelected;
        tableTitle.value = diseases.value[0].isTract
          ? tableTractTitle
          : tableCountyTitle;
      })
      .catch((err) => {
        console.log(err);
      });
  });

  return {
    selected,
    diseasePrevalence,
    diseases,
    tableTitle,
    stateOptions,
    countiesOptions,
    stateCounties,
    updateState,
    updateDisease,
    updateCounty,
    updatePrevalence,
  };
}
