# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
import dataiku
from dataiku import pandasutils as pdu
import pandas as pd
import requests
import json

# custom function that can be found within Libraries tab > G+L
from census_api_functions import get_project_variables, get_query_text, state_name_list, get_tracts_code_table

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# variable specifications
census_api_key = get_project_variables('standard','api_key')
# census API variables specifications
census_code = 'S0101'
year = 2022

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# url path to call census API services
url_path = f"https://api.census.gov/data/{year}/acs/acs5/profile?get=NAME&for=state:*&key={census_api_key}"

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# numerical list of US states
state_nums_list = state_name_list(url_path)

# API request to gather the dataset from U.S. Census Bureau
all_tracts_df = pd.DataFrame()

for state in state_nums_list:
    print(f"Processing state: {state}")
    try:
        # Construct API URL
        state_all_tracts_query_url = (
            f"https://api.census.gov/data/{year}/acs/acs5/subject"
            f"?get=NAME,group({census_code})&for=tract:*&in=state:{state}&key={census_api_key}"
        )
        # Fetch the data using requests
        response = requests.get(state_all_tracts_query_url)
        response.raise_for_status()  # Raise exception for HTTP errors
        
        # Parse the JSON response
        state_all_tract_names_query_result_list = response.json()
        
        # Convert JSON to DataFrame
        state_all_tract_names_df = pd.DataFrame(
            state_all_tract_names_query_result_list[1:], 
            columns=state_all_tract_names_query_result_list[0]
        )
        
        # Exclude NAME column (first column)
        df = state_all_tract_names_df.iloc[:, 1:]
        
        # Concatenate to main DataFrame
        all_tracts_df = pd.concat([all_tracts_df, df], ignore_index=True)
        
    except requests.exceptions.RequestException as req_err:
        print(f"HTTP error for state {state}: {req_err}")
    except json.JSONDecodeError as json_err:
        print(f"JSON decode error for state {state}: {json_err}")
    except Exception as e:
        print(f"Unexpected error for state {state}: {e}")


# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# data preprocessing: column mapping, feature generation
all_tracts_df = all_tracts_df[['GEO_ID', 'S0101_C01_030E', 'S0101_C02_030E']]
all_tracts_df = all_tracts_df.rename(columns = {'S0101_C01_030E':'E_AGE65','S0101_C02_030E':'EP_AGE65'})

# -------------------------------------------------------------------------------- NOTEBOOK-CELL: CODE
# Write recipe outputs

S0101_all_tracts = dataiku.Dataset("S0101_svi_tracts")
S0101_all_tracts.write_with_schema(all_tracts_df)