from other_api_utils.google_docs.typing_commons import Credentials


def get_google_credentials(client_id: str, client_secret: str, token: str, refresh_token: str,
                           scopes: list=["documents", "spreadsheets", "presentations", "drive"])->Credentials:
    """Retrieves a google presentation credentials object.
    
    :param client_id: str: A Google API 'Client ID' string.
    :param client_secret: str: A Google API 'Client secret' string.
    :param token: str: A Google API 'Access token' string.
    :param refresh_token: str: A Google API 'Refresh token' string.
    :param scopes: list: The list of the Google API scopes to associate with your credentials.
    
    :returns: google_credentials: Credentials: The google credentials object. 
    """
    ALLOWED_SCOPES = ["documents", "spreadsheets", "presentations", "drive"]
    invalid_scopes = set(scopes).difference(set(ALLOWED_SCOPES))
    if invalid_scopes:
        raise ValueError(f"You provided the invalid scopes '{list(invalid_scopes)}'! Please choose scopes that are in '{ALLOWED_SCOPES}'")
    
    detailed_scopes = [
        f"https://www.googleapis.com/auth/{scope}" for scope in scopes
        if scope in ALLOWED_SCOPES
    ]
    user_info = {
        "token": token,
        "refresh_token": refresh_token,
        "token_uri": "https://oauth2.googleapis.com/token",
        "client_id": client_id,
        "client_secret": client_secret,
        "scopes": detailed_scopes,
        "universe_domain": "googleapis.com",
    }
    google_credentials = Credentials.from_authorized_user_info(user_info, detailed_scopes)
    print(f"Google credentials successfully created on the scopes: '{detailed_scopes}'")
    return google_credentials