/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.transactions.fs;

import com.dataiku.dip.transactions.fs.ifaces.StreamSupplier;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dss.shadelib.org.apache.commons.io.IOUtils;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.Closeable;
import java.io.File;
import java.io.FileDescriptor;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.zip.CRC32;
import java.util.zip.CheckedInputStream;

public class FSyncUtils
implements Closeable {
    private static final int NB_LOGGED_ERRORS_LIMIT = 5;
    private static final DKULogger logger = DKULogger.getLogger(FSyncUtils.class);
    private final ThreadLocal<byte[]> threadLocalBuffers = ThreadLocal.withInitial(() -> new byte[65536]);
    private final ExecutorService pool;
    private final LinkedList<Future<Void>> futures = new LinkedList();
    private final boolean enableCRC32Check;
    private final boolean enableFSync;

    public FSyncUtils(int nThreads, boolean enableCRC32Check, boolean enableFSync) {
        this.pool = Executors.newFixedThreadPool(nThreads);
        this.enableFSync = enableFSync;
        this.enableCRC32Check = enableCRC32Check;
        logger.info((Object)("Setting up fsync pool with " + nThreads + " threads and CRC32 check is " + (enableCRC32Check ? "enabled" : "disabled")));
    }

    public void writeAndSync(File file, StreamSupplier streamSupplier) throws IOException {
        this.futures.add(this.pool.submit(() -> {
            this.writeAndSyncNow(file, streamSupplier);
            return null;
        }));
    }

    @Override
    public void close() {
        this.pool.shutdown();
    }

    public void flush() throws IOException {
        int nbErrors = 0;
        ExecutionException storedException = null;
        Iterator iterator = this.futures.iterator();
        while (iterator.hasNext()) {
            Future future = (Future)iterator.next();
            try {
                iterator.remove();
                Uninterruptibles.getUninterruptibly((Future)future);
            }
            catch (ExecutionException e) {
                ++nbErrors;
                if (storedException == null) {
                    storedException = e;
                    continue;
                }
                if (nbErrors >= 5) continue;
                logger.error((Object)"Error while writing file", e.getCause());
            }
        }
        if (storedException != null) {
            String nbOtherErrors = nbErrors > 5 ? " (and " + (nbErrors - 5) + " more)" : "";
            throw new IOException("Error while writing file" + nbOtherErrors, storedException);
        }
    }

    private void writeAndSyncNow(File file, StreamSupplier streamSupplier) throws IOException {
        try (InputStream dataStream = streamSupplier.openStream();){
            this.writeAndSyncNow(file, dataStream);
        }
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    private void writeAndSyncNow(File file, InputStream dataStream) throws IOException {
        logger.debug((Object)("writeAndSync: " + file.getAbsolutePath()));
        if (file.exists()) {
            if (file.isFile()) {
                logger.warn((Object)("The file " + file.getAbsolutePath() + " already exists. This is unexpected, but we can still try to delete it ."));
                file.delete();
                if (file.exists()) {
                    throw new IOException("Unable to delete file " + file.getAbsolutePath());
                }
            } else {
                if (!file.isDirectory()) throw new IOException("The path " + file.getAbsolutePath() + " represents neither a file, nor a directory. This is highly unexpected.");
                throw new IOException("This should not be a directory: " + file.getAbsolutePath());
            }
        }
        byte[] buffer = this.threadLocalBuffers.get();
        CRC32 expectedCrc32 = new CRC32();
        InputStream checkedDataStream = this.enableCRC32Check ? new CheckedInputStream(dataStream, expectedCrc32) : dataStream;
        try (FileOutputStream s = new FileOutputStream(file);
             FileChannel c = s.getChannel();){
            c.truncate(0L);
            try (OutputStream os = Channels.newOutputStream(c);){
                IOUtils.copyLarge((InputStream)checkedDataStream, (OutputStream)os, (byte[])buffer);
                if (this.enableFSync) {
                    c.force(true);
                    FileDescriptor fd = s.getFD();
                    if (!fd.valid()) {
                        throw new IOException("The file descriptor for " + file.getAbsolutePath() + " isn't valid");
                    }
                    fd.sync();
                }
            }
        }
        CRC32 writtenCrc32 = new CRC32();
        if (!file.exists()) throw new IOException("The file we just wrote doesn't exist : " + file.getAbsolutePath());
        if (file.isFile()) {
            if (!this.enableCRC32Check) return;
            try (FileInputStream writtenFileStream = new FileInputStream(file);
                 CheckedInputStream writtenFileCheckedStream = new CheckedInputStream(writtenFileStream, writtenCrc32);){
                while (((InputStream)writtenFileCheckedStream).read(buffer) != -1) {
                }
                if (writtenCrc32.getValue() == expectedCrc32.getValue()) return;
                throw new IOException("The file we just wrote is corrupted: " + file.getAbsolutePath());
            }
        }
        if (!file.isDirectory()) throw new IOException("The impossible happens: we wrote a file and we created something which is neither a file or a directory : " + file.getAbsolutePath());
        throw new IOException("The impossible happens: we wrote a file and we created a directory : " + file.getAbsolutePath());
    }
}

