/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.cli;

import com.dataiku.dip.DKUApp;
import com.dataiku.dip.cli.CLICommand;
import com.dataiku.dip.cli.DKU;
import com.dataiku.dip.data.currency.CurrencyCodes;
import com.dataiku.dip.util.XMLUtils;
import com.dataiku.dip.utils.DKUFileUtils;
import com.dataiku.dss.shadelib.com.google.common.io.Files;
import com.dataiku.dss.shadelib.org.apache.commons.io.IOUtils;
import com.dataiku.dss.shadelib.org.apache.http.HttpEntity;
import com.dataiku.dss.shadelib.org.apache.http.client.methods.CloseableHttpResponse;
import com.dataiku.dss.shadelib.org.apache.http.client.methods.HttpGet;
import com.dataiku.dss.shadelib.org.apache.http.client.methods.HttpUriRequest;
import com.dataiku.dss.shadelib.org.apache.http.client.utils.URIBuilder;
import com.dataiku.dss.shadelib.org.apache.http.impl.client.CloseableHttpClient;
import com.dataiku.dss.shadelib.org.apache.http.impl.client.HttpClients;
import com.dataiku.dss.shadelib.org.apache.http.util.EntityUtils;
import com.dataiku.dss.shadelib.org.joda.time.LocalDate;
import com.dataiku.dss.shadelib.org.joda.time.ReadablePartial;
import com.dataiku.dss.shadelib.org.joda.time.format.DateTimeFormat;
import com.dataiku.dss.shadelib.org.joda.time.format.DateTimeFormatter;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.zip.GZIPOutputStream;
import org.apache.commons.cli.Options;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

public class UpdateCurrenciesRatesCommand
extends CLICommand {
    public static final String CACHE_PATH = "/caches/eurorates";
    private final File euroRatesDirectory = new File(System.getenv("DIP_HOME") + "/resources/eurorates");
    private static final Logger logger = Logger.getLogger((String)"dku.currenciesRates");

    public static void main(String[] args) throws Exception {
        new UpdateCurrenciesRatesCommand().execute(new String[]{"update-currency-rates"});
    }

    @Override
    public String usageString() {
        return "update-currency-rates";
    }

    @Override
    public String description() {
        return "Updates eurorates/data.csv.gz";
    }

    @Override
    public Options opts() {
        Options options = new Options();
        DKU.addGlobalOptions(options);
        return options;
    }

    @Override
    public int execute(String[] args) throws Exception {
        CurrencyData imfCurrencyData;
        this.parseAndMiniSetup(args, "update-currency-rates", true);
        if (!this.euroRatesDirectory.exists() && !this.euroRatesDirectory.mkdirs()) {
            throw new IOException("Unable to create directory " + this.euroRatesDirectory.getAbsolutePath());
        }
        CurrencyData ecbCurrencyData = new ECBCurrencyLoader().fetch();
        try {
            imfCurrencyData = new IMFCurrencyLoader().fetch(ecbCurrencyData.getDateRange());
        }
        catch (IOException | RuntimeException e) {
            logger.warn((Object)"Unable to retrieve currency data from IMF. Some exchange rates will not be valued", (Throwable)e);
            imfCurrencyData = new CurrencyData();
        }
        CurrencyData currencyData = ecbCurrencyData.merge(imfCurrencyData);
        new CsvGzWriter().write(new File(this.euroRatesDirectory, "data.csv.gz"), currencyData);
        return 0;
    }

    private static class ECBCurrencyLoader {
        private ECBCurrencyLoader() {
        }

        public CurrencyData fetch() throws IOException, SAXException, ParseException {
            Map<String, BigDecimal> lastRates = null;
            SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd");
            LocalDate lastDate = null;
            Map<String, Map<String, BigDecimal>> data = this.loadData();
            CurrencyData result = new CurrencyData();
            for (Map.Entry<String, Map<String, BigDecimal>> entry : data.entrySet()) {
                LocalDate date = new LocalDate((Object)simpleDateFormat.parse(entry.getKey()));
                Map<String, BigDecimal> rates = entry.getValue();
                if (lastDate != null) {
                    LocalDate iterDate = lastDate.minusDays(1);
                    while (iterDate.isAfter((ReadablePartial)date)) {
                        result.data.put(iterDate, lastRates);
                        iterDate = iterDate.minusDays(1);
                    }
                }
                result.data.put(date, rates);
                lastDate = date;
                lastRates = rates;
            }
            return result;
        }

        private Map<String, Map<String, BigDecimal>> loadData() throws SAXException, IOException {
            logger.info((Object)"Fetching rates from ECB");
            URL url = new URL(DKUApp.getProperty((String)"dku.currenciesRates.ecbURL", (String)"https://www.ecb.europa.eu/stats/eurofxref/eurofxref-hist.xml"));
            Document document = XMLUtils.parse(url.openStream());
            return this.parseCurrencyXML(document);
        }

        private Map<String, Map<String, BigDecimal>> parseCurrencyXML(Document document) {
            LinkedHashMap<String, Map<String, BigDecimal>> data = new LinkedHashMap<String, Map<String, BigDecimal>>();
            NodeList dateList = document.getDocumentElement().getElementsByTagName("Cube");
            String currentDate = "";
            HashMap<String, BigDecimal> dateData = new HashMap<String, BigDecimal>();
            for (int i = 0; i < dateList.getLength(); ++i) {
                Element dateElement = (Element)dateList.item(i);
                String date = dateElement.getAttribute("time");
                String currency = dateElement.getAttribute("currency");
                String rate = dateElement.getAttribute("rate");
                if (!StringUtils.isBlank((String)date)) {
                    if (!StringUtils.isBlank((String)currentDate)) {
                        data.put(currentDate, dateData);
                        dateData = new HashMap();
                    }
                    currentDate = date;
                    continue;
                }
                if (StringUtils.isBlank((String)currency) || StringUtils.isBlank((String)rate)) continue;
                dateData.put(currency, new BigDecimal(rate));
            }
            data.put(currentDate, dateData);
            return data;
        }
    }

    private static class CurrencyData {
        public Map<LocalDate, Map<String, BigDecimal>> data = new LinkedHashMap<LocalDate, Map<String, BigDecimal>>();

        private CurrencyData() {
        }

        public DateRange getDateRange() {
            ArrayList<LocalDate> dates = new ArrayList<LocalDate>(this.data.keySet());
            dates.sort(LocalDate::compareTo);
            return new DateRange((LocalDate)dates.get(0), (LocalDate)dates.get(dates.size() - 1));
        }

        private Set<String> getCurrencies() {
            return this.data.values().stream().flatMap(v -> v.keySet().stream()).collect(Collectors.toSet());
        }

        public void addRate(LocalDate date, String currencyCode, BigDecimal value) {
            this.data.computeIfAbsent(date, k -> new HashMap()).put(currencyCode, value);
        }

        private CurrencyData merge(CurrencyData additionalData) {
            CurrencyData result = new CurrencyData();
            for (LocalDate date : this.data.keySet()) {
                LinkedHashMap<String, BigDecimal> mergedRates = new LinkedHashMap<String, BigDecimal>(this.data.get(date));
                Map<String, BigDecimal> imfRates = additionalData.data.get(date);
                if (imfRates != null) {
                    for (String currency : imfRates.keySet()) {
                        mergedRates.computeIfAbsent(currency, imfRates::get);
                    }
                }
                result.data.put(date, mergedRates);
            }
            return result;
        }
    }

    private static class IMFCurrencyLoader {
        private final DateTimeFormatter imfDateParser = DateTimeFormat.forPattern((String)"MMMM dd, yyyy").withLocale(Locale.US);

        private IMFCurrencyLoader() {
        }

        public CurrencyData fetch(DateRange dateRange) throws IOException {
            LocalDate startMonth = dateRange.start.withDayOfMonth(1);
            LocalDate endMonth = dateRange.end.withDayOfMonth(1);
            CurrencyData result = new CurrencyData();
            LocalDate curMonth = endMonth;
            while (curMonth.isAfter((ReadablePartial)startMonth) || curMonth.isEqual((ReadablePartial)startMonth)) {
                try {
                    CurrencyData ratesForMonth = this.fetchMonth(curMonth);
                    for (LocalDate day : ratesForMonth.data.keySet()) {
                        if (day.isBefore((ReadablePartial)dateRange.start) || day.isAfter((ReadablePartial)dateRange.end)) continue;
                        result.data.put(day, ratesForMonth.data.get(day));
                    }
                }
                catch (Exception e) {
                    logger.warn((Object)String.format("Unable to retrieve currency data from IMF for %s. Exchange rates for some currencies will not be accurate for this month.", IMFCurrencyLoader.printMonth(curMonth)));
                }
                curMonth = curMonth.minusMonths(1);
            }
            this.fillGaps(result, dateRange);
            return this.convertSdrToEuroRates(result);
        }

        private CurrencyData fetchMonth(LocalDate month) throws IOException {
            CurrencyData out = new CurrencyData();
            String content = this.getMonthContent(month);
            ArrayList<Integer> startPageIndexes = new ArrayList<Integer>();
            int index = content.indexOf("SDRs per Currency unit for");
            while (index >= 0) {
                startPageIndexes.add(index);
                index = content.indexOf("SDRs per Currency unit for", index + 1);
            }
            for (int i = 0; i < startPageIndexes.size(); ++i) {
                String page = i == startPageIndexes.size() - 1 ? content.substring((Integer)startPageIndexes.get(i)) : content.substring((Integer)startPageIndexes.get(i), (Integer)startPageIndexes.get(i + 1));
                this.parsePage(out, page);
            }
            return out;
        }

        private void fillGaps(CurrencyData rates, DateRange dateRange) {
            Set<String> currencies = rates.getCurrencies();
            Map<Object, Object> lastRates = new HashMap();
            LocalDate currentDay = dateRange.start;
            while (currentDay.isBefore((ReadablePartial)dateRange.end) || currentDay.isEqual((ReadablePartial)dateRange.end)) {
                Map<String, BigDecimal> ratesForDate = rates.data.get(currentDay);
                if (ratesForDate == null) {
                    ratesForDate = new HashMap<String, BigDecimal>(lastRates);
                    rates.data.put(currentDay, ratesForDate);
                } else {
                    for (String currency : currencies) {
                        BigDecimal rate = ratesForDate.get(currency);
                        if (rate != null) continue;
                        rate = (BigDecimal)lastRates.get(currency);
                        ratesForDate.put(currency, rate);
                    }
                }
                lastRates = ratesForDate;
                currentDay = currentDay.plusDays(1);
            }
        }

        private CurrencyData convertSdrToEuroRates(CurrencyData input) {
            CurrencyData output = new CurrencyData();
            for (Map.Entry<LocalDate, Map<String, BigDecimal>> e : input.data.entrySet()) {
                LocalDate date = e.getKey();
                Map<String, BigDecimal> sdrRates = e.getValue();
                BigDecimal euroRate = sdrRates.get("EUR");
                if (euroRate == null) continue;
                HashMap<String, BigDecimal> euroRates = new HashMap<String, BigDecimal>();
                for (String currencyCode : sdrRates.keySet()) {
                    if ("EUR".equals(currencyCode)) continue;
                    BigDecimal sdrRate = sdrRates.get(currencyCode);
                    BigDecimal convertedRate = sdrRate == null ? null : euroRate.divide(sdrRate, RoundingMode.HALF_UP).setScale(4, RoundingMode.HALF_UP);
                    euroRates.put(currencyCode, convertedRate);
                }
                output.data.put(date, euroRates);
            }
            return output;
        }

        private void parsePage(CurrencyData data, String page) {
            List<String> lines = Arrays.asList(page.split("\n"));
            List rows = lines.stream().map(String::trim).filter(line -> !line.isBlank() && !line.startsWith("SDRs per") && line.contains("\t")).map(line -> Arrays.asList(line.split("\t"))).collect(Collectors.toList());
            List headers = (List)rows.remove(0);
            ArrayList<LocalDate> dates = new ArrayList<LocalDate>();
            for (int i = 1; i < headers.size(); ++i) {
                dates.add(this.imfDateParser.parseLocalDate((String)headers.get(i)));
            }
            for (List row : rows) {
                String currency = (String)row.get(0);
                String currencyCode = IMFCurrencyLoader.getCurrencyCodeByName(currency);
                if (currencyCode == null) continue;
                for (int colIndex = 1; colIndex < row.size(); ++colIndex) {
                    LocalDate date = (LocalDate)dates.get(colIndex - 1);
                    String cell = (String)row.get(colIndex);
                    try {
                        BigDecimal value = IMFCurrencyLoader.isNullValue(cell) ? null : new BigDecimal(cell);
                        data.addRate(date, currencyCode, value);
                        continue;
                    }
                    catch (NumberFormatException e) {
                        logger.warn((Object)String.format("Unable to parse value for %s %s: %s", date, currencyCode, cell));
                    }
                }
            }
        }

        private String getMonthContent(LocalDate month) throws IOException {
            if (month.isBefore((ReadablePartial)new LocalDate(2003, 4, 1))) {
                return "";
            }
            boolean isCurrentMonth = LocalDate.now().withDayOfMonth(1).isEqual((ReadablePartial)month.withDayOfMonth(1));
            File cache = new File(new File(System.getenv("DIP_HOME") + UpdateCurrenciesRatesCommand.CACHE_PATH), "IMF_" + String.valueOf(month) + ".txt");
            if (cache.isFile() && !isCurrentMonth) {
                return Files.asCharSource((File)cache, (Charset)StandardCharsets.UTF_8).read();
            }
            logger.info((Object)("Fetching rates from IMF for " + IMFCurrencyLoader.printMonth(month)));
            String content = IMFCurrencyLoader.getContent(IMFCurrencyLoader.getUri(month));
            try {
                DKUFileUtils.mkdirsParent((File)cache);
                Files.write((byte[])content.getBytes(StandardCharsets.UTF_8), (File)cache);
            }
            catch (IOException e) {
                logger.warn((Object)("Unable to cache IMF exchange rates data for " + String.valueOf(month)), (Throwable)e);
            }
            return content;
        }

        private static String getUri(LocalDate month) {
            try {
                return new URIBuilder(DKUApp.getProperty((String)"dku.currenciesRates.imfURL", (String)"https://www.imf.org/external/np/fin/data/rms_mth.aspx")).addParameter("SelectDate", month.toString()).addParameter("reportType", "SDRCV").addParameter("tsvflag", "Y").toString();
            }
            catch (URISyntaxException e) {
                throw new IllegalStateException("Failed to build URI", e);
            }
        }

        /*
         * Enabled aggressive block sorting
         * Enabled unnecessary exception pruning
         * Enabled aggressive exception aggregation
         */
        private static String getContent(String uri) throws IOException {
            HttpGet get;
            try {
                get = new HttpGet(uri);
                logger.trace((Object)("GET " + uri));
            }
            catch (Exception e) {
                throw new IOException("Failed to create request", e);
            }
            try (CloseableHttpClient httpClient = HttpClients.createDefault();){
                CloseableHttpResponse resp = httpClient.execute((HttpUriRequest)get);
                try {
                    int code = resp.getStatusLine().getStatusCode();
                    if (code == 204) {
                        String string = "";
                        return string;
                    }
                    String payload = IOUtils.toString((InputStream)resp.getEntity().getContent(), (Charset)StandardCharsets.UTF_8);
                    if (code == 200 || code == 201) {
                        if ("text/html".equals(resp.getEntity().getContentType().getValue())) {
                            logger.debug((Object)String.format("Server error %s", payload));
                            throw new IOException("Server error");
                        }
                        String string = payload;
                        return string;
                    }
                    logger.debug((Object)String.format("HTTP %d %s", code, payload));
                    throw new IOException(String.format("HTTP %d", code));
                }
                finally {
                    EntityUtils.consume((HttpEntity)resp.getEntity());
                }
            }
        }

        private static boolean isNullValue(String v) {
            return v == null || "na".equals(v.toLowerCase(Locale.ROOT).replaceAll("[. /\\\\]", ""));
        }

        private static String getCurrencyCodeByName(String currencyName) {
            HashMap<String, String> mapping = new HashMap<String, String>();
            mapping.put("Chinese yuan", "CNY");
            mapping.put("Euro", "EUR");
            mapping.put("Japanese yen", "JPY");
            mapping.put("U.K. pound", "GBP");
            mapping.put("U.S. dollar", "USD");
            mapping.put("Algerian dinar", "DZD");
            mapping.put("Australian dollar", "AUD");
            mapping.put("Botswana pula", "BWP");
            mapping.put("Brazilian real", "BRL");
            mapping.put("Brunei dollar", "BND");
            mapping.put("Canadian dollar", "CAD");
            mapping.put("Chilean peso", "CLP");
            mapping.put("Czech koruna", "CZK");
            mapping.put("Danish krone", "DKK");
            mapping.put("Indian rupee", "INR");
            mapping.put("Israeli New Shekel", "ILS");
            mapping.put("Korean won", "KRW");
            mapping.put("Kuwaiti dinar", "KWD");
            mapping.put("Malaysian ringgit", "MYR");
            mapping.put("Mauritian rupee", "MUR");
            mapping.put("Mexican peso", "MXN");
            mapping.put("New Zealand dollar", "NZD");
            mapping.put("Norwegian krone", "NOK");
            mapping.put("Omani rial", "OMR");
            mapping.put("Peruvian sol", "PEN");
            mapping.put("Philippine peso", "PHP");
            mapping.put("Polish zloty", "PLN");
            mapping.put("Qatari riyal", "QAR");
            mapping.put("Russian ruble", "RUB");
            mapping.put("Saudi Arabian riyal", "SAR");
            mapping.put("Singapore dollar", "SGD");
            mapping.put("South African rand", "ZAR");
            mapping.put("Swedish krona", "SEK");
            mapping.put("Swiss franc", "CHF");
            mapping.put("Thai baht", "THB");
            mapping.put("Trinidadian dollar", "TTD");
            mapping.put("U.A.E. dirham", "AED");
            mapping.put("Uruguayan peso", "UYU");
            mapping.put("Bolivar Soberano", "VES");
            mapping.put("Bahrain dinar", "BHD");
            mapping.put("Colombian peso", "COP");
            mapping.put("Libyan dinar", "LYD");
            mapping.put("Nepalese rupee", "NPR");
            mapping.put("Pakistani rupee", "PKR");
            mapping.put("Sri Lankan rupee", "LKR");
            mapping.put("Tunisian dinar", "TND");
            mapping.put("Hungarian forint", "HUF");
            mapping.put("Icelandic krona", "ISK");
            mapping.put("Indonesian rupiah", "IDR");
            mapping.put("Iranian rial", "IRR");
            mapping.put("Kazakhstani tenge", "KZT");
            return (String)mapping.get(currencyName);
        }

        private static String printMonth(LocalDate month) {
            return DateTimeFormat.forPattern((String)"MMMM yyyy").withLocale(Locale.US).print((ReadablePartial)month);
        }
    }

    private static class DateRange {
        public LocalDate start;
        public LocalDate end;

        public DateRange(LocalDate start, LocalDate end) {
            this.start = start;
            this.end = end;
        }
    }

    private static class CsvGzWriter {
        private CsvGzWriter() {
        }

        public void write(File file, CurrencyData currencyData) throws IOException {
            logger.info((Object)("Writing " + file.getAbsolutePath()));
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter((OutputStream)new GZIPOutputStream(new FileOutputStream(file)), StandardCharsets.UTF_8));){
                writer.append(this.getHeader());
                for (Map.Entry<LocalDate, Map<String, BigDecimal>> entry : currencyData.data.entrySet()) {
                    writer.append(this.getCSVLine(entry.getKey(), entry.getValue()));
                }
            }
        }

        private String getHeader() {
            StringBuilder result = new StringBuilder("Date");
            for (String currency : CurrencyCodes.getAvailableCurrencies()) {
                if ("EUR".equals(currency)) continue;
                result.append(",").append(currency);
            }
            result.append("\n");
            return result.toString();
        }

        private String getCSVLine(LocalDate date, Map<String, BigDecimal> rates) {
            StringBuilder line = new StringBuilder(date.toString());
            for (String currency : CurrencyCodes.getAvailableCurrencies()) {
                if ("EUR".equals(currency)) continue;
                line.append(",");
                BigDecimal value = rates.get(currency);
                line.append(value == null ? "NA" : value);
            }
            line.append("\n");
            return line.toString();
        }
    }
}

