(function() {
    'use strict';

    const app = angular.module('dataiku.analysis.mlcore');

    app.component("tableManager", {
        bindings: {
            getRowsForIdentifier: '<',
            getHeaders: "<",
            fullModelId: "@",
            unparsedTimeseriesIdentifiers: "<",
            insightId: "<",
            setFlags: "<?",
            timeseriesIdentifierColumns: "<",
            onUiChange: "<?",
            hideColumnsWithField: "<?",
            getAdditionalFilters: "<?",
            getPinnedRows: "<?",
        },
        templateUrl: "/templates/ml/prediction-model/timeseries/per-ts-table.html",
        controller: function($scope, $stateParams, PerTimeseriesService) {
            const $ctrl = this;
            $ctrl.uiState = {};
            $ctrl.timeseriesTableFilters = {};
            $ctrl.columnFieldsToHide = [];
            $ctrl.initialized = false;
            $ctrl.tableHeaders = [];
            $ctrl.filters = {};

            $ctrl.$onInit = () => {
                $ctrl.initialized = true;

                const isDashboardTile = !!$stateParams.dashboardId;
                $ctrl.objectId = isDashboardTile ? `insightId.${$ctrl.insightId}` : `fullModelId.${$ctrl.fullModelId}`;

                if (isFunction($ctrl.setFlags)) {
                    $ctrl.setFlags($ctrl.uiState);
                }

                if (isFunction($ctrl.hideColumnsWithField)) {
                    $ctrl.columnFieldsToHide = $ctrl.hideColumnsWithField($ctrl.uiState);
                }

                refreshTable();
            };

            $ctrl.$onChanges = function(changes) {
                if (!$ctrl.initialized) return;

                if (changes.unparsedTimeseriesIdentifiers) {
                    if (isFunction($ctrl.setFlags)) {
                        $ctrl.setFlags($ctrl.uiState);
                    }

                    if (changes.unparsedTimeseriesIdentifiers.currentValue.length > 0) {
                        if (isFunction($ctrl.hideColumnsWithField)) {
                            $ctrl.columnFieldsToHide = $ctrl.hideColumnsWithField($ctrl.uiState);
                        }
                        refreshTable();
                    }
                }
            }

            $ctrl.updateColumnsToHide = function() {
                if (isFunction($ctrl.hideColumnsWithField)) {
                    $ctrl.columnFieldsToHide = $ctrl.hideColumnsWithField($ctrl.uiState);
                    $ctrl.tableHeaders = getAndConfigureHeaders();
                }
            }

            $ctrl.fireEvent = function(e) {
                if (isFunction($ctrl.onUiChange)) {
                    $ctrl.onUiChange($ctrl.uiState);
                }
            }

            $ctrl.updateFilters = function() {
                let additionalFilters = {};
                if (isFunction($ctrl.getAdditionalFilters)) {
                    additionalFilters = $ctrl.getAdditionalFilters($ctrl.uiState);
                }

                $ctrl.filters = { ...$ctrl.timeseriesTableFilters, ...additionalFilters }; // necessary to create a new dict here so that the ag-grid onChanges fires
            };

            $scope.$on('timeseriesIdentifiersFiltersUpdated', function(event, filters) {
                $ctrl.timeseriesTableFilters = filters;
                $ctrl.updateFilters();
            });

            $ctrl.onFilterChange = function(gridApi)  {
                if ($ctrl.getPinnedRows && gridApi) {
                    let headers = gridApi.getColumnDefs();
                    const pinnedRows = $ctrl.getPinnedRows(gridApi, headers);
                    gridApi.setGridOption('pinnedBottomRowData', pinnedRows || []);

                    // Set pinnedRow tooltip
                    headers.forEach(header => {
                        header.tooltipValueGetter = (p) => {
                            if (p.node.rowPinned && header.headerName) return "Average (± stddev) value of "+header.headerName;
                        }
                    })
                    gridApi.setGridOption('columnDefs', headers);
                }
            };

            function hideHeaders(currentHeaders) {
                if (!$ctrl.columnFieldsToHide.length) {
                    return currentHeaders;
                }

                for (const header of currentHeaders) {
                    header.hide = false;
                    for (const field of $ctrl.columnFieldsToHide) {
                        if (header[field]) {
                            header.hide = true;
                        }
                    }
                }
                return currentHeaders;
            }

            const dateFilterParams = {
                comparator: (filterLocalDateAtMidnight, cellValue) => {
                    const dateAsString = cellValue;
                    if (dateAsString == null) return -1;
                    const d = moment(cellValue).toDate();

                    if (filterLocalDateAtMidnight.getTime() === d.getTime()) {
                        return 0;
                    }

                    if (d < filterLocalDateAtMidnight) {
                        return -1;
                    }

                    if (d > filterLocalDateAtMidnight) {
                        return 1;
                    }
                    return 0;
                }
            };

            function getAndConfigureHeaders() {
                const tempHeaders = $ctrl.getHeaders($ctrl);

                for (const header of tempHeaders) {
                    if ("date" === header.type) {
                        header.filter = 'agDateColumnFilter';
                        header.filterParams = dateFilterParams;
                    }
                }
                return hideHeaders(tempHeaders);
            }

            function refreshTable() {
                if (!$ctrl.unparsedTimeseriesIdentifiers || $ctrl.unparsedTimeseriesIdentifiers.length === 0) {
                    return;
                }

                $ctrl.tableHeaders = getAndConfigureHeaders();

                // Create the rows of the fattable + fill allTimeseriesIdentifierValuesMap (mapping from each ts identifier column to the list of their values)
                $ctrl.allTableRows = [];
                $ctrl.allTimeseriesIdentifierValuesMap = PerTimeseriesService.initTimeseriesIdentifiersValues($ctrl.timeseriesIdentifierColumns);

                $ctrl.unparsedTimeseriesIdentifiers.forEach(unparsedTimeseriesIdentifier => {
                    if ("__single_timeseries_identifier" === unparsedTimeseriesIdentifier) {
                        const rowsForIdentifier = $ctrl.getRowsForIdentifier("", unparsedTimeseriesIdentifier, $ctrl.uiState)
                        $ctrl.allTableRows.push(...rowsForIdentifier);
                    } else {
                        const parsedTimeseriesIdentifier = JSON.parse(unparsedTimeseriesIdentifier);
                        PerTimeseriesService.addIdentifierValues(parsedTimeseriesIdentifier, $ctrl.timeseriesIdentifierColumns, $ctrl.allTimeseriesIdentifierValuesMap);
                        const rowsForIdentifier = $ctrl.getRowsForIdentifier(parsedTimeseriesIdentifier, unparsedTimeseriesIdentifier, $ctrl.uiState)
                        $ctrl.allTableRows.push(...rowsForIdentifier);
                    }
                });

                if ($ctrl.allTableRows.length > 0) {
                    PerTimeseriesService.removeDuplicatesAndSortIdentifierValuesForFilterDropdowns($ctrl.allTimeseriesIdentifierValuesMap);
                    $ctrl.updateFilters();
                    $ctrl.tableRows = $ctrl.allTableRows;
                    $scope.$emit('timeseriesTableEvent', { headers: $ctrl.tableHeaders, rows: $ctrl.allTableRows });
                }
            }

            function isFunction(maybeFunction) {
                return 'function' === typeof maybeFunction;
            }
        }
    });
})();
