(function() {
'use strict';

const app = angular.module('dataiku.common.nav', ['dataiku.notebooks']);


app.service("TopNav", function($stateParams, $rootScope, Logger) {
    const svc = this;

    let currentPageTitle = "Dataiku";
    $rootScope.topNav = { item : {}, isProjectAnalystRO : true, homeSearchFilter: "" }

    function getItemKey(item) {
        const projectKey = item.projectKey || $stateParams.projectKey;
        const type = item.type || "";
        return projectKey + ':' + type + ':' + item.id;
    }

    function sameItem(item1, item2) {
        return item1 && item2 && (getItemKey(item1) == getItemKey(item2));
    }

    svc.setOverrideLeftType = function(type) {
        $rootScope.topNav.overrideLeftType = type;
    };

    svc.setPageTitle = function(title) {
        currentPageTitle = title;
        svc.refreshPageTitle();
    };

    svc.refreshPageTitle = function(){
        var pn = "";
        if ($rootScope.totalUnreadNotifications && $rootScope.totalUnreadNotifications > 0) {
            pn = "(" + $rootScope.totalUnreadNotifications + ") "
        }
        var it = "";
        if ($rootScope.appConfig && $rootScope.appConfig.isAutomation) {
            it = " (Automation)";
        }
        document.title = pn + currentPageTitle + " | Dataiku" + it;
    }

    /**
     * Valid "top" elements
     * Frontend only
     */
    svc.LOGIN = "LOGIN";
    svc.DSS_HOME = "DSS_HOME";
    svc.TOP_HOME = "HOME";
    svc.TOP_ACCESS_REQUEST = "ACCESS_REQUEST";
    svc.TOP_FLOW = "FLOW";
    svc.TOP_ANALYSES = "ANALYSES";
    svc.TOP_NOTEBOOKS = "NOTEBOOKS";
    svc.TOP_JOBS = "JOBS";
    svc.TOP_DASHBOARD = "DASHBOARD";
    svc.TOP_WIKI = "WIKI";
    svc.TOP_DEPLOYER = "DEPLOYER";
    svc.TOP_API_DEPLOYER = "API_DEPLOYER";
    svc.TOP_PROJECT_DEPLOYER = "PROJECT_DEPLOYER";
    svc.TOP_MORE = "MORE";
    svc.TOP_SAVED_MODELS = "SAVED_MODELS";
    svc.TOP_GENAI_MODELS = "GENAI_MODELS";
    svc.TOP_MODEL_EVALUATION_STORES = "MODEL_EVALUATION_STORES";
    svc.TOP_RETRIEVABLE_KNOWLEDGE = "RETRIEVABLE_KNOWLEDGE";
    svc.TOP_MODEL_COMPARISONS = "MODEL_COMPARISONS";
    svc.TOP_EXPERIMENT_TRACKING = "EXPERIMENT_TRACKING";
    svc.TOP_LABELING_TASK = "LABELING_TASK";
    svc.TOP_PROMPT_STUDIOS = "PROMPT_STUDIOS";
    svc.TOP_UNIFIED_MONITORING = "UNIFIED_MONITORING";
    svc.TOP_AGENT_TOOLS = "AGENT_TOOLS";
    svc.TOP_AGENT_HUB = "AGENT_HUB";

    /**
     *  Valid "left" elements
     *  Frontend only
     */
    svc.LEFT_FLOW = "flow";
    svc.LEFT_DATASETS = "datasets";
    svc.LEFT_RECIPES = "recipes";
    svc.LEFT_STREAMING_ENDPOINT = "streaming_endpoints";
    svc.LEFT_ANALYSES = "analyses";
    svc.LEFT_SAVED_MODELS = "saved models";
    svc.LEFT_GENAI_MODELS = "genai models";
    svc.LEFT_AGENT_HUB = "agent hub";
    svc.LEFT_MODEL_EVALUATION_STORES = "model evaluation stores";
    svc.LEFT_RETRIEVABLE_KNOWLEDGE = "knowledge banks";
    svc.LEFT_MODEL_COMPARISONS = "model comparisons";
    svc.LEFT_EXPERIMENT_TRACKING = "experiment tracking";
    svc.LEFT_PROMPT_STUDIOS = "prompt studios";
    svc.LEFT_AGENT_TOOLS = "agent tools";
    svc.LEFT_NOTEBOOKS = "notebooks";
    svc.LEFT_WEB_APPS = "webapps";
    svc.LEFT_REPORTS = "reports";
    svc.LEFT_LIBRARIES = "libraries";
    svc.LEFT_JOBS = "jobs";
    svc.LEFT_SCENARIOS = "scenarios";
    svc.LEFT_MONITORING = "monitoring";
    svc.LEFT_CONTINUOUS_ACTIVITIES = "continuous-activities";
    svc.LEFT_WIKI = "wiki";
    svc.LEFT_DASHBOARDS = "dashboards";
    svc.LEFT_INSIGHTS = "insights";
    svc.LEFT_LAMBDA_SERVICE = "lambda";
    svc.LEFT_BUNDLES_DESIGN = "bundlesdesign";
    svc.LEFT_BUNDLE_AUTOMATION = "bundlesautomation";
    svc.LEFT_VARIABLES = "variables";
    svc.LEFT_MACROS = "runnables";
    svc.LEFT_VERSION_CONTROL = "version-control";
    svc.LEFT_APP_DESIGNER = "appdesigner";
    svc.LEFT_SETTINGS = "settings";
    svc.LEFT_SECURITY = "security";


    /**
     * Valid item types
     * Shared with backend
     */
    svc.ITEM_DATASET = "DATASET";
    svc.ITEM_RECIPE = "RECIPE";
    svc.ITEM_ANALYSIS = "ANALYSIS";
    svc.ITEM_DASHBOARD = "DASHBOARD";
    svc.ITEM_WEB_APP = "WEB_APP";
    svc.ITEM_CODE_STUDIO = "CODE_STUDIO";
    svc.ITEM_REPORT = "REPORT";
    svc.ITEM_SQL_NOTEBOOK = "SQL_NOTEBOOK";
    svc.ITEM_SEARCH_NOTEBOOK = "SEARCH_NOTEBOOK";
    svc.ITEM_JUPYTER_NOTEBOOK = "JUPYTER_NOTEBOOK";
    svc.ITEM_SAVED_MODEL = "SAVED_MODEL";
    svc.ITEM_MODEL_EVALUATION_STORE = "MODEL_EVALUATION_STORE";
    svc.ITEM_RETRIEVABLE_KNOWLEDGE = "RETRIEVABLE_KNOWLEDGE";
    svc.ITEM_MODEL_COMPARISON = "MODEL_COMPARISON";
    svc.ITEM_MANAGED_FOLDER = "MANAGED_FOLDER";
    svc.ITEM_STREAMING_ENDPOINT = "STREAMING_ENDPOINT";
    svc.ITEM_JOB = "JOB";
    svc.ITEM_CONTINUOUS_ACTIVITY = "CONTINUOUS_ACTIVITY";
    svc.ITEM_INSIGHT = "INSIGHT";
    svc.ITEM_LAMBDA_SERVICE = "LAMBDA_SERVICE";
    svc.ITEM_SCENARIO = "SCENARIO";
    svc.ITEM_MONITORING = "MONITORING";
    svc.ITEM_PROJECT = "PROJECT";
    svc.ITEM_ARTICLE = "ARTICLE";
    svc.ITEM_LABELING_TASK = "LABELING_TASK";
    svc.ITEM_PROMPT_STUDIO = "PROMPT_STUDIO";
    svc.ITEM_AGENT_TOOL = "AGENT_TOOL";
    svc.ITEM_UNIFIED_MONITORING = "UNIFIED_MONITORING";

    /**
     * Valid tabs-type
     * Frontend only
     */
    svc.TABS_NONE = "NONE";
    svc.TABS_ANALYSIS = "ANALYSIS";
    svc.TABS_DATASET = "DATASET";
    svc.TABS_STREAMING_ENDPOINT = "STREAMING_ENDPOINT";
    svc.TABS_NEW_DATASET = "NEW-DATASET";
    svc.TABS_SAVED_MODEL = "SAVED_MODEL";
    svc.TABS_SAVED_MODEL_VERSION = "SAVED_MODEL-VERSION";
    svc.TABS_MODEL_EVALUATION_STORE = "MODEL_EVALUATION_STORE";
    svc.TABS_RETRIEVABLE_KNOWLEDGE = "RETRIEVABLE_KNOWLEDGE";
    svc.TABS_MODEL_COMPARISON = "MODEL_COMPARISON";
    svc.TABS_RECIPE = "RECIPE";
    svc.TABS_SQL_NOTEBOOK = "SQL_NOTEBOOK";
    svc.TABS_SEARCH_NOTEBOOK = "SEARCH_NOTEBOOK";
    svc.TABS_JUPYTER_NOTEBOOK = "JUPYTER_NOTEBOOK";
    svc.TABS_DASHBOARD = "DASHBOARD";
    svc.TABS_INSIGHT = "INSIGHT";
    svc.TABS_JOB = "JOB";
    svc.TABS_CONTINUOUS_ACTIVITY = "CONTINUOUS_ACTIVITY";
    svc.TABS_MANAGED_FOLDER = "MANAGED_FOLDER";
    svc.TABS_LAMBDA = "LAMBDA";
    svc.TABS_SCENARIO = "SCENARIO";
    svc.TABS_CODE_STUDIO = "CODE_STUDIO";
    svc.TABS_MONITORING = "MONITORING";
    svc.TABS_RUNNABLE = "RUNNABLE";
    svc.TABS_LABELING_TASK = "LABELING_TASK";
    svc.TABS_UNIFIED_MONITORING = "UNIFIED_MONITORING";


    svc.refreshPageTitle = function() {
        let pn = "";
        if ($rootScope.totalUnreadNotifications && $rootScope.totalUnreadNotifications > 0) {
            pn = "(" + $rootScope.totalUnreadNotifications + ") "
        }
        let it = "";
        if ($rootScope.appConfig && $rootScope.appConfig.isAutomation) {
            it = " (Automation)";
        }
        document.title = pn + currentPageTitle + " | Dataiku" + it;
    };

    /**
     * top = which universe is highlighted in the global nav (+color for item icon)
     * left = which "sub-universe" is active (e.g. Project Home > Settings)
     * tabsType = which tabs to show on the right on secondary nav
     * tab = which tab is active on the right on secondary nav
     */
    svc.setLocation = function setLocation(top, left, tabsType, tab) {
        $rootScope.topNav.top = top;
        $rootScope.topNav.left = left;
        $rootScope.topNav.tabsType = tabsType;
        $rootScope.topNav.tab = tab;

        Logger.debug("Set location to ", $rootScope.topNav);
    };

    svc.setTopLocation = function(top) {
        $rootScope.topNav.top = top;
    }

    svc.setProjectData = function(projectSummary, projectCurrentBranch) {
        $rootScope.topNav.project = projectSummary;
        $rootScope.topNav.projectCurrentBranch = (projectCurrentBranch && projectCurrentBranch !== "master" && projectCurrentBranch !== "main") ? projectCurrentBranch : "";
    };

    svc.setItem = function(type, id, data) {
        Logger.debug("Set item", type, id, data);
        const oldItem = $rootScope.topNav.item;
        const newItem = {
            type: type,
            id: id,
            data: data
        }
        const same = svc.sameItem(oldItem, newItem);
        $rootScope.topNav.item = newItem;
        // If we change item and don't have data yet, show "Loading..." state
        if (type && !same && !data) {
            $rootScope.topNav.item.data = {name: "Loading ...", loading: true };
        }
    };

    svc.getItem = function() {
        return $rootScope.topNav.item;
    };

    svc.setNoItem = function() {
        svc.setItem(null, null, null);
    };

    // Only changes the tab, nothing elses
    svc.setTab = function(tab) {
        $rootScope.topNav.tab = tab;
    };

    svc.sameItem = function(item1, item2) {
        return item1 && item2 && (getItemKey(item1) == getItemKey(item2));
    };

    svc.sameItem = sameItem;
});


app.factory("StateUtils", function($state, $stateParams, $filter, $rootScope, objectTypeFromNodeFlowType, SmartId,
    ActiveProjectKey, FullModelLikeIdUtils, LoggerProvider) {

    const logger = LoggerProvider.getLogger('common.nav.state.utils');

    function makeStateService(handlingFunction) {
        const that = {
            project: function(projectKey, options = {}) {
                const page = options.page || 'home.regular';
                return handlingFunction('projects.project.' + page, {
                    projectKey: projectKey,
                    discussionId: options.discussionId,
                    selectedTab: options.selectedTab
                });
            },
            pinboard: function(projectKey) {
                return handlingFunction('projects.project.pinboard', {
                    projectKey: projectKey
                });
            },
            dataset: function(datasetName, contextProject, options = {}) {
                contextProject = contextProject || ActiveProjectKey.get();
                const ref = SmartId.resolve(datasetName, contextProject);
                const tab = options.tab || 'explore';

                if (contextProject && ref.projectKey != contextProject && !options.moveToTargetProject) {
                    return handlingFunction('projects.project.foreigndatasets.dataset.' + tab, {
                        datasetFullName: datasetName,
                        projectKey: contextProject,
                        discussionId: options.discussionId,
                        chartId: options.chartId,
                    });
                } else {
                    return handlingFunction('projects.project.datasets.dataset.' + tab, {
                        projectKey: ref.projectKey,
                        datasetName: ref.id,
                        discussionId: options.discussionId,
                        chartId: options.chartId,
                    });
                }
            },
            managedFolder: function(id, projectKey, contextProject, options = {}) {
                //Note that foreign dataset view is not implemented
                contextProject = contextProject || ActiveProjectKey.get();
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'view'

                if (contextProject && ref.projectKey != contextProject && !options.moveToTargetProject) {
                    return handlingFunction('projects.project.foreignmanagedfolders.managedfolder.' + tab, {
                        projectKey: contextProject,
                        sourceProjectKey: ref.projectKey,
                        odbId: ref.id,
                        discussionId: options.discussionId
                    });
                } else {
                    return handlingFunction('projects.project.managedfolders.managedfolder.' + tab, {
                        projectKey: ref.projectKey,
                        odbId: ref.id,
                        discussionId: options.discussionId
                    });
                }
            },
            streamingEndpoint: function(id, projectKey, contextProject, options = {}) {
                //Note that foreign streaming endpoint view is not implemented
                contextProject = contextProject || ActiveProjectKey.get();
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'settings' // TODO:

                if (contextProject && ref.projectKey != contextProject && !options.moveToTargetProject) {
                    return handlingFunction('projects.project.foreignstreaming-endpoints.streaming-endpoint.' + tab, {
                        projectKey: contextProject,
                        sourceProjectKey: ref.projectKey,
                        streamingEndpointId: ref.id,
                        discussionId: options.discussionId
                    });
                } else {
                    return handlingFunction('projects.project.streaming-endpoints.streaming-endpoint.' + tab, {
                        projectKey: ref.projectKey,
                        streamingEndpointId: ref.id,
                        discussionId: options.discussionId
                    });
                }
            },
            savedModel: function(id, projectKey, options = {}) {
                //Note that foreign saved model view is not implemented
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'versions';
                return handlingFunction('projects.project.savedmodels.savedmodel.' + tab, {
                    projectKey: ref.projectKey,
                    smId: ref.id,
                    discussionId: options.discussionId
                });
            },
            savedModelVersion: function(taskType, smId, fullModelId, projectKey, savedModelType, options = {}) {
                const tab = options.tab || '';
                let state;
                if (savedModelType === "PYTHON_AGENT" || savedModelType === "PLUGIN_AGENT" || savedModelType === "TOOLS_USING_AGENT") {
                    state = 'projects.project.savedmodels.savedmodel.agent.design';
                } else if (savedModelType === "LLM_GENERIC") {
                    state = 'projects.project.savedmodels.savedmodel.llmGeneric.report';
                }  else if (savedModelType === "RETRIEVAL_AUGMENTED_LLM") {
                    state = 'projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design';
                } else {
                    state = 'projects.project.savedmodels.savedmodel.' +  taskType.toLowerCase() + '.report';
                }
                return handlingFunction(state + tab, { projectKey: projectKey || ActiveProjectKey.get(), smId, fullModelId });
            },
            modelEvaluationStore: function(id, projectKey, options = {}) {
                //Note that foreign model evaluation store view is not implemented
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction('projects.project.modelevaluationstores.modelevaluationstore.evaluations', {
                    projectKey: ref.projectKey,
                    mesId: ref.id,
                    discussionId: options.discussionId
                });
            },
            retrievableKnowledge: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction('projects.project.retrievableknowledges.retrievableknowledge.usage', {
                    projectKey: ref.projectKey,
                    retrievableKnowledgeId: ref.id,
                    discussionId: options.discussionId
                });
            },
            modelComparison: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction('projects.project.modelcomparisons.modelcomparison.summary', {
                    projectKey: ref.projectKey,
                    modelComparisonId: ref.id,
                    discussionId: options.discussionId
                });
            },
            modelEvaluation: function(mesId, evaluationId, projectKey, isLLM) {
                const ref = SmartId.resolve(mesId, projectKey);
                const subRoute = isLLM ? 'evaluation.report.llm-evaluation-summary' : 'evaluation.report';
                const route = 'projects.project.modelevaluationstores.modelevaluationstore.' + subRoute;

                return handlingFunction(route, {
                    projectKey: ref.projectKey,
                    mesId: ref.id,
                    evaluationId: evaluationId
                });
            },
            promptStudio: function(promptStudioId, projectKey, options = {}) {
                const ref = SmartId.resolve(promptStudioId, projectKey);
                return handlingFunction('projects.project.promptstudios.promptstudio', {
                    projectKey: ref.projectKey,
                    promptStudioId: ref.id,
                    discussionId: options.discussionId
                });
            },
            agentTool: function(agentToolId, projectKey, options = {}) {
                const ref = SmartId.resolve(agentToolId, projectKey);
                return handlingFunction('projects.project.agenttools.agenttool', {
                    projectKey: ref.projectKey,
                    agentToolId: ref.id,
                    discussionId: options.discussionId
                });
            },
            projectStandards: function(checkId) {
                return handlingFunction('admin.projectstandards.checks', {checkId});
            },
            connection: function(connectionName) {
                return handlingFunction('admin.connections.edit', {
                    connectionName: connectionName,
                });
            },
            labelingTask: function(labelingTask, projectKey, options = {}) {
                const ref = SmartId.resolve(labelingTask, projectKey);
                return handlingFunction('projects.project.labelingtasks.labelingtask', {
                    projectKey: ref.projectKey,
                    labelingTaskId: ref.id,
                    discussionId: options.discussionId
                });
            },
            recipe: function(recipeName, projectKey, options = {}) {
                const ref = SmartId.resolve(recipeName, projectKey);
                return handlingFunction('projects.project.recipes.recipe', {
                    projectKey: ref.projectKey,
                    recipeName: ref.id,
                    discussionId: options.discussionId
                });
            },
            analysis: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'script';
                return handlingFunction('projects.project.analyses.analysis.' + tab, {
                    projectKey: ref.projectKey,
                    analysisId: ref.id,
                    discussionId: options.discussionId
                });
            },
            //deprecated use analysis
            analysisChart: function(chartId, analysisId, projectKey) {
                return handlingFunction('projects.project.analyses.analysis.charts', {
                    chartId,
                    analysisId: analysisId,
                    projectKey: projectKey || ActiveProjectKey.get()
                });
            },
            mlTask: function(id, projectKey, mlTaskType, mlTaskId) {
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction(
                    'projects.project.analyses.analysis.ml.' + (mlTaskType.toUpperCase() === 'CLUSTERING' ? 'clustmltask' : 'predmltask') + '.list.results',
                    {
                        projectKey: ref.projectKey,
                        analysisId: ref.id,
                        mlTaskId: mlTaskId
                    }
                );
            },
            analysisModelReport: function(mlTaskType, projectKey, analysisId, mlTaskId, fullModelId) {
                return handlingFunction(
                    'projects.project.analyses.analysis.ml.' + (mlTaskType.toUpperCase() === 'CLUSTERING' ? 'clustmltask' : 'predmltask') + '.model.report',
                    {
                        projectKey: projectKey,
                        analysisId: analysisId,
                        mlTaskId: mlTaskId,
                        fullModelId: fullModelId
                    }
                );
            },
            sqlNotebook: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction('projects.project.notebooks.sql_notebook', {
                    projectKey: ref.projectKey,
                    notebookId: ref.id,
                    discussionId: options.discussionId,
                    cellId: options.cellId
                });
            },
            searchNotebook: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction('projects.project.notebooks.search_notebook', {
                    projectKey: ref.projectKey,
                    notebookId: ref.id,
                    discussionId: options.discussionId
                });
            },
            jupyterNotebook: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                return handlingFunction('projects.project.notebooks.jupyter_notebook', {
                    projectKey: ref.projectKey,
                    notebookId: ref.id,
                    discussionId: options.discussionId
                });
            },
            notebook: function(notebookType, notebookId, projectKey, options = {}) {
                if (notebookType.toUpperCase().includes('SQL')) {
                    return that.sqlNotebook(notebookId, projectKey, options);
                } else if (notebookType.toLowerCase() === 'search') {
                    return that.searchNotebook(notebookId, projectKey, options);
                }
                return that.jupyterNotebook(notebookId, projectKey, options);
            },
            webapp: function(webAppId, projectKey, options = {}) {
                const ref = SmartId.resolve(webAppId, projectKey);
                const tab = options.tab || 'view';
                return handlingFunction("projects.project.webapps.webapp." + tab, {
                    projectKey: ref.projectKey || ActiveProjectKey.get(),
                    webAppId: ref.id,
                    webAppName: options.name,
                    discussionId: options.discussionId
                });
            },
            codeStudio: function(codeStudioObjectId, projectKey, options = {}) {
                const ref = SmartId.resolve(codeStudioObjectId, projectKey);
                const tab = options.tab || 'view';
                return handlingFunction("projects.project.code-studios.code-studio." + tab, {
                    projectKey: ref.projectKey || ActiveProjectKey.get(),
                    codeStudioObjectId: ref.id,
                    codeStudioObjectName: options.name,
                    discussionId: options.discussionId
                });
            },
            codeStudioTemplate: function(codeStudioTemplateId) {
                return handlingFunction("admin.code-studios.code-studio", {
                    codeStudioTemplateId: codeStudioTemplateId
                });
            },
            plugin: function(pluginId) {
                return handlingFunction("plugin.summary", {
                    pluginId: pluginId
                });
            },
            report: function(reportId, projectKey, options = {}) {
                const ref = SmartId.resolve(reportId, projectKey);
                const tab = options.tab || 'view';
                return handlingFunction("projects.project.reports.report." + tab, {
                    projectKey: ref.projectKey || ActiveProjectKey.get(),
                    reportId: ref.id,
                    reportName: options.name,
                    discussionId: options.discussionId
                });
            },
            scenario: function(scenarioId, projectKey, options = {}) {
                const ref = SmartId.resolve(scenarioId, projectKey);
                const tab = options.tab || 'runs.list';
                return handlingFunction("projects.project.scenarios.scenario." + tab, {
                    projectKey: ref.projectKey || ActiveProjectKey.get(),
                    scenarioId: ref.id,
                    discussionId: options.discussionId,
                    runId: options.runId
                });
            },

            dashboard: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'view';
                return handlingFunction('projects.project.dashboards.dashboard.' + tab, {
                    projectKey: ref.projectKey,
                    dashboardId: ref.id,
                    dashboardName: options.name ? $filter('slugify')(options.name) : '',
                    pageId: options.pageId ? options.pageId : '',
                    fullScreen: options.fullScreen,
                    separator: '_',
                    discussionId: options.discussionId,
                    filters: options.filters
                });
            },
            insight: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'view';
                return handlingFunction('projects.project.dashboards.insights.insight.' + tab, {
                    projectKey: ref.projectKey,
                    insightId: ref.id,
                    insightName: options.name ? $filter('slugify')(options.name) : '',
                    fullScreen: options.fullScreen,
                    discussionId: options.discussionId,
                    originDashboard: options.originDashboard,
                    tabSelect: options.tabSelect,
                    sections: options.sections
                });
            },
            lambdaService: function(id, projectKey, options = {}) {
                const ref = SmartId.resolve(id, projectKey);
                const tab = options.tab || 'endpoints';
                return handlingFunction('projects.project.lambdaservices.service.' + tab, {
                    projectKey: ref.projectKey,
                    serviceId: ref.id,
                    discussionId: options.discussionId
                });
            },
            article: function(articleId, projectKey, options = {}) {
                const ref = SmartId.resolve(articleId, projectKey);
                const tab = options.tab || 'view';
                return handlingFunction("projects.project.wiki.article." + tab, {
                    projectKey: ref.projectKey || ActiveProjectKey.get(),
                    articleId: ref.id,
                    articleName: options.articleName,
                    discussionId: options.discussionId,
                    '#': ''
                });
            },
            cluster: function(clusterId, options = {}) {
                return handlingFunction('admin.clusters.cluster', {
                    clusterId: clusterId,
                    discussionId: options.discussionId
                });
            },
            statisticsWorksheet: function(worksheetId, contextProject) {
                const ref = SmartId.resolve(worksheetId, contextProject);
                return handlingFunction('projects.project.statisticsWorksheet', {
                    projectKey: ref.projectKey,
                    worksheetId: ref.id
                });
            },
            flowZone: function(id, contextProject, options = {}) {
                contextProject = contextProject || ActiveProjectKey.get();
                const ref = SmartId.resolve(id, contextProject);

                return handlingFunction('projects.project.flow', {
                    projectKey: ref.projectKey,
                    zoneId: ref.id,
                });
            },
            continuousActivity: function(id, contextProject, options = {}) {
                contextProject = contextProject || ActiveProjectKey.get();
                const ref = SmartId.resolve(id, contextProject);

                return handlingFunction('projects.project.continuous-activities.continuous-activity.runs', {
                    projectKey: ref.projectKey,
                    continuousActivityId: ref.id,
                });
            },
            chart: function(objectType, projectKey, ownerName, index) {
                if (objectType === 'dataset') {
                    return handlingFunction('projects.project.datasets.dataset.visualize', {
                        projectKey: projectKey,
                        datasetName: ownerName,
                        chartIdx: index
                    });
                } else if (objectType === 'analysis') {
                    return handlingFunction('projects.project.analyses.analysis.charts', {
                        projectKey: projectKey,
                        analysisId: ownerName,
                        chartIdx: index
                    });
                } else {
                    return null;
                }
            },
            /* for all taggable types */
            dssObject: function(type, id, projectKey, options = {}) { //if moveToTargetProject is falsish, explore as foreign object
                projectKey = projectKey || ActiveProjectKey.get();
                switch (type) {
                    case 'DATASET':
                        return that.dataset(id, projectKey, options);
                    case 'SAVED_MODEL':
                        return that.savedModel(id, projectKey, options);
                    case 'MODEL_EVALUATION_STORE':
                        return that.modelEvaluationStore(id, projectKey, options);
                    case 'MODEL_COMPARISON':
                        return that.modelComparison(id, projectKey, options);
                    case 'MANAGED_FOLDER':
                        return that.managedFolder(id, projectKey, null, options);
                    case 'STREAMING_ENDPOINT':
                        return that.streamingEndpoint(id, projectKey, null, options);
                    case 'RECIPE':
                        return that.recipe(id, projectKey, options);
                    case 'ANALYSIS':
                        if (options.mlTaskId && options.subType) {
                            return that.mlTask(id, projectKey, options.subType, options.mlTaskId);
                        }
                        return that.analysis(id, projectKey, options);
                    case 'STATISTICS_WORKSHEET':
                        return that.statisticsWorksheet(id, projectKey, options);
                    case 'SQL_NOTEBOOK':
                        return that.sqlNotebook(id, projectKey, options);
                    case 'SEARCH_NOTEBOOK':
                        return that.searchNotebook(id, projectKey, options);
                    case 'JUPYTER_NOTEBOOK':
                        return that.jupyterNotebook(id, projectKey, options);
                    case 'INSIGHT':
                        return that.insight(id, projectKey, options);
                    case 'WEB_APP':
                        return that.webapp(id, projectKey, options);
                    case 'CODE_STUDIO':
                        return that.codeStudio(id, projectKey, options);
                    case 'REPORT':
                        return that.report(id, projectKey, options);
                    case 'ARTICLE':
                        return that.article(id, projectKey, options);
                    case 'SCENARIO':
                        return that.scenario(id, projectKey, options);
                    case 'DASHBOARD':
                        return that.dashboard(id, projectKey, options);
                    case 'PROJECT':
                        return that.project(id || projectKey, options);
                    case 'APP':
                        return that.app(id);
                    case 'LAMBDA_SERVICE':
                        return that.lambdaService(id, projectKey, options);
                    case 'CLUSTER':
                        return that.cluster(id, options);
                    case 'JOB':
                        return that.job(projectKey, id);
                    case 'FLOW_ZONE':
                        return that.flowZone(id, projectKey, options);
                    case 'CONTINUOUS_ACTIVITY':
                        return that.continuousActivity(id, projectKey, options);
                    case 'LABELING_TASK':
                        return that.labelingTask(id, projectKey, options);
                    case 'DATA_COLLECTION':
                        return that.dataCollection(id, options);
                    case 'PROMPT_STUDIO':
                        return that.promptStudio(id, projectKey, options);
                    case 'AGENT_TOOL':
                        return that.agentTool(id, projectKey, options);
                    case 'PROJECT_STANDARDS_CHECK':
                        return that.projectStandards(id);
                    case 'RETRIEVABLE_KNOWLEDGE':
                        return that.retrievableKnowledge(id, projectKey, options);
                    case 'CONNECTION':
                        return that.connection(id);
                    case 'WORKSPACE':
                        return that.workspace(id);
                }
                throw new Error("Unknown object type: '" + type + "' for " + projectKey + '.' + id);
            },
            taggableObject: function(tor, options = {}) { //if moveToTargetProject is falsish, explore as foreign object
                if (tor.workspaceKey) {
                    if (tor.projectKey) {
                        return that.workspaceObject(tor.workspaceKey, tor.projectKey, tor.type, tor.id, options);
                    }
                    return that.workspace(tor.workspaceKey, options);
                }
                return that.dssObject(tor.type, tor.id, tor.projectKey, options);
            },
            table: function(connection, catalog, schema, table) {
                return handlingFunction(
                    'external-table',
                    { connection, catalog, schema, table }
                );
            },
            node: function(flowNode) {
                if (!flowNode) return;
                switch (flowNode.nodeType) {
                    case 'LOCAL_DATASET':
                        return that.dataset(flowNode.name, flowNode.projectKey);
                    case 'FOREIGN_DATASET':
                        return that.dataset(flowNode.projectKey + '.' + flowNode.name, ActiveProjectKey.get());
                    case 'LOCAL_SAVEDMODEL':
                    case 'FOREIGN_SAVEDMODEL':
                        return that.savedModel(flowNode.name, flowNode.projectKey);
                    case 'LOCAL_MODELEVALUATIONSTORE':
                    case 'FOREIGN_MODELEVALUATIONSTORE':
                        return that.modelEvaluationStore(flowNode.name, flowNode.projectKey);
                    case 'LOCAL_RETRIEVABLE_KNOWLEDGE':
                    case 'FOREIGN_RETRIEVABLE_KNOWLEDGE':
                        return that.retrievableKnowledge(flowNode.name, flowNode.projectKey);
                    case 'LOCAL_MANAGED_FOLDER':
                    case 'FOREIGN_MANAGED_FOLDER':
                        return that.managedFolder(flowNode.name, flowNode.projectKey);
                    case "LOCAL_STREAMING_ENDPOINT":
                    case "FOREIGN_STREAMING_ENDPOINT":
                        return that.streamingEndpoint(flowNode.name, flowNode.projectKey);
                    case "LABELING_TASK":
                        return that.labelingTask(flowNode.name, flowNode.projectKey);
                    case 'RECIPE':
                        return that.recipe(flowNode.name, flowNode.projectKey);
                    case 'ZONE':
                        return that.flowZone(flowNode.name, flowNode.projectKey);
                    case 'JUPYTER_NOTEBOOK':
                        return that.jupyterNotebook(flowNode.id, flowNode.projectKey);
                    case 'SQL_NOTEBOOK':
                        return that.sqlNotebook(flowNode.id, flowNode.projectKey);
                    case 'SEARCH_NOTEBOOK':
                        return that.searchNotebook(flowNode.id, flowNode.projectKey);
                    default:
                        logger.error('Unknown Flow node nodeType: ' + flowNode.nodeType);
                }
            },
            // contextProjectKey is the project the foreign object currently is in
            flowLink: function(flowNode, contextProjectKey) {
                const type = objectTypeFromNodeFlowType(flowNode.nodeType).toLowerCase();
                return that.flowLinkFromProps(type, flowNode.projectKey, flowNode.name, contextProjectKey);
            },
            // contextProjectKey is the project the foreign object currently is in
            flowLinkFromProps: function(objectType, objectProjectKey, objectId, contextProjectKey) {
                let id;
                switch (objectType.toLowerCase()) {
                    case 'dataset':
                        id = "dataset_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'managed_folder':
                        id = "managedfolder_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'model_evaluation_store':
                        id = "modelevaluationstore_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'saved_model':
                        id = "savedmodel_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'retrievable_knowledge':
                        id = "retrievableknowledge_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'streaming_endpoint':
                        id = "streamingendpoint_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'labeling_task':
                        id = "labelingtask_" + objectProjectKey + '.' + objectId;
                        break;
                    case 'recipe':
                        id = "recipe_" + objectId;
                        break;
                }

                return handlingFunction('projects.project.flow', {
                    id,
                    projectKey: contextProjectKey || objectProjectKey
                });
            },
            job: function(projectKey, jobId) {
                return handlingFunction("projects.project.jobs.job", {
                    projectKey : projectKey || ActiveProjectKey.get(),
                    jobId : jobId
                });
            },
            home: function() {
                return handlingFunction("home", {});
            },
            workspaces: function() {
                return handlingFunction("workspaces.home", {});
            },
            workspace: function(workspaceKey, options = {}) {
                return handlingFunction("workspaces.workspace", {
                    workspaceKey: workspaceKey,
                    discussionId: options.discussionId,
                });
            },
            workspaceObject: function(workspaceKey, projectKey, objectType, objectId, options = {}) {
                return handlingFunction("workspaces.object", {
                    workspaceKey,
                    projectKey,
                    objectType,
                    objectId,
                    discussionId: options.discussionId,
                });
            },
            workspaceDashboardInsight: function(workspaceKey, projectKey, dashboardId, insightId) {
                return handlingFunction("workspaces.object.insight.view", {
                    workspaceKey,
                    projectKey,
                    objectType: 'DASHBOARD',
                    objectId: dashboardId,
                    insightId,
                });
            },
            workspaceApp: function(workspaceKey, appId) {
                return handlingFunction("workspaces.app", {
                    workspaceKey,
                    appId
                })
            },
            app: function(appId) {
                let appRef = SmartId.resolve(appId);
                return handlingFunction("apps.app", {
                    appId: appRef.id
                });
            },
            projectFolder: function(folderId) {
                // TODO @homepage cleanup one migration done
                if($rootScope.featureFlagEnabled('homepageRedesign')) {
                    return handlingFunction("homeV2.projects.folder", {
                        folderId: folderId === 'ROOT' ? '' : folderId
                    });
                } else {
                    return handlingFunction("project-list", {
                        folderId: folderId === 'ROOT' ? '' : folderId
                    });
                }

            },
            inboxRequest: function(requestId) {
                return handlingFunction("inbox.requests.selected", {
                    requestId: requestId
                });
            },
            pluginSummary: function(pluginId) {
                return handlingFunction('plugin.summary', {
                    pluginId: pluginId
                });
            },
            pluginStore: function(pluginId) {
                return handlingFunction('plugins.store', {
                    pluginid: pluginId
                });
            },
            pluginDefinition: function(pluginId) {
                return handlingFunction('plugindev.definition', {
                    pluginId: pluginId
                });
            },
            pluginEditor: function(pluginId, path) {
                return handlingFunction('plugindev.editor', {
                    pluginId: pluginId,
                    filePath: path
                });
            },
            projectLibEditor: function(projectKey, path) {
                return handlingFunction('projects.project.libedition.versioned', {
                    projectKey: projectKey,
                    initialPath: path
                });
            },
            globalLibEditor: function(path) {
                return handlingFunction('libedition.libpython', {
                    initialPath: path
                });
            },
            codeEnvCreation: function(id) {
                return handlingFunction('admin.codeenvs-design.create', {
                    draftId: id
                });
            },
            codeEnvEdit: function(envName, envLang) {
                if (envLang === "PYTHON") {
                    return handlingFunction('admin.codeenvs-design.python-edit', {
                        envName: envName
                    });
                } else if (envLang === "R") {
                    return handlingFunction('admin.codeenvs-design.r-edit', {
                        envName: envName
                    });
                } else {
                    logger.error(`Unknown envLang: ${envLang}`);
                    return;
                }
            },
            fullModelLikeId: function(fullId, mlTaskType = 'PREDICTION', isLLM) {
                const item = FullModelLikeIdUtils.parse(fullId);
                if (fullId.startsWith("ME-")) {
                    return that.modelEvaluation(item.id, item.evaluationId, item.projectKey, isLLM);
                } else if (fullId.startsWith("S-")) {
                    return that.savedModelVersion(mlTaskType, item.savedModelId, fullId, item.projectKey);
                } else if (fullId.startsWith("A-")) {
                    return that.analysisModelReport(mlTaskType, item.projectKey, item.analysisId, item.mlTaskId, fullId);
                }
                throw new Error("Unexpected type");
            },
            dataCollection: function(dataCollectionId, fromProject = false) {
                // when using fromProject, You MUST already be in the project (it's not valid to go directly to an in-project view from outside)
                if (fromProject) {
                    return handlingFunction('projects.project.datacatalog.datacollections.datacollection', {dataCollectionId: dataCollectionId})
                } else {
                    return handlingFunction('homeV2.data-catalog.data-collections.data-collection', {dataCollectionId: dataCollectionId})
                }
            },
            apiDeployer: {
                deployment: function(deploymentId, options = {}) {
                    const tab = options.tab || 'status';
                    return handlingFunction('apideployer.deployments.deployment.' + tab, {
                        deploymentId: deploymentId
                    });
                },
                serviceVersions: function(serviceId, versions) {
                    return handlingFunction('apideployer.services.service.status', {
                        serviceId: serviceId,
                        versions: versions
                    });
                }
            },
            projectDeployer: {
                bundle: function(projectKey, bundleId) {
                    return handlingFunction('projectdeployer.projects.project.bundle.status', {
                        publishedProjectKey: projectKey,
                        bundleId: bundleId
                    });
                }
            }
        };
        return that;
    }

    function getDefaultTab(alternate) {
        const scd = $state.current.data;
        return scd && scd.tab ? scd.tab : alternate;
    }

    return {
        href: makeStateService($state.href.bind($state)),
        go: makeStateService($state.go.bind($state)),
        uiSref: makeStateService((ref, params) => (`${ref}(${JSON.stringify(params)})`)),
        // Added to pass options to the underlying $state.go call, such as {reload: true}
        goWithOptions: (options) => makeStateService((to, params) => {
            $state.go(to, params, options)
        }),
        defaultTab: getDefaultTab
    };
});


app.directive("stdObjectBreadcrumb", function($rootScope, $state, Navigator, DatasetCustomFieldsService, translate) {
    return {
        templateUrl: '/templates/widgets/std-object-breadcrumb.html',
        scope: {
            jobDef: "=jobDef",
            jobStatus: "=jobStatus",
            projectAppView: '<?', // for projects / app instances, says if it's a project view or app view
        },
        link: function(scope) {
            scope.topNav = $rootScope.topNav;
            scope.$state = $state;
            scope.Navigator = Navigator;
            scope.DatasetCustomFieldsService = DatasetCustomFieldsService;
            scope.translate = translate;
        }
    };
});

app.component('computableParentLink', {
    bindings: {
        computableFullInfo: '<',
    },
    controller: function ctrlComputableParentLink($state) {
        this.$state = $state;
    },
    templateUrl: '/templates/widgets/computable-parent-link.html'
});

app.component('copyToClipboardIcon', {
    bindings: {
        name: '<',
        value: '<',
        tooltipPosition: '<',
        iconClass: '<',
        iconSize: '<',
        disabled: '<',
    },
    controller: function($scope, $timeout, Logger, translate, ClipboardUtils){
        const ctrl = this;

        ctrl.$onInit = function () {
            ctrl.copied = false;
            ctrl.error = false;
            ctrl.iconSize = ctrl.iconSize || '16';
            ctrl.name = (ctrl.name !== undefined && ctrl.name !== null) ? ctrl.name : translate("COPY_TO_CLIPBOARD_ICON.NAME", "name");
            $scope.translate = translate;
        }

        ctrl.copyContent = function() {
            if (ctrl.disabled) return;

            ClipboardUtils.copyToClipboard(ctrl.value, '', '').then(() => {
                ctrl.copied = true;
                ctrl.inFlightTimeout = $timeout(() => {
                    ctrl.copied = false;
                    $scope.$apply();
                }, 2000);
                $scope.$apply();
            }, (exception) => {
                Logger.error("Failed to copy to clipboard", exception);
                ctrl.error = true;
                ctrl.inFlightTimeout = $timeout(() => {
                    ctrl.error = false;
                    $scope.$apply();
                }, 10000);
                $scope.$apply();
            });
        }

        ctrl.$onChanges = function() {
            if (ctrl.inFlightTimeout) {
                $timeout.cancel(ctrl.inFlightTimeout);
            }
            ctrl.copied = false;
            ctrl.error = false;
        }
    },
    templateUrl: '/templates/widgets/copy-to-clipboard-icon.html'
});


app.directive("itemHeader", function() {
    return {
        scope: {
            item: '=',
            href: '=',
            color: '@',
            icon: '@',
            title: '@',
            class: '@',
            flowLink: '=?',
            exposeObjectFn: '=?',
            exposeIcon: '=?',
            exposeLabel: '=?',
            exposeDisabled: '=?',
            importLbl: '=?',
            navigatorFn: '=?',
            editable: '=?',
            edit: '&?',
            deletable: '=?',
            delete: '&?',
            copyToClipboardValue: '@?'
        },
        transclude: true,
        template:
            `<div class="{{class}} item-header horizontal-flex aic">
                <div class="noflex object-icon universe-background {{color}}" style="margin-bottom:0">
                    <div class="middle"><i class="icon {{icon}}"></i></div>
                </div>
                <h2 class="flex" title="{{title}}">
                    <a href="{{href}}" ng-if="href"><ng-transclude></ng-transclude></a>
                   <span ng-if="!href"><ng-transclude></ng-transclude></span>
                   <copy-to-clipboard-icon ng-if="copyToClipboardValue" value="copyToClipboardValue" class="mleft4" />
                </h2>
                <div class="btn-items">
                    <span title="{{exposeLabel}}" toggle="tooltip"><button disabled-if='exposeDisabled' ng-if="exposeObjectFn" ng-click="exposeObjectFn()" class="btn btn--secondary" container="button"><i class="{{exposeIcon || 'icon-dku-share'}}"></i> {{importLbl || "Use"}}</button></span>
                    <button ng-if="navigatorFn" ng-click="navigatorFn()" class="btn btn--secondary btn--icon" alt="Navigate around" title="Navigate around">
                        <i class="icon-compass"></i>
                    </button>
                    <a ng-if="flowLink" class="btn btn--secondary btn--icon" href="{{flowLink}}" alt="See in flow" title="See in flow">
                        <i class="icon-dku-nav_flow"></i>
                    </a>
                    <button ng-if="editable" ng-click="edit()" class="btn btn--secondary btn--icon" alt="Edit">
                        <i class="icon-pencil"></i>
                    </button>
                    <button ng-if="deletable" ng-click="delete()" class="btn btn--secondary btn--icon" alt="Delete">
                        <i class="icon-trash"></i>
                    </button>
                </div>
            </div>`,
        link: function(scope, element, attrs, ctrl, transclude) {
          //NOSONAR ng1.6 doesnt work; doesnt appear necessary probably due to https://github.com/angular/angular.js/commit/32aa7e7395527624119e3917c54ee43b4d219301 //element.find('ng-transclude').replaceWith(transclude());
        }
    };
});


app.directive("simpleRightColActionHref", function($rootScope) {
    return {
        scope: {
            href: '@',
            label: '@',
            icon: '@',
            title: '@',
            target: '@'
        },
        replace: true,
        template:   `<div class="action-icon" role="button">
                        <a href="{{href}}" target="{{target}}">
                            <i class="{{icon}}"></i>
                            <label>{{label}}</label>
                        </a>
                    </div>`
    };
});


app.directive("simpleRightColActionClick", function($rootScope) {
    return {
        scope: {
            onClick : '&',
            label : '@',
            icon : '@',
            title : '@'
        },
        replace: true,
        template:   `<div class="action-icon" ng-click="onClick()" role="button">
                        <i class="{{icon}}"></i>
                        <label>{{label}}</label>
                    </div>`
    };
});

app.component("simpleRightColShareButton", { // simple-right-col-share-button
        bindings: {
            // for conditions that should hide the button regardless of the share capability of the user (example this is a foreign dataset)
            // use ng-if on the component to make conditions simpler and less redundant.

            // Share options
            canShare: '<', // will show the share button
            shareToProject: '&',
            // Request share options
            canRequestShare: '<', // will show the request share button if canShare is false
            requestShareToProject:'&',
            // Disabled state
            showIfDisabled: '<', // show the share button, but disabled it neither share nor request share is possible
            disabledTooltip : '@',
            // General
            enableWt1ClickIds: '<',
            label: '<' // Not actually necessary for the component by itself but used by the ellipsedList
        },
        templateUrl: '/templates/right-col-share-button-menu.html',
        controller: function($element){
            const ctrl = this;
            /** Update component wrapper visibility */
            const updateComponentWrapperVisibility = () => {
                if (!ctrl.showShareButton && !ctrl.showRequestAccessButton){
                    $element.addClass('hide');
                } else {
                    $element.removeClass('hide');
                }
                if (ctrl.disableShareButton) {
                    $element.addClass('disabled');
                } else {
                    $element.removeClass('disabled');
                }
            }

            this.$onChanges = function () {
                ctrl.showShareButton = ctrl.canShare || (ctrl.showIfDisabled && !ctrl.canRequestShare);
                ctrl.disableShareButton = !ctrl.canShare && !ctrl.canRequestShare && ctrl.showIfDisabled;
                ctrl.showRequestAccessButton = !ctrl.canShare && ctrl.canRequestShare;
                updateComponentWrapperVisibility();
            }
        }
});

app.directive("clickNext", function($timeout, $stateParams, $state, Dialogs) {
    return {
        scope: false,
        restrict: 'A',
        link: function(scope, element, attrs) {
            const $e = $(element);
            $e.on('click', function(evt) {
                $e.next().trigger(evt);
            });
        }
    };
});

app.component('noAccessScreen', {
    bindings: {
        message: '@'
    },
    template: `<div class="no-access-screen">
                   <i class="icon-4x icon-warning-sign"></i>
                   <span class="no-access-screen__text">{{$ctrl.message || 'You do not have the permission to access this page.'}}</span>
               </div>`
});

app.component('requestAccessTopBar', {
    bindings: {
        accessInfo: '<',
        objectType: '<',
        objectName: '<',
        autoOpenModal: '<',
        objectId: '<',
    },
    templateUrl: '/templates/request-access-top-bar.html',
    controller: function ctrlRequestAccessTopBar($scope, CreateModalFromTemplate, DataikuAPI, RequestCenterService) {
        const ctrl = this;
        ctrl.HAS_REQUEST_STATUS = {
            PENDING: 'PENDING',
            HAS_REQUEST:'HAS_REQUEST',
            NO_REQUEST: 'NO_REQUEST'
        };
        ctrl.pendingRequest = ctrl.HAS_REQUEST_STATUS.PENDING;

        function getObjectProjectKey() {
            switch (ctrl.objectType) {
                case 'PROJECT':
                    return ctrl.objectId;
                case 'APP':
                    if (ctrl.objectId.startsWith("PROJECT_")) {
                        return ctrl.objectId.substring("PROJECT_".length);
                    } else if (ctrl.objectId.startsWith("PLUGIN_")) {
                        return ctrl.objectId.substring("PLUGIN_".length);
                    }
                    break;
                default:
                    throw "Unhandled objectType: " + ctrl.objectType;
            }
        }

        ctrl.getLatestRequestForCurrentUser = function() {
            DataikuAPI.requests.getLatestRequestForCurrentUser(ctrl.objectId, ctrl.objectType, getObjectProjectKey()).then(response => {
                ctrl.request = response.data;
                if(ctrl.request.status === 'PENDING'){
                    ctrl.pendingRequest = ctrl.HAS_REQUEST_STATUS.HAS_REQUEST;
                } else {
                    ctrl.pendingRequest = ctrl.HAS_REQUEST_STATUS.NO_REQUEST;
                    if (ctrl.accessInfo.isAccessRequestsEnabled && ctrl.autoOpenModal) {
                        ctrl.openRequestAccessModal();
                    }
                }
            }, error => {
                if(error.status === 404){
                    ctrl.pendingRequest = ctrl.HAS_REQUEST_STATUS.NO_REQUEST;
                    if (ctrl.accessInfo.isAccessRequestsEnabled && ctrl.autoOpenModal) {
                        ctrl.openRequestAccessModal();
                    }
                } else {
                    setErrorInScope.bind($scope)(error);
                }
            });
        }

        ctrl.openRequestAccessModal = () => {
            CreateModalFromTemplate("/templates/request-access-modal.html", $scope, null, (newScope) => {
                newScope.objectType = ctrl.objectType;
                newScope.objectId = ctrl.objectId;
                newScope.objectName = ctrl.objectName;
                newScope.ui = { message: ""};

                newScope.sendRequest = (requestMessage) => {
                    DataikuAPI.requests.createAccessRequest(ctrl.objectType, getObjectProjectKey(), ctrl.objectId, requestMessage).success((data) => {
                        RequestCenterService.WT1Events.onRequestSent(ctrl.objectType, getObjectProjectKey(), ctrl.objectId, requestMessage, data.id);
                        ctrl.getLatestRequestForCurrentUser();
                    }).error(() => {
                        setErrorInScope.bind($scope);
                    });
                    newScope.dismiss();
                };
            });
        };

        ctrl.$onChanges = (changes) => {
            if(changes.accessInfo) {
                ctrl.actionMessage = ctrl.accessInfo.isAccessRequestsEnabled
                    ? ` Contact the project administrators for access.`
                    : ` Contact the project administrators for more information.`;
            }

            if(changes.objectType || changes.objectId) {
                ctrl.getLatestRequestForCurrentUser();
            }
        }
    }
});

app.directive("tabModel", function($timeout, $stateParams, $state, Dialogs) {
    return {
        scope: false,
        restrict: 'A',
        link: function(scope, element, attrs) {
            const $e = $(element),
                expr = attrs.tabModel + ' = $tab',
                klass = attrs.tabActiveClass || 'active',
                notify = attrs.tabModelNotify === "true" ? true : false,
                disableTransition = attrs.disableTransition === "true" ? true : false;

            function transition(evt, e) {
                const tab = e.getAttribute('tab-set');
                if (disableTransition) {
                    scope.$eval.bind(scope, expr, {$tab: tab})();
                } else {
                    $state.go('.', {selectedTab: tab}, {location: true, inherit: true, relative: $state.$current, notify: notify }).then(function() {
                        scope.$eval.bind(scope, expr, {$tab: tab})();
                    });
                }
            }

            $e.on('click', '[tab-set]', function(evt) {
                const that = this;
                if(scope.hooks && scope.hooks.dirty && scope.hooks.dirty()) {
                    Dialogs.confirm(scope, 'Unsaved changes', "You have unsaved changes. Are you sure you want to leave this page ?")
                        .then( _ => transition(evt, that) );
                } else {
                    scope.$apply( _ => transition(evt, that) );
                }
            });

            let activeTab = attrs.tabModel; // Since `updateActiveTab` is called by `$watch`ers below, it..
            const updateActiveTab = () => { // ..shouldn't rely on `attrs.tabModel`. We use `activeTab` instead.
                $e.find('[tab-active]').each(function() {
                    let validTabs = [this.getAttribute('tab-active')];
                    const alternativeTabs = this.getAttribute('alternative-tabs');
                    if (alternativeTabs) {
                        validTabs = validTabs.concat(JSON.parse(alternativeTabs.replace(/'/g, '"')));
                    }
                    this.classList[validTabs.includes(activeTab) ? 'add' : 'remove'](klass);
                });
            }
            // `attrs.tabModel` may be set before some tabs are loaded.
            // Therefore, we `$watch` both `attrs.tabModel` and the DOM.
            const getTabNames = () => $e.find('[tab-active]').toArray().map(e => e.innerText);
            scope.$watchCollection(getTabNames, updateActiveTab);
            scope.$watch(attrs.tabModel, (newVal, _) => {
                activeTab = newVal;
                updateActiveTab();
            });
        }
    };
});


app.service("QuickView", function(CreateCustomElementFromTemplate, $rootScope, $timeout) {
    const svc = this;

    let elScope, removeListener;

    this.show = function(projectKey, objectType, objectId) {
        if (projectKey === false) return;

        if (!elScope) {
            CreateCustomElementFromTemplate("/templates/object-details/quick-view.html", $rootScope, null, function(newScope) {
                elScope = newScope;
                elScope.hasObject = false;
                elScope.objectType = objectType;
                elScope.objectId = objectId;
                elScope.projectKey = projectKey;
                $timeout(function() { $('.object-quick-view-wrapper').addClass('visible'); });
            });
        } else {
            elScope.hasObject = false;
            elScope.objectType = objectType;
            elScope.objectId = objectId;
            elScope.projectKey = projectKey;
        }

        $('.object-quick-view-wrapper').removeClass('loading');
        removeListener = $rootScope.$on("$stateChangeStart", svc.hide);
    };

    this.showObject = function(object, objectType) {
        if (object === false) return;

        const objectData = {};
        objectData[objectType.toLowerCase()] = object;

        if (!elScope) {
            CreateCustomElementFromTemplate("/templates/object-details/quick-view.html", $rootScope, null, function(newScope) {
                elScope = newScope;
                elScope.hasObject = true;
                elScope.objectType = objectType;
                elScope.objectData = objectData;
                $timeout(function() { $('.object-quick-view-wrapper').addClass('visible'); });
            });
        } else {
            elScope.hasObject = true;
            elScope.objectType = objectType;
            elScope.objectData = objectData;
        }

        $('.object-quick-view-wrapper').removeClass('loading');
        removeListener = $rootScope.$on("$stateChangeStart", svc.hide);
    };

    this.hide = function() {
        if (elScope && elScope.dismiss) {
            elScope.dismiss();
            elScope = null;
        }

        if (removeListener) removeListener();
    };

    this.setLoading = function() {
        $('.object-quick-view-wrapper').addClass('loading');
    };
});

app.directive("objectDetails", function ($rootScope, $q, $timeout, $state, $stateParams, $filter, translate, DataikuAPI, DatasetUtils, ActivityIndicator,
    RecipesUtils, QuickView, NotebooksUtils, StateUtils, LoggerProvider, CreateModalFromTemplate, TaggingService, TAGGABLE_TYPES, ActiveProjectKey,
    _SummaryHelper, ObjectDetailsUtils, SavedModelsService, EditDatasetDataStewardModalService, ProjectStandardsService, TypeMappingService) {
    const logger = LoggerProvider.getLogger('objectDetails');

    return {
        restrict: 'E',
        template: '<div ng-if="objectType" ng-include="getTemplateFile()" />',
        scope: {
            projectKey: '=',
            objectType: '@',
            objectId: '=?',
            data: '=?objectData',
            context: '@', // 'right-column', 'right-column-workspace', 'navigator', 'new-insight'
            hoverIntentCallback: '=?',
            editable: '=?',
            editCustomFields: '=',
            showTopBorder: '<'
        },
        link: function($scope, element, attrs) {
            $scope.translate = translate;
            _SummaryHelper.addEditBehaviour($scope, element);
            _SummaryHelper.addInterestsManagementBehaviour($scope);

            $scope.getObjectId = function() {
             return $scope.object.id || $scope.object.name;
            };

            $scope.uiState = {
                isHoverEdit: false
            };
            $scope.appConfig = $rootScope.appConfig;

            $scope.resolveObjectSmartId = resolveObjectSmartId;

            $scope.QuickView = QuickView;
            $scope.StateUtils = StateUtils;

            $scope.canWriteProject = () => $scope.editable;

            $scope.canReadProject = $rootScope.projectSummary && $rootScope.projectSummary.canReadProjectContent;

            $scope.isProjectAdmin = () => $rootScope.projectSummary && $rootScope.projectSummary.isProjectAdmin;

            $scope.saveCustomFields = function (customFields) {
                $scope.$emit('customFieldsSummaryEdited', customFields);
            };

            $scope.inNavigator = function() {
                return ObjectDetailsUtils.inNavigator($scope.context);
            };

            $scope.setHoverEdit = function (on)  {
                $scope.uiState.isHoverEdit = on;
            };

            $scope.inRightColumn = function() {
                return $scope.context == 'right-column' || $scope.context == 'right-column-workspace';
            };

            $scope.inWorkspace = function() {
                return ObjectDetailsUtils.inWorkspace($scope.context)
            };

            $scope.inQuickView = function() {
                return ObjectDetailsUtils.inQuickView($scope.context)
            };

            if (!$scope.context) $scope.context = '';
            if ($scope.inQuickView()) {
                $scope.maxListItems = 5;
            }

            $scope.getTemplateFile = function() {
                return '/templates/object-details/' + $scope.objectType.toLowerCase() + '.html';
            };

            $scope.getTaggableObject = function() {
                return {
                    type: $scope.objectType.toUpperCase(),
                    projectKey: $scope.object.projectKey,
                    id: $scope.object.id || $scope.object.name,
                    displayName: $scope.object.displayName || $scope.object.name
                }
            };

            $rootScope.$on('toggleActiveRightCol', function(){
                $scope.object.active = !$scope.object.active;
            });

            $scope.toggleActive = function(scenario) {
                var message = scenario.active ? 'Activate ' : 'Deactivate ';
                $rootScope.$emit('toggleActiveList');
                message = message + 'auto-triggers of ' + scenario.projectKey + '.' + (scenario.name || scenario.id);
                DataikuAPI.scenarios.saveNoParams(scenario.projectKey, scenario, {commitMessage:message}).success(function(data){
                    // save the expanded states
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind($scope));
            };

            $scope.isOnEditableTab = function() {
                return $state.current.name == "projects.project.datasets.dataset.edit"
            }

            $scope.isMetaDataEditable = function () {
                return !$scope.inNavigator() && $stateParams.projectKey;
            };

            $scope.isMetaDataSupported = function(){
                return TAGGABLE_TYPES.includes($scope.objectType.toUpperCase());
            }

            $scope.getAllTagsForProject = function () {
                const deferred = $q.defer();
                deferred.resolve(TaggingService.getProjectTags());
                return getRewrappedPromise(deferred);
            }

            $scope.setUpdatedMetaData = function(update) {
                const o = $scope.object;
                o.tags = update.tags;
                o.shortDesc = update.shortDesc;
                o.description = update.description;
            };

            $scope.$watch("data", function(nv) {
                if (!nv) return;
                enrichData();
                if($scope.object) {
                    $scope.object.isFlowObj = isShownInFlow($scope.objectType.toUpperCase());
                    $scope.isLocalObject = $scope.object.projectKey === ActiveProjectKey.get();
                    $scope.typeBadgeList = $scope.object.typeBadges ? Object.keys($scope.object.typeBadges) : [];
                }
            });

            $scope.isDataStewardEditable = function () {
                // objectAuthorizations is only there after an api call
                return $scope.data.objectAuthorizations && $scope.data.objectAuthorizations.canWriteObject;
            };

            $scope.getEffectiveDataSteward = function() {
                return $scope.data.dataSteward || $scope.data.defaultDataSteward || {displayName: 'None', login: undefined};
            }

            $scope.shouldDisplayVersions = function() {
                return $scope.data.model.savedModelType !== "RETRIEVAL_AUGMENTED_LLM";
            };

            $scope.showEditDataStewardModal = function() {
                const dataset = $scope.getTaggableObject();
                EditDatasetDataStewardModalService.showEditDataStewardModal(
                    $scope, dataset.projectKey, dataset.id, $scope.data.dataSteward, $scope.data.defaultDataSteward,
                    { from: 'dataset-right-panel' }
                ).then((dataSteward) => {
                    $scope.data.dataSteward = dataSteward;
                }).catch(() => {}); // rejects if modal is closed
            };

            $scope.showEditMetadataModal = function () {
                if (!$scope.editable) {
                    return;
                }

                // _SummaryHelper add edit behavior and create a shortDesc obj
                // but we can erase safely the things related to the description
                $scope.state.shortDesc = '';
                $scope.state.description = '';
                $scope.state.writeDescriptionsAsSQLComment = $scope?.data?.dataset?.params?.writeDescriptionsAsSQLComment || false;
                $scope.state.customFields = {};

                CreateModalFromTemplate(
                    "/templates/widgets/edit-metadata-modal.html",
                    $scope,
                    "EditMetadataModalController",
                    undefined,
                    true
                );
            };

            $scope.generateDatasetDescription = function() {
                DataikuAPI.datasets.get($scope.object.projectKey, $scope.object.name, ActiveProjectKey.get()).noSpinner()
                    .success(function(data) {
                        $scope.object = data;
                        CreateModalFromTemplate(
                            "/static/dataiku/ai-dataset-descriptions/generate-documentation-modal/generate-documentation-modal.html",
                            $scope,
                            "AIDatasetDescriptionsModalController"
                        )
                    })
                    .error(setErrorInScope.bind($scope));
            };

            function findAgentConnectUsage(objectData) {
                const fieldsMap = {
                    "TOOLS_USING_AGENT": {modelType: "agent", agentConnectField: "agents_ids"},
                    "RETRIEVAL_AUGMENTED_LLM": {modelType: "retrieval-augmented-llm", agentConnectField: "augmented_llms"},
                };
                const fields = fieldsMap[objectData.model.savedModelType];
                const modelId = fields.modelType + ":" + objectData.model.id;
                function filterWebappUsage(webapp) {
                    return webapp.type === "webapp_agent-connect_portal"
                            && ((webapp.config.llm_id === modelId)
                                || ((webapp.config[fields.agentConnectField] || []).includes(objectData.model.projectKey + ":" + modelId)));
                }
                DataikuAPI.webapps.list(objectData.model.projectKey).success(function(data) {
                    objectData.model.summary.webappsUsage = data.filter(filterWebappUsage);
                }).error(setErrorInScope.bind($scope));
            }

            $scope.$on("objectMetaDataChanged", (event, data) => {
                $scope.object.shortDesc = data.shortDesc;
                $scope.object.description = data.description;
            });

            if ($scope.objectType === 'CODE_STUDIO') {
                $scope.showCurrentUsers = function(users) {
                    const modalScope = $scope.$new();
                    modalScope.usersList = users;
                    modalScope.icon = "icon-cogs";
                    modalScope.title = users.length + ' user' + (users.length===1 ? ' is' : 's are') + ' currently using this Code Studio';
                    CreateModalFromTemplate("/templates/interested-users.html", modalScope);
                };
                $scope.showCurrentWebApps = function(webapps) {
                    const modalScope = $scope.$new();
                    modalScope.webappsList = webapps;
                    modalScope.icon = "icon-cogs";
                    modalScope.title = webapps.length + ' webapp' + (webapps.length===1 ? ' is' : 's are') + ' based on this Code Studio';
                    CreateModalFromTemplate("/templates/webapps-list.html", modalScope);
                };
            }
            $scope.getObjectIcon = function(object) {
                switch(object.type) {
                    case 'SAVED_MODEL':
                        return 'dku-icon-machine-learning-regression-16 saved-model';
                    case 'MANAGED_FOLDER':
                        return 'dku-icon-folder-open-16 managed-folder';
                    case 'MODEL_EVALUATION_STORE':
                        return 'dku-icon-model-evaluation-store-16';
                    case 'RETRIEVABLE_KNOWLEDGE':
                        return 'dku-icon-cards-stack-16 retrievable-knowledge';
                    default:
                        return $filter('toModernIcon')($filter('datasetTypeToIcon')(object.type, 16), 16) + ' dataset';
                }
            };

            $scope.getObjectLink = function(object) {
                switch(object.type) {
                    case 'SAVED_MODEL':
                        return StateUtils.href.savedModel(object.id, object.projectKey);
                    case 'MANAGED_FOLDER':
                        return StateUtils.href.managedFolder(object.id, object.projectKey);
                    case 'MODEL_EVALUATION_STORE':
                        return StateUtils.href.modelEvaluationStore(object.id, object.projectKey);
                    case 'RETRIEVABLE_KNOWLEDGE':
                        return StateUtils.href.retrievableKnowledge(object.id, object.projectKey);
                    default:
                        return StateUtils.href.dataset(object.id);
                }
            };

            $scope.$watchCollection('[objectType, projectKey, objectId]', function(nv, ov) {
                if (!nv[0]) return;
                if (!attrs.hasOwnProperty('objectData') && (!nv[1] || !nv[2])) return;

                if (!attrs.hasOwnProperty('objectData')) {
                    switch ($scope.objectType.toUpperCase()) {
                    case 'CONNECTION':
                        return;

                    case 'DATASET_CONTENT': {
                        let projectKey = $scope.projectKey,
                            name = $scope.objectId;
                        const parts = $scope.objectId.split('.');
                        if (parts.length == 2) {
                            projectKey = parts[0];
                            name = parts[1];
                        }
                        DataikuAPI.datasets.get(projectKey, name, ActiveProjectKey.get()).noSpinner()
                            .success(function(data) {
                                $scope.data = {dataset_content: data}
                            })
                            .error(setErrorInScope.bind($scope));
                        return;
                    }
                    case 'RECIPE':
                        DataikuAPI.flow.recipes.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = data;
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'MANAGED_FOLDER':
                        DataikuAPI.managedfolder.getWithStatus($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {folder: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'STREAMING_ENDPOINT':
                        DataikuAPI.streamingEndpoints.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data){
                                $scope.data = data;
                            }).error(setErrorInScope.bind($scope));
                        return;

                    case 'SAVED_MODEL':
                        DataikuAPI.savedmodels.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {model: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'MODEL_EVALUATION_STORE':
                        DataikuAPI.modelevaluationstores.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {model: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'PROMPT_STUDIO':
                        DataikuAPI.promptStudios.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = data;
                            })
                            .error(setErrorInScope.bind($scope));
                        return;
                    case 'AGENT_TOOL':
                        DataikuAPI.agentTools.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = data;
                            })
                            .error(setErrorInScope.bind($scope));
                        return;
                    case 'RETRIEVABLE_KNOWLEDGE':
                        DataikuAPI.retrievableknowledge.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = data;
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'LABELING_TASK':
                        DataikuAPI.labelingtasks.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {labelingTask: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'MODEL_COMPARISON':
                        DataikuAPI.modelcomparisons.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {model: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'ANALYSIS':
                        DataikuAPI.analysis.getCore($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                 $scope.data = {analysis: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'SQL_NOTEBOOK':
                        DataikuAPI.sqlNotebooks.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {notebook: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'SEARCH_NOTEBOOK':
                        DataikuAPI.searchNotebooks.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                $scope.data = {notebook: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'JUPYTER_NOTEBOOK':
                        DataikuAPI.jupyterNotebooks.getNotebook($scope.projectKey, $scope.objectId, undefined).noSpinner()
                            .success(function(data) {
                                $scope.data = {notebook: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'INSIGHT':
                        DataikuAPI.dashboards.insights.get($scope.projectKey, $scope.objectId).noSpinner()
                            .success(function(data) {
                                 $scope.data = {insight: data};
                            })
                            .error(setErrorInScope.bind($scope));
                        return;

                    case 'DASHBOARD':
                        DataikuAPI.dashboards.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                          .success(function(data) {
                              $scope.data = data;
                          })
                          .error(setErrorInScope.bind($scope));
                        return;

                    case 'FLOW_ZONE':
                        DataikuAPI.zones.getFullInfo($scope.projectKey, $scope.objectId).noSpinner()
                          .success(function(data) {
                              $scope.data = data;
                          })
                          .error(setErrorInScope.bind($scope));
                        return;

                    case 'CONTINUOUS_ACTIVITY':

                        DataikuAPI.continuousActivities.getState(ActiveProjectKey.get(), $stateParams.continuousActivityId).noSpinner()
                            .success(function(data) {
                                $scope.data = data;
                            }).error(setErrorInScope.bind($scope));
                        return;
                    }
                    logger.error("Unknown type: "+$scope.objectType.toUpperCase());
                }
            });

            function refreshMetadata(o) {
                if (o.metaRefreshed) return;
                DataikuAPI.taggableObjects.getMetadata($scope.getTaggableObject()).success(function(data) {
                    o.description = data.description;
                    o.shortDesc = data.shortDesc;
                    o.tags = data.tags; //mjt suspicious that this was missing....
                    o.metaRefreshed = true;
                });
            }

            function isShownInFlow(objType) {

                switch(objType) {
                    case 'DATASET_CONTENT':
                    case 'RECIPE':
                    case 'DATASET':
                    case 'SAVED_MODEL':
                    case 'MODEL_EVALUATION_STORE':
                    case 'RETRIEVABLE_KNOWLEDGE':
                    case 'MANAGED_FOLDER':
                    case 'STREAMING_ENDPOINT':
                    case 'LABELING_TASK':
                        return true;

                    // 'CONNECTION', 'PROJECT', 'SCENARIO', 'ANALYSIS', 'SQL_NOTEBOOK', 'SEARCH_NOTEBOOK',
                    // 'JUPYTER_NOTEBOOK', 'INSIGHT', 'DASHBOARD', 'WEB_APP', 'REPORT', 'CODE_STUDIO'
                    default:
                        return false;
                }
            }

            const Status = {
                RUN_SUCCESS: 'RUN_SUCCESS',
                RUN_ERROR: 'RUN_ERROR',
                // ajouter d'autres statuts si nécessaire
            };

            function mapSeverity(result) {
                switch (result.status) {
                    case Status.RUN_SUCCESS:
                        return result.severity;
                    default:
                        return 0;
                }
            }

            function enrichData() {
                const objectData = $scope.data;
                const objType = $scope.objectType.toUpperCase()

                switch(objType) {

                case 'CONNECTION':
                    return;
                case 'BUNDLE':
                    if ($scope.data.projectStandardsRunReport) {
                        $scope.object = objectData;
                        objectData.projectStandardsSummary = ProjectStandardsService.getProjectStandardsSummaryFromReport($scope.data.projectStandardsRunReport);
                    }
                    return;
                case 'PROJECT':
                    $scope.object = objectData;
                    return;
                case 'SCENARIO':
                    $scope.object = objectData.object;
                    refreshMetadata(objectData.object);
                    return;
                case 'DATASET_CONTENT':
                    if ($scope.data.dataset_content) {
                        $scope.dataset = $scope.data.dataset_content;
                    }
                    $scope.object = $scope.dataset;
                    return;

                case 'ANALYSIS':
                    if (!objectData.mlTasks) {
                        DataikuAPI.analysis.listMLTasks(objectData.analysis.projectKey, objectData.analysis.id)
                            .success(function (data) {
                                objectData.mlTasks = data;
                            })
                            .error(setErrorInScope.bind($scope));
                    }

                    if (!objectData.timeline) {
                        objectData.timeline = {};
                        if (objectData.analysis.creationTag) {
                            objectData.timeline.createdBy = objectData.analysis.creationTag.lastModifiedBy;
                            objectData.timeline.createdOn = objectData.analysis.creationTag.lastModifiedOn;
                        }
                        if (objectData.analysis.versionTag) {
                            objectData.timeline.lastModifiedBy = objectData.analysis.versionTag.lastModifiedBy;
                            objectData.timeline.lastModifiedOn = objectData.analysis.versionTag.lastModifiedOn;
                        }
                    }
                    $scope.object = objectData.analysis;
                    return;
                case 'RECIPE':
                    RecipesUtils.parseScriptIfNeeded(objectData);
                    //fetchTimeline(objectData.recipe.projectKey, objectData.recipe.name);
                    $scope.object = objectData.recipe;
                    return;
                case 'DATASET':
                    //fetchTimeline(objectData.dataset.projectKey, objectData.dataset.name);
                    $scope.object = objectData.dataset;
                    return;
                case 'STREAMING_ENDPOINT':
                    //fetchTimeline(objectData.streamingEndpoint.projectKey, objectData.streamingEndpoint.id);
                    $scope.object = objectData.streamingEndpoint;
                    return;
                case 'SQL_NOTEBOOK':
                    if (!objectData.notebook) objectData.notebook = objectData.sql_notebook;
                    $scope.object = objectData.notebook;
                    if (!objectData.notebook.niceConnection) {
                        objectData.notebook.niceConnection = NotebooksUtils.parseConnection(objectData.notebook.connection).niceConnection;
                    }
                    refreshMetadata(objectData.notebook);
                    return;
                case 'SEARCH_NOTEBOOK':
                    if (!objectData.notebook) {
                        objectData.notebook = objectData.search_notebook;
                    }
                    $scope.object = objectData.notebook;
                    if (!objectData.notebook.niceConnection && objectData.notebook.connection) {
                        objectData.notebook.niceConnection = objectData.notebook.connection;
                    }
                    refreshMetadata(objectData.notebook);
                    return;
                case 'JUPYTER_NOTEBOOK':
                    if (!objectData.notebook) objectData.notebook = objectData.jupyter_notebook;
                    $scope.object = objectData.notebook;
                    refreshMetadata(objectData.notebook);
                    return;
                case 'INSIGHT':
                    $scope.object = objectData.insight;
                    return;
                case 'DASHBOARD':
                    $scope.object = objectData.dashboard;
                    return;
                case 'WEB_APP':
                    $scope.object = objectData.webapp;
                    return;
                case 'CODE_STUDIO':
                    $scope.object = objectData.codeStudioObject;
                    return;
                case 'REPORT':
                    $scope.object = objectData.report;
                    return;
                case 'LABELING_TASK':
                    $scope.object = objectData.labelingTask;
                    return;
                case 'LAMBDA_SERVICE':
                    $scope.object = objectData.object;
                    $scope.packages = objectData.packages;
                    SavedModelsService.listAPIServiceableModels(objectData.object.projectKey).then(function(serviceableModels) {
                        $scope.savedModels = serviceableModels;
                    });
                    return;
                case 'LAMBDA_PACKAGE':
                    $scope.object = objectData.object;

                    SavedModelsService.listAPIServiceableModels($scope.projectKey).then(function(serviceableModels) {
                        $scope.allSavedModels = serviceableModels;
                    });
                    $scope.typeBadges = objectData.config.typeBadges;
                    return;
                case 'FLOW_ZONE':
                    $scope.object = objectData.zone;
                    return;
                case 'SAVED_MODEL':
                    if (!objectData.model) objectData.model = objectData.saved_model;
                    if (!objectData.status) {
                        switch (objectData.model.miniTask.taskType) {
                            case 'CLUSTERING':
                                DataikuAPI.savedmodels.clustering.getStatus(objectData.model.projectKey, objectData.model.id)
                                    .success(function (data) {
                                        objectData.status = data;
                                    })
                                    .error(setErrorInScope.bind($scope));
                                break;
                            case 'PREDICTION':
                                DataikuAPI.savedmodels.prediction.getStatus(objectData.model.projectKey, objectData.model.id)
                                    .success(function (data) {
                                        objectData.status = data;
                                    })
                                    .error(setErrorInScope.bind($scope));
                                break;
                        }
                    }

                    // Agents display a Summary section
                    if (["TOOLS_USING_AGENT", "PYTHON_AGENT", "PLUGIN_AGENT"].includes(objectData.model.savedModelType)) {
                        objectData.model.summary = {
                            webappsUsage: [],
                            // visual agents specific fields
                            prompt: undefined,
                            llm: undefined,
                            tools: []
                        };

                        if (objectData.model.savedModelType === "TOOLS_USING_AGENT") {
                            const agentVersion = objectData.model.inlineVersions.find(v => v.versionId === objectData.model.activeVersion);
                            if (agentVersion) {
                                objectData.model.summary.prompt = agentVersion.toolsUsingAgentSettings.systemPromptAppend;

                                // find the LLM used by the agent
                                DataikuAPI.pretrainedModels.listAvailableLLMs(objectData.model.projectKey, "GENERIC_COMPLETION").success(function(data) {
                                    objectData.model.summary.llm = data.identifiers.find(llm => llm.id === agentVersion.toolsUsingAgentSettings.llmId);
                                }).error(setErrorInScope.bind($scope));

                                // enrich each tool object with their type, name and more
                                DataikuAPI.agentTools.listAvailable(objectData.model.projectKey).success(function(data) {
                                    objectData.model.summary.tools = agentVersion.toolsUsingAgentSettings.tools.map(tool => {
                                        const projectTool = data.find(t => t.id === tool.toolRef);
                                        if (projectTool) {
                                            Object.assign(tool, projectTool);
                                        }
                                        return tool;
                                    });
                                }).error(setErrorInScope.bind($scope));
                                // find agent connect usage (current project only)
                                findAgentConnectUsage(objectData);
                            }
                        }
                    }
                    // Retrieval Augmented LLMs display a Summary section
                    if (objectData.model.savedModelType === "RETRIEVAL_AUGMENTED_LLM") {
                        const ragllmVersion = objectData.model.inlineVersions.find(v => v.versionId === objectData.model.activeVersion);
                        objectData.model.summary = {
                            version: ragllmVersion,
                            settings: ragllmVersion.ragllmSettings,
                            llm: undefined,
                            kb: undefined,
                            webappsUsage: [],
                        };
                        // find the LLM used by the ra-llm
                        DataikuAPI.pretrainedModels.listAvailableLLMs(objectData.model.projectKey, "GENERIC_COMPLETION").success(function(data) {
                            objectData.model.summary.llm = data.identifiers.find(llm => llm.id === ragllmVersion.ragllmSettings.llmId);
                        }).error(setErrorInScope.bind($scope));
                        // find knowledge bank
                        DataikuAPI.retrievableknowledge.get(objectData.model.projectKey, ragllmVersion.ragllmSettings.kbRef).success(knowledgeBank => {
                            objectData.model.summary.kb = knowledgeBank;
                        }).error(setErrorInScope.bind($scope));
                        // find agent connect usage (current project only)
                        findAgentConnectUsage(objectData);
                    }
                    $scope.object = objectData.model;
                    return;
                case 'MODEL_EVALUATION_STORE':
                    if (!objectData.evaluationStore) objectData.evaluationStore = objectData.model_evaluation_store;
                    $scope.object = objectData.evaluationStore;
                    return;
                case 'MODEL_COMPARISON':
                    $scope.object = objectData.modelComparison;
                    return;
                case 'PROMPT_STUDIO':
                    $scope.object = objectData.promptStudio;
                    return;
                case 'AGENT_TOOL':
                    $scope.object = objectData.agentTool;

                    $scope.object.summary = {
                        usedByAgents: [],
                        usesKnowledgeBank: undefined, // Knowledge Bank Search
                        usesSavedModel: undefined, // Model Predict
                        usesDataset: undefined, // Dataset Append and Dataset Lookup
                    };

                    // find agents that use this tool
                    DataikuAPI.agentTools.getUsage($scope.object.projectKey, $scope.object.id).success(data => {
                        $scope.object.summary.usedByAgents = data.agents.map(agent => {
                            agent.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(agent.type, agent.backendType, agent.predictionType, agent.savedModelType, null, 20);
                            return agent;
                        });
                    }).error(setErrorInScope.bind($scope));

                    // load knowledge bank if any
                    if ($scope.object.type === 'VectorStoreSearch' && $scope.object.params.knowledgeBankRef) {
                        DataikuAPI.retrievableknowledge.get($scope.object.projectKey, $scope.object.params.knowledgeBankRef).success(knowledgeBank => {
                            $scope.object.summary.usesKnowledgeBank = knowledgeBank;
                        }).error(setErrorInScope.bind($scope));
                    }

                    // load saved model if any
                    if ($scope.object.type === 'ClassicalPredictionModelPredict' && $scope.object.params.smRef) {
                        DataikuAPI.savedmodels.get($scope.object.projectKey, $scope.object.params.smRef).success(savedModel => {
                            $scope.object.summary.usesSavedModel = savedModel;
                            $scope.object.summary.usesSavedModel.computedIcon = TypeMappingService.mapSavedModelSubtypeToIcon(savedModel.type, savedModel.backendType, savedModel.predictionType, savedModel.savedModelType, null, 20);
                        }).error(setErrorInScope.bind($scope));
                    }

                    // load dataset if any
                    if (($scope.object.type === 'DatasetRowAppend' || $scope.object.type === 'DatasetRowLookup') && $scope.object.params.datasetRef) {
                        const datasetLoc = DatasetUtils.getLocFromSmart($scope.object.projectKey, $scope.object.params.datasetRef);
                        DataikuAPI.datasets.get(datasetLoc.projectKey, datasetLoc.name, $scope.object.projectKey).success(dataset => {
                            $scope.object.summary.usesDataset = dataset;
                        }).error(setErrorInScope.bind($scope));
                    }
                    return;
                case 'RETRIEVABLE_KNOWLEDGE':
                    $scope.object = objectData.retrievableKnowledge;
                    return;
                case 'MANAGED_FOLDER':
                    if (!objectData.folder) objectData.folder = objectData.managed_folder;
                    $scope.object = objectData.folder;
                    if (!objectData.timeline) {
                        objectData.timeline = {};
                        if (objectData.folder.creationTag) {
                            objectData.timeline.createdBy = objectData.folder.creationTag.lastModifiedBy;
                            objectData.timeline.createdOn = objectData.folder.creationTag.lastModifiedOn;
                        }
                        if (objectData.folder.versionTag) {
                            objectData.timeline.lastModifiedBy = objectData.folder.versionTag.lastModifiedBy;
                            objectData.timeline.lastModifiedOn = objectData.folder.versionTag.lastModifiedOn;
                        }
                    }
                    return;
                case "ARTICLE":
                    $scope.object = objectData.article ? objectData.article.data : (objectData.object ? objectData.object : undefined);
                    return
                case 'CONTINUOUS_ACTIVITY':
                    $scope.object = objectData.object;
                    return;
                case 'WORKSPACE':
                    $scope.object = objectData;
                    return;
                }

                logger.error("Unknown type: "+$scope.objectType.toUpperCase());
            }

            function fetchTimeline(projectKey, objectId) {
                if (!$scope.data.timeline) {
                    const objectType = $scope.objectType == 'DATASET_CONTENT' ? 'DATASET' : $scope.objectType.toUpperCase();
                    DataikuAPI.timelines.getForObject(projectKey, objectType, objectId)
                        .success(function(data) {
                            $scope.data.timeline = data;
                        })
                        .error(setErrorInScope.bind($scope));
                }
            }
        }
    };
});


app.directive("rightColumnTab", function(QuickView) {
    return {
        scope: false,
        link: function(scope, element, attrs) {
            function updateActiveTab(tabName = "actions", update = false) {
                scope.uiState = { activeTab: tabName };
                if(scope.setCurrentTab !== undefined){ // old right panel still need to work
                    scope.setCurrentTab(tabName, update);
                }
            }

            attrs.$observe("rightColumnTab", value => {
              updateActiveTab(value);
            });

            Mousetrap.bind('space', function() {
                const nextTab = scope.displayedTabs[((scope.displayedTabs.findIndex(tab => tab.name === scope.currentTab) + 1) % scope.displayedTabs.length)];
                if (nextTab !== undefined) {
                    updateActiveTab(nextTab.name, true);
                }
            });
            scope.$on('$destroy', function() {
                Mousetrap.unbind('space');
                QuickView.hide();
            });
        }
    };
});


app.constant("STANDARDIZED_SIDE_PANEL_KEY", "dss.standardizedSidePanel");

app.service("ActivateOldRightPanel", function($state){
    return {
        isActivated: function(){
            return !["projects.project.analyses.list",
                     "projects.project.datasets.list",
                     "projects.project.streaming-endpoints.list",
                     "projects.project.bundlesdesign.list",
                     "projects.project.lambdaservices.list",
                     "projects.project.lambdaservices.service.packages",
                     "projects.project.recipes.list",
                     "projects.project.labelingtasks.list",
                     "projects.project.notebooks.list",
                     "projects.project.scenarios.list",
                     "projects.project.continuous-activities.list",
                     "projects.project.webapps.list",
                     "projects.project.agenthub.list",
                     "projects.project.code-studios.list",
                     "projects.project.reports.list",
                     "projects.project.dashboards.list",
                     "projects.project.dashboards.insights.list",
                     "projects.project.flow",
                     "projects.project.savedmodels.list",
                     "projects.project.genai.list",
                     "projects.project.modelevaluationstores.list",
                     "projects.project.modelcomparisons.list",
                     "projects.project.promptstudios.list",
                     "projects.project.agenttools.list",
                     "projects.project.datasets.dataset.settings",
                     "projects.project.datasets.new_with_type.settings"].includes($state.current.name);
        }
    }
});

app.directive('standardizedSidePanel', function (LocalStorage, STANDARDIZED_SIDE_PANEL_KEY, $rootScope, QuestionnaireService, $stateParams, translate, $timeout) {
    return {
        restrict: "E",
        scope: true,
        templateUrl: '/templates/standardized-right-panel.html',
        link: function($scope, $element, attrs) {
            let allTabs = [
              {
                name: "actions",
                icon: "dku-icon-plus-circle-fill-24"
              },
              {
                name: "details",
                icon: "dku-icon-info-circle-fill-24"
              },
              {
                name: "preview",
                icon: "dku-icon-eye-circle-fill-24"
              },
              {
                name: "schema",
                icon: "dku-icon-text-bullet-list-circle-24"
              },
              {
                name: "Data Quality",
                icon: "dku-icon-checkmark-circle-fill-24"
              },
              {
                name: "lab",
                icon: "dku-icon-microscope-circle-fill-24"
              },
              {
                name: "timeline",
                icon: "dku-icon-clock-circle-fill-24"
              },
              {
                name: "discussions",
                icon: "dku-icon-comment-circle-fill-24"
              },
              {
                name: "Generate recipe",
                icon: "dku-icon-stars-circle-fill-24"
              }
            ];
            const defaultDatasetTabNames = ["actions", "details", "schema", "Data Quality", "lab", "timeline", "discussions"]

            let objectsToTabsMapping = [
              {
                objectTypes: ["DATASET", "LOCAL_DATASET", "FOREIGN_DATASET"],
                tabNames: !$rootScope.appConfig.prepareAICompletionEnabled  ? defaultDatasetTabNames : [...defaultDatasetTabNames, "Generate recipe"]
              },
              {
                objectTypes: ["RECIPE"],
                tabNames: ["actions", "details", "discussions", "timeline"] // TO DO : add & implement preview tab
              },
              {
                objectTypes: ["ANALYSIS"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["SQL_NOTEBOOK"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["SEARCH_NOTEBOOK"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["JUPYTER_NOTEBOOK"],
                tabNames: ["actions", "details", "discussions"]
              },
              {
                objectTypes: ["SCENARIO"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["WEB_APP"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["CODE_STUDIO"],
                tabNames: ["actions", "details", "discussions"]
              },
              {
                objectTypes: ["REPORT"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["SAVED_MODEL", "LOCAL_SAVEDMODEL", "FOREIGN_SAVEDMODEL"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["MODEL_EVALUATION_STORE", "LOCAL_MODELEVALUATIONSTORE", "FOREIGN_MODELEVALUATIONSTORE"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["PROMPT_STUDIO", "AGENT_TOOL"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["RETRIEVABLE_KNOWLEDGE", "LOCAL_RETRIEVABLE_KNOWLEDGE", "FOREIGN_RETRIEVABLE_KNOWLEDGE"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["MODEL_COMPARISON"],
                tabNames: ["actions", "details", "discussions"]
              },
              {
                objectTypes: ["MANAGED_FOLDER", "LOCAL_MANAGED_FOLDER", "FOREIGN_MANAGED_FOLDER"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["DASHBOARD"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["LABELING_TASK"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["INSIGHT"],
                tabNames: ["actions", "details", "discussions", "timeline"]
              },
              {
                objectTypes: ["ZONE"],
                tabNames: ["details", "discussions"]
              },
              {
                objectTypes: ["CONTINUOUS_ACTIVITY"],
                tabNames: ["actions", "details"]
              },
              {
                objectTypes: ["STREAMING_ENDPOINT", "LOCAL_STREAMING_ENDPOINT", "FOREIGN_STREAMING_ENDPOINT"], // the foreign version doesn't exist yet
                tabNames: ["actions", "details", "discussions"]
              },
              {
                objectTypes: ["BUNDLES_DESIGN"],
                tabNames: ["actions", "details"]
              },
              {
                objectTypes: ["PROJECT"],
                tabNames: ["details"]
              },
              {
                objectTypes: ["BUNDLES_DESIGN_MULTI"], // not a taggable object, has own multi
                tabNames: ["actions"]
              },
              {
                objectTypes: ["WORKSPACE"],
                tabNames: ["timeline", "discussions"]
              },
              {
                objectTypes: ["WORKSPACE_DATASET", "WORKSPACE_WEB_APP", "WORKSPACE_ARTICLE", "WORKSPACE_DASHBOARD"],
                tabNames: ["actions", "details", "discussions"]
              },
              {
                objectTypes: ["WORKSPACE_APP"],
                tabNames: ["actions"]
              },
              {
                objectTypes: ["LAMBDA_SERVICE"],
                tabNames: ["actions", "details"]
              },
              {
                objectTypes: ["LAMBDA_PACKAGE"],
                tabNames: ["actions", "details"]
              },
              {
                objectTypes: ["LAMBDA_PACKAGE_MULTI"], // not a taggable object, has own multi
                tabNames: ["actions"]
              },
              {
                objectTypes: ["MULTI"],
                tabNames: ["actions"]
              },
              {
                objectTypes: ["MULTI_DATASETS"],
                tabNames: !$rootScope.appConfig.prepareAICompletionEnabled  ? ["actions"] : ["actions", "Generate recipe"],
              },
              {
                objectTypes: ["MY_ITEM"],
                tabNames: [""]
              },
              {
                objectTypes: [""],
                tabNames: []
              }
            ];
            $scope.standardizedSidePanel.tabToToggle = '';

            function getLatestUsedTab(objectType, tabs) {
                const key = !objectType ? `${STANDARDIZED_SIDE_PANEL_KEY}.tab` : `${STANDARDIZED_SIDE_PANEL_KEY}.${objectType}.tab`;
                const lastTabState = LocalStorage.get(key);
                let lastGeneralTab = undefined;
                if (objectType) {
                    lastGeneralTab = getLatestUsedTab(null, tabs);
                }
                if (lastGeneralTab !== undefined) {
                    return lastGeneralTab;
                }
                if (lastTabState !== undefined && Array.isArray(tabs)) {
                    const found = tabs.map(tab => tab.name).find(tabName => tabName === lastTabState);
                    return found !== undefined || objectType === null ? found : $scope.defaultTab;
                }
                return Array.isArray(tabs) && tabs.length > 0 ? tabs[0].name : $scope.defaultTab;
            }

            function updateLatestUsedTab(objectType, tab) {
                const key = !objectType ? `${STANDARDIZED_SIDE_PANEL_KEY}.tab` : `${STANDARDIZED_SIDE_PANEL_KEY}.${objectType}.tab`;
                LocalStorage.set(key, tab);
                if (objectType) {
                    updateLatestUsedTab(null, tab);
                }
            }

            let panel = document.getElementsByClassName('right-panel')[0];
            $scope.standardizedSidePanel.opened = false;
            $scope.defaultTab = "actions";

            let openPanelOnLoad;
            switch (attrs.page) {
                case 'flow':
                    openPanelOnLoad = !QuestionnaireService.isFromQuestionnaire() ? getLastPanelState(true) : false
                    break;
                case 'objects_list':
                    openPanelOnLoad = true;
                    break;
                case 'home':
                    openPanelOnLoad = getLastPanelState(false);
                    break;
                case 'object':
                default:
                    openPanelOnLoad = false;
            }

            if (attrs.panelClosedTitle) {
                if (attrs.panelClosedTitleTranslate) {
                    $scope.panelClosedTitle = translate(attrs.panelClosedTitleTranslate, attrs.panelClosedTitle);
                } else {
                    $scope.panelClosedTitle = attrs.panelClosedTitle;
                }
            }

            function computeTabsToDisplay() {
                for (let mapping of objectsToTabsMapping) {
                    if (mapping.objectTypes.includes(attrs.objectType)) {
                        $scope.displayedTabs = allTabs
                            .filter(tab => mapping.tabNames.includes(tab.name))
                            .sort((a, b) => mapping.tabNames.indexOf(a.name) - mapping.tabNames.indexOf(b.name));
                        if (mapping.tabNames && mapping.tabNames.length > 0 && !mapping.tabNames.includes($scope.defaultTab)) {
                            $scope.defaultTab = mapping.tabNames[0];
                        }
                        break;
                    }
                }
            }
            computeTabsToDisplay();

            attrs.$observe("closeOnClickOutside", newValue => {
                if (newValue === "true") {
                    // Uses timeout to wait the end of page creation in order to have all elements
                    // otherwise the event listener callback will not be called
                    $timeout(() => {
                        let mainPanes = document.getElementsByClassName("main-panel");
                        if (mainPanes.length > 0) {
                            let mainPane = mainPanes[0];
                            // Possible leak here but the event listener will be cleaned when the main-panel element will be destroyed
                            mainPane.addEventListener("click", event => {
                                if (event && event.target && event.target.id === "qa_generic_actions-dropdown") {
                                    return;
                                }
                                // Uses timeout to apply change in next digest cycle
                                $timeout(() => {
                                    $scope.closePanel();
                                });
                            });
                        }
                    });
                }
            });

            attrs.$observe("page", newValue => {
                computeTabsToDisplay();
                $scope.page = newValue;
            });

            attrs.$observe("objectType", newValue => {
                computeTabsToDisplay();
                $scope.objectType = newValue;
                const mayChooseDiscussionsTab = $stateParams.discussionId && $scope.displayedTabs.find(tab => tab.name == 'discussions');
                const lastUsedTab = (mayChooseDiscussionsTab && 'discussions') || getLatestUsedTab(newValue, $scope.displayedTabs);
                if (lastUsedTab !== $scope.currentTab && newValue !== '') {
                    $scope.currentTab = lastUsedTab;
                    // Here we do not update the latest tab when switching to a project/zone because the user did not actively select the tab
                    // and we don't want it to override the tab for other items when unselecting then re-selecting them (see [sc-208324])
                    if (!["PROJECT", "ZONE"].includes(newValue)) {
                        updateLatestUsedTab(newValue, $scope.currentTab);
                    }
                }
                if (mayChooseDiscussionsTab) {
                    $scope.openPanel();
                }
            });

            attrs.$observe("toggleTab", tabName => {
                if (tabName && tabName != "") {
                    $scope.clickTab(tabName, true);
                }
            });

            attrs.$observe("singleType", newValue => {
                $scope.singleType = newValue;
            });

            $scope.getTooltipText = function (tab) {
                return translate("RIGHT_PANEL.TABS." + tab.name.replace(/\s+/g, "_").toUpperCase(), tab.name.charAt(0).toUpperCase() + tab.name.slice(1));
            }

            $scope.togglePanel = function () {
                if ($scope.isPanelOpened()) {
                    $scope.closePanel();
                } else {
                    $scope.openPanel();
                }
            }

            $scope.openPanel = function(saveState) {
                $scope.changePanelStateIfNeeded(true, saveState);
                if (!angular.isDefined($scope.currentTab) || !$scope.currentTab) {
                    $scope.setCurrentTab($scope.defaultTab, false);
                }
            }

            $scope.closePanel = function(saveState) {
                $scope.changePanelStateIfNeeded(false, saveState);
            }

            $scope.clickTab = function (tabName, forceOpen=false) {
                if ($scope.isPanelOpened() && $scope.isCurrentTab(tabName) && !forceOpen) {
                    $scope.closePanel();
                } else {
                    $scope.setCurrentTab(tabName);
                    $scope.openPanel();
                }
            }

            $scope.setCurrentTab = function (tabName, update = true) {
                $scope.currentTab = tabName;
                if (update === true) {
                    if (tabName === 'actions' && attrs.objectType && attrs.objectType.includes('DATASET') && attrs.objectType !== "MULTI_DATASETS") {
                        $rootScope.$broadcast('rightPanelSummary.triggerFullInfoUpdate');
                    }
                    updateLatestUsedTab($scope.objectType, tabName);
                }
            }

            $scope.getCurrentTab = function() {
                return $scope.currentTab;
            };

            $scope.isCurrentTab = function (tabName) {
                return $scope.currentTab === tabName;
            }

            $scope.isPanelOpened = function () {
                return $scope.standardizedSidePanel.opened;
            }

            $scope.isTabActive = function (tabName) {
                return ($scope.isPanelOpened() && $scope.isCurrentTab(tabName));
            }

            $scope.changePanelStateIfNeeded = function(newState, saveState) {
                let currentState = $scope.isPanelOpened();
                if (currentState != newState) {
                    $scope.standardizedSidePanel.slidePanel();
                    if (saveState !== false) {
                        savePanelState();
                    }
                }
            };

            // Init states
            if (openPanelOnLoad) {
                const saveState = false;
                $scope.openPanel(saveState);
            }

            $scope.$on("standardizedSidePanelContentChanged", function() {
                if (getLastPanelState() === undefined) {
                    const saveState = false;
                    $scope.openPanel(saveState);
                }
            });

            $scope.$watchGroup(
                ["standardizedSidePanel.opened", "displayedTabs.length", "currentTab"],
                ([panelOpened, numDisplayedTabs, currentTab]) => {
                    $scope.$emit("opalsCurrentRightPanelTab", panelOpened && numDisplayedTabs > 0 ? currentTab : null);
                }
            );

            $scope.$on("objectMetaDataChanged", $scope.clickTab.bind($scope, "details", true));

            // Activate transitions only after page load
            // N.B. : If we start using this mecanism in other places in the code it would be better
            // to place the following block in a stateChange event in DataikuController
            let httpRequestsListener = $rootScope.$watch('httpRequests.length', (newVal) => {
                if (newVal !== 0) {
                    return;
                }

                const noTransitionsOnLoadClass = 'no-transitions-on-load';
                let noTransitionsElements = document.getElementsByClassName(noTransitionsOnLoadClass);
                const nbElmts = noTransitionsElements.length;

                for (let i = 0; i < nbElmts; i++) {
                    noTransitionsElements[0].classList.remove(noTransitionsOnLoadClass);
                }

                httpRequestsListener();
            });

            // Refresh fat-repeats of the object list views
            panel.addEventListener('transitionend', (ev) => {
                // To do : add event only if the page contains fat-repeat
                if(ev.target === panel) {// we don't want the panel child elements to trigger this
                    $rootScope.$broadcast("reflow"); // To do : call reflow only on the fat-repeat scope ?
                }
            });

            // Settings using Local Storage
            function getLastPanelState(defaultValue) {
                let key = STANDARDIZED_SIDE_PANEL_KEY + '.' + attrs.page + 'Panel';
                let lastPanelState = LocalStorage.get(key);

                if (lastPanelState != undefined) {
                    return lastPanelState;
                } else {
                    return defaultValue;
                }
            }

            function savePanelState() {
                let key = STANDARDIZED_SIDE_PANEL_KEY + '.' + attrs.page + 'Panel';
                LocalStorage.set(key, $scope.isPanelOpened());
            }

        }
    };
});

app.directive("hideIfNoFilter", function($controller) {
    return {
        restrict : 'A',
        link : function(scope, element, attrs) {

            scope.$watch("noTags", function(nv) {
                let elmts = document.getElementsByClassName('list-page__filter');
                let c = 0;

                for(let i = 0; i < elmts.length; i++) {
                    c += elmts[i].childNodes.length;
                }

                if (c == 0 && nv) {
                    element.addClass('display-none');
                } else {
                    element.removeClass('display-none');
                }
            });

        }
    }
});

app.service('EditDatasetDataStewardModalService', function(DataikuAPI, CreateModalFromTemplate, WT1) {
    function showEditDataStewardModal(parentScope, projectKey, datasetId, currentDataSteward, defaultDataSteward, wt1Context) {
        return CreateModalFromTemplate("/templates/widgets/edit-data-steward-modal.html", parentScope, null, function(modalScope) {
            const getLogin = (dataSteward) => dataSteward ? dataSteward.login : undefined;
            modalScope.datasetId = datasetId;
            modalScope.state = {
                selectedDataSteward: currentDataSteward,
                defaultDataStewardName: defaultDataSteward ? ( defaultDataSteward.displayName || defaultDataSteward.login ) : null,
            }

            DataikuAPI.security.listUsers().success(function(data) {
                modalScope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                modalScope.allUsersLogin = data.map(user => '@' + user.login);
            }).error(setErrorInScope.bind(modalScope));

            modalScope.removeDataSteward = () => modalScope.state.selectedDataSteward = undefined;
            modalScope.canSave = () => getLogin(modalScope.state.selectedDataSteward) !== getLogin(currentDataSteward);

            modalScope.cancel = function() {
                modalScope.dismiss();
            };

            modalScope.save = function() {
                const dataSteward = modalScope.state.selectedDataSteward; // undefined if removed
                DataikuAPI.datasets.setDataSteward(projectKey, datasetId, getLogin(dataSteward))
                .then(function() {
                    const eventType = dataSteward ? 'set-data-steward' : 'remove-data-steward';
                    WT1.event(eventType, {
                        ...wt1Context,
                        objectType: 'DATASET',
                        objectProjectKeyh: md5(projectKey),
                        objectIdh: md5(datasetId),
                    })
                    modalScope.resolveModal(dataSteward);
                })
                .catch(setErrorInScope.bind(modalScope));
            };

        }, true);
    }

    return {
        showEditDataStewardModal,
    };
})

app.service('ObjectDetailsUtils', function() {
    function inWorkspace(context) {
        return context == 'right-column-workspace';
    }

    function inQuickView(context) {
        return context && context.toLowerCase() === 'quick-view';
    }

    function inNavigator(context) {
        return context && context != 'right-column' && context != 'right-column-workspace' && context != 'feature-store';
    }

    function inFeatureStore(context) {
        return context && context.toLowerCase() === 'feature-store';
    }

    return {
        inWorkspace,
        inQuickView,
        inNavigator,
        inFeatureStore
    }
});

app.service('DatasetDetailsUtils', function(ActiveProjectKey, DataikuAPI, FutureProgressModal, Dialogs, FlowTool) {
    function isLocalDataset(data) {
        return data.dataset.projectKey == ActiveProjectKey.get();
    };

    function isPartitioned(data) {
        return data && isDatasetPartitioned(data.dataset);
    }

    function isDatasetPartitioned(dataset) {
        return dataset
            && dataset.partitioning
            && dataset.partitioning.dimensions
            && dataset.partitioning.dimensions.length > 0;
    }

    function refreshAndGetStatus($scope, datasetData, computeRecords, forceRecompute) {
        DataikuAPI.datasets.getRefreshedSummaryStatus(ActiveProjectKey.get(), datasetData.dataset.name, computeRecords, forceRecompute).success(function(data) {
            FutureProgressModal.show($scope, data, "Refresh dataset status").then(function(result){
                datasetData.status = result;
                if (result) { // undefined in case of abort
                    Dialogs.infoMessagesDisplayOnly($scope, "Computation result", result.messages);
                    FlowTool.refreshFlowStateWhenViewIsActive(['COUNT_OF_RECORDS', 'FILESIZE']);
                    var event = new CustomEvent('datasetStatusRefreshed');
                    document.dispatchEvent(event);
                }
            });
        }).error(error => Dialogs.displaySerializedError($scope, error));
    };

    return {
        isLocalDataset,
        isDatasetPartitioned,
        isPartitioned,
        refreshAndGetStatus
    }
});

app.directive('datasetStatus', function(ObjectDetailsUtils, DatasetDetailsUtils, DatasetUtils, translate) {
    return {
        restrict: 'E',
        templateUrl: '/templates/object-details/dataset_status.html',
        transclude: true,
        scope: {
            objectName: '<',
            checklists: '<',
            data: '<',
            context: '<',
            readOnly: '<'
        },
        link: function(scope) {
            scope.DatasetUtils = DatasetUtils;
            scope.translate = translate;

            scope.inWorkspace = function() {
                return ObjectDetailsUtils.inWorkspace(scope.context);
            };
            scope.inQuickView = function() {
                return ObjectDetailsUtils.inQuickView(scope.context);
            };
            scope.inNavigator = function() {
                return ObjectDetailsUtils.inNavigator(scope.context);
            };
            scope.inFeatureStore = function() {
                return ObjectDetailsUtils.inFeatureStore(scope.context);
            };
            scope.isLocalDataset = function() {
                return DatasetDetailsUtils.isLocalDataset(scope.data);
            };
            scope.isPartitioned = function() {
                return DatasetDetailsUtils.isPartitioned(scope.data);
            };
            scope.refreshAndGetStatus = function(datasetData, computeRecords, forceRecompute) {
                return DatasetDetailsUtils.refreshAndGetStatus(scope, datasetData, computeRecords, forceRecompute);
            };
        }
    }
});

app.controller("EditMetadataModalController", function(
    $q,
    $scope,
    $rootScope,
    $timeout,
    DataikuAPI,
    AIExplanationService,
    TaggingService
) {
    const ctrl = this;

    $scope.cancel = function() {
        $scope.dismiss();
    };

    if (!$scope.object) {
        return;
    }

    if (!$scope.state) {
        $scope.state = {};
    }

    $scope.state.tags = {
        newVal: undefined,
        savedVal: angular.copy($scope.object.tags),
        editing: false,
    };
    $scope.state.shortDesc = angular.copy($scope.object.shortDesc);
    $scope.state.description = angular.copy($scope.object.description);
    $scope.state.customFields = angular.copy($scope.object.customFields);

    $scope.startEditTags = function() {
        $scope.state.tags.newVal = angular.copy($scope.state.tags.savedVal);
        $scope.state.tags.editing = true;
    };
    $scope.cancelEditTags = function() {
        $scope.state.tags.newVal = null;
        $scope.state.tags.editing = false;
    };
    $scope.validateEditTags = function() {
        if ($scope.state.tags.editing) {
            $scope.state.tags.savedVal = $scope.state.tags.newVal;
            $scope.state.tags.editing = false;
        }
    };

    $scope.getAllTagsForProject = function() {
        const deferred = $q.defer();
        deferred.resolve(TaggingService.getProjectTags());
        return getRewrappedPromise(deferred);
    };

    ctrl.$onInit = function() {
        $scope.generateDescriptionButtonTitle = AIExplanationService.getObjectNotExplainableReason($scope.objectType);
        $scope.generateDescriptionButtonDisabled = !!$scope.generateDescriptionButtonTitle;
    };

    $scope.generateDescription = function() {
        $scope.generateDescriptionForObject().then(function(description) {
            $scope.state.description = description;
            // The CodeMirror gets refreshed when the edit-metadata-modal is
            // still in the "perspective-modified" state which messes it up
            // when the modal is brought back to the front by the 500ms
            // animation. Let's refresh it again on focus and also after 750ms.
            $scope.refreshDescriptionOnFocus = true;
            $timeout(function() {
                $scope.descriptionUIRefreshToggle = !$scope.descriptionUIRefreshToggle;
            }, 750);
        });
    };

    $scope.onDescriptionFocus = function(codeMirrorInstance) {
        if ($scope.refreshDescriptionOnFocus) {
            codeMirrorInstance.refresh();
            $scope.refreshDescriptionOnFocus = false;
        }
    };

    $scope.save = function() {
        const ui = $scope.state;

        if ($scope.state.tags.editing) {
            $scope.validateEditTags();
        }

        let apiPromise;
        const update = {
            shortDesc: ui.shortDesc,
            description: ui.description,
            customFields: ui.customFields,
            tags: ui.tags.newVal ? ui.tags.newVal : ui.tags.savedVal,
        };
        if ($scope.objectType === "PROJECT") {
            const projectSummary = angular.copy($scope.object);
            Object.assign(projectSummary, update);
            apiPromise = DataikuAPI.projects.saveSummary($scope.object.projectKey, projectSummary);
        } else {
            apiPromise = DataikuAPI.taggableObjects.setMetaData(
                {
                    type: $scope.objectType.toUpperCase(),
                    projectKey: $scope.object.projectKey,
                    id: $scope.object.id || $scope.object.name,
                },
                update
            ).then(function(resp) {
                $rootScope.$broadcast.bind($rootScope, "objectMetaDataChanged", update)
                $scope.warningMessages = resp?.data?.anyMessage ? resp.data : null;
            })
        }
        apiPromise
            .then(function() {
                $scope.setUpdatedMetaData(update);
                if ($scope.warningMessages === null) {
                    $scope.resolveModal();
                }
            })
            .catch(setErrorInScope.bind($scope));
    };
});

})();
