(function() {
'use strict';

/*
* This file defines a set of "Standard" flow views
*
* They implement a common API, in particular they are essentially a mapping: node -> single value
* (the single value can be structured but will be displayed as a single value as opposed to multi-valued views like tags)
*
*/
const app = angular.module('dataiku.flow.tools');

app.service('WatchView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('WATCH', 'Watched and starred items', {
            getRepr: val => val.w ? 'Watching' : undefined,
            totem: function(val) {
                return {
                    class: val.s ? 'icon-star' : '',
                    style: val.s ? 'color: gold; font-size: 32px;' : ''
                }
            }
        });
    }
});


app.service('SparkConfigView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('SPARK_CONFIG', 'Spark configurations', {
            getRepr: val => val.inheritConf,
            totem: function(val) {
                return {
                    class: val.conf.length ? 'icon-plus flow-totem-ok' : '',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/spark-config-view-tooltip.html'
        });
    }
});


app.service('ConnectionsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('CONNECTIONS', 'Connections', {
            getRepr: val => val.connection,
            tooltipTemplate: '/templates/flow-editor/tools/connections-view-tooltip.html'
        });
    }
});

app.service('FlowZonesView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('FLOW_ZONES', 'Flow Zones', {
            getRepr: function(val) {
                return val.id;
            },
            tooltipTemplate: '/templates/flow-editor/tools/flow-zones-view-tooltip.html',
            settingsTemplate: '/templates/flow-editor/tools/flow-zones-settings.html'
        });
    }
});

/*
* Note that the tag view is a multi-valued one (each node has several labels)
*/
app.service('TagsView', function($rootScope, $filter, $injector, $stateParams, translate,
    DataikuAPI, CreateModalFromTemplate, TaggableObjectsUtils, TaggingService,
    FlowTool, FlowGraph, FlowToolsUtils, StandardFlowViews) {

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('TAGS', 'Tags', {
            postInit: function(tool) {
                tool.manageTags = function() {
                    CreateModalFromTemplate("/templates/widgets/edit-tags-modal.html", $rootScope, null, function(modalScope) {
                        modalScope.translate = translate;
                        modalScope.tagsDirty = angular.copy(TaggingService.getProjectTags());

                        modalScope.save = function() {
                            TaggingService.saveToBackend(modalScope.tagsDirty)
                                .then(modalScope.resolveModal)
                                .catch(setErrorInScope.bind(modalScope));
                        };
                        modalScope.cancel = function() {modalScope.dismiss();};
                    });
                };
                tool.displayGlobalTags = true;
            },
            getRepr: function(val) {
                return val;
            },
            postProcessNode: function(tags, nodeElt, tool) {
                if (!tags) return;
                tags.forEach(function(tag, idx) {
                    function onClick() {
                        tool.user.focus(tag);
                        $rootScope.$digest();
                        d3.event.stopPropagation();
                        d3.event.preventDefault();
                    }   
                    FlowToolsUtils.addViewValueIndicator(nodeElt, $filter('tagToColor')(tag), idx, onClick);
                });
            },
            actions: {
                setTags: function(tags, nodes, mode) { // mode = TOGGLE, ADD or REMOVE
                    const request = {
                        elements: nodes.map(TaggableObjectsUtils.fromNode),
                        operations: [{mode: mode, tags: tags}]
                    };

                    DataikuAPI.taggableObjects.applyTagging($stateParams.projectKey, request).success(function(data) {
                        TaggingService.bcastTagUpdate(false, true);
                    }).error(FlowGraph.setError());
                }
            },
            autoSelectFirstOnly: true,
            tooltipTemplate: '/templates/flow-editor/tools/tags-view-tooltip.html'
        });
    }
});


app.service('CustomFieldsView', function($rootScope, FlowTool, StandardFlowViews, objectTypeFromNodeFlowType) {
    function getSelectedOption(value, fromLabel) {
        if (!FlowTool.getCurrent().currentSession || !FlowTool.getCurrent().currentSession.options || FlowTool.getCurrent().currentSession.options.selectedCustomField) {
            return null;
        }
        // TODO: selectedCustomField  come from Angular 2 now adays
        let selectedCustomField = FlowTool.getCurrent().currentSession.options.selectedCustomField;
        for (let taggableType in $rootScope.appConfig.customFieldsMap) {
            if ($rootScope.appConfig.customFieldsMap.hasOwnProperty(taggableType)) {
                let componentList = $rootScope.appConfig.customFieldsMap[taggableType];
                for (let i = 0; i < componentList.length; i++) {
                    let paramDesc = (componentList[i].customFields.filter(cf => cf.type == 'SELECT' && cf.selectChoices) || []).find(cf => cf.name == selectedCustomField);
                    if (paramDesc) {
                        let selOpt = (paramDesc.selectChoices || []).find(function(choice) {
                            if (fromLabel) {
                                return value && choice.label == value;
                            } else {
                                return value ? choice.value == value : (paramDesc.defaultValue && choice.value == paramDesc.defaultValue);
                            }
                        });
                        if (selOpt) {
                            return selOpt;
                        }
                    }
                }
            }
        }
        return null;
    }

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('CUSTOM_FIELDS', 'Metadata fields', {
            getRepr: function(val) {
                let selOpt = getSelectedOption(val);
                return (selOpt && (selOpt.label || selOpt.value)) || val;
            },
            postInit: function(tool) {
                tool.objectTypeFromNodeFlowType = objectTypeFromNodeFlowType;
            },
            tooltipTemplate: '/templates/flow-editor/tools/custom-fields-view-tooltip.html'
        });
    };
});


/*
* Note that the scenarios view is a multi-valued one (each node has several labels)
*/
app.service('ScenariosView', function($rootScope, FlowToolsUtils, StandardFlowViews) {
    const ACTIONS = {
        'build_flowitem': 'Build',
        'clear_items': 'Clear',
        'check_dataset': 'Verify rules or run checks',
        'compute_metrics': 'Compute metrics',
        'sync_hive': 'Synchronize Hive',
        'update_from_hive': 'Update from Hive'
    };

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('SCENARIOS', 'Scenarios', {
            getRepr: function(uses) {
                return uses.map(use => use.scenarioName+' ('+use.scenarioId+')');
            },
            postProcessNode: function(uses, nodeElt, tool) {
                if (!uses) return;
                uses.forEach(function(use, idx) {
                    function onClick() {
                        tool.user.focus(use);
                        $rootScope.$digest();
                        d3.event.stopPropagation();
                        d3.event.preventDefault();
                    }
                    const fullId = use.scenarioName+' ('+use.scenarioId+')';
                    FlowToolsUtils.addViewValueIndicator(
                        nodeElt,
                        tool.user.getColor(fullId),
                        idx,
                        onClick
                    );
                });
            },
            actions: {
                getActionsNames(actions) {
                    if (!actions) return;
                    return actions.map(a => ACTIONS[a]);
                }
            },
            autoSelectFirstOnly: true,
            tooltipTemplate: '/templates/flow-editor/tools/scenarios-view-tooltip.html'
        });
    }
});


app.service('FileformatsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('FILEFORMATS', 'File format', {
            getRepr: val => val.formatType,
            tooltipTemplate: '/templates/flow-editor/tools/fileformats-view-tooltip.html'
        });
    }
});


app.service('PipelinesView', function(StandardFlowViews) {
    this.getService = function(toolName) {
        let displayName;
        if (toolName === "SPARK_PIPELINES") {
            displayName = "Spark pipelines";
        } else if (toolName === "SQL_PIPELINES") {
            displayName = "SQL pipelines";
        }
        return {
            getDefinition: function() {
                return StandardFlowViews.getDefinition(toolName, displayName, {
                    getRepr: function(val) {
                        if (val.pipelineId) {
                            return val.pipelineId
                        }
                        if (val.virtualizable) {
                            return null;// Dataset
                        }
                        return;
                    },
                    totem: function(val) {
                        return {
                            class: val.virtualizable ? 'icon-forward flow-totem-ok' : '',
                            style: ''
                        }
                    },
                    tooltipTemplate: '/templates/flow-editor/tools/pipeline-view-tooltip.html',
                });
            }
        }
    };
});


app.service('ImpalaWriteModeView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('IMPALA_WRITE_MODE', 'Impala write mode', {
            getRepr: val => val
        });
    }
});


app.service('HiveModeView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('HIVE_MODE', 'Hive mode', {
            getRepr: function(val) {
                if (val == "HIVECLI_LOCAL") return "Hive CLI (isolated metastore)";
                if (val == "HIVECLI_GLOBAL") return "Hive CLI (global metastore)";
                if (val == "HIVESERVER2") return "HiveServer2";
                return val;
            }
        });
    }
});


app.service('PartitioningView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('PARTITIONING',  'Partitioning schemes', {
            getRepr: function(val) {
                if (val.dimensions.length) {
                    return val.dimensions.map(x => x.name).sort().join(', ');
                } else {
                    return 'Not partitioned';
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/partitioning-view-tooltip.html'
        });
    }
});


app.service('PartitionsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('PARTITIONS', 'Partitions count', {
            getRepr: function(val) {
                return val;
            },
            tooltipTemplate: '/templates/flow-editor/tools/partitions-view-tooltip.html'
        });
    }
});


app.service('ColumnUsageView', function(StandardFlowViews, ColorPalettesService, FlowToolsUtils, $rootScope) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('COLUMN_USAGE', 'Column usage', {
            getRepr: function(val) {
                return val;
            },
            postProcessNode: function(values, nodeElt, tool) {
                if (!values) return;
                values.forEach(function(value, idx) {
                    function onClick() {
                        tool.user.focus(value);
                        $rootScope.$digest();
                        d3.event.stopPropagation();
                        d3.event.preventDefault();
                    }
                    FlowToolsUtils.addViewValueIndicator(nodeElt, tool.user.getColor(value), idx, onClick);
                });
            },
            tooltipTemplate: '/templates/flow-editor/tools/column-usage-tooltip.html',
        });
    }
});

app.service('DataQualityView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('DATA_QUALITY', 'Data quality', {
            postInit: function(tool) {
                tool.currentSession.options.includeForeignObjects = tool.currentSession.options.includeForeignObjects || false;
            },
            getRepr: function(val) {
                return {
                    ERROR: 'Error',
                    WARNING: 'Warning',
                    OK: 'OK',
                    EMPTY: 'Empty',
                    _: 'Not Computed'
                } [val.lastOutcome || '_'];
            },
            tooltipTemplate: '/templates/flow-editor/tools/data-quality-tooltip.html',
        });
    }
});


app.service('RecentActivityView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('RECENT_ACTIVITY', 'Recent modifications', {
            getRepr: function(val) {
                return val.numberOfModifications;
            },
            tooltipTemplate: '/templates/flow-editor/tools/recent-activity-view-tooltip.html',
        });
    }
});


app.service('FilesizeView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('FILESIZE', 'File size', {
            getRepr: function(val) {
                let totalValue = parseFloat(val.size.totalValue);
                if (isNaN(totalValue) || totalValue <= 0) {
                    return 'Unknown';
                }
                return totalValue;
            },
            tooltipTemplate: '/templates/flow-editor/tools/filesize-view-tooltip.html',
        });
    }
});


app.service('CountOfRecordsView', function($filter, StandardFlowViews, DataikuAPI, $stateParams, FlowGraph, $rootScope,DatasetsService, Dialogs) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('COUNT_OF_RECORDS', 'Records count', {
            getRepr: function(val) {
                let totalValue = parseFloat(val.countOfRecords.totalValue);
                if (totalValue == -1) {
                    return 'Unknown';
                }
                return totalValue;
            },
            postInit: function(tool) {
                const listDatasetsToCompute = function (datasets, computeOnlyMissingRecordsCount) {
                    if (!computeOnlyMissingRecordsCount) {
                        return datasets
                    }
                    const datasetsToCount = [];
                    for (const dataset of datasets) {
                        if (dataset.id) {
                            const datasetGraphId = graphVizEscape(`dataset_${dataset.id}`);
                            const isRecordCountComputed = (tool.user.state.valueByNode && tool.user.state.valueByNode[datasetGraphId] && tool.user.state.valueByNode[datasetGraphId].countOfRecords && tool.user.state.valueByNode[datasetGraphId].countOfRecords.hasData);
                            if (!isRecordCountComputed) {
                                datasetsToCount.push(dataset);
                            }
                        }
                    }
                    return datasetsToCount
                }
                // Action used in Angular 2
                tool.compute = function (computeOnlyMissingRecordsCount) {
                    DataikuAPI.flow.listUsableComputables($stateParams.projectKey, {datasetsOnly: true})
                        .success((datasets) => {
                            const datasetsToCount = listDatasetsToCompute(datasets, computeOnlyMissingRecordsCount);
                            DatasetsService.refreshSummaries($rootScope, datasetsToCount, true, false, true)
                                .then(function (result) {
                                    tool.refreshState();
                                    return result;
                                })
                                .then(function (result) {
                                    if (result && result.anyMessage) {
                                        Dialogs.infoMessagesDisplayOnly($rootScope, "Datasets statuses update results", result);
                                    }
                                }).catch(FlowGraph.setError())
                        }).error(FlowGraph.setError())
                }
            },
            totem: function(val) {
                const countOfRecordsText = sanitize(val.countOfRecords.totalValue < 0 ? '-' : $filter('longSmartNumber')(val.countOfRecords.totalValue));
                return {
                    class: 'icon-refresh '+(val.autoCompute ? 'flow-totem-ok' : 'flow-totem-disabled'),
                    style: '',
                    text: countOfRecordsText
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/count-of-records-view-tooltip.html'
        });
    }
});



const DATE_REPR = [
    'Just now',
    'Past hour',
    'Past 24h',
    'Past week',
    'Past month',
    'Past year',
    'More than a year ago',
    'Unknown'
];
function simpleTimeDelta(timestamp) {
    if (typeof timestamp == 'string' && !isNaN(parseFloat(timestamp))) { //TODO @flow dirty
        timestamp = parseFloat(timestamp);
    }
    if (!timestamp || typeof timestamp != 'number') {
        return DATE_REPR[7];
    }
    const seconds = (new Date().getTime() - timestamp)/1000;
    if (seconds < 60) {
        return DATE_REPR[0];
    }
    if (seconds < 3600) {
        return DATE_REPR[1];
    }
    if (seconds < 3600*24) {
        return DATE_REPR[2];
    }
    if (seconds < 3600*24*7) {
        return DATE_REPR[3];
    }
    if (seconds < 3600*24*30) {
        return DATE_REPR[4];
    }
    if (seconds < 3600*24*365) {
        return DATE_REPR[5];
    }
    return DATE_REPR[6];
}



app.service('CreationView', function(StandardFlowViews, UserImageUrl, FlowTool) {

    const viewByUser = {
        getRepr: function(val) {
            return val.userLogin;
        }
    };
    const viewByDate = {
        getRepr: function(val) {
            const time = parseFloat(val.time);
            return simpleTimeDelta(time);
        }
    };

    function getSubview() {
        const mode = FlowTool.getCurrent().currentSession.options.mode;
        if (mode == 'BY_USER') {
            return viewByUser;
        }
        return viewByDate;
    }

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('CREATION', 'Creation', {
            postInit: function(tool) {
                tool.currentSession.options.mode = tool.currentSession.options.mode || 'BY_DATE';
            },
            getRepr: function(val) {
                if (!val) return;
                return getSubview().getRepr(val);
            },
            totem: function(val) {
                return {
                    class: 'avatar32',
                    style: "background-image: url('" + UserImageUrl(val.userLogin, 128) + "'); box-sizing: border-box;"
                };
            },
            tooltipTemplate: '/templates/flow-editor/tools/creation-view-tooltip.html',
            settingsTemplate: '/templates/flow-editor/tools/creation-view-settings.html'
        });
    };
});


app.service('LastModifiedView', function(StandardFlowViews, UserImageUrl, FlowTool) {

    const viewByUser = {
        getRepr: function(val) {
            return val.userLogin;
        }
    };
    const viewByDate = {
        getRepr: function(val) {
            const time = parseFloat(val.time);
            return simpleTimeDelta(time);
        }
    };

    function getSubview() {
        const mode = FlowTool.getCurrent().currentSession.options.mode;
        if (mode == 'BY_USER') {
            return viewByUser;
        }
        return viewByDate;
    }

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('LAST_MODIFICATION', 'Last modification', {
            postInit: function(tool) {
                tool.currentSession.options.mode = tool.currentSession.options.mode || 'BY_DATE';
            },
            getRepr: function(val) {
                if (!val) return;
                return getSubview().getRepr(val);
            },
            totem: function(val) {
                return {
                    class: 'avatar32',
                    style: "background-image: url('" + UserImageUrl(val.userLogin, 128) + "'); box-sizing: border-box;"
                };
            },
            tooltipTemplate: '/templates/flow-editor/tools/last-modification-view-tooltip.html',
        });
    };
});


app.service('LastBuildView', function(StandardFlowViews) {

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('LAST_BUILD', 'Last build', {
            getRepr: function(val) {
                if (!val) return;
                const time = parseFloat(val.buildEndTime);
                return simpleTimeDelta(time);
            },
            totem: function(val) {
                return {
                    class: val.buildSuccess ? 'icon-ok flow-totem-ok' : 'icon-remove flow-totem-error',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/last-build-view-tooltip.html',
        });
    };
});


app.service('LastBuildDurationView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('LAST_BUILD_DURATION', 'Last build duration', {
            getRepr: function(val) {
                if (val < 0) return;
                return val;
            },
            tooltipTemplate: '/templates/flow-editor/tools/last-build-duration-view-tooltip.html'
        });
    };
});

app.service('RecipesEnginesView', function($filter, StandardFlowViews) {
    const RECIPE_ENGINE_NAMES = {
        'DSS': 'DSS',
        'USER_CODE': 'User code',
        'PLUGIN_CODE': 'Plugin code',
        'HADOOP_MAPREDUCE': 'Hadoop MapReduce',
        'SPARK': 'Spark',
        'PIG': 'Pig',
        'SQL': 'SQL',
        'HIVE': 'Hive',
        'IMPALA': 'Impala',
        'S3_TO_REDSHIFT': 'S3 to Redshift',
        'REDSHIFT_TO_S3': 'Redshift to S3',
        'AZURE_TO_SQLSERVER': 'Azure to SQLServer',
        'TDCH': 'Teradata Hadoop Connector',
        'GCS_TO_BIGQUERY': 'GCS to BigQuery',
        'BIGQUERY_TO_GCS': 'BigQuery to GCS',
        'S3_TO_SNOWFLAKE': 'S3 to Snowflake',
        'SNOWFLAKE_TO_S3': 'Snowflake to S3',
        'WASB_TO_SNOWFLAKE': 'WASB to Snowflake',
        'SNOWFLAKE_TO_WASB': 'Snowflake to WASB',
        'GCS_TO_SNOWFLAKE': 'GCS to Snowflake',
        'SNOWFLAKE_TO_GCS': 'Snowflake to GCS',
        'AZURE_TO_DATABRICKS': 'Azure to Databricks',
        'DATABRICKS_TO_AZURE': 'Databricks to Azure',
        'S3_TO_DATABRICKS': 'S3 to Databricks',
        'DATABRICKS_TO_S3': 'Databricks to S3',
        'DOCKER/KUBERNETES': 'Docker/Kubernetes' // Fake engine
    };
    const RUN_IN_CONTAINER_SUFFIX = " in container";

    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('RECIPES_ENGINES', 'Recipe engines', {
            getRepr: function(engineStatus) {
                let engineType = engineStatus.type;
                let runInContainer = engineType.endsWith(RUN_IN_CONTAINER_SUFFIX);
                if (runInContainer) {
                    engineType = engineType.slice(0, -RUN_IN_CONTAINER_SUFFIX.length);
                }
                let result = RECIPE_ENGINE_NAMES[engineType];
                if (!result) {
                    return $filter('capitalize')(engineStatus.type.toLowerCase().replaceAll('_', ' '));
                }
                return result + (runInContainer ? RUN_IN_CONTAINER_SUFFIX : '');
            },
            totem: function(val) {
                return {
                    class: val.statusWarnLevel === 'ERROR' ? 'icon-remove flow-totem-error' : '',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/recipes-engines-view-tooltip.html'
        });
    }
});

app.service('RecipesCodeEnvsView', function(StandardFlowViews) {
    this.getDefinition = function() {
        return StandardFlowViews.getDefinition('RECIPES_CODE_ENVS', 'Recipe code environments', {
            getRepr: function(codeEnvState) {
                return codeEnvState.selectedEnvName || codeEnvState.envName || 'DSS builtin env';
            },
            totem: function(val) {
                return {
                    class: val.preventedByProjectSettings ? 'icon-remove flow-totem-error' : '',
                    style: ''
                }
            },
            tooltipTemplate: '/templates/flow-editor/tools/recipes-code-envs-view-tooltip.html',
        });
    }
});


app.service('StandardFlowViews', function ($stateParams, Ng1ToolBridgeService, Debounce, FlowViewsUtils, FlowGraph, FlowGraphHighlighting) {
    this.getDefinition = function (name, displayName, {
        // TODO every view is holding a definition of this method, but it should come from Angular 2 so we have a single source of truth
        getRepr,
        totem,
        tooltipTemplate,
        settingsTemplate,
        postInit,
        postProcessNode,
        actions
    }) {

    return {
        getName: () => name,
        getToolDisplayName: () => displayName,

        initFlowTool: function (tool, viewData) {
            tool.user = {};
            tool.projectKey = $stateParams.projectKey;
            tool.update = (viewData) => {
                tool.user.state = viewData;
                tool.currentSession.options.mode = tool.user.state.mode;

                const countByValue = {};
                $.each(tool.user.state.valueByNode, function (nodeId, val) {
                    const repr = getRepr(val);
                    if (angular.isArray(repr)) {
                        repr.forEach(function (it) {
                            countByValue[it] = (countByValue[it] || 0) + 1;
                        });
                    } else if (repr !== null && repr !== undefined) {
                        countByValue[repr] = (countByValue[repr] || 0) + 1;
                    }
                });
                tool.user.state.countByValue = countByValue;
                tool.user.state.values = Object.keys(countByValue);

                if (postInit) {
                    postInit(tool);
                }

                tool.drawHooks.updateFlowToolDisplay();
                Ng1ToolBridgeService().viewLoaded();
            }

            tool.refreshState = function () {
                Ng1ToolBridgeService().emitRefreshView();
            };

            tool.refreshStateLater = Debounce().withDelay(400, 400).wrap(tool.refreshState);

            FlowViewsUtils.addFocusBehavior(tool, true);
            tool.user.getColor = function (repr) {
                if (!repr || repr === 'Unknown' || !Ng1ToolBridgeService().activeView) {
                    return '#333';
                }
                return Ng1ToolBridgeService().activeView.getColor(repr);
            };

            tool.drawHooks.updateFlowToolDisplay = function () {
                if (!tool.user.state) return; // protect against slow state fetching
                if (!FlowGraph.ready()) return; // protect against slow graph fetching

                // TODO @flow too slow?
                $.each(FlowGraph.get().nodes, function (nodeId, node) {
                    let realNodeId = node.realId || nodeId;
                    const nodeElt = FlowGraph.d3NodeWithId(nodeId);
                    if (tool.user.state.valueByNode[realNodeId] === undefined) {
                        $('.node-totem span', nodeElt[0]).removeAttr('style').removeClass();
                        $('.nodecounter__text div span', nodeElt[0]).text('');
                        $('.never-built-computable *', nodeElt[0]).removeAttr('style');
                    }
                    $('.nodecounter__wrapper', nodeElt[0]).removeClass('nodecounter__wrapper--shown');
                });

                $('.tool-simple-zone', FlowGraph.getSvg()).empty();

                // We first iterate over all non-recipes then on recipes,
                // This is because in some cases, recipes color their outputs
                function styleNodes(recipesOnly) {
                    $.each(FlowGraph.get().nodes, function (nodeId, node) {
                        let val = tool.user.state.valueByNode[node.realId];
                        if (!node) { // If some nodes are broken, they might not be rendered in the flow
                            return;
                        }
                        const isRecipe = node.nodeType == 'RECIPE';
                        if (recipesOnly != isRecipe) {
                            return;
                        }
                        const isZone = node.nodeType == "ZONE";
                        if (isZone && Ng1ToolBridgeService().activeView && tool.user.state.focusMap && tool.user.state.focusMap[node.name] && !$stateParams.zoneId) {
                            FlowGraphHighlighting.highlightZoneCluster(d3.select(`g[id=cluster_zone_${node.name}]`)[0][0], Ng1ToolBridgeService().activeView.getColor(node.name));
                        }
                        if (val !== undefined) {
                            const nodeElt =  FlowGraph.d3NodeWithIdFromType(nodeId, node.nodeType);
                            if (!isZone && FlowGraph.rawNodeWithId(nodeId) === undefined) {
                                return;
                            }
                            const nodeTotem = $('.node-totem span', nodeElt[0]);
                            nodeTotem.removeAttr('style').removeClass();
                            if (totem && totem(val)) {
                                nodeTotem.attr('style', totem(val).style).addClass(totem(val).class);
                            }

                            const nodeCounter = $('.nodecounter__text div span', nodeElt[0]);
                            const nodeCounterWrapper = $('.nodecounter__wrapper', nodeElt[0]);
                            nodeCounter.text('');
                            if (totem && totem(val)) {
                                if (totem(val).text !== undefined) {
                                    const nodeCounterText = totem(val).text;
                                    nodeCounter.text(nodeCounterText);
                                    nodeCounterWrapper.addClass("nodecounter__wrapper--shown");
                                }
                            }

                            if (postProcessNode) {
                                postProcessNode(val, nodeElt, tool);
                            }
                        }
                    });
                }
                styleNodes(false);
                styleNodes(true);
            };

            tool.drawHooks.setupTooltip = function (node) {
                const activeView = Ng1ToolBridgeService().activeView;
                if (!tool.user || !tool.user.state || !activeView) return;
                const tooltip = {};
                tooltip.val = tool.user.state.valueByNode[node.realId];
                tooltip.template = tooltipTemplate || '/templates/flow-editor/tools/default-view-tooltip.html';
                const nodeViewCategories = activeView.getNodeCategories(node);
                if (!tooltip.val || nodeViewCategories.length === 0) {
                    return tooltip;
                }
                if (nodeViewCategories.length === 1) {
                    const category = nodeViewCategories[0];
                    let bulletText = tool.type == "FLOW_ZONES" ? tooltip.val.name : category;
                    tooltip.bullets = [{ text: bulletText, color: activeView.getColor(category) }];
                } else if (nodeViewCategories.length > 1) {
                    const focused = tool.user.getFocusedAsList();
                    const matchedValues = nodeViewCategories.filter(_ => focused.indexOf(_) !== -1);
                    if (matchedValues.length === 1) {
                        tooltip.bullets = [{text: matchedValues[0], color: activeView.getColor(matchedValues[0])}];
                    } else {
                        tooltip.bullets = []
                        matchedValues.forEach((value) => {
                            tooltip.bullets.push({ text: value, color: activeView.getColor(value) });
                        });
                    }
                }
                return tooltip;
            };

            tool.getRepr = getRepr;
            tool.def.settingsTemplate = settingsTemplate;
            tool.actions = actions;

            tool.update(viewData);
        }
    };
};
});


app.controller("StandardFlowViewsMainController", function($scope, FlowTool) {
    $scope.tool = FlowTool.getCurrent();
});
})();
