(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('GroupedColumnsDrawer', GroupedColumnsDrawer);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/column.js
     */
    function GroupedColumnsDrawer(ChartDimension, Fn, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartColorUtils, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, CHART_TYPES, HierarchicalChartsUtils) {
        const backgroundXMargin = 4;

        return function(g, chartDef, chartHandler, chartData, chartBase, groupsData, f, labelCollisionDetectionHandler, drawReferenceLines = true) {

            const xDimension = chartDef.genericDimension0[0],
                xLabels = chartData.getAxisLabels('x'),
                xAxis = chartBase.xAxis,
                rightAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes),
                leftAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes),
                isAxisLogScale = function(d) {
                    const displayAxis = chartDef.genericMeasures[d.measure].displayAxis == 'axis1' ? leftAxis : rightAxis;
                    return ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, displayAxis.id);
                },
                getScale = function(d) {
                    return chartDef.genericMeasures[d.measure].displayAxis == 'axis1' ? leftAxis.scale() : rightAxis.scale();
                },
                zeroIsInDomain = function(d) {
                    const axisDomain = getScale(d).domain();
                    return (axisDomain[0] > 0) != (axisDomain[1] > 0);
                },
                getRectValue = function(d) {
                    return chartData.aggr(d.measure).get(d);
                };

            const nonEmptyBinsCounts = {};
            // do a first pass to compute nonEmptyBinsIdx, we have to do it here, and not in the "prepareData" method, because it would not have the current animation frame or subchart
            groupsData.forEach(group => {
                group.columns.forEach(c => {
                    if (_.isNil(nonEmptyBinsCounts[c.x])) {
                        nonEmptyBinsCounts[c.x] = 0;
                    }
                    const value = chartData.aggr(c.measure).get(c);
                    let nonEmptyIdx = undefined;
                    if (value !== 0) {
                        nonEmptyIdx = nonEmptyBinsCounts[c.x];
                        nonEmptyBinsCounts[c.x] += 1;
                    }
                    c.nonEmptyIdx = nonEmptyIdx;
                });
            });
            // second pass to set nbNonEmptyBins
            groupsData.forEach(function(group) {
                group.columns.forEach(c => c.nbNonEmptyBins = nonEmptyBinsCounts[c.x]);
            });

            const nbMaxColumns = Math.max(...groupsData.map(group => group.columns.filter(c => !_.isNil(c.nonEmptyIdx)).length));

            let groupWidth = ChartDimension.isUngroupedNumerical(xDimension) ? 10 : Math.max(1, xAxis.ordinalScale.rangeBand());
            // Adjust barWidth if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
            groupWidth = Math.max(groupWidth * ChartAxesUtils.getCustomExtentRatio(chartDef.xAxisFormatting.customExtent), 1);

            let barWidth = groupWidth;
            if (groupsData.length) {
                if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                    barWidth = nbMaxColumns ? groupWidth / nbMaxColumns : 0;
                } else {
                    barWidth = groupWidth / groupsData[0].columns.length;
                }
            }


            // Wrapper to contain all rectangles (used to apply clip-path)
            let wrapper = g.selectAll('g.group-wrapper');
            if (wrapper.empty()) {
                g.append('g').attr('class', 'group-wrapper');
                wrapper = g.selectAll('g.group-wrapper');
            }
            const groups = wrapper.selectAll('g.group-bars').data(groupsData);
            groups.enter().append('g')
                .attr('class', 'group-bars');
            groups.exit().remove();
            groups.attr('transform', BarChartUtils.translate('x', xAxis, xLabels, groupWidth));

            const positionRects = function(rects) {
                return rects.attr('transform', function(d, i) {
                    const yScale = getScale(d);
                    let value = chartData.aggr(d.measure).get(d),
                        s;
                    if (isAxisLogScale(d) && value === 0) {
                        value = 1;
                    }
                    if (!isAxisLogScale(d) && zeroIsInDomain(d)) {
                        s = Math.min(yScale(value), yScale(0));
                    } else {
                        s = value <= 0 ? yScale(0) : yScale(value);
                    }
                    let xTranslate = barWidth * i;
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        xTranslate = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * d.nonEmptyIdx + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2);
                    }
                    return 'translate(' + xTranslate + ', ' + s + ')';
                }).attr('height', function(d) {
                    const yScale = getScale(d);
                    let v = chartData.aggr(d.measure).get(d);
                    let h;
                    if (isAxisLogScale(d)) {
                        if (v === 0) {
                            v = 1;
                        }
                        h = chartBase.vizHeight - yScale(v);
                    } else {
                        h = Math.abs(yScale(v) - yScale(0));
                    }
                    return Math.max(h, 1);
                });
            };
            const hasColorDim = ChartColorUtils.getColorDimensionOrMeasure(chartDef) !== undefined;

            const labelCollisionDetectionHandlersByColorScaleIndex = {};
            const getFinalColorScaleIndex = (d) => hasColorDim ? d.colorScaleIndex : d.colorScaleIndex + d.measure;

            const rects = groups.selectAll('rect').data(Fn.prop('columns'));
            rects.enter().append('rect')
                .attr('fill', function(d) {
                    return chartBase.colorScale(getFinalColorScaleIndex(d));
                })
                .attr('opacity', chartDef.colorOptions.transparency)
                .each(function(d) {
                    chartBase.tooltips.registerEl(this, angular.extend({}, d, { facet: f }), 'fill', undefined, hasColorDim);
                    chartBase.contextualMenu.addContextualMenuHandler(this, angular.extend({}, d, { facet: f }));
                })
                .call(positionRects);
            rects.exit().remove();
            rects
                .attr('width', barWidth)
                .transition().ease('easeOutQuad')
                .call(positionRects);

            //mix chart uses the same drawer, hence the condition on the chartType
            if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                const getLabelYPosition = (d) => {
                    const rectValue = getRectValue(d);
                    const scaleValue = getScale(d)(rectValue);
                    const defaultSpacing = chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES ? 5 : 2;
                    const spacing = d.valuesInChartDisplayOptions.spacing ?? defaultSpacing;
                    const endBarY = isNaN(scaleValue) ? 0 : scaleValue - (rectValue >= 0 ? spacing : -spacing);

                    if (rectValue < 0) {
                        // for negative values, the label is below the bar
                        return endBarY;
                    }
                    return endBarY - d.height;
                };

                const getLabelXPosition = (d, i) => {
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        // nonEmptyIdx undefined means that the bin is empty
                        const xPos = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * (d.nonEmptyIdx + 0.5) + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2);
                        return xPos;
                    }
                    return barWidth * (i + 0.5);
                };

                const getBackgroundXPosition = (d, i) => {
                    if (HierarchicalChartsUtils.shouldIgnoreEmptyBinsForColorDimension(chartDef)) {
                        // nonEmptyIdx undefined means that the bin is empty
                        const xPos = (_.isNil(d.nonEmptyIdx) || _.isNil(d.nbNonEmptyBins)) ? 0 : (barWidth * (d.nonEmptyIdx + 0.5) + groupWidth / 2 - (barWidth * d.nbNonEmptyBins) / 2) - d.width / 2 - backgroundXMargin;
                        return xPos;
                    }
                    return barWidth * (i + 0.5) - d.width / 2 - backgroundXMargin;
                };

                const getBackgroundYPosition = (d) => {
                    const y = getLabelYPosition(d);
                    return SVGUtils.getSubTextYPosition(d, y);
                };

                const getLabelText = (d) => {
                    return BarChartUtils.shouldDisplayBarLabel(chartData.getCount(d), getRectValue(d)) ? chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d)) : '';
                };

                const labelsDrawContext = {
                    node: wrapper,
                    data: groupsData.map(d => d.columns),
                    axisName: 'x',
                    axis: xAxis,
                    labels: xLabels,
                    thickness: groupWidth,
                    opacity: chartDef.colorOptions.transparency,
                    overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                    colorScale: chartBase.colorScale,
                    getExtraData: function(d, i, n, onHoverMode) {
                        const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, labelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, chartDef.valuesInChartDisplayOptions.overlappingStrategy, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);

                        return { labelPosition: extraData.labelPosition };
                    },
                    getLabelXPosition: (d) => {
                        return d.labelPosition.x;
                    },
                    getLabelYPosition: (d) => {
                        return d.labelPosition.y;
                    },
                    getLabelText,
                    getBackgroundXPosition,
                    getBackgroundYPosition,
                    backgroundXMargin,
                    theme: chartHandler.getChartTheme()
                };

                BarChartUtils.drawLabels(labelsDrawContext);
            }

            // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
            SVGUtils.clipPaths(chartBase, g, wrapper);

            if (drawReferenceLines) {
                const isPercentScaleOnLeftYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis === 'axis1' && ChartDimension.isPercentScale([measure]));
                const isPercentScaleOnRightYAxis = chartDef.genericMeasures.some(measure => measure.displayAxis !== 'axis1' && ChartDimension.isPercentScale([measure]));

                const d3RightYAxis = ChartAxesUtils.getRightYAxis(chartBase.yAxes);
                const d3LeftYAxis = ChartAxesUtils.getLeftYAxis(chartBase.yAxes);
                const leftYFormattingOptions = d3LeftYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, d3LeftYAxis.id);
                const rightYFormattingOptions = d3RightYAxis && ChartAxesUtils.getYAxisNumberFormatting(chartDef.yAxesFormatting, d3RightYAxis.id);

                const refLinesXAxis = { ...chartBase.xAxis, isPercentScale: false, formattingOptions: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting };
                const refLinesYAxes = [];

                if (d3LeftYAxis) {
                    refLinesYAxes.push({ ...d3LeftYAxis, isPercentScale: isPercentScaleOnLeftYAxis, formattingOptions: leftYFormattingOptions });
                }
                if (d3RightYAxis) {
                    refLinesYAxes.push({ ...d3RightYAxis, isPercentScale: isPercentScaleOnRightYAxis, formattingOptions: rightYFormattingOptions });
                }

                const dataSpec = chartHandler.getDataSpec();
                const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
                const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
                ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

                const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, chartDef.$axisSpecs && chartDef.$axisSpecs.x, undefined),
                    referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures);

                ReferenceLines.drawReferenceLines(
                    wrapper,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    refLinesXAxis,
                    refLinesYAxes,
                    displayedReferenceLines,
                    referenceLinesValues
                );
            }

            const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, wrapper, d3, g);
            ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
        };
    };
})();
