(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('StackedBarsChart', StackedBarsChart);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/bars.js
     */
    function StackedBarsChart(ChartManager, ChartDimension, ChartDataWrapperFactory, Fn, StackedChartUtils, BarChartUtils, SVGUtils, ChartAxesUtils, ReferenceLines, ColumnAvailability, ChartYAxisPosition, ChartCustomMeasures, ChartUsableColumns, ColorFocusHandler, ChartDataUtils) {
        return function($container, chartDef, chartHandler, axesDef, data) {
            const backgroundXMargin = 4;
            const isPercentScale = ChartDimension.isPercentScale(chartDef.genericMeasures) || chartDef.variant == 'stacked_100';
            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const yAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const yDimension = chartDef.genericDimension0[0];
            const ySpec = { id: yAxisID, type: 'DIMENSION', name: 'y', mode: 'COLUMNS', dimension: yDimension, minRangeBand: 18, ascendingDown: true, customExtent: ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, yAxisID), position: ChartYAxisPosition.LEFT };

            const dataSpec = chartHandler.getDataSpec();
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));
            ColumnAvailability.updateAvailableColumns(chartDef.genericMeasures, allMeasures, customMeasures);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, undefined, { [yAxisID]: ySpec }),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { x: { isPercentScale }, [yAxisID]: { isPercentScale: false } });

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, ySpec.name, ySpec.dimension), referenceLinesExtents[yAxisID], ySpec);

            const animationData = StackedChartUtils.prepareData(chartDef, chartData, 'y'),
                yLabels = chartData.getAxisLabels('y'),
                xExtent = ReferenceLines.getExtentWithReferenceLines([animationData.minTotal, animationData.maxTotal], referenceLinesExtents.x);

            const drawFrame = function(frameIdx, chartBase) {
                ReferenceLines.removeReferenceLines($container[0]);

                animationData.frames[frameIdx].facets.forEach(function(facetData, f) {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    StackedBarsChartDrawer(g, facetData, chartBase, f);
                });
            };

            // Hack: by design bar charts start at 0, but if log scale is checked start at 1 (to make it possible)
            if (chartDef.xAxisFormatting.isLogScale) {
                xExtent[0] = 1;
            }

            const xSpec = { type: 'MEASURE', extent: xExtent, isPercentScale: isPercentScale, measure: chartDef.genericMeasures, customExtent: chartDef.xAxisFormatting.customExtent };
            const axisSpecs = {
                x: xSpec,
                [yAxisID]: ySpec
            };

            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'X_AXIS', isDisplayed: ['X_AXIS'], isNumerical: true, isPercentScale },
                { axis: 'LEFT_Y_AXIS', isDisplayed: ['LEFT_Y_AXIS'], isNumerical: ChartAxesUtils.isNumerical(ySpec) && !ChartDimension.hasOneTickPerBin(ySpec.dimension), isPercentScale: false, isContinuousDate: ChartAxesUtils.isContinuousDate(ySpec) }
            ]);

            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                { type: 'DIMENSION', name: 'color', dimension: chartDef.genericDimension1[0] });

            function StackedBarsChartDrawer(g, stacksData, chartBase, f) {
                const percentFormatter = d3.format('.0%');

                let barHeight = ChartDimension.isUngroupedNumerical(yDimension) ? 10 : Math.max(1, chartBase.yAxes[0].ordinalScale.rangeBand());
                // Adjust barHeight if there is a custom extent, (mainly to prevent bars from overlapping each others when extent is reduced)
                const yAxisCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, chartBase.yAxes[0].id);
                barHeight = Math.max(barHeight * ChartAxesUtils.getCustomExtentRatio(yAxisCustomExtent), 1);

                const stacks = g.selectAll('.stack-bars').data(stacksData.stacks);
                stacks.enter().append('g').attr('class', 'stack-bars');
                stacks.exit().remove();
                stacks.attr('transform', BarChartUtils.translate('y', chartBase.yAxes[0], yLabels, barHeight));

                /*
                 * Display total, not enabled for now TODO:
                 * if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.variant !== 'stacked_100') {
                 *  var totals = stacks.selectAll('text.total').data(function(d) { return [d]; });
                 *  totals.enter().append('text')
                 *      .attr('class', 'total')
                 *      .attr('y', barHeight/2)
                 *      .attr('text-anchor', 'middle')
                 *      .attr('dominant-baseline', 'middle')
                 *      .attr('font-weight', 500);
                 *  totals
                 *      .text(function(d) { return d.count > 0 ? chartBase.measureFormatters[0](d.total) : ''; })
                 *      .each(function(d) {
                 *          var bbox = this.getBoundingClientRect();
                 *          if (bbox.height > barHeight) {
                 *              d3.select(this).attr('visibility', 'hidden');
                 *          } else {
                 *              d3.select(this).attr('visibility', null);
                 *          }
                 *      })
                 *      .transition()
                 *      .attr('x', function(d) {
                 *          var bbox = this.getBoundingClientRect();
                 *          return chartBase.xAxis.scale()(d.total) + bbox.width/2 + 5;
                 *      });
                 * }
                 */

                const rects = stacks.selectAll('rect').data(Fn.prop('data'));
                const getFinalColorScaleIndex = (d) => d.color + d.measure;
                rects.enter().append('rect')
                    .attr('height', barHeight)
                    .attr('y', 0)
                    .attr('fill', function(d) {
                        return chartBase.colorScale(getFinalColorScaleIndex(d));
                    })
                    .attr('opacity', chartDef.colorOptions.transparency)
                    .each(function(d) {
                        chartBase.tooltips.registerEl(this, { measure: d.measure, y: d.y, color: d.color, facet: f }, 'fill');
                        chartBase.contextualMenu.addContextualMenuHandler(this, { y: d.y, color: d.color, facet: f });
                    });

                if (chartDef.valuesInChartDisplayOptions && chartDef.valuesInChartDisplayOptions.displayValues) {
                    const xScale = chartBase.xAxis.scale();

                    const getLabelYPosition = (d) => {
                        return (barHeight / 2) - (d.height / 2);
                    };

                    const getLabelXPosition = (d, i) => {
                        return xScale((d.top + d.base) / 2);
                    };

                    const getRectValue = (d) => {
                        return chartData.aggr(d.measure).get(d);
                    };

                    // init the collision detection
                    const globalLabelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);

                    const labelCollisionDetectionHandlersByColorScaleIndex = {};
                    const labelsDrawContext = {
                        node: g,
                        data: stacksData.stacks.map(stack => stack.data),
                        axisName: 'y',
                        axis: chartBase.yAxes[0],
                        labels: yLabels,
                        thickness: barHeight,
                        opacity: chartDef.colorOptions.transparency,
                        overlappingStrategy: chartDef.valuesInChartDisplayOptions.overlappingStrategy,
                        colorScale: chartBase.colorScale,
                        getLabelXPosition,
                        getLabelYPosition,
                        getLabelText: (d) => {
                            if (BarChartUtils.shouldDisplayBarLabel(d.count, d.value)) {
                                if (chartDef.variant === 'stacked_100') {
                                    return percentFormatter(d.value);
                                } else {
                                    return chartBase.measureFormatters[d.aggregationIndex](chartData.aggr(d.aggregationIndex).get(d));
                                }
                            } else {
                                return '';
                            }
                        },
                        getExtraData: function(d, i, n, onHoverMode) {
                            const extraData = BarChartUtils.getExtraData(d, i, this, chartBase, globalLabelCollisionDetectionHandler, labelCollisionDetectionHandlersByColorScaleIndex, getFinalColorScaleIndex, getLabelXPosition, getLabelYPosition, chartDef.valuesInChartDisplayOptions.overlappingStrategy, chartDef.valuesInChartDisplayOptions.textFormatting.fontSize, onHoverMode);

                            return { labelPosition: extraData.labelPosition };
                        },
                        getBackgroundXPosition: (d) => {
                            return getLabelXPosition(d) - d.width / 2 - backgroundXMargin;
                        },
                        getBackgroundYPosition: (d) => {
                            const y = getLabelYPosition(d);
                            return SVGUtils.getSubTextYPosition(d, y);
                        },
                        backgroundXMargin,
                        hasBackground: true,
                        theme: chartHandler.getChartTheme()
                    };

                    BarChartUtils.drawLabels(labelsDrawContext);
                }

                rects.transition()
                    .attr('width', function(d) {
                        return Math.abs(chartBase.xAxis.scale()(d.top) - chartBase.xAxis.scale()(d.base));
                    })
                    .attr('x', function(d) {
                        return chartBase.xAxis.scale()(Math.min(d.base, d.top));
                    });

                // Clip paths to prevent bars from overlapping axis when user chose a custom range which results in the bar being cropped (out of visible range)
                ChartAxesUtils.isCroppedChart(chartDef) && SVGUtils.clipPaths(chartBase, g, stacks);

                const xAxis = chartBase.xAxis ? { ...chartBase.xAxis, isPercentScale, formattingOptions: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting } : null;
                const yAxis = chartBase.yAxes[0] ? { ...chartBase.yAxes[0], formattingOptions: chartDef.yAxesFormatting[0] && chartDef.yAxesFormatting[0].axisValuesFormatting.numberFormatting } : null;

                ReferenceLines.drawReferenceLines(
                    g,
                    chartBase.vizWidth,
                    chartBase.vizHeight,
                    xAxis,
                    [yAxis],
                    displayedReferenceLines,
                    referenceLinesValues
                );

                const colorFocusHandler = ColorFocusHandler.create(chartDef, chartHandler, stacks, d3, g);
                ColorFocusHandler.appendFocusUnfocusMecanismToLegend(chartHandler.legendsWrapper.getLegend(0), colorFocusHandler);
            }
        };
    }
})();
