(function(){
'use strict';

    const app = angular.module('dataiku.widgets.drawers', ['dataiku.filters', 'dataiku.services']);

    app.constant('TOPBAR_DRAWER_IDS', {
        ADMIN: 'admin',
        OPALS_HELP: 'opalsHelp',
        APPS: 'apps',
        HELP: 'help',
        USER: 'user',
        ACTIVITY: 'activity',
        TRIAL: 'trial',
    });

    app.constant('TOPBAR_DRAWER_DOCKED_STATES', {
        NONE: 'none',
        LEFT: 'left',
        RIGHT: 'right',
    });

    app.constant('TOPBAR_DRAWER_MINIMIZED_STATES', {
        NONE: 'none',
        MINIMIZED: 'minimized',
    });

    let drawers = {};

    app.service('TopbarDrawersService', function() {
        this.getDrawer = function(drawerId) {
            return drawers[drawerId];
        };
    });

    app.directive('topbarDrawer', function($rootScope, $timeout, LocalStorage, TOPBAR_DRAWER_DOCKED_STATES, TOPBAR_DRAWER_MINIMIZED_STATES) {
        return {
            scope: true,
            link: function(scope, element, attrs) {
                // the space between the drawer and the parent edges for the
                // initial drawer position (and to calculate the drawer height)
                const MARGIN = 10;
                const MIN_HEIGHT = 57;
                const DOCK_THRESHOLD_PARENT_WIDTH = 1280 + 500; // 1280px is the min. supported screen width
                const MIN_POSITION_X_DIFF = 89;
                const MAX_POSITION_X_OFFSET = -90;
                const MIN_POSITION_Y = 0;
                const MAX_POSITION_Y_OFFSET = -56;
                const defaultState = {
                    shown: false,
                    docked: TOPBAR_DRAWER_DOCKED_STATES.RIGHT,
                    minimized: TOPBAR_DRAWER_MINIMIZED_STATES.NONE,
                };
                const localStorageKey = attrs.localStorageKey ? `dss.topbarDrawer.${attrs.localStorageKey}` : null;
                const parentDomElement = element.parent()[0];

                scope.DOCKED_STATES = TOPBAR_DRAWER_DOCKED_STATES;

                const storedState = localStorageKey ? LocalStorage.get(localStorageKey) : null;
                scope.state = !storedState || typeof storedState !== "object" ? defaultState : {
                    shown: storedState.shown && $rootScope.appConfig && $rootScope.appConfig.loggedIn && attrs.autoHide === "false",
                    docked: storedState.docked || defaultState.docked,
                    minimized: storedState.minimized || defaultState.minimized,
                    positionX: typeof storedState.positionX === "number" ? storedState.positionX : undefined,
                    positionY: typeof storedState.positionY === "number" ? storedState.positionY : undefined,
                };
                scope.moving = false;
                element.hide();

                let onDockedCallback, onMinimizedCallback;

                function getDesiredDrawerWidth() {
                    // the drawer width when not docked (needs to be aligned with
                    // .main-view__drawer--sticky.main-view__drawer--not-docked)
                    return Math.min(Math.max(Math.round(window.innerWidth / 4) + 41, 361), 501);
                }

                function updateActualDockedState(parentDomElement) {
                    scope.actualDockedState = parentDomElement.offsetWidth >= DOCK_THRESHOLD_PARENT_WIDTH
                        ? scope.state.docked
                        : TOPBAR_DRAWER_DOCKED_STATES.NONE;
                    if (onDockedCallback) {
                        onDockedCallback(scope.actualDockedState);
                    }
                }

                function saveStateIfRequired() {
                    if (localStorageKey) {
                        LocalStorage.set(localStorageKey, scope.state);
                    }
                }

                function setPosition(parentDomElement) {
                    if (scope.actualDockedState !== TOPBAR_DRAWER_DOCKED_STATES.NONE) {
                        return;
                    }

                    const desiredDrawerWidth = getDesiredDrawerWidth();
                    if (scope.state.positionX === undefined || scope.state.positionY === undefined) {
                        scope.state.positionX = Math.max(MARGIN, parentDomElement.offsetWidth - (desiredDrawerWidth + MARGIN));
                        scope.state.positionY = MARGIN;
                    } else {
                        scope.state.positionX = Math.min(
                            Math.max(scope.state.positionX, MIN_POSITION_X_DIFF - desiredDrawerWidth),
                            parentDomElement.offsetWidth + MAX_POSITION_X_OFFSET
                        );
                        scope.state.positionY = Math.min(
                            Math.max(scope.state.positionY, MIN_POSITION_Y),
                            parentDomElement.offsetHeight + MAX_POSITION_Y_OFFSET
                        );
                    }

                    saveStateIfRequired();
                }

                function updateHeight(parentDomElement) {
                    // TODO: Allow the user to resize the drawer themselves
                    scope.height = scope.state.minimized === TOPBAR_DRAWER_MINIMIZED_STATES.NONE
                        ? Math.max(MIN_HEIGHT, parentDomElement.offsetHeight - 2 * MARGIN)
                        : MIN_HEIGHT;
                }

                function assertDocked() {
                    if (scope.actualDockedState === TOPBAR_DRAWER_DOCKED_STATES.NONE) {
                        throw new Error("Drawer not docked");
                    }
                }

                function assertNotDocked() {
                    if (scope.actualDockedState !== TOPBAR_DRAWER_DOCKED_STATES.NONE) {
                        throw new Error("Drawer docked");
                    }
                }

                function minimize(state) {
                    scope.state.minimized = state;
                    if (onMinimizedCallback) {
                        onMinimizedCallback(scope.state.minimized);
                    }
                    updateHeight(parentDomElement);
                    saveStateIfRequired();
                }

                const MOVE_OVERLAY_ELEMENT_CLASS = "main-view__drawer__move-overlay";
                const MOVE_OVERLAY_POINTER_BUTTON = 0; // the left button
                const DOUBLE_CLICK_THRESHOLD = 500; // in milliseconds

                let moveOverlayGeometry = [
                    // the default geometry (it can be overriden, e.g. OPALS)
                    { top: "1px", left: "1px", width: "calc(100% - 82px)", height: "55px" },
                ];

                let viewportMoveOverlayElement = null;
                let usedPointerId = null;
                let moveOffsetX = null;
                let moveOffsetY = null;

                function isMoving() {
                    return viewportMoveOverlayElement !== null && usedPointerId !== null && moveOffsetX !== null && moveOffsetY !== null;
                }

                function assertMoving() {
                    if (!isMoving()) {
                        throw new Error("Drawer not moving");
                    }
                }

                function assertNotMoving() {
                    if (isMoving()) {
                        throw new Error("Drawer moving");
                    }
                }

                let moveOverlayPointerDownEventCount = 0;
                let moveOverlayLastPointerDownEventTimestamp = 0;

                function handleViewportMoveOverlayPointerMove(event) {
                    if (!isMoving() || event.target !== viewportMoveOverlayElement[0] || event.pointerId !== usedPointerId) {
                        return;
                    }
                    scope.$apply(() => {
                        scope.moving = true;
                        scope.state.positionX = event.clientX - moveOffsetX;
                        scope.state.positionY = event.clientY - moveOffsetY;
                        setPosition(parentDomElement);
                    });
                    // Cancel any "in progress" double click when drawer is
                    // moved
                    moveOverlayPointerDownEventCount = 0;
                }

                function handleViewportMoveOverlayPointerUp(event) {
                    if (event.target !== viewportMoveOverlayElement[0] || event.pointerId !== usedPointerId || event.button !== MOVE_OVERLAY_POINTER_BUTTON) {
                        return;
                    }
                    assertNotDocked();
                    assertMoving();
                    scope.$apply(() => {
                        scope.moving = false;
                        if (moveOverlayPointerDownEventCount === 2) {
                            minimize(
                                scope.state.minimized === TOPBAR_DRAWER_MINIMIZED_STATES.NONE
                                    ? TOPBAR_DRAWER_MINIMIZED_STATES.MINIMIZED
                                    : TOPBAR_DRAWER_MINIMIZED_STATES.NONE
                            );
                        }
                    });
                    viewportMoveOverlayElement[0].releasePointerCapture(usedPointerId);
                    viewportMoveOverlayElement.remove();
                    viewportMoveOverlayElement = usedPointerId = moveOffsetX = moveOffsetY = null;
                }

                function createViewportMoveOverlayElement() {
                    const element = angular.element("<div>");
                    element.on("pointermove", handleViewportMoveOverlayPointerMove);
                    element.on("pointerup", handleViewportMoveOverlayPointerUp);
                    element.addClass("main-view__drawer__viewport-move-overlay");
                    return element;
                }

                function handleMoveOverlayPointerDown(event) {
                    if (event.button !== MOVE_OVERLAY_POINTER_BUTTON) {
                        return;
                    }
                    assertNotDocked();
                    assertNotMoving();
                    moveOffsetX = event.clientX - scope.state.positionX;
                    moveOffsetY = event.clientY - scope.state.positionY;
                    usedPointerId = event.pointerId;
                    viewportMoveOverlayElement = createViewportMoveOverlayElement();
                    element.append(viewportMoveOverlayElement);
                    viewportMoveOverlayElement[0].setPointerCapture(usedPointerId);
                    // We don't receive any click event (and so can't read
                    // the detail property of a click event to check for double
                    // clicks) because the pointerdown and pointerup events
                    // create and remove a viewport overlay, respectively, and
                    // thus each are fired with a different target. We rely on
                    // these events and detect double clicks ourselves.
                    if (moveOverlayLastPointerDownEventTimestamp + DOUBLE_CLICK_THRESHOLD >= event.timeStamp) {
                        moveOverlayPointerDownEventCount++;
                    } else {
                        moveOverlayPointerDownEventCount = 1;
                    }
                    moveOverlayLastPointerDownEventTimestamp = event.timeStamp;
                    event.preventDefault();
                }

                function showMoveOverlay() {
                    for (const rectangle of moveOverlayGeometry) {
                        const moveOverlayElement = angular.element("<div>");
                        moveOverlayElement.on("pointerdown", handleMoveOverlayPointerDown);
                        moveOverlayElement.addClass(MOVE_OVERLAY_ELEMENT_CLASS);
                        moveOverlayElement.css(rectangle);
                        element.append(moveOverlayElement);
                    }
                }

                function destroyMoveOverlay() {
                    element.find(`.${MOVE_OVERLAY_ELEMENT_CLASS}`).remove();
                }

                function isMoveOverlayShown() {
                    return element.has(`.${MOVE_OVERLAY_ELEMENT_CLASS}`).length > 0;
                }

                function showOrDestroyMoveOverlay(newGeometry) {
                    // TODO: Turn topbarDrawer into a component and define the
                    // overlay in the template rather than with direct DOM calls
                    // here
                    if (scope.actualDockedState === TOPBAR_DRAWER_DOCKED_STATES.NONE) {
                        if (newGeometry && !angular.equals(moveOverlayGeometry, newGeometry)) {
                            destroyMoveOverlay();
                            moveOverlayGeometry = newGeometry;
                        }
                        if (!isMoveOverlayShown()) {
                            showMoveOverlay();
                        }
                    } else {
                        destroyMoveOverlay();
                        if (newGeometry) {
                            moveOverlayGeometry = newGeometry;
                        }
                    }
                }

                const resizeObserver = new ResizeObserver((entries) => {
                    scope.$apply(() => {
                        updateActualDockedState(entries[0].target);
                        setPosition(entries[0].target);
                        updateHeight(entries[0].target);
                    });
                    showOrDestroyMoveOverlay();
                });

                function hideIfClickElsewhere(event) {
                    if (event.target) {
                        var e = event.target;
                        // go back up to find a bootstrap-select or absence of <body> (that would indicate we're after a bootstrap-select closed itself)
                        var i = 0;
                        while (e) {
                            if (e.classList && e.classList.indexOf && e.classList.indexOf("bootstrap-select") >= 0) return;
                            if (e.tagName.toLowerCase() == 'body') break;
                            e = e.parentElement;
                        }
                        if (e == null) return;
                    }
                    if (!element.get(0).contains(event.target) && !event.target.classList.contains("dropdown-menu")) {
                        scope.hide();
                    }
                }

                function broadcastStateChangedEvent() {
                    const args = {
                        autoHide: attrs.autoHide !== "false",
                        shown: scope.state.shown,
                    };
                    $rootScope.$broadcast("topbarDrawerStateChanged", args);
                    if (!args.autoHide) {
                        $rootScope.$broadcast("reflow");
                    }
                }

                scope.hide = function(isImplicit) {
                    if (scope.state.shown || isImplicit) {
                        element.hide();
                        if (attrs.autoHide !== "false") {
                            $("html").unbind("click", hideIfClickElsewhere);
                        } else {
                            resizeObserver.unobserve(parentDomElement);
                        }
                        scope.state.shown = false;
                        if (!isImplicit && attrs.autoHide === "false") {
                            saveStateIfRequired();
                        }
                        broadcastStateChangedEvent();
                    }
                };

                scope.show = function(isImplicit) {
                    if (!scope.state.shown || isImplicit) {
                        scope.state.shown = true;
                        if (attrs.autoHide === "false") {
                            updateActualDockedState(parentDomElement);
                            if (scope.actualDockedState === TOPBAR_DRAWER_DOCKED_STATES.NONE && !isImplicit) {
                                scope.state.minimized = TOPBAR_DRAWER_MINIMIZED_STATES.NONE;
                                if (scope.state.positionX !== undefined && scope.state.positionY !== undefined) {
                                    const maxPositionX = parentDomElement.offsetWidth - getDesiredDrawerWidth();
                                    scope.state.positionX = Math.min(Math.max(scope.state.positionX, 0), maxPositionX);
                                    const maxPositionY = 2 * MARGIN;
                                    scope.state.positionY = Math.min(Math.max(scope.state.positionY, 0), maxPositionY);
                                }
                            }
                            if (onMinimizedCallback) {
                                onMinimizedCallback(scope.state.minimized);
                            }
                            setPosition(parentDomElement);
                            updateHeight(parentDomElement);
                            showOrDestroyMoveOverlay();
                        }
                        element.show();
                        if (attrs.autoHide !== "false") {
                            window.setTimeout(function() { $("html").click(hideIfClickElsewhere); }, 0);
                        } else {
                            resizeObserver.observe(parentDomElement);
                        }
                        if (!isImplicit) {
                            // Focus the Close button, if present
                            $timeout(() => {
                                element.find(".drawer-button--hide").focus();
                            });
                            if (attrs.autoHide === "false") {
                                saveStateIfRequired();
                            }
                        }
                        broadcastStateChangedEvent();
                    }
                };

                scope.toggle = function() {
                    if (scope.state.shown) {
                        scope.hide();
                    } else {
                        scope.show();
                    }
                };

                scope.dock = function(state) {
                    if (state === TOPBAR_DRAWER_DOCKED_STATES.NONE) {
                        throw new Error("Cannot explicitly undock drawer");
                    }

                    assertDocked();
                    scope.state.docked = scope.actualDockedState = state;
                    if (onDockedCallback) {
                        onDockedCallback(scope.state.docked);
                    }
                    saveStateIfRequired();
                };

                scope.minimize = function(state) {
                    assertNotDocked();
                    minimize(state);
                };

                scope.setMoveOverlayGeometry = function(geometry) {
                    showOrDestroyMoveOverlay(geometry);
                };

                scope.registerDrawer = function(drawer, onDocked, onMinimized) {
                    if (!drawer.toggle) {
                        drawer.toggle = scope.toggle;
                    }

                    if (!drawer.show) {
                        drawer.show = scope.show
                    }

                    if (!drawer.hide) {
                        drawer.hide = scope.hide;
                    }

                    drawer.isToggledOn = () => scope.state.shown;

                    if (!drawer.dock) {
                        drawer.dock = scope.dock;
                    }

                    if (!drawer.minimize) {
                        drawer.minimize = scope.minimize;
                    }

                    if (!drawer.setMoveOverlayGeometry) {
                        drawer.setMoveOverlayGeometry = scope.setMoveOverlayGeometry;
                    }

                    drawer.getActualDockedState = function() {
                        return scope.actualDockedState;
                    };

                    drawer.getMinimizedState = function() {
                        return scope.state.minimized;
                    };

                    if (!drawer.isEnabled) {
                        drawer.isEnabled = () => Promise.resolve(true);
                    }

                    onDockedCallback = onDocked;
                    onMinimizedCallback = onMinimized;

                    // Watch for stateChange for not kept alive drawers
                    if (attrs.autoHide !== "false") {
                        scope.$on('$stateChangeSuccess', scope.hide);
                    }

                    drawers[drawer.id] = drawer;

                    drawer.isEnabled().then(function(isEnabled) {
                        if (!isEnabled) {
                          scope.state.shown = false;
                        }

                        const isImplicit = true;
                        if (scope.state.shown) {
                            drawer.show(isImplicit);
                        } else {
                            drawer.hide(isImplicit);
                        }
                    });
                };
            }
        };
    });

    app.directive('adminDrawer', function(TOPBAR_DRAWER_IDS) {
        return {
            restrict: 'A',
            templateUrl : '/templates/widgets/topbar_drawers/admin-drawer.html',
            link : function(scope, element, attrs) {
                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.ADMIN,
                });
            }
        };
    });

    app.directive('appsDrawer', function(TOPBAR_DRAWER_IDS) {
        return {
            restrict: 'A',
            templateUrl: '/templates/widgets/topbar_drawers/apps-drawer.html',
            link: function (scope, element, attrs) {
                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.APPS,
                });

                // when clicking on a link in the menu, hide the menu right away so that it doesn't hide possible 'unsaved content' modals
                scope.hideBeforeNavigate = ($event) => {
                    const containingLink = $($event.target).parents('a');
                    if(containingLink.length) {
                        scope.hide();
                    }
                };
            }
        };
    });
    
    app.directive('trialDrawer', function(TOPBAR_DRAWER_IDS, DataikuAPI, $rootScope, $filter, RequestCenterService) {
        return {
            restrict: 'A',
            templateUrl: '/templates/widgets/topbar_drawers/trial-drawer.html',
            link: function (scope) {
                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.TRIAL,
                    toggle: () => {
                        scope.toggle();
                        const trialMenu = $('.request-drawer-menu');
                        const trialButton = $('#master-nav__trial-drawer-trigger');
                        const right = $(window).width() - trialButton.offset().left - trialMenu.outerWidth();
                        // position the drawer relative to the button or the end of the screen if it would be out of the screen
                        trialMenu.css({ 
                            right: right > 0 ? right : 200
                        });
                    }
                });
                
                DataikuAPI.requests.getLatestRequestForCurrentUser("", "INSTANCE", "")
                .then(response => {
                    scope.hasPreviousRequest = response.data.status === "PENDING";
                }, error => {
                    scope.hasPreviousRequest = false;
                    if (error.status !== 404) {
                        setErrorInScope.bind(scope)(error);
                    }
                });
                
                scope.canRequestAccess = $rootScope.appConfig.noneUsersCallToActionBehavior === 'ALLOW_REQUEST_ACCESS' || ($rootScope.appConfig.noneUsersCallToActionBehavior === 'ALLOW_START_TRIAL' && $rootScope.appConfig.allowRequestAccessWithStartedTrial);
                scope.trialPopupTitle = $filter('dateDayDiff')($rootScope.appConfig.trialStatus.expiresOn) <= 10 ? "Your trial will expire soon" : "Request access before your trial expires";
                scope.expiringTime = $filter('date')(new Date($rootScope.appConfig.trialStatus.expiresOn), "MMMM dd, yyyy 'at' HH:mm");
                
                scope.requestLicenseFromTrialUser = (message) => {
                    DataikuAPI.requests.createInstanceAccessRequest(message).success((data) => {
                        RequestCenterService.WT1Events.onRequestSent("INSTANCE", null, null, message, data.id);
                        scope.hasPreviousRequest = true;
                    }).error(setErrorInScope.bind(scope));
                };
            }
        };
    });    

    app.directive('opalsHelpDrawer', function($timeout, TOPBAR_DRAWER_IDS, OpalsService, QuestionnaireService, PageSpecificTourService) {
        return {
            restrict: 'A',
            templateUrl : '/templates/widgets/topbar_drawers/opals-help-drawer.html',
            link : function(scope) {
                scope.initialDisplay = false;

                const show = function(isImplicit) {
                    if (!scope.state.shown || isImplicit) {
                        if (!scope.initialDisplay) {
                            scope.initialDisplay = true;
                        }
                        scope.show(isImplicit);
                        OpalsService.sendWT1Event("opals-help-open", { isImplicit: !!isImplicit });
                    }
                    if (!isImplicit) {
                        $timeout(() => {
                            OpalsService.focusHelpCenterFrame();
                        });
                    }
                };

                const hide = function(isImplicit) {
                    if (scope.state.shown || isImplicit) {
                        scope.hide(isImplicit);
                        if (!isImplicit) {
                            OpalsService.sendWT1Event("opals-help-close");
                            if (QuestionnaireService.isFromQuestionnaire()) {
                                PageSpecificTourService.displayReopenTutorialPopup();
                                setTimeout(() => QuestionnaireService.setIsFromQuestionnaire(false)); // timeout is needed so it's still true when showing the Reopen Tutorial popup
                            }
                        }
                    }
                };

                let lastDockedState;
                const onDocked = function(state) {
                    if (lastDockedState !== state) {
                        OpalsService.notifyDrawerDocked(state);
                        lastDockedState = state;
                    }
                };

                let lastMinimizedState;
                const onMinimized = function(state) {
                    if (lastMinimizedState !== state) {
                        OpalsService.notifyDrawerMinimized(state);
                        lastMinimizedState = state;
                    }
                };

                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.OPALS_HELP,
                    toggle: function() {
                        if (scope.state.shown) {
                            hide();
                        } else {
                            show();
                        }
                    },
                    show: show,
                    hide: hide,
                    dock: function(state) {
                        scope.dock(state);
                        OpalsService.sendWT1Event("opals-help-dock", { state });
                    },
                    minimize: function(state) {
                        scope.minimize(state);
                        OpalsService.sendWT1Event("opals-help-minimize", { state });
                    },
                    isEnabled: OpalsService.isEnabled,
                },
                onDocked,
                onMinimized);
            }
        };
    });

    app.directive('helpDrawer', function ($rootScope, TOPBAR_DRAWER_IDS, ContextualMenu, WT1, OpalsService) {
        return {
            restrict: 'A',
            templateUrl : '/templates/widgets/topbar_drawers/help-drawer.html',
            link : function(scope, element, attrs) {

                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.HELP,
                    toggle: function() {
                        scope.toggle();
                        if (scope.state.shown) {
                            let wt1EventParams;
                            if ($rootScope.wl) {
                                wt1EventParams = {
                                    wlContextualHelpSearchEnabled: !!$rootScope.wl.contextualHelpSearchEnabled,
                                    wlGiveFeedbackModalEnabled: !!$rootScope.wl.giveFeedbackModalEnabled,
                                    wlAdditionalHelpMenuItemsNotEmpty:
                                            $rootScope.wl.additionalHelpMenuItems && $rootScope.wl.additionalHelpMenuItems.length > 0,
                                };
                            } else {
                                wt1EventParams = {};
                            }
                            if ($rootScope.appConfig) {
                                wt1EventParams.offlineFrontend = !!$rootScope.appConfig.offlineFrontend;
                            }
                            WT1.event('help-open-menu', wt1EventParams);
                        }
                    },
                    isEnabled: () => OpalsService.isEnabled().then((isEnabled) => !isEnabled),
                });

        	    /* ********** UI ************ */

            	scope.closeMenu = function(){
            		ContextualMenu.prototype.closeAny();
            	}

                OpalsService.getHelpCenterUrl().then(function(url) {
                    scope.helpCenterUrl = url;
                });
            }
        };
    });

    app.directive('userDrawer', function($filter, TOPBAR_DRAWER_IDS, DataikuAPI, Notification) {
        return {
            restrict: 'A',
            templateUrl : '/templates/widgets/topbar_drawers/user-drawer.html',
            link : function(scope, element, attrs) {
                scope.context = "drawer";

                function getPluginsList() {
                    DataikuAPI.plugins.list().success(function(data) {
                        if (scope.orderedItemsWithDays) {
                            scope.orderedItemsWithDays.forEach(function(x, i) {
                                if (x.evt) {
                                    x.evt.pluginsList = data;
                                }
                            });
                        }
                    });
                }

                function getNotifications() {
                    DataikuAPI.notifications
                        .get()
                        .success((data) => {
                            update(data);
                            ack();
                        })
                        .error(setErrorInScope.bind(scope));
                }


                //we want to be aware of new unread notifications while the notification panel
                //is open to automatically load and display them
                const releaseFn = Notification.registerEvent("update-notifications-count", (_evt, message) => {
                    if (scope.state.shown && message.totalUnread > 0) {
                        getNotifications();
                    }
                });
                scope.$on("$destroy", releaseFn);

                function ack() {
                    //we wanna ack only if the panel is open, and there's unread notifications
                    if (scope.state.shown && scope.pnotifications && scope.pnotifications.totalUnread > 0) {
                        //the acknowledge api call set as "read" all user's message
                        DataikuAPI.notifications
                            .ack(scope.pnotifications.timestamp)
                            .success(() => {
                                //we sync the local state only if the server did it
                                //to avoid unsync issues in case of network/downtime/... errors
                                scope.pnotifications.totalUnread = 0;
                                scope.pnotifications.notifications = scope.pnotifications.notifications.map((n) => {
                                    n.unread = false;
                                    return n;
                                });
                            })
                            .error(setErrorInScope.bind(scope));
                    }
                }

                function humanReadableObjectType(objectType) {
                    if (!objectType) return;
                    switch(objectType) {
                    case "MANAGED_FOLDER":
                        return "folder";
                    case "SAVED_MODEL":
                        return "model";
                    case "MODEL_EVALUATION_STORE":
                        return "evaluation store";
                    case "LAMBDA_SERVICE":
                        return "API service";
                    default:
                        return objectType.toLowerCase().replace('_', ' ');
                    }
                }

                function update(data) {
                    scope.timelineReady = true;
                    scope.pnotifications = data;
                    if (!data || !data.notifications) {
                        return;
                    }

                    //add a "day" attribute to all the menu elements to show grouping by day
                    $.each(data.notifications, function(idx, elt) {
                        elt.day = $filter('friendlyDate')(elt.timestamp);
                        if (elt.evt.objectType) {
                            elt.evt.humanReadableObjectType = humanReadableObjectType(elt.evt.objectType);
                        }
                        if (elt.evt.item) {
                            elt.evt.item.humanReadableObjectType = humanReadableObjectType(elt.evt.item.objectType);
                        }
                    });

                    var orderedItems = $filter('orderBy')(data.notifications, '-timestamp');
                    orderedItems = orderedItems.slice(0, maxItems);

                    /* Insert days separators */
                    scope.orderedItemsWithDays = [];
                    orderedItems.forEach(function(x, i) {
                        if (x.evt.objectType === "PLUGIN") {
                            x.evt.pluginsList = scope.pluginsList;
                        }
                        if (x.evtType === 'code-env-request-granted') {
                            x.evt.isInstalledCodeEnv = true;
                        }
                        if (i === 0) {
                            scope.orderedItemsWithDays.push({isSeparator: true, day : x.day});
                            scope.orderedItemsWithDays.push({...x}); //we clone it, to keep the unread for css
                        } else if (x.day === orderedItems[i-1].day) {
                            scope.orderedItemsWithDays.push({...x});
                        } else {
                            scope.orderedItemsWithDays.push({isSeparator: true, day : x.day});
                            scope.orderedItemsWithDays.push({...x});
                        }
                    });
                }

                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.USER,
                    toggle: function() {
                        scope.toggle();
                        if (scope.state.shown) {
                            maxItems = 15;
                            scope.timelineReady = false;
                            getNotifications();
                            getPluginsList();
                        }
                    },
                });

                var maxItems = 15;
                scope.scroll = function() {
                    if (!scope.state.shown || !scope.pnotifications || !scope.pnotifications.notifications) {
                        return;
                    }
                    maxItems += 5;
                    update(scope.pnotifications);
                };
            }
        };
    });

    app.directive('activityDrawer', function(TOPBAR_DRAWER_IDS, DataikuAPI, ActiveProjectKey) {
        return {
            restrict: 'A',
            templateUrl : '/templates/widgets/topbar_drawers/activity-drawer.html',
            link : function(scope) {
                scope.runnings = [];

                function isScenarioFuture(future) {
                    try {
                        return future.payload.action == 'run_scenario';
                    } catch (e) { /* Nothing for now */ }
                    return false;
                }

                scope.getActivityInfo = function() {
                    DataikuAPI.running.listPersonal().success(function(data) {
                        var runnings = [];
                        //scenario have normal futures, but we put them in another tab
                        data.futures.filter(function(f){return isScenarioFuture(f);}).forEach(function(o){
                            o.$runningType = 'scenario';
                            o.$id = o.scenarioId;
                            runnings.push(o);
                        });
                        data.futures.filter(function(f){return !isScenarioFuture(f);}).forEach(function(o){
                            o.$runningType = 'future';
                            o.$id = o.jobId;
                            runnings.push(o);
                        });
                        data.jobs.forEach(function(o){
                            o.$runningType = 'job';
                            o.$id = o.jobId;
                            runnings.push(o);
                        });
                        data.notebooks.forEach(function(o){
                            o.$runningType = 'notebook';
                            o.$id = o.name;
                            runnings.push(o)
                        });
                        scope.runnings = runnings;
                    }).error(setErrorInScope.bind(scope));
                };

                scope.getConnectedUsers = function() {
                    DataikuAPI.security.listConnectedUsers(ActiveProjectKey.get())
                        .success(connectedUsers => scope.connectedUsers = connectedUsers.filter(u => u.login !== scope.appConfig.user.login))
                        .error(setErrorInScope.bind(scope));
                }

                scope.Math = window.Math; // for the display of the running time

                // TODO : auto refresher but beware of huge calls
                // var refresher;
                // scope.$on('$destroy', function(){
                //     $interval.cancel(refresher);
                // });

                // scope.$watch('state.shown', function(nv) {
                //     if (nv) { refresher = $interval(scope.getActivityInfo,5000) }
                //     else { $interval.cancel(refresher) };
                // })

                scope.refreshActivityDrawer = () => {
                    scope.getConnectedUsers();
                    scope.getActivityInfo();
                };

                scope.registerDrawer({
                    id: TOPBAR_DRAWER_IDS.ACTIVITY,
                    toggle: function() {
                        scope.toggle();
                        if (scope.state.shown) {
                            scope.refreshActivityDrawer();
                        }
                    },
                });
            }
        };
    });

    app.directive('conflictIcon',function($rootScope, $timeout, ConflictDetector) {
        return {
            restrict: 'A',
            transclude: true,
            templateUrl : '/templates/widgets/conflict-icon.html',
            link : function(scope, element) {

                var updateWarningState = function() {
                    scope.warn = false;
                    for(var k in scope.conflicts) {
                        if(scope.conflicts[k].warn) {
                            scope.warn = true;
                        }
                    }
                    if(scope.warn) {
                        element.addClass('warn');
                        if (scope.needToDisplayWarningPopUp) {
                            // First time there is a conflict on this page, we want to display the popup to let the user know
                            $timeout(function() {
                                var popoverIcon = element.find('>span');
                                if(popoverIcon && popoverIcon[0] && popoverIcon[0].showPopover) {
                                    popoverIcon[0].showPopover();
                                }
                            });

                            // Set the needToDisplayWarningPopUp variable to false, so we don't spam the user
                            scope.needToDisplayWarningPopUp = false;
                        }
                    } else {
                        // No more conflict
                        element.removeClass('warn');
                        scope.needToDisplayWarningPopUp = true;
                    }
                }
                $rootScope.$watch('appConfig.login',function(nv) {
                    scope.currentUserLogin = nv;
                });
                $rootScope.$on('conflict-list-changed',function() {
                    scope.conflicts = ConflictDetector.listConflicts();
                    updateWarningState();
                });
                scope.conflicts = ConflictDetector.listConflicts();
                scope.needToDisplayWarningPopUp = true;
                updateWarningState();
            }
        };

    });

    app.directive('futureMainTarget', function($filter) {
        return {
            scope: false,
            link: function($scope, element, attrs) {
                $scope.icon = null;
                var updateIcon = function() {
                    var future = $scope.$eval(attrs.futureMainTarget);
                    if (future && future.payload) {
                        // special cases first
                        if (future.payload.action == 'run_sql') {
                            $scope.icon = 'icon-code_sql_recipe';
                        } else if (future.payload.action == 'export') {
                            $scope.icon = 'icon-download';
                        } else if (future.payload.targets.length > 0) {
                            var main = future.payload.targets[0];
                            $scope.icon = $filter('typeToIcon')(main.objectType);
                        }
                    }
                };
                $scope.$watch(attrs.futureMainTarget, updateIcon);
            }
        };
    });
    app.directive('futureMainPayload', function($filter) {
        return {
            scope: false,
            link: function($scope, element, attrs) {
                $scope.futurePayload = null;
                $scope.futureDisplayName = null;
                var updatePayload = function() {
                    var future = $scope.$eval(attrs.futureMainPayload);
                    if (future && future.payload) {
                        if (future.payload.action == 'remote') {
                            $scope.futurePayload = future.payload.extras.remotePayload;
                            $scope.futureDisplayName = future.payload.extras.remotePayload.displayName;
                        } else {
                            $scope.futurePayload = future.payload;
                            $scope.futureDisplayName = future.jobDisplayName;
                        }
                    }
                };
                $scope.$watch(attrs.futureMainPayload, updatePayload);
            }
        };
    });
    app.directive('futureProgressBar', function(ProgressStackMessageBuilder) {
        return {
            scope: false,
            link: function($scope, element, attrs) {
                $scope.bar = {};
                var updateBar = function() {
                    var progress = $scope.$eval(attrs.futureProgressBar);
                    if (progress && progress.states && progress.states.length > 0) {
                        $scope.bar.percentage = ProgressStackMessageBuilder.getPercentage(progress);
                        $scope.bar.perpetual = false;
                    } else {
                        $scope.bar.perpetual = true;
                    }
                };
                $scope.$watch(attrs.futureProgressBar, updateBar, true);
            }
        };
    });

    app.directive('futureAbortConfirmation', function() {
        return {
            restrict: 'AE',
            scope : {
                future:'=futureAbortConfirmation',
                abortFn: '&',
                abortMsg: '=',
                abortTitle: '='
            },
            templateUrl: "/templates/widgets/topbar_drawers/future-abort-confirmation.html",
            link : function($scope, element, attrs) {
            }
        }
    });

    app.directive('activityFutureDisplay', function(DataikuAPI, StateUtils) {
        return {
            restrict: 'AE',
            scope : {
                future:'=activityFutureDisplay',
                refreshList:'=',
                inAdmin:'='
            },
            templateUrl: "/templates/widgets/topbar_drawers/activity-future-display.html",
            link : function($scope, element, attrs) {
                $scope.StateUtils = StateUtils;
                $scope.abortFuture = function(jobId) {
                    DataikuAPI.futures.abort(jobId).success(function(data) {
                        $scope.refreshList();
                    }).error(setErrorInScope.bind($scope));
                };
            }
        }
    });

    app.directive('activityScenarioDisplay', function(DataikuAPI, StateUtils) {
        return {
            restrict: 'AE',
            scope : {
                scenario:'=activityScenarioDisplay',
                refreshList:'=',
                inAdmin:'='
            },
            templateUrl: "/templates/widgets/topbar_drawers/activity-scenario-display.html",
            link : function($scope, element, attrs) {
                $scope.StateUtils = StateUtils;
                $scope.abortFuture = function(jobId) {
                    DataikuAPI.futures.abort(jobId).success(function(data) {
                        $scope.refreshList();
                    }).error(setErrorInScope.bind($scope));
                };
            }
        }
    });

    app.directive('activityClusterKernelDisplay', function(DataikuAPI, StateUtils) {
        return {
            restrict: 'AE',
            scope : {
                clusterKernel:'=activityClusterKernelDisplay',
                refreshList:'=',
                inAdmin:'='
            },
            templateUrl: "/templates/widgets/topbar_drawers/activity-cluster-kernel-display.html",
            link : function($scope, element, attrs) {
                $scope.StateUtils = StateUtils;
                $scope.abortKernel = function(clusterKernel) {
                    clusterKernel.aborted = true;
                    DataikuAPI.admin.clusters.abortKernel(clusterKernel.prefix, clusterKernel.kernelId).success(function(data) {
                        $scope.refreshList();
                    }).error(setErrorInScope.bind($scope));
                };
            }
        }
    });

    app.directive('activityNotebookDisplay', function(DataikuAPI, StateUtils) {
        return {
            restrict: 'AE',
            scope : {
                notebook:'=activityNotebookDisplay',
                refreshList:'=',
                inAdmin:'='
            },
            templateUrl: "/templates/widgets/topbar_drawers/activity-notebook-display.html",
            link : function($scope, element, attrs) {
                $scope.StateUtils = StateUtils;
                $scope.abortNotebook = function(jobId) {
                    DataikuAPI.jupyterNotebooks.unload(jobId).success(function(data) {
                        $scope.refreshList();
                    }).error(setErrorInScope.bind($scope));
                };
            }
        }
    });

    app.directive('activityJobDisplay', function(DataikuAPI, StateUtils) {
        return {
            restrict: 'AE',
            scope : {
                job:'=activityJobDisplay',
                refreshList:'=',
                inAdmin:'='
            },
            templateUrl: "/templates/widgets/topbar_drawers/activity-job-display.html",
            link : function($scope, element, attrs) {
                $scope.StateUtils = StateUtils;
                $scope.abortJob = function(projectKey, jobId) {
                    DataikuAPI.flow.jobs.abort(projectKey, jobId).success(function(data) {
                        $scope.refreshList();
                    }).error(setErrorInScope.bind($scope));
                };
            }
        }
    });

})();
