// Always put beforeEach(module("dataiku.mock")); last in tests
(function() {

    const defaultTheme = {
        colors: [],
        palettes: {}
    };

    angular.module('dataiku.mock', [])

        .instantiable('Logger', function(LoggerProvider) {
            return LoggerProvider.getLogger();
        })

        .factory('LoggerProvider', function() {
            return {
                getLogger: function() {
                    var ret = {};
                    ['log', 'warn', 'debug', 'info', 'error'].forEach(function(v) {
                        ret[v] = function() {
                        };
                    });
                    return ret;
                }
            }
        })

        .factory('translate', function() {
            return function(translateID, defaultValue) {
                return defaultValue;
            }
        })

        .factory('DataikuAPI', function() {
            return {};
        })

        // mock of chart-tensor-data-wrapper.model.ts
        .value('ChartTensorDataWrapper', function(data, axesDef) {
            let that = {
                coords: [],
                data: data,
                getAggrExtent: function(aggrIdx) {
                    return d3.extent(data.aggregations[aggrIdx].tensor);
                }
            }

            if (!axesDef) {
                return that;
            }

            that = {
                ...that,
                numAxes: data.axisLabels.length,
                axesDef: axesDef,
                subtotalCoords: data.axisLabels.map(axisLabels => axisLabels.findIndex(axisLabel => axisLabel.label === '___dku_total_value___')),
                aggr: function(aggrIdx) {
                    return {
                        get: function(coordsDict) {
                            return that.getAggrPoint(aggrIdx, coordsDict);
                        },

                        getAxisValue: function(axisName, axisCoord) {
                            return data.aggregations[aggrIdx].axes[that.axesDef[axisName]][axisCoord];
                        }
                    }
                },
                getCount: function(coordsDictOrBin) {
                    if (typeof coordsDictOrBin === 'number') {
                        return this.data.counts.tensor[coordsDictOrBin];
                    }
                    return this.getPoint(this.data.counts, this.getCoordsArray(coordsDictOrBin));
                },
                getNonNullCount: function(coordsDictOrBin, aggrIdx) {
                    let bin;
                    if (typeof coordsDictOrBin !== 'number') {
                        bin = that.getCoordsLoc(that.data.aggregations[aggrIdx], that.getCoordsArray(coordsDictOrBin));
                    } else {
                        bin = coordsDictOrBin;
                    }
            
                    if (that.data.aggregations[aggrIdx].nonNullCounts) {
                        return that.data.aggregations[aggrIdx].nonNullCounts[bin];
                    } else {
                        // When the aggregation has no null value, nonNullCounts isn't sent because nonNullCounts == counts
                        return that.data.counts.tensor[bin];
                    }
                },
                isBinMeaningful(aggrFn, coordsDictOrBin, measureIdx) {
                    if (this.getNonNullCount(coordsDictOrBin, measureIdx) > 0) {
                        return true;
                    }

                    const count = this.getCount(coordsDictOrBin);
                    return (aggrFn === 'COUNT' || aggrFn === 'COUNTD') && count !== 0;
                },
                getAggrPoint: function(aggrIdx, coordsDict) {
                    return that.getPoint(data.aggregations[aggrIdx], that.getCoordsArray(coordsDict));
                },
                getAggrLoc(aggrIdx, coordsDict) {
                    return this.getCoordsLoc(this.data.aggregations[aggrIdx], this.getCoordsArray(coordsDict));
                },
                getSubtotalPoint: function(aggrIdx, coordsDict) {
                    const aggregation = data.aggregations[aggrIdx];
                    return that.getSubtotal(aggregation, that.getSubTotalCoordsArray(coordsDict));
                },
                getGrandTotalPoint: function(aggrIdx) {
                    const aggregation = data.aggregations[aggrIdx];
                    return that.getSubtotal(aggregation, that.getSubTotalCoordsArray({}));
                },
                getCoordsLoc: function(tensor, coordsArray) {
                    let loc = 0;
                    for (let i = 0; i < that.numAxes; i++) {
                        loc += coordsArray[i] * tensor.multipliers[i];
                    }
                    return loc;
                },
                getPoint: function(tensor, coordsArray) {
                    return tensor.tensor[that.getCoordsLoc(tensor, coordsArray)];
                },
                getSubtotal: function(tensor, coordsArray) {
                    return tensor.tensor[that.getCoordsLoc(tensor, coordsArray)];
                },
                getCoordsArray: function(coordsDict) {
                    for (const axisName in coordsDict) {
                        that.coords[that.axesDef[axisName]] = coordsDict[axisName];
                    }
                    return that.coords;
                },
                hasSubtotalCoords: function() {
                    return that.aggr.length === 1 || that.subtotalCoords && that.subtotalCoords.length > 1 && that.subtotalCoords[0] !== -1;
                },
                getSubTotalCoordsArray: function(coordsDict) {
                    if (!that.hasSubtotalCoords()) {
                        throw new Error('Option `computeSubTotals` must be enabled to fetch subtotals in pivot tensor request');
                    }
                    const coords = {};
                    for (const axisName in that.axesDef) {
                        if (coordsDict[axisName] !== undefined) {
                            coords[that.axesDef[axisName]] = coordsDict[axisName];
                        } else {
                            coords[that.axesDef[axisName]] = that.subtotalCoords[that.axesDef[axisName]];
                        }
                    }
                    return coords;
                },
                getAxisLabels: function(axisName) {
                    return data.axisLabels[that.axesDef[axisName]];
                },
                getSubtotalLabelIndex: function(axisName) {
                    return that.subtotalCoords[that.axesDef[axisName]];
                },
                getMinValue: function(axisName) {
                    const axisLabels = this.getAxisLabels(axisName);
                    if (!axisLabels || !axisLabels.length) {
                        return null;
                    }
                    const { min, max, sortValue } = axisLabels[0];
                    return min === max ? sortValue : min;
                },
                getMaxValue: function(axisName) {
                    const axisLabels = this.getAxisLabels(axisName);
                    if (!axisLabels || !axisLabels.length) {
                        return null;
                    }
                    const { min, max, sortValue } = axisLabels[axisLabels.length - 1];
                    return min === max ? sortValue : max;
                },
                getNumValues: function(axisName) {
                    const axisLabels = this.getAxisLabels(axisName);
                    if (!axisLabels || !axisLabels.length) {
                        return null;
                    }
                    return axisLabels.length;
                },
                getAxisDef: function(axisName) {
                    return data.axisDefs[that.axesDef[axisName]];
                },
                getLabels: function() {
                    return data.axisLabels;
                },
                getAxisIdx: function(axisName) {
                    return that.axesDef[axisName];
                },
                fixAxis: function(axisName, binIdx) {
                    that.coords[that.axesDef[axisName]] = binIdx;
                    return that;
                },
                getCurrentCoord: function(axisName) {
                    return that.coords[that.axesDef[axisName]];
                }
            };

            return that;
        })

        // mock of chart-tensor-data-wrapper-factory.service.ts
        .factory('ChartDataWrapperFactory', function(ChartTensorDataWrapper, ChartColorUtils, ChartLabels) {
            return {
                chartTensorDataWrapper(data, axesDef) {
                    return new ChartTensorDataWrapper(data, axesDef, ChartColorUtils, ChartLabels, this);
                }
            }
        })

        // mock of number-formatter.service.ts
        .factory('NumberFormatter', function() {
            const ret = {
                longSmartNumberFilter: function() {
  
                    return '';
                }
            };

            return ret;
        })

        // mock of chart-static-data.service.ts
        .factory('ChartsStaticData', function() {

            function createDateMode(value, label, group) {
                return {
                    group,
                    value,
                    label
                };
            }

            function buildBinNumberConfiguration(chartType, valueForMainDimension, valueForOtherDimension) {
                return {
                    chartType,
                    valueForMainDimension,
                    valueForOtherDimension
                };
            }

            const GROUP_FIXED_TIMELINE = 'Fixed timeline';
            const GROUP_DYNAMIC_TIMELINE = 'Dynamic timeline';
            const GROUP_REGROUP = 'Regroup';
            const AUTOMATIC = createDateMode('AUTOMATIC', 'Automatic', GROUP_DYNAMIC_TIMELINE);
            const YEAR = createDateMode('YEAR', 'Year', GROUP_FIXED_TIMELINE);
            const QUARTER = createDateMode('QUARTER', 'Quarter', GROUP_FIXED_TIMELINE);
            const MONTH = createDateMode('MONTH', 'Month', GROUP_FIXED_TIMELINE);
            const WEEK = createDateMode('WEEK', 'Week', GROUP_FIXED_TIMELINE);
            const DAY = createDateMode('DAY', 'Day', GROUP_FIXED_TIMELINE);
            const HOUR = createDateMode('HOUR', 'Hour', GROUP_FIXED_TIMELINE);
            const MINUTE = createDateMode('MINUTE', 'Minute', GROUP_FIXED_TIMELINE);
            const SECOND = createDateMode('SECOND', 'Second', GROUP_FIXED_TIMELINE);
            const QUARTER_OF_YEAR = createDateMode('QUARTER_OF_YEAR', 'Quarter of year', GROUP_REGROUP);
            const MONTH_OF_YEAR = createDateMode('MONTH_OF_YEAR', 'Month of year', GROUP_REGROUP);
            const WEEK_OF_YEAR = createDateMode('WEEK_OF_YEAR', 'Week of year', GROUP_REGROUP);
            const DAY_OF_MONTH = createDateMode('DAY_OF_MONTH', 'Day of month', GROUP_REGROUP);
            const DAY_OF_WEEK = createDateMode('DAY_OF_WEEK', 'Day of week', GROUP_REGROUP);
            const HOUR_OF_DAY = createDateMode('HOUR_OF_DAY', 'Hour of day', GROUP_REGROUP);

            const TIMELINE_DATE_MODES = [
                YEAR,
                QUARTER,
                MONTH,
                WEEK,
                DAY,
                HOUR,
                MINUTE,
                SECOND
            ];
            const GROUPED_DATE_MODES = [
                QUARTER_OF_YEAR,
                MONTH_OF_YEAR,
                WEEK_OF_YEAR,
                DAY_OF_MONTH,
                DAY_OF_WEEK,
                HOUR_OF_DAY
            ];

            const BACKEND_ONLY_DATE_MODES = [
                createDateMode('QUARTER_OF_DAY', 'Quarter of day', 'NA'),
                createDateMode('QUARTER_OF_HOUR', 'Quarter of hour', 'NA'),
                createDateMode('QUARTER_OF_MINUTE', 'Quarter of minute', 'NA')
            ];

            const DATE_MODES = [AUTOMATIC].concat(TIMELINE_DATE_MODES).concat(GROUPED_DATE_MODES);

            const AUTO_EXTENT_MODE = 'AUTO';
            const MANUAL_EXTENT_MODE = 'MANUAL';

            const ALL_MULTIPLIERS = {
                'NONE': { label: 'None', powerOfTen: 0 },
                'THOUSANDS': { label: 'Thousands', powerOfTen: 3, symbol: 'k' },
                'MILLIONS': { label: 'Millions', powerOfTen: 6, symbol: 'M' },
                'BILLIONS': { label: 'Billions', powerOfTen: 9, symbol: 'B' },
                'TRILLIONS': { label: 'Trillions', powerOfTen: 12, symbol: 'T' },
                'QUADRILLIONS': { label: 'quadrillions', powerOfTen: 15, symbol: 'P' },
                'QUINTILLIONS': { label: 'quintillions', powerOfTen: 18, symbol: 'E' },
                'SEXTILLIONS': { label: 'sextillions', powerOfTen: 21, symbol: 'Z' },
                'SEPTILLIONS': { label: 'septillions', powerOfTen: 24, symbol: 'Y' }
            };

            const AVAILABLE_MULTIPLIERS = {
                'AUTO': { label: 'Auto' },
                'NONE': ALL_MULTIPLIERS.NONE,
                'THOUSANDS': ALL_MULTIPLIERS.THOUSANDS,
                'MILLIONS': ALL_MULTIPLIERS.MILLIONS,
                'BILLIONS': ALL_MULTIPLIERS.BILLIONS
            };

            const DEFAULT_COLOR_OPTIONS = {
                singleColor: '#2678B1',
                transparency: 0.75
            };
            const DEFAULT_FILL_OPACITY = 0.6;
            const DEFAULT_STROKE_WIDTH = 2;

            const svc = {
                stdAggrMeasureComputeModes: {
                    'NORMAL': ['NORMAL', 'Normal'],
                    // "INDICE_100": ["INDICE_100", "100-indexed"],
                    'CUMULATIVE': ['CUMULATIVE', 'Cumulative values'],
                    'DIFFERENCE': ['DIFFERENCE', 'Differential values'],
                    'LOG_SCALE': ['LOG_SCALE', 'Log scale'],
                    'PERCENTAGE': ['PERCENTAGE', 'Percentage scale'],
                    'CUMULATIVE_PERCENTAGE': ['CUMULATIVE_PERCENTAGE', 'Cumulative percentage scale'],
                    'AVG_RATIO': ['AVG_RATIO', 'Ratio to average']
                },

                mapAdminLevels: [
                    [2, 'Country'],
                    [4, 'Region/State'],
                    [6, 'Department/County'],
                    [7, 'Metropolis'],
                    [8, 'City']
                ],
                dateModes: DATE_MODES,
                defaultDateMode: AUTOMATIC,
                AUTOMATIC_DATE_MODE: AUTOMATIC,
                availableMultipliers: AVAILABLE_MULTIPLIERS,
                allMultipliers: ALL_MULTIPLIERS,
                highestMultiplier: ALL_MULTIPLIERS.SEPTILLIONS,
                autoMultiplier: AVAILABLE_MULTIPLIERS.Auto,
                AUTO_EXTENT_MODE: AUTO_EXTENT_MODE,
                MANUAL_EXTENT_MODE: MANUAL_EXTENT_MODE,
                extentEditModes: {
                    AUTO_EXTENT_MODE: { value: AUTO_EXTENT_MODE, label: 'Auto' },
                    MANUAL_EXTENT_MODE: { value: MANUAL_EXTENT_MODE, label: 'Manual' }
                },
                pivotTableMeasureDisplayMode: {
                    ROWS: 'ROWS',
                    COLUMNS: 'COLUMNS'
                },
                defaultPivotDisplayTotals: {
                    subTotals: { rows: true, columns: true },
                    grandTotal: { row: true, column: true }
                },
                GEOM_AGGREGATIONS: {
                    DEFAULT: undefined,
                    DISTINCT: 'DISTINCT'
                },

                ANY_AND_NUMERICAL_RESULT_AGGREGATION: ['COUNTD', 'COUNT'],
                ALPHANUM_AND_ALPHANUMERICAL_RESULT_AGGREGATION: ['OBJECT_MIN', 'OBJECT_MAX'],

                AVAILABLE_AGGREGATIONS: {
                    DATE: ['COUNTD', 'COUNT', 'MIN', 'MAX'],
                    ALPHANUM: ['COUNTD', 'COUNT', 'OBJECT_MIN', 'OBJECT_MAX'],
                    NUMERICAL: ['COUNTD', 'COUNT', 'SUM', 'AVG', 'MIN', 'MAX', 'STDEV', 'STDEV_POPULATION', 'VARIANCE', 'VARIANCE_POPULATION']
                },

                LABEL_POSITIONS: {
                    BOTTOM: 'BOTTOM',
                    TOP: 'TOP'
                },

                DEFAULT_COLOR_OPTIONS,
                DEFAULT_STROKE_WIDTH,
                DEFAULT_FILL_OPACITY,
                MEASURES_PERCENT_MODES: ['PERCENT', 'PERCENTAGE', 'CUMULATIVE_PERCENTAGE'],
                YEAR,
                QUARTER_OF_YEAR,
                MONTH_OF_YEAR,
                WEEK_OF_YEAR,
                DAY_OF_MONTH,
                DAY_OF_WEEK,
                HOUR_OF_DAY,
                INDIVIDUAL: createDateMode('INDIVIDUAL', 'Individual dates', GROUP_REGROUP),
                RELATIVE_CUSTOM_YEAR: createDateMode('YEAR', 'Year', GROUP_FIXED_TIMELINE),
                RELATIVE_CUSTOM_QUARTER: createDateMode('QUARTER_OF_YEAR', 'Quarter', GROUP_FIXED_TIMELINE),
                RELATIVE_CUSTOM_MONTH: createDateMode('MONTH_OF_YEAR', 'Month', GROUP_FIXED_TIMELINE),
                RELATIVE_CUSTOM_WEEK: createDateMode('WEEK_OF_MONTH', 'Day', GROUP_FIXED_TIMELINE),
                RELATIVE_CUSTOM_DAY: createDateMode('DAY_OF_MONTH', 'Day', GROUP_FIXED_TIMELINE),
                RELATIVE_CUSTOM_HOUR: createDateMode('HOUR_OF_DAY', 'Hour', GROUP_FIXED_TIMELINE),
                NUMERICAL_BIN_NUMBERS: [
                    buildBinNumberConfiguration('grouped_columns', 10, 5),
                    buildBinNumberConfiguration('stacked_bars', 10, 5),
                    buildBinNumberConfiguration('stacked_columns', 10, 5),
                    buildBinNumberConfiguration('binned_xy', 10, 10)
                ],
                AUTOMATIC_MAX_BIN_NUMBERS: [
                    buildBinNumberConfiguration('grouped_columns', 30, 5),
                    buildBinNumberConfiguration('stacked_bars', 30, 5),
                    buildBinNumberConfiguration('stacked_columns', 30, 5),
                    buildBinNumberConfiguration('binned_xy', 10, 10),
                    buildBinNumberConfiguration('lines', 1000, 10),
                    buildBinNumberConfiguration('stacked_area', 1000, 10),
                    buildBinNumberConfiguration('multi_columns_lines', 30, 10),
                    buildBinNumberConfiguration('pie', 30, 10)
                ],
                DEFAULT_DATE_RANGE_FILTER_TYPE: createDateMode('RANGE', 'Date range'),
                DEFAULT_DATE_RELATIVE_FILTER_TYPE: createDateMode('RELATIVE', 'Relative range'),
                DEFAULT_DATE_PART_FILTER_TYPE: createDateMode('PART', 'Date part'),
                TIMELINE_AND_AUTOMATIC_DATE_MODES: [AUTOMATIC].concat(TIMELINE_DATE_MODES),
                DATE_MODES_WITH_BACKEND_ONLY: DATE_MODES.concat(BACKEND_ONLY_DATE_MODES),
                BIN_NUMBER_DEFAULT: buildBinNumberConfiguration('default', 30, 30)
            };

            return svc;
        })

        // mock of chart-ua-dimension.service.ts
        .factory('ChartUADimension', function(ChartsStaticData) {
            const svc = {
                isTrueNumerical: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return dimension.type === 'NUMERICAL' && !dimension.treatAsAlphanum;
                },
                isAlphanumLike: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return dimension.type === 'ALPHANUM' || (dimension.type === 'NUMERICAL' && dimension.treatAsAlphanum);
                },
                isDiscreteDate: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return dimension.type === 'DATE' && dimension.dateMode !== 'RANGE';
                },
                isDateRange: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return dimension.type === 'DATE' && dimension.dateMode === 'RANGE';
                },
                isDate: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return dimension.type === 'DATE';
                },
                getDateModes() {
                    return [
                        ChartsStaticData.DEFAULT_DATE_RANGE_FILTER_TYPE,
                        ChartsStaticData.YEAR,
                        ChartsStaticData.QUARTER_OF_YEAR,
                        ChartsStaticData.MONTH_OF_YEAR,
                        ChartsStaticData.WEEK_OF_YEAR,
                        ChartsStaticData.DAY_OF_MONTH,
                        ChartsStaticData.DAY_OF_WEEK,
                        ChartsStaticData.HOUR_OF_DAY
                    ];
                },
                areAllNumericalOrDate(chartDef) {
                    if (chartDef.type === 'scatter_multiple_pairs') {
                        const completePairs = this.getDisplayableDimensionPairs(chartDef.uaDimensionPair);
                        return completePairs && completePairs.length && completePairs.every(pair => {
                            return (this.isTrueNumerical(pair.uaXDimension[0]) && this.isTrueNumerical(pair.uaYDimension[0])) || (this.isDate(pair.uaXDimension[0]) && this.isDate(pair.uaYDimension[0]));
                        })
                    } else {
                        return chartDef.uaXDimension && chartDef.uaXDimension[0]
                            && chartDef.uaYDimension && chartDef.uaYDimension[0]
                            && ((this.isTrueNumerical(chartDef.uaXDimension[0]) && this.isTrueNumerical(chartDef.uaYDimension[0])) || (this.isDate(chartDef.uaXDimension[0]) && this.isDate(chartDef.uaYDimension[0])));
                    }
                }
            }

            return svc;
        })

        // mock of chart-formatting.service.ts
        .factory('ChartFormatting', function(NumberFormatter) {
            const svc = {
                createMeasureFormatters: function(chartDef, chartData, labelsResolution) {
                    return [];
                },

                getForIsolatedNumber: function() {
                    return '';
                },
                getForOrdinalAxis(value) {
                    return value;
                },
                getFormattedSpecialLabel(value) {
                    return value;
                },
            };

            return svc;
        })

        // mock of chart-dimension.service.ts
        .factory('ChartDimension', function(ChartsStaticData) {
            function findBinNumberOrDefault(chartType, binNumbers) {
                return binNumbers.find(automaticMaxBin => automaticMaxBin.chartType === chartType) || ChartsStaticData.BIN_NUMBER_DEFAULT;
            }

            function getBinNumber(chartType, isMainDimension, binNumbers) {
                const binNumber = findBinNumberOrDefault(chartType, binNumbers);
                if (isMainDimension) {
                    return binNumber.valueForMainDimension;
                }
                return binNumber.valueForOtherDimension;
            }

            function isTimelineable(dimension) {
                if (dimension && dimension.type === 'DATE') {
                    if (!dimension.dateParams) {
                        return false;
                    }
                    return ChartsStaticData.TIMELINE_AND_AUTOMATIC_DATE_MODES.map(dateMode => dateMode.value).includes(dimension.dateParams.mode);
                }
                return false;
            }

            function isOrdinalDateScale(dimension) {
                return dimension && dimension.type === 'DATE' && dimension.oneTickPerBin;
            }

            function isUnaggregated(dimension) {
                return dimension && dimension.isA === 'ua';
            }

            function isAutomatic(dimension) {
                if (!isTimelineable(dimension)) {
                    return false;
                }
                return dimension.dateParams.mode === ChartsStaticData.AUTOMATIC_DATE_MODE.value;
            }

            function getMaxBinNumberForAutomaticMode(chartType, isMainDimension) {
                return getBinNumber(chartType, isMainDimension, ChartsStaticData.AUTOMATIC_MAX_BIN_NUMBERS);
            }

            function isMainDateAxisAutomatic(chartDef) {
                return chartDef.genericDimension0.length > 0 && isAutomatic(chartDef.genericDimension0[0]);
            }

            function getMainDateAxisBinningMode(response) {
                return response.axisDefs[0].dateParams.mode;
            }

            function getNumberFormattingOptions(dimension) {
                if (!dimension || !isTrueNumerical(dimension)) {
                    return null;
                }
                return {
                    multiplier: dimension.multiplier || ChartsStaticData.autoMultiplier.label,
                    decimalPlaces: dimension.decimalPlaces,
                    prefix: dimension.prefix,
                    suffix: dimension.suffix
                };
            }

            function hasFixedNumberOfBins(dimension) {
                return dimension.numParams.mode === 'FIXED_NB';
            }

            function hasFixedBinSize(dimension) {
                return dimension.numParams.mode == 'FIXED_SIZE';
            }

            function isNumerical(dimension) {
                return dimension && dimension.type == 'NUMERICAL';
            }

            function isTrueNumerical(dimension) {
                return isNumerical(dimension) && ((dimension.numParams && dimension.numParams.mode != 'TREAT_AS_ALPHANUM') || dimension.numParams == undefined);
            }

            return {
                isTimelineable,
                isUnaggregated,
                isAutomatic,
                getNumberFormattingOptions,
                hasFixedNumberOfBins,
                hasFixedBinSize,
                isNumerical,
                isTrueNumerical,
                isTimeline: function(dimension) {
                    return !isOrdinalDateScale(dimension) && isTimelineable(dimension);
                },
                containsInteractiveDimensionCandidate: function(chartDef) {
                    if (chartDef.genericDimension0.length === 0) {
                        return false;
                    }
                    return this.isCandidateForInteractivity(chartDef.genericDimension0[0]);
                },
                isInteractiveChart: function(chartDef) {
                    if (chartDef.type !== 'lines') {
                        return false;
                    }
                    return this.containsInteractiveDimensionCandidate(chartDef);
                },
                isCandidateForInteractivity: function(dimension) {
                    return isAutomatic(dimension) && !isOrdinalDateScale(dimension);
                },
                isPercentScale: function(measures) {
                    return measures.every((measure) => ChartsStaticData.MEASURES_PERCENT_MODES.includes(measure.computeMode));
                },
                getDateModeDescription: function(mode) {
                    const result = ChartsStaticData.DATE_MODES_WITH_BACKEND_ONLY.filter(dateMode => dateMode.value === mode);
                    if (result.length === 1) {
                        return result[0].label;
                    }
                    return 'Unknown';
                },
                getComputedMainAutomaticBinningModeLabel: function(response, chartDef) {
                    if (!this.isInteractiveChart(chartDef)) {
                        return undefined;
                    }
                    if (isMainDateAxisAutomatic(chartDef)) {
                        return `(${this.getDateModeDescription(getMainDateAxisBinningMode(response))})`;
                    } else {
                        return undefined;
                    }
                },
                getNumericalBinNumber: function(chartType, isMainDimension) {
                    return getBinNumber(chartType, isMainDimension, ChartsStaticData.NUMERICAL_BIN_NUMBERS);
                },
                buildDateParamsForAxis: function(dimension, chartType, isMainInteractiveDateAxis) {
                    const dateParams = Object.assign({}, dimension.dateParams);
                    if (isAutomatic(dimension)) {
                        dateParams.maxBinNumberForAutomaticMode =
                            getMaxBinNumberForAutomaticMode(chartType, isMainInteractiveDateAxis);
                    }
                    return dateParams;
                },
                buildZoomRuntimeFilter: function(interactiveDimension, zoomUtils) {
                    return {
                        column: interactiveDimension.column,
                        columnType: 'DATE',
                        filterType: 'INTERACTIVE_DATE_FACET',
                        dateFilterType: 'RANGE',
                        minValue: Math.round(zoomUtils.displayInterval[0]),
                        maxValue: Math.round(zoomUtils.displayInterval[1])
                    };
                },
                isAlphanumLike: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return dimension.type == 'ALPHANUM' || (dimension.type == 'NUMERICAL' && dimension.numParams && dimension.numParams.mode == 'TREAT_AS_ALPHANUM');
                },
                isGroupedNumerical: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return this.isTrueNumerical(dimension) && ((dimension.numParams && dimension.numParams.mode != 'NONE') || dimension.numParams == undefined);
                },
                isUngroupedNumerical: function(dimension) {
                    if (!dimension) {
                        return;
                    }
                    return this.isTrueNumerical(dimension) && !this.isGroupedNumerical(dimension);
                },
                supportOneTickPerBin: function(dimension) {
                    return isTimelineable(dimension) || this.isGroupedNumerical(dimension);
                },    
                hasOneTickPerBin: function(dimension) {
                    if (!dimension) {
                        return;
                    }
            
                    if (dimension.$isInteractiveChart) {
                        return dimension.oneTickPerBin === 'YES';
                    }
            
                    if (this.supportOneTickPerBin(dimension) && dimension.oneTickPerBin && dimension.numParams) {
                        const AUTO_TRESHOLD = 20;
                        const numberOfBins = this.hasFixedBinSize(dimension) || this.isTimelineable(dimension) ? dimension.$numberOfBins : dimension.numParams.nbBins;
                        return dimension.oneTickPerBin === 'YES' || (dimension.oneTickPerBin === 'AUTO' && numberOfBins <= AUTO_TRESHOLD);
                    } else {
                        return false;
                    }
                }
            };
        })

        .factory('DateUtilsService', function() {
            return {
                convertDateToTimezone: function(date, timezone) {
                    const dateString = date.toLocaleString('en-US', { timeZone: timezone || 'UTC' });
                    return new Date(dateString);
                },
                convertDateFromTimezone: function(date, timezone) {
                    const offset = this.convertDateToTimezone(date, timezone).getTime() - date.getTime();
                    const result = new Date(date.getTime() - offset);
                    const checkDate = this.convertDateToTimezone(result, timezone);
                    if (date.getTime() === checkDate.getTime()) {
                        return result;
                    } else {
                        // DST bites us
                        const dstOffset = date.getTime() - checkDate.getTime();
                        return new Date(result.getTime() + dstOffset);
                    }
                },
                formatDateToISOLocalDate(date) {
                    const year = date.getFullYear();
                    const month = `0${(date.getMonth() + 1)}`.slice(-2);
                    const day = `0${date.getDate()}`.slice(-2);
                    return `${year}-${month}-${day}`;
                }      ,
                formatDateToISOLocalTime(date) {
                    const hours = `0${date.getHours()}`.slice(-2);
                    const minutes = `0${date.getMinutes()}`.slice(-2);
                    const seconds = `0${date.getSeconds()}`.slice(-2);
                    const milliseconds = `000${date.getMilliseconds()}`.slice(-3);
                    return `${hours}:${minutes}:${seconds}.${milliseconds}`;
                },
                formatDateToISOLocalDateTime(date) {
                    return this.formatDateToISOLocalDate(date) + 'T' + this.formatDateToISOLocalTime(date);
                }
            };
        })

        .factory('ChartFilters', function($filter, ChartColumnTypeUtils, DateUtilsService, ChartFilterUtils, ChartDataUtils, ChartLabels, DKU_OTHERS_VALUE) {

            function isNumberWithinRange(value, responseFacet) {
                return (typeof value === 'number' && value >= responseFacet.minValue && value <= responseFacet.maxValue) || value === null;
            }

            function hasWithinRangeMinAndMax(filter, responseFacet) {
                return filter && isNumberWithinRange(filter.minValue, responseFacet) && isNumberWithinRange(filter.maxValue, responseFacet);
            }

            function hasChangedAlphanumColumnSelectionType(filterTmpData) {
                return filterTmpData.filterType !== 'ALPHANUM_FACET'
                    || filterTmpData.filterSelectionType !== 'MULTI_SELECT';
            }

            function hasChangedNumericalColumnSelectionType(filterTmpData) {
                return filterTmpData.filterType !== 'NUMERICAL_FACET'
                    || filterTmpData.filterSelectionType !== 'RANGE_OF_VALUES';
            }

            function hasChangedDateColumnSelectionType(filterTmpData) {
                return filterTmpData.dateFilterType !== 'RANGE'
                    || filterTmpData.filterSelectionType !== 'RANGE_OF_VALUES';
            }

            function hasChangedFacetSelectionType(filterTmpData) {
                if (ChartColumnTypeUtils.isNumericalColumnType(filterTmpData.columnType)) {
                    return hasChangedNumericalColumnSelectionType(filterTmpData);
                } else if (ChartColumnTypeUtils.isDateColumnType(filterTmpData.columnType)) {
                    return hasChangedDateColumnSelectionType(filterTmpData);
                } else if (ChartColumnTypeUtils.isAlphanumColumnType(filterTmpData.columnType)) {
                    return hasChangedAlphanumColumnSelectionType(filterTmpData);
                }
                return false;
            }

            function getDefaultExcludeOtherValues(isAGlobalFilter, facetValues) {
                // Charts are in include mode by default.
                if (!isAGlobalFilter) {
                    return false;
                }
                // Dashboards are in exclude mode if the facet contains less than 200 values or in include mode otherwise.
                return (facetValues || []).length < 200;
            }

            function hasChangedExcludeOtherValuesOption(filterTmpData, filterFacet) {
                const values = (filterTmpData && filterFacet && filterFacet.values) || [];
                return getDefaultExcludeOtherValues(filterTmpData.isAGlobalFilter, values) !== filterTmpData.excludeOtherValues;
            }

            function hasChangedAlphanumericalValue(filterTmpData) {
                if (!filterTmpData || !ChartFilterUtils.isAlphanumericalFilter(filterTmpData)) {
                    return false;
                }
                // For multi select filters, a filter has changed if all values aren't selected.
                if (filterTmpData.filterSelectionType === 'MULTI_SELECT') {
                    return !!(filterTmpData.values || []).some(({ included }) => !included);
                }
                // For single select filters, a filter has changed if the selected value isn't the first one.
                return !!(filterTmpData.values && filterTmpData.values.length && !filterTmpData.values[0].included);
            }

            function hasChangedNumericalValue(filterTmpData, filterFacet) {
                if (!ChartFilterUtils.isNumericalFilter(filterTmpData)) {
                    return false;
                }
                if (filterTmpData.isAGlobalFilter) {
                    // On dashboard filters, if range has been modified it is necessarily within the current facet available range
                    const hasChanged = (filterTmpData.minValue !== filterFacet.minValue && filterTmpData.minValue !== null) || (filterTmpData.maxValue !== filterFacet.maxValue && filterTmpData.maxValue !== null);
                    return hasChanged && hasWithinRangeMinAndMax(filterTmpData, filterFacet);
                } else {
                    // On charts filters (explore and insights), no change means values have not been set
                    return typeof filterTmpData.minValue !== 'number' && typeof filterTmpData.maxValue !== 'number';
                }
            }

            function hasChangedTimezone(filterTmpData) {
                return filterTmpData.timezone !== 'UTC';
            }
        
            function hasChangedRelativeDateValue(filterTmpData) {
                return filterTmpData.dateFilterOption !== 'THIS';
            }

            function hasChangedFacetValues(filterTmpData, filterFacet) {
                if (ChartFilterUtils.isAlphanumericalFilter(filterTmpData)) {
                    return hasChangedAlphanumericalValue(filterTmpData);
                } else if (ChartFilterUtils.isDateRangeFilter(filterTmpData)) {
                    return hasChangedNumericalValue(filterTmpData, filterFacet) || hasChangedTimezone(filterTmpData);
                } else if (ChartFilterUtils.isNumericalFilter(filterTmpData)) {
                    return hasChangedNumericalValue(filterTmpData, filterFacet);
                } else if (ChartFilterUtils.isRelativeDateFilter(filterTmpData)) {
                    return hasChangedRelativeDateValue(filterTmpData);
                }
                return false;
            }

        function getForRange(
            minValue,
            maxValue,
            numValues,
            commaSeparator,
            hideTrailingZeros,
            measureOptions,
            shouldFormatInPercentage
        ) {
    
            shouldFormatInPercentage = shouldFormatInPercentage != undefined ? shouldFormatInPercentage : this.shouldFormatInPercentage(measureOptions);
    
            if (shouldFormatInPercentage) {
                minValue = minValue * 100;
                maxValue = maxValue * 100;
            }
           
            return (value) => {
                const digitGroupingOptions = measureOptions && measureOptions.digitGrouping && this.chartStaticDataService.availableDigitGrouping[measureOptions.digitGrouping];
                const formattingOptions = {
                    multiplier: measureOptions && measureOptions.multiplier,
                    prefix: measureOptions && measureOptions.prefix,
                    suffix: measureOptions && measureOptions.suffix,
                    decimalPlaces: measureOptions && measureOptions.decimalPlaces,
                    digitGrouping: measureOptions && measureOptions.digitGrouping,
                    shouldFormatInPercentage,
                    minDecimals: this.getMinDecimals(minValue, maxValue, numValues),
                    // For obscure reasons, d3 expects "comma" to be "," if you want to group digits (no matter if it's a comma or not)...
                    comma: (digitGroupingOptions && digitGroupingOptions.groupingSymbol !== 'NONE') ? ',' : (commaSeparator ? ',' : ''),
                    hideTrailingZeros: hideTrailingZeros
                }
                return this.get(formattingOptions)(value);
            }
        }
    

            return {
                onDateRangeChange: function(facetUiState, filterTmpData) {
                    if (!filterTmpData) {
                        return;
                    }
            
                    const from = facetUiState.fromDateRangeModel;
                    const to = facetUiState.toDateRangeModel;
                    // If a boundary is undefined it means that this boundary is invalid.
                    if (from === undefined || to === undefined) {
                        return;
                    }
                    const tz = facetUiState.timezoneDateRangeModel;
            
                    if (tz) {
                        filterTmpData.timezone = tz;
                        filterTmpData.minValue = from != null ? DateUtilsService.convertDateFromTimezone(from, tz).getTime() : null;
                        filterTmpData.maxValue = to != null ? DateUtilsService.convertDateFromTimezone(to, tz).getTime() : null;
                    }
                },
                canClearFilter: function(filterTmpData, filterFacet) {
                     return hasChangedFacetSelectionType(filterTmpData) || hasChangedExcludeOtherValuesOption(filterTmpData) || hasChangedFacetValues(filterTmpData, filterFacet);
                },
                getFacetSummary: function(filterTmpData, filterFacet) {
                    if (ChartFilterUtils.isAlphanumericalFilter(filterTmpData)) {
                        if (!filterTmpData.values || filterTmpData.values.length === 0) {
                            return 'No value available';
                        }
                        const valuesCount = Object.values(filterTmpData.values).filter(({ included }) => included).length;
                        if (valuesCount === 0) {
                            return 'None selected';
                        }
            
                        return `${valuesCount} selected`;
                    } else if (ChartFilterUtils.isNumericalFilter(filterTmpData)) {
                        if (filterFacet.minValue === 0 && filterFacet.maxValue === 0) {
                            return 'No value available';
                        }
            
                        /*
                         * Stored filter values can be out of range compared to new facets values received from response
                         * so we always display the minimal range from both
                         */
                        const areMinAndMaxWithinRange = hasWithinRangeMinAndMax(filterTmpData, filterFacet);
                        let summaryValues;
                        if (areMinAndMaxWithinRange) {
                            summaryValues = {
                                minValue: filterTmpData.minValue != null ? filterTmpData.minValue : filterFacet.minValue,
                                maxValue: filterTmpData.maxValue != null ? filterTmpData.maxValue : filterFacet.maxValue
                            };
                        } else {
                            summaryValues = filterFacet;
                        }
            
                        let formatter;
                        if (filterTmpData.columnType === 'DATE') {
                            const dateDisplayUnit = ChartDataUtils.computeDateDisplayUnit(summaryValues.minValue, summaryValues.maxValue);
                            formatter = (value) => $filter('date')(value, dateDisplayUnit.dateFilterOption, dateDisplayUnit.dateFilterOptionTimezone);
                        } else {
                            formatter = getForRange(summaryValues.minValue, summaryValues.maxValue, 2);
                        }
                        return `${formatter(summaryValues.minValue)} to ${formatter(summaryValues.maxValue)}`;
                    } else if (ChartFilterUtils.isRelativeDateFilter(filterTmpData) && filterTmpData.dateFilterOption && filterTmpData.dateFilterPart) {
                        return ChartFilterUtils.computeRelativeDateLabel(filterTmpData.dateFilterOption, filterTmpData.dateFilterPart, filterTmpData.includeEmptyValues);
                    }
            
                    return '';
                },
                getFacetUiState: function(filterTmpData, filterFacet) {
                    const facetUiState = {};
                    const isResponseAlphanumerical = !!filterFacet && !!filterFacet.values.length;

                    if (filterTmpData.filterSelectionType === 'RANGE_OF_VALUES') {
                        const lb = !isResponseAlphanumerical ? filterFacet.minValue : undefined;
                        const ub = !isResponseAlphanumerical ? filterFacet.maxValue : undefined;

                        facetUiState.sliderLowerBound = lb !== undefined ? lb : facetUiState.sliderLowerBound;
                        facetUiState.sliderUpperBound = ub !== undefined ? ub : facetUiState.sliderUpperBound;
                        facetUiState.sliderModelMin = filterTmpData.minValue;
                        facetUiState.sliderModelMax = filterTmpData.maxValue;

                        if (ChartColumnTypeUtils.isNumericalColumnType(filterTmpData.columnType) && facetUiState.sliderModelMax != null && facetUiState.sliderModelMin != null) {
                            // 10000 ticks
                            let sliderStep = Math.round(10000 * (facetUiState.sliderModelMax - facetUiState.sliderModelMin)) / 100000000;
                            // Handle min=max
                            sliderStep = sliderStep === 0 ? 1 : sliderStep;

                            const sliderDecimals = Math.max(String(sliderStep - Math.floor(sliderStep)).length - 2, 0);

                            facetUiState.sliderStep = sliderStep;
                            facetUiState.sliderDecimals = sliderDecimals;
                        }
                    }
                    if (filterTmpData.filterType === 'DATE_FACET') {
                        facetUiState.dateFilterType = filterTmpData.dateFilterType;
                        if (filterTmpData.dateFilterType === 'RANGE') {
                            const response = filterFacet || {};
                            const minValue = filterTmpData.minValue !== undefined ? filterTmpData.minValue : (!isResponseAlphanumerical ? response.minValue : null);
                            const maxValue = filterTmpData.maxValue !== undefined ? filterTmpData.maxValue : (!isResponseAlphanumerical ? response.maxValue : null);

                            facetUiState.timezoneDateRangeModel = filterTmpData.timezone || 'UTC';
                            facetUiState.fromDateRangeModel = minValue !== null ? DateUtilsService.convertDateToTimezone(new Date(minValue), facetUiState.timezoneDateRangeModel) : undefined;
                            facetUiState.toDateRangeModel = maxValue !== null ? DateUtilsService.convertDateToTimezone(new Date(maxValue), facetUiState.timezoneDateRangeModel) : undefined;
                        } else {
                            facetUiState.dateFilterPart = filterTmpData.dateFilterPart;
                        }
                    }

                    return facetUiState;
                },
                convertFilterTmpDataToPivotFilter(filterTmpData, filterFacet) {
                    const pivotFilter = {
                        active: filterTmpData.active,
                        isA: 'filter',
                        column: filterTmpData.column,
                        columnType: filterTmpData.columnType,
                        filterType: filterTmpData.filterType,
                        filterSelectionType: filterTmpData.filterSelectionType,
                        isAGlobalFilter: filterTmpData.isAGlobalFilter,
                        allValuesInSample: filterTmpData.allValuesInSample,
                        excludeOtherValues: filterTmpData.excludeOtherValues
                    };
            
                    if (ChartColumnTypeUtils.isDateColumnType(pivotFilter.columnType)) {
                        pivotFilter.dateFilterType = filterTmpData.dateFilterType;
                        pivotFilter.dateFilterPart = filterTmpData.dateFilterPart;
            
                        switch (pivotFilter.dateFilterType) {
                            case 'RELATIVE':
                                pivotFilter.dateFilterOption = filterTmpData.dateFilterOption;
                                pivotFilter.minValue = filterTmpData.minValue;
                                pivotFilter.maxValue = filterTmpData.maxValue;
                                break;
                            case 'RANGE':
                                pivotFilter.timezone = filterTmpData.timezone;
                                break;
                        }
                    }
                    if (ChartFilterUtils.isAlphanumericalFilter(pivotFilter)) {
                        pivotFilter.excludeOtherValues = filterTmpData.excludeOtherValues;
                        delete pivotFilter.minValue;
                        delete pivotFilter.maxValue;
            
                        if (filterTmpData.excludeOtherValues) {
                            pivotFilter.selectedValues = this.getFacetValues(filterTmpData, true);
                            delete pivotFilter.excludedValues;
                        } else {
                            pivotFilter.excludedValues = this.getFacetValues(filterTmpData, false);
                            delete pivotFilter.selectedValues;
                        }
                        pivotFilter.$totalValuesCount = filterTmpData.values ? filterTmpData.values.length : 0;
                    } else if (ChartFilterUtils.isNumericalFilter(pivotFilter)) {
                        delete pivotFilter.selectedValues;
                        delete pivotFilter.excludedValues;
                        pivotFilter.minValue = filterTmpData.minValue !== filterFacet.minValue ? filterTmpData.minValue : null;
                        pivotFilter.maxValue = filterTmpData.maxValue !== filterFacet.maxValue ? filterTmpData.maxValue : null;
                        pivotFilter.$globalMinValue = filterFacet.globalMinValue;
                        pivotFilter.$globalMaxValue = filterFacet.globalMaxValue;
                    }
            
                    return pivotFilter;
                },
                getFilterableOptions(store, chartData, coords, chartDef) {
                    let disableMultiDimensionalFiltering = false;
                    const filterableElements = Object.keys(coords).flatMap(axisName => {
                        const coord = coords[axisName];
                        const isSubtotal = coord === chartData.getSubtotalLabelIndex(axisName);
                        const dimension = store.get('dimensionDict').get(axisName);
                        const axis = chartData.getAxisLabels(axisName);
                        if (!isSubtotal && dimension != null && coord != null && axis != null) {
                            const axisElt = axis[coord];
                            if (axisElt.label === DKU_OTHERS_VALUE) {
                                disableMultiDimensionalFiltering = true;
                                return [];
                            }
                            const dimensionLabel = ChartLabels.getDimensionLabel(dimension);
                            const value = ChartLabels.getAxisEltFormattedLabel({
                                chartDef,
                                chartData,
                                dimension,
                                axisName,
                                axisElt
                            });
                            return [{ axisElt, dimension, dimensionLabel, value }];
                        }
                        return [];
                    });
                    return { filterableElements, disableMultiDimensionalFiltering };
                }
            };
        })

        .factory('ChartFilterUtils', function(ChartsStaticData, ChartColumnTypeUtils) {
            function toDateFilterType(dateMode) {
                let label = dateMode.label;
                if (dateMode.suffix) {
                    label += ` (${dateMode.suffix})`;
                }
                return [dateMode.value, label];
            }

            function buildDateFreeRangeFilterType(suffix) {
                return toDateFilterType({ ...ChartsStaticData.DEFAULT_DATE_RANGE_FILTER_TYPE, suffix });
            }

            function buildDateRelativeFilterType(suffix) {
                return toDateFilterType({ ...ChartsStaticData.DEFAULT_DATE_RELATIVE_FILTER_TYPE, suffix });
            }

            function buildDatePartFilterType(suffix) {
                return toDateFilterType({ ...ChartsStaticData.DEFAULT_DATE_PART_FILTER_TYPE, suffix });
            }

            return {
                isDateRangeFilter: function(filter) {
                    if (!filter) {
                        return false;
                    }
                    return filter.columnType == 'DATE' && filter.dateFilterType == 'RANGE';
                },
                isRelativeDateFilter: function(filter) {
                    if (!filter) {
                        return false;
                    }
                    return filter.columnType == 'DATE' && filter.dateFilterType == 'RELATIVE';
                },
                isDatePartFilter: function(filter) {
                    if (!filter) {
                        return false;
                    }
                    return filter.columnType == 'DATE' && filter.dateFilterType == 'PART';
                },
                getDateFilterTypes: function() {
                    return [
                        buildDateFreeRangeFilterType(),
                        buildDateRelativeFilterType(),
                        buildDatePartFilterType()
                    ];
                },
                getDateChartFilterParts: function() {
                    return [
                        toDateFilterType(ChartsStaticData.YEAR),
                        toDateFilterType(ChartsStaticData.QUARTER_OF_YEAR),
                        toDateFilterType(ChartsStaticData.MONTH_OF_YEAR),
                        toDateFilterType(ChartsStaticData.WEEK_OF_YEAR),
                        toDateFilterType(ChartsStaticData.DAY_OF_MONTH),
                        toDateFilterType(ChartsStaticData.DAY_OF_WEEK),
                        toDateFilterType(ChartsStaticData.HOUR_OF_DAY)
                    ];
                },
                getDateChartRangeFilterTypes: function() {
                    return [
                        buildDateFreeRangeFilterType(),
                        buildDateRelativeFilterType()
                    ];
                },
                getDateFilterParts: function() {
                    return [
                        toDateFilterType(ChartsStaticData.YEAR),
                        toDateFilterType(ChartsStaticData.QUARTER_OF_YEAR),
                        toDateFilterType(ChartsStaticData.MONTH_OF_YEAR),
                        toDateFilterType(ChartsStaticData.WEEK_OF_YEAR),
                        toDateFilterType(ChartsStaticData.DAY_OF_MONTH),
                        toDateFilterType(ChartsStaticData.DAY_OF_WEEK),
                        toDateFilterType(ChartsStaticData.HOUR_OF_DAY),
                        toDateFilterType(ChartsStaticData.INDIVIDUAL)
                    ]
                },
                getCustomDateRelativeFilterParts: function() {
                    return [
                        toDateFilterType(ChartsStaticData.RELATIVE_CUSTOM_YEAR),
                        toDateFilterType(ChartsStaticData.RELATIVE_CUSTOM_QUARTER),
                        toDateFilterType(ChartsStaticData.RELATIVE_CUSTOM_MONTH),
                        toDateFilterType(ChartsStaticData.RELATIVE_CUSTOM_WEEK),
                        toDateFilterType(ChartsStaticData.RELATIVE_CUSTOM_DAY),
                        toDateFilterType(ChartsStaticData.RELATIVE_CUSTOM_HOUR)
                    ]
                },
                isAlphanumericalFilter: function(filter) {
                    return filter.filterType === 'ALPHANUM_FACET'
                    || ChartColumnTypeUtils.isAlphanumColumnType(filter.columnType)
                    || this.isDatePartFilter(filter);
                },
                isNumericalRangeFilter: function(filter) {
                    return (filter.filterType === 'NUMERICAL_FACET'
                        && filter.columnType === 'NUMERICAL');
                },
                isNumericalFilter: function(filter) {
                    return this.isNumericalRangeFilter(filter)
                    || this.isDateRangeFilter(filter);
                }
            };
        })

        .factory('ColumnAvailability', function() {
            return {};
        })

        .factory('ChartLegendsWrapper', function() {

            class EventEmitter {
                constructor() {
                  this.eventTarget = new EventTarget();
                }

                subscribe(callback) {
                  this.eventTarget.addEventListener('xxx', (event) => {
                    callback(event);
                  });
                }
              
                next(data) {
                  const event = new CustomEvent('xxx', data);
                  this.eventTarget.dispatchEvent(event);
                }
              }

              
            return function() {
                return {
                    legends: [],
                    change: new EventEmitter(),
        
                    hasLegends() {
                        return this.legends.length > 0;
                    },
                    deleteLegends() {
                        this.legends.length = 0;
                        this.change.next(this.legends);
                    },
                    pushLegend(legend) {
                        this.legends.push(legend);
                        this.change.next(this.legends);
                    },
                    setLegends(legends) {
                        this.legends = legends;
                        this.change.next(this.legends);
                    },
                    getLegend(index) {
                        return this.legends[index];
                    }
                };
            };
        })
        

        // mock of chart-column-type-utils.service.ts
        .factory('ChartColumnTypeUtils', function() {
           return {
                isGeometryColumnType: function(columnType) {
                    return columnType === 'GEOMETRY';
                },
            
                isGeopointColumnType: function(columnType) {
                    return columnType === 'GEOPOINT';
                },

                isRBND: function(column) {
                    return column.isRBND;
                },
            
                isGeoColumnType: function(columnType) {
                    return this.isGeopointColumnType(columnType) || this.isGeometryColumnType(columnType);
                },
            
                isDateColumnType: function(columnType) {
                    return columnType === 'DATE';
                },
            
                isAlphanumColumnType: function(columnType) {
                    return columnType === 'ALPHANUM';
                },
            
                isNumericalColumnType: function(columnType) {
                    return columnType === 'NUMERICAL';
                },
            
                isCount: function(column) {
                    return !!column && (column.column === '__COUNT__' || column.column == null && column.function === 'COUNT');
                },
            
                isNonColumnData: function(column) {
                    return !column || !column.type;
                },

                isCustomColumnType: function(columnType) {
                    return columnType === 'CUSTOM';
                },

                isCustomMeasureAndNonNumerical: function(column){
                    return column.type === 'CUSTOM' && column.inferredType !== 'NUMERICAL';
                }
            };
        })

        // ChartLabels service is tested Angular-side, there is no reason to test any of its code here.
        // Mainly mocking for existing pivot table tests.
        .factory('ChartLabels', function() {
            return {
                getDimensionLabel: (data) => { return data && (data.displayLabel || data.column) },
                getFormattedLabel: (data) => { return data && data.label },
                getLongMeasureLabel: (measure) => { return measure && (measure.displayLabel || measure.column) },
                getUALabel: (data) => { return data && data.label },
                getKpiLabelPositions: () => { },
                getAxisEltFormattedLabel: ({axisElt}) => axisElt.label,
                hasLabelFormatting: (chartType) => { return !['radar', 'sankey'].includes(chartType) }
            };
        })

        .factory('ChartCustomColors', function() {
            return {
                createColorOptions: () => { return; },
                updateColorOptions: () => { return; },
                removeColorOptions: () => { return; },
            };
        })

        .factory('ChartColorUtils', function() {
            return {
                getDiscretePalette(colorPaletteId, customPalette, theme) {
                    if (colorPaletteId === '__dku_custom__') {
                        return customPalette;
                    }
                    if (colorPaletteId.includes('default_theme') && theme && theme.themePalettes.discrete.id === colorPaletteId) {
                        return theme.themePalettes.discrete;
                    }
                    return window.dkuColorPalettes.discrete.find(p => p.id === colorPaletteId);
                },
                getContinuousPalette(colorPaletteId, customPalette, isDiverging, theme) {
                    if (colorPaletteId === '__dku_custom__') {
                        return customPalette;
                    }
                    if (colorPaletteId.includes('default_theme') && theme) {
                        if (isDiverging && theme.themePalettes.diverging.id === colorPaletteId) {
                            return theme.themePalettes.diverging;
                        } else if (!isDiverging) {
                            const themePalettes = theme.themePalettes.continuous;
                            return themePalettes.find(palette => palette.id === colorPaletteId);
                        }
                    }
                    return (isDiverging ? window.dkuColorPalettes.diverging : window.dkuColorPalettes.continuous).find(p => p.id === colorPaletteId);
                },
                getColorDimensionOrMeasure(chartDef) {
                    switch (chartDef.type) {
                        case 'scatter':
                        case 'scatter_map':
                        case 'geom_map':
                            return chartDef.uaColor[0];
                        case 'pie':
                            return chartDef.genericDimension0[0];
                        case 'pivot_table':
                            return chartDef.colorMeasure;
                        case 'treemap':
                            return chartDef.colorMeasure[0] || chartDef.yDimension[0];
                        default:
                            return chartDef.genericDimension1[0];
                    }
                },
                getDefaultCustomPalette() {
                    return {
                        id: '__dku_custom__',
                        name: 'Custom Palette',
                        colors: [],
                        values: [],
                        fixedValues: false
                    }
                }
            };
        })

        .factory('ColorUtils', function() {
            return {
                generateThemePaletteColors() {
                    return { foregroundColors: [], backgroundColors: [] };
                }
            };
        })
        
        .factory('ChartZoomControlAdapter', function() {
            return {

            };
        })

        .factory('D3ChartZoomControl', function() {
            return {

            };
        })

        .factory('$stateParams', function() {
            return {
            };
        })

        .value('ChartYAxisPosition', {
            LEFT: 'left',
            RIGHT: 'right'
        })
        
        .factory('ChartCustomMeasures', function() {
            return {
            };
        })

        .factory('ChartUsableColumns', function() {
            return {
            };
        })

        .factory('GridlinesAxisType', function() {
            return {
            };
        })

        .factory('ValueType', function() {
            return {
                Constant: 'Constant'
            };
        })

        .factory('ColorMode', function() {
            return {
            };
        })

        .factory('MultiplotUtils', function() {
            return {

            };
        })

        .factory('CHART_COLOR_AUTO', function() {
            return "AUTO"
        })

        .factory('ChartFormattingPane', function() {
            const svc = {
                folds: {},

                open: function (fold, scrollIntoView) {
                    svc.set(svc.folds, fold, true);
            
                    if (scrollIntoView) {
                        //  Scroll to opened fold if exists.
                        setTimeout(() => {
                            const fold = document.querySelector(`#${fold}`);
                            if (fold) {
                                fold.scrollIntoView();
                            }
                        });
                    }
                },
            
                close: function(fold) {
                    svc.set(svc.folds, fold, false);
                },
            
                toggle: function(fold) {
                    //  Retrieves fold state, if undefined, return false (meaning closed).
                    const foldState = svc.get(svc.folds, fold, false);
                    svc.set(svc.folds, fold, !foldState);
                },
            
                set: function(fold, value) {
                    svc.set(svc.folds, fold, value);
                }
            };

            return svc;
        })

        .factory('ConditionalFormattingOptions', function() {
            return {
                getMeasureId: (value) => {
                    return `${ value.column || '' } ${ value.function || '' } ${ value.percentile || 50 } ${ value.inferredType || '' }`;
                }
            }
        })

        .factory('AISqlGenerationService', function() {})
        .factory('DSSVisualizationThemeUtils', function() {
            return {
                applyToChart: () => {
                },
                getThemeOrDefault: (theme) => {
                    if (theme) {
                        return theme;
                    }
                    return defaultTheme;
                },
                applyToLegend: () => {},
                applyAxisTitleFormatting: () => {},
                applyAxisValueFormatting: () => {},
                applyToTableFormatting: () => {},
                applyToScatterOptions: () => {},
                applyToSankeyNodeLabels: () => {}
            };
        })
        .factory('DefaultDSSVisualizationTheme', () => defaultTheme)
})();
