from abc import ABCMeta
from abc import abstractmethod

import numpy as np
import six


@six.add_metaclass(ABCMeta)
class ScoreToExplain(object):
    """
    Object representing the data from which explanations will be computed. It will be made of:
        * Predictions for regression problems
        * Log odds of probabilities for classification problems
    """

    @property
    @abstractmethod
    def score(self):
        """
        :return: 1d array representing the score to explain
        """


class OneDimensionScoreToExplain(ScoreToExplain):

    def __init__(self, score):
        self._score = score

    @property
    def score(self):
        return self._score


class MulticlassScoreToExplain(ScoreToExplain):

    def __init__(self, per_class_score, indices_for_global_score):
        """
        :type per_class_score: np.ndarray
        """
        self._per_class_score = per_class_score
        self._indices_for_global_score = indices_for_global_score

        self._score = None

    @property
    def score(self):
        if self._score is None:
            self._score = np.take_along_axis(self._per_class_score,
                                             self._indices_for_global_score[:, np.newaxis], axis=1).flatten()
        return self._score

    @property
    def per_class_score(self):
        return self._per_class_score

    @staticmethod
    def build_from_best_per_class_score(per_class_score):
        """
        :type per_class_score: np.ndarray
        :rtype: MulticlassScoreToExplain
        """
        return MulticlassScoreToExplain(per_class_score, np.argmax(per_class_score, axis=1))

    @staticmethod
    def build_from_other_score_to_explain(per_class_score, other_score_to_explain, indices_in_other_score):
        """
        :type per_class_score: np.ndarray
        :type other_score_to_explain: MulticlassScoreToExplain
        :type indices_in_other_score: np.ndarray
        :rtype: MulticlassScoreToExplain
        """
        return MulticlassScoreToExplain(per_class_score,
                                        other_score_to_explain._indices_for_global_score[indices_in_other_score])
