import json
import logging

import pandas as pd

from dataiku.external_ml.proxy_model.common.inputformat import BaseWriter
from dataiku.external_ml.proxy_model.common.utils import decode_base64
from dataiku.external_ml.utils import convert_dataframe_to_records


class DeployAnywhereWriter(BaseWriter):
    NAME = "INPUT_DEPLOY_ANYWHERE_ROW_ORIENTED_JSON"

    def __init__(self, client):
        super(DeployAnywhereWriter, self).__init__(client)

    def write(self, input_df):
        return self.client.call_endpoint(self.build_request_payload(input_df), {"ContentType": 'application/json'})

    def can_write(self, input_df):
        return self.write(input_df)

    def build_request_payload(self, input_df):
        records = convert_dataframe_to_records(input_df)
        payload = {"items": records}
        return json.dumps(payload)

    @staticmethod
    def decode_input_logs(logs_input_data, feature_dtypes):
        dfs_list = []
        logs_input_data_decoded = logs_input_data.apply(decode_base64).apply(json.loads)
        for index, raw_inference_query in enumerate(logs_input_data_decoded):
            try:
                items = raw_inference_query["items"]
                inference_query_df = pd.DataFrame([item.get("features") for item in items])
                inference_query_df.index = [(index, i) for i in range(len(inference_query_df))]
                inference_query_df = inference_query_df.dropna()

                for column, dtype in feature_dtypes.items():
                    if column in inference_query_df:
                        inference_query_df[column] = inference_query_df[column].astype(dtype)
                    else:
                        inference_query_df[column] = pd.Series(dtype=dtype)

                dfs_list.append(inference_query_df)
            except Exception as e:
                logging.info("Logs input data cannot be read for row %s" % str(raw_inference_query))
                logging.debug("Logs input reading exception: {}".format(e))

        return pd.concat(dfs_list, ignore_index=False)
