(function() {
'use strict';


const app = angular.module('dataiku.admin.clusters', []);


app.controller("ClusterCoreController", function($scope, CreateModalFromTemplate, KubernetesClusterService) {
    $scope.openDeleteClusterModal = function(clusterId, actionAfterDeletion) {
        var newScope = $scope.$new();
        newScope.clusterId = clusterId;
        newScope.actionAfterDeletion = actionAfterDeletion || function(){};
        CreateModalFromTemplate("/templates/admin/clusters/delete-cluster-modal.html", newScope, "ClusterDeleteController");
    }

    $scope.getStateDisplayString = KubernetesClusterService.getStateDisplayString;

    $scope.getStartStopButtonDisplayString = KubernetesClusterService.getStartStopButtonDisplayString;

    $scope.getStateDisplayClass = function(cluster) {
        var classes = {NONE:'icon-off none', RUNNING:'icon-play running', STARTING:'icon-play starting', STOPPING:'icon-unlink stopping', '': 'icon-question unknown'};
        if (cluster.state == null || cluster.type === 'manual') return classes[''];
        return classes[cluster.state];
    }
    $scope.getStateDisplayClassNoIcon = function(cluster) {
        var classes = {NONE:'none', RUNNING:'running', STARTING:'starting', STOPPING:'stopping', '': 'question unknown'};
        if (cluster.state == null || cluster.type === 'manual') return classes[''];
        return classes[cluster.state];
    }

    $scope.getCreatedByString = function(cluster) {
        if (cluster.origin == null) {
            return null;
        } else if (cluster.origin.type === 'MANUAL') {
            return "Created by " + cluster.origin.identifier;
        } else if (cluster.origin.type === 'SCENARIO') {
            return "Created by scenario " + cluster.origin.scenarioProjectKey + "." + cluster.origin.scenarioId;
        } else {
            return null;
        }
    }
});

app.controller("ClusterDeleteController", function($scope, Assert, DataikuAPI, FutureProgressModal) {
    $scope.uiState = {stop:true};

    DataikuAPI.admin.clusters.getStatus($scope.clusterId).success(function(data) {
        $scope.clusterStatus = data;
    }).error(setErrorInScope.bind($scope));

    $scope.doesntNeedStop = function() {
        return $scope.clusterStatus && $scope.clusterStatus.clusterType != 'manual' && $scope.clusterStatus.state == 'NONE';
    };
    $scope.mayNeedStop = function() {
        return $scope.clusterStatus && $scope.clusterStatus.clusterType != 'manual' && $scope.clusterStatus.state != 'NONE';
    };

    $scope.delete = function() {
        Assert.inScope($scope, 'clusterStatus');
        if ($scope.mayNeedStop() && $scope.uiState.stop) {
            var parentScope = $scope.$parent;
            DataikuAPI.admin.clusters.stop($scope.clusterId, true, false).success(function(data){
                $scope.dismiss();
                FutureProgressModal.show(parentScope, data, "Stop cluster", undefined, 'static', 'false').then(function(result){
                    if (result) { // undefined in case of abort
                        $scope.actionAfterDeletion();
                    }
                });
            }).error(setErrorInScope.bind($scope));
        } else {
            DataikuAPI.admin.clusters.delete($scope.clusterId).success(function(data){
                $scope.dismiss();
                $scope.actionAfterDeletion();
            }).error(setErrorInScope.bind($scope));
        }
    };
});

app.controller("ClustersController", function ($scope, $controller, DataikuAPI, CreateModalFromTemplate, TopNav,
    FutureWatcher, KubernetesClusterService, ActivityIndicator, $rootScope, Dialogs) {
    $controller("ClusterCoreController", {$scope:$scope});

    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {query:null};

    $scope.clusters = [];

    $scope.getK8SClusterData = (cluster) => {
        let handleError = () => {
            cluster.data.state = 'ERROR';
        };
        DataikuAPI.admin.clusters.countNodesAndPods(cluster.id).success((initialResponse) => {
                let setResponse = (response) => {
                    cluster.data.count = response.result;
                    cluster.data.state = 'LOADED';
                };
                if (initialResponse.hasResult) {
                    setResponse(initialResponse);
                } else {
                    FutureWatcher.watchJobId(initialResponse.jobId)
                        .success(setResponse)
                        .error(handleError);
                }
            }).error(handleError);
    };

    $scope.getNodesMetrics = (cluster) => {
        const clusterInScope = $scope.clusters.find(el => el.id === cluster.id);

        let handleError = () => {
            clusterInScope.metrics.state = 'ERROR';
        }

        DataikuAPI.admin.clusters.getNodesMetrics(cluster.id).success((initialResponse) => {
            let parseResponse = (response) => {
                clusterInScope.metrics = response.result.reduce((accumulator, current) => {
                    accumulator.cpuUsage += current.cpuUsage;
                    accumulator.memoryUsage += current.memoryUsage;
                    return accumulator;
                }, {cpuUsage: 0, memoryUsage:0});
                clusterInScope.metrics.cpuUsage = clusterInScope.metrics.cpuUsage / response.result.length | 0;
                clusterInScope.metrics.memoryUsage = clusterInScope.metrics.memoryUsage / response.result.length | 0;
                clusterInScope.metrics.state = 'LOADED';
            };
            if (initialResponse.hasResult) {
                parseResponse(initialResponse);
            } else {
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(parseResponse)
                    .error(handleError);
            }
        }).error(handleError);
    };

    $scope.refreshList = function() {
        $scope.refreshing = true;
        DataikuAPI.admin.clusters.list().success(function (data) {
            // sorting to display K8S clusters first
            data.sort((el1, el2) => {
                if (el1.architecture === el2.architecture || el1.architecture !== 'KUBERNETES' && el2.architecture !== 'KUBERNETES') {
                    return 0;
                } else {
                    return el1.architecture === 'KUBERNETES' ? -1 : 1;
                }
            });

            data.forEach(el => {
                el.ui = {};
                el['ui']['typeName'] = KubernetesClusterService.getTypeDisplayName(el.type);
                el['ui']['icon'] = KubernetesClusterService.getIcon(el.architecture, el.type);
                el.metricsServer = 'NOT_INSTALLED';
                if ($scope.isK8sAndIsRunning(el)) {
                    if (el.canUpdateCluster) {
                        el.data = { state: 'LOADING' };
                        el.metrics = { state: 'LOADING' };
                        el.metricsServer = 'CHECKING';
                        $scope.getK8SClusterData(el);

                        const handleError = () => {
                            el.metricsServer = 'ERROR';
                            el.metrics = { state: 'ERROR' }
                        }
                        const parseResponse = (response) => {
                            el.metricsServer = response.result ? 'INSTALLED' : 'NOT_INSTALLED';
                            if (el.metricsServer === 'INSTALLED') {
                                $scope.getNodesMetrics(el);
                            }
                        }

                        DataikuAPI.admin.clusters.hasMetricsServer(el.id).success((response) => {
                            if (response.hasResult) {
                                parseResponse(response);
                            } else {
                                return FutureWatcher.watchJobId(response.jobId)
                                    .success(parseResponse)
                                    .error(handleError)
                            }
                        }).error(handleError);

                    } else {
                        el.metricsServer = 'INSUFFICIENT_PERMISSION';
                        el.data = { state: 'INSUFFICIENT_PERMISSION' };
                    }
                }
            });

            $scope.clusters = data;

            // Populating create cluster options for the empty state page
            if (data.length === 0) {
                const basicClusterCreateOption = {
                    title: 'Add Cluster',
                    action: $scope.createCluster
                }

                if ($scope.isPluginInstalled('eks') || $scope.isPluginInstalled('aks') || $scope.isPluginInstalled('gke')) {
                    $scope.createClusterOptions = [];
                    const addPluginOption = (name) => {
                        if ($scope.isPluginInstalled(name)) {
                            const currentOption = {
                                title: `Create ${name.toUpperCase()} cluster`,
                                action : () => $scope.createPluginCluster(name)
                            }
                            if (!$scope.mainCreateOption) {
                                $scope.mainCreateOption = currentOption;
                            } else {
                                $scope.createClusterOptions.push(currentOption)
                            }
                        }
                    }

                    addPluginOption('eks');
                    addPluginOption('aks');
                    addPluginOption('gke');

                    $scope.createClusterOptions.push(basicClusterCreateOption);

                } else {
                    $scope.mainCreateOption = basicClusterCreateOption;
                }

            }
            $scope.refreshing = false;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.filterClusters = function(clusterObject) {
        if (!$scope.uiState.query) return true;
        const query = $scope.uiState.query.toLowerCase();
        return clusterObject && clusterObject.name.toLowerCase().includes(query)
            || clusterObject.ui.typeName && clusterObject.ui.typeName.toLowerCase().includes(query)
            || clusterObject.architecture && clusterObject.architecture.toLowerCase().includes(query);
    };

    $scope.isK8sAndIsRunning = (cluster) => {
        return cluster.architecture === 'KUBERNETES' && (cluster.state === 'RUNNING' || cluster.type === 'manual')
    }

    $scope.isK8sAndIsStopped = (cluster) => {
        return cluster.architecture === 'KUBERNETES' && cluster.state !== 'RUNNING' && cluster.type !== 'manual';
    }

    $scope.refreshList();

    $scope.createCluster = function() {
        CreateModalFromTemplate("/templates/admin/clusters/new-cluster-modal.html", $scope, "NewClusterController")
    };

    $scope.deleteCluster = (clusterId, architecture, type) => {
        KubernetesClusterService.sendWt1Event('clusters-action-delete', architecture, type);
        $scope.openDeleteClusterModal(clusterId, function() {
            $scope.refreshList();
            KubernetesClusterService.sendWt1Event('clusters-status-deleted', architecture, type);
        });
    };

    $scope.downloadClusterDiagnostic = (clusterId, architecture, type) => {
        ActivityIndicator.success("Preparing cluster diagnosis ...");
        KubernetesClusterService.sendWt1Event('clusters-action-diagdownload', architecture, type);
        downloadURL(DataikuAPI.admin.clusters.getDiagnosisURL(clusterId));
        KubernetesClusterService.sendWt1Event('clusters-status-diagdownloaded', architecture, type);
    };

    $scope.markStoppedCluster = (clusterId, architecture, type) => {
        KubernetesClusterService.sendWt1Event('clusters-action-markstopped', architecture, type);
        DataikuAPI.admin.clusters.markStopped(clusterId).success(function(){
            $scope.refreshList();
            KubernetesClusterService.sendWt1Event('clusters-status-markstopped', architecture, type);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.forceStopCluster = (clusterId, architecture, type) => {
        Dialogs.confirm($scope, "Confirm cluster force stop", "Are you sure you want to force stop the cluster?").then(() => {
            KubernetesClusterService.stopCluster(true, $scope, clusterId, $scope.refreshList, architecture, type);
        }, () => {});
    };

    const getPluginCreateClusterType = function(cloud) {
        return "pycluster_" + cloud + "-clusters_create-" + cloud + "-cluster";
    };

    $scope.isPluginInstalled = function(cloud) {
        return $rootScope.appConfig.customPythonPluginClusters.some(
            pluginCluster => pluginCluster.clusterType === getPluginCreateClusterType(cloud));
    };

    $scope.createPluginCluster = function(cloud) {
        CreateModalFromTemplate("/templates/admin/clusters/new-cluster-modal.html", $scope, "NewClusterController", function(newScope) {
            newScope.title = "Create " + cloud.toUpperCase() + " cluster";
            newScope.newCluster = {type: getPluginCreateClusterType(cloud), architecture: 'KUBERNETES', params: {}}
            newScope.typeAlreadyKnown = true;
        });
    };
});

app.controller("NewClusterController", function($scope, $rootScope, $state, DataikuAPI, KubernetesClusterService, PluginConfigUtils) {

    const visibilityFilter = PluginConfigUtils.shouldComponentBeVisible($rootScope.appConfig.loadedPlugins);
    $scope.title = $scope.title || 'Add cluster';
    $scope.newCluster = $scope.newCluster || {type:'manual', params: {}};
    $scope.clusterArchitectures = [{id:'HADOOP', label:'Hadoop'}, {id:'KUBERNETES', label:'K8S'}];
    $scope.clusterTypes = [
        {id:'manual', label:'Non managed', architecture:'MANUAL'},
        ...$rootScope.appConfig.customPythonPluginClusters
            .filter(visibilityFilter)
            .filter(cluster => // Don't add "create cluster" options from our plugins since those have dedicated buttons
                    !['pycluster_eks-clusters_create-eks-cluster',
                        'pycluster_aks-clusters_create-aks-cluster',
                        'pycluster_gke-clusters_create-gke-cluster'].includes(cluster.clusterType))
            .map(cluster => ({ id: cluster.clusterType, label: cluster.desc.meta.label || cluster.id, architecture: cluster.desc.architecture || 'HADOOP' }))
    ];

    $scope.$watch('newCluster.type', function() {
        if ($scope.newCluster.type && $scope.newCluster.type != 'manual') {
            let clusterType = $scope.clusterTypes.filter(function(t) {return $scope.newCluster.type == t.id;})[0];
            $scope.newCluster.architecture = clusterType ? clusterType.architecture : null;
        }
    });

    $scope.create = function(){
        var parentScope = $scope.$parent.$parent;
        KubernetesClusterService.sendWt1Event('clusters-action-create', $scope.newCluster.architecture, $scope.newCluster.type);
        DataikuAPI.admin.clusters.create($scope.newCluster).success(function(data){
            $scope.dismiss();
            parentScope.refreshList();
            $state.go("admin.clusters.cluster", {clusterId:data.id});
            KubernetesClusterService.sendWt1Event('clusters-status-created', data.architecture, data.type);
        }).error(setErrorInScope.bind($scope));
    }
});

app.controller("ClusterController", function($scope, $controller, $stateParams, Assert, DataikuAPI, $state, TopNav, FutureProgressModal, ActivityIndicator, $q,
    Logs, CreateModalFromTemplate, FutureWatcher, KubernetesClusterService, StateUtils, Dialogs) {
    $controller("ClusterCoreController", {$scope:$scope});

    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {
        active: $state.params.action ? 'actions' : 'info',
        logsQuery: '',
        detailedNodeData: false,
        detailedPodData: false
    };

    $scope.cluster = {};
    $scope.origCluster = {};

    $scope.listLogs = function(){
        DataikuAPI.admin.clusters.listLogs($scope.cluster.id).success(function(data) {
            $scope.logs = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refreshItem = function() {
        DataikuAPI.admin.clusters.get($stateParams.clusterId).success(function(data) {
            $scope.cluster = data;
            $scope.origCluster = angular.copy(data);
            $scope.cluster.params = $scope.cluster.params || {};
            $scope.listLogs();
            $scope.refreshStatus();
        }).error(setErrorInScope.bind($scope));
    };
    $scope.refreshItem();

    $scope.getDisplayType = clusterType => KubernetesClusterService.getTypeDisplayName(clusterType);

    $scope.refreshStatus = function() {
        if (!$scope.cluster.canUpdateCluster) return;
        DataikuAPI.admin.clusters.getStatus($stateParams.clusterId).success(function(data) {
            $scope.clusterStatus = data;
        }).error(setErrorInScope.bind($scope));
    };

    const dssObjectColumns = [
        {type: "dss-user", key: 'dssSubmitter', columnName: 'DSS user'},
        {type: "dss-project", key: 'dssProjectKey', columnName: 'DSS project'},
        {type: "dss-object", key: 'dssExecutionType', columnName: 'DSS object'}
    ];

    const podMonitoringColumns = {
        simple : [
            "ready",
            {type:"pod-status", key:'status'},
            ...dssObjectColumns
        ],
        detailed: [
            "ready",
            {type:"pod-status", key:'status'},
            "cpuRequestMillis", "cpuLimitMillis", "cpuCurrentMillis",
            "memoryRequestMB", "memoryLimitMB", "memoryCurrentMB",
            ...dssObjectColumns
        ]
    };

    const nodeMonitoringColumns = {
        simple : [
            {type:"node-status", key:'status'}
        ],
        detailed: [
            {type:"node-status", key:'status'},
            "cpuCapacityMillis", "cpuCurrentMillis", "cpuLoad",
            "memoryCapacityMB", "memoryCurrentMB", "memoryLoad"
        ]
    };

    function calcPercent(value, max){
        const m = parseFloat(max), v = parseFloat(value);
        return isNaN(m) || isNaN(v) || m==0 ? '' : Math.floor(100 * Math.min(1, v/m)) + '%';
    }

    // the angularjs default comparator use the row index
    // when a value is not defined (?!), making weird behaviors.
    // we want a comparison with 0 / "" as default value.
    // we also handle the sorting of percentage and fractions
    const isNumericalRegex = /^\d+%$|^\d+ \/ \d+$/; //match "52%" or "2 / 4"
    $scope.resourceSort = function(a,b){
        if (a.type === 'string' || b.type === 'string') {
            let va = a.value || '';
            let vb = b.value || '';
            if (isNumericalRegex.test(va) || isNumericalRegex.test(vb)) {
                va = parseFloat(va) || 0;
                vb = parseFloat(vb) || 0;
                return va - vb;
            }
            return va.localeCompare(vb);
        } else if (a.type === 'number' || b.type === 'number') {
            return (a.value || 0) - (b.value || 0);
        }
        return 0;
    }

    function buildDssObjectLink(item) {
        if (!item['dssExecutionType'] || !item['onLocalDssNode']) {
            return '';
        }

        const projectKey = item['dssProjectKey'];
        // From com.dataiku.dip.resourceusage.ComputeResourceUsageContext.ComputeResourceUsageContextType
        switch (item['dssExecutionType']) {
            case 'ANALYSIS_ML_TRAIN':
                return StateUtils.href.analysis(item['analysisId'], projectKey, {tab: 'ml.list'});
            case 'WEBAPP_BACKEND':
                return StateUtils.href.webapp(item['webappId'], projectKey, {tab: 'edit'});
            case 'JUPYTER_NOTEBOOK_KERNEL':
                return StateUtils.href.jupyterNotebook(item['notebookId'], projectKey);
            case 'EDA':
                return StateUtils.href.dataset(item['datasetName'], projectKey, {tab: 'statistics'});
            case 'JOB_ACTIVITY':
                return StateUtils.href.job(projectKey, item['jobId']);
            case 'CONTINUOUS_ACTIVITY':
                return StateUtils.href.continuousActivity(item['activityId'], projectKey);
            case 'API_DEPLOYER_DEPLOYMENT':
                return StateUtils.href.apiDeployer.deployment(item['deploymentId']);
            case 'CODE_STUDIO':
                return StateUtils.href.codeStudio(item['codeStudioId'], projectKey);
            default:
                return '';
        }
    }

    function addLinksToObjects(data) {
        data.forEach(item => {
            item.dssLink = buildDssObjectLink(item);
        });
    }

    $scope.clusterObjects = [
        {
            name: "Pod",
            state: 'NONE',
            cols: podMonitoringColumns.simple,
            api: () => DataikuAPI.admin.clusters.monitoring.getPods(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name,
                $scope.uiState.detailedPodData
            ),
            hasNamespace: true,
            parseClusterObjectData: (data) => {
                data.forEach(pod => {
                    pod.ready = `${pod.readyContainerCount} / ${pod.containerCount}`;
                });
                addLinksToObjects(data);
            }
        },
        {
            name: "Job",
            state: 'NONE',
            cols: dssObjectColumns,
            api: () => DataikuAPI.admin.clusters.monitoring.getJobs(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name
            ),
            parseClusterObjectData: addLinksToObjects,
            hasNamespace: true
        },
        {
            name: "Deployment",
            state: 'NONE',
            cols: dssObjectColumns,
            api: () => DataikuAPI.admin.clusters.monitoring.getDeployments(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name
            ),
            parseClusterObjectData: addLinksToObjects,
            hasNamespace: true
        },
        {
            name: "Service",
            state: 'NONE',
            cols: [
                "type",
                ...dssObjectColumns
            ],
            api: () => DataikuAPI.admin.clusters.monitoring.getServices(
                $stateParams.clusterId,
                $scope.namespaces.selectedMonitoringNamespaceFilter.filterType,
                $scope.namespaces.selectedMonitoringNamespaceFilter.name
            ),
            parseClusterObjectData: addLinksToObjects,
            hasNamespace: true
        },
        {
            name: "Node",
            state: 'NONE',
            cols: nodeMonitoringColumns.simple,
            api: () => DataikuAPI.admin.clusters.monitoring.getNodes(
                $stateParams.clusterId,
                $scope.uiState.detailedNodeData
            ),
            hasNamespace: false,
            parseClusterObjectData: (data) => {
                data.forEach(node => {
                    node.cpuLoad = calcPercent(node.cpuCurrentMillis, node.cpuCapacityMillis);
                    node.memoryLoad = calcPercent(node.memoryCurrentMB, node.memoryCapacityMB);
                });
            }
        }
    ];

    $scope.namespaces = {
        loading: false,
        loaded: false,
        monitoringNamespaceFilterOptions: [
            {name: 'None specified (kubectl default)', filterType: 'NONE'},
            {name: 'All', filterType: 'ALL'},
            {name: 'All but kube-system', filterType: 'ALL_BUT_SYSTEM'} // Default to match what is seen on tile
        ],
        parseClusterObjectData: (data) => {
            $scope.namespaces.monitoringNamespaceFilterOptions =
                $scope.namespaces.monitoringNamespaceFilterOptions.filter(namespace => namespace.filterType !== 'LOADING');
            $scope.namespaces.loaded = true;
            data.forEach(namespace =>
                $scope.namespaces.monitoringNamespaceFilterOptions.push({name: namespace, filterType: 'EXPLICIT'}));
        },
        api: () => DataikuAPI.admin.clusters.monitoring.getNamespaces($stateParams.clusterId)
    }
    $scope.namespaces.selectedMonitoringNamespaceFilter = $scope.namespaces.monitoringNamespaceFilterOptions[2];

    $scope.$watch("uiState.detailedPodData", function(nv, ov) {
        if (nv === true) {
            // Dirty
            $scope.clusterObjects[0].cols = podMonitoringColumns.detailed;
            $scope.refreshMonitoring();
        } else if (nv=== false) {
            $scope.clusterObjects[0].cols = podMonitoringColumns.simple;
        }
    });

    $scope.$watch("uiState.detailedNodeData", function(nv, ov) {
        if (nv) {
            // Dirty
            $scope.clusterObjects[4].cols = nodeMonitoringColumns.detailed;
            $scope.refreshMonitoring();
        } else {
            $scope.clusterObjects[4].cols = nodeMonitoringColumns.simple;
        }
    });

    $scope.getClusterObjectData = (clusterObject) => {
        if (!$scope.cluster.canUpdateCluster) return;
        clusterObject.state = 'NONE';

        const parseResponse = (data) => {
            clusterObject.state = 'LOADED';
            if (clusterObject.parseClusterObjectData) {
                clusterObject.parseClusterObjectData(data);
            }
            clusterObject.data = data;
        }

        let handleError = (data, status, headers) => {
            clusterObject.state = 'ERROR';
            setErrorInScope.bind($scope)(data, status, headers);
        }

        clusterObject.api().success(function(initialResponse) {
            if (initialResponse.hasResult) {
                parseResponse(initialResponse.result);
            } else {
                clusterObject.state = 'LOADING';
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(data => parseResponse(data.result))
                    .error(handleError);
            }
        }).error(handleError);
    }

    $scope.goToClusterObjectMonitoring = function(clusterObject) {
        $scope.uiState.activeMonitoring = clusterObject.name
        if (($scope.cluster.type === 'manual' || $scope.cluster.state === 'RUNNING')) {
            $scope.getClusterObjectData(clusterObject);
        }
        KubernetesClusterService.sendWt1Event(`clusters-monitoring-${clusterObject.name}s-viewed`.toLowerCase(),
                                              $scope.cluster.architecture, $scope.cluster.type);
    }

    $scope.refreshMonitoring = function() {
        let clusterObject = $scope.clusterObjects.find((item) => item.name === $scope.uiState.activeMonitoring);
        $scope.getClusterObjectData(clusterObject);
    }

    $scope.getNamespaces = function() {
        if (($scope.cluster.type === 'manual' || $scope.cluster.state === 'RUNNING') && !$scope.namespaces.loading && !$scope.namespaces.loaded) {
            $scope.namespaces.monitoringNamespaceFilterOptions.push({name: 'Loading cluster namespaces...', filterType: 'LOADING'});
            $scope.getClusterObjectData($scope.namespaces);
        }
    }

    $scope.setMonitoringNamespaceFilter = function(nv) {
        $scope.namespaces.selectedMonitoringNamespaceFilter = nv;
        $scope.refreshMonitoring();
    }

    $scope.filterMonitoring = function(clusterObject) {
        if (!$scope.uiState.monitoringQuery) return true;
        const query = $scope.uiState.monitoringQuery.toLowerCase();
        return clusterObject && clusterObject.name.toLowerCase().includes(query)
            || clusterObject.status && clusterObject.status.toLowerCase().includes(query)
            || clusterObject.namespace && clusterObject.namespace.toLowerCase().includes(query)
            || clusterObject.type && clusterObject.type.toLowerCase().includes(query)
            || clusterObject.dssSubmitter && clusterObject.dssSubmitter.toLowerCase().includes(query)
            || clusterObject.dssExecutionType && clusterObject.dssExecutionType.toLowerCase().includes(query)
            || clusterObject.dssProjectKey && clusterObject.dssProjectKey.toLowerCase().includes(query);
    };

    $scope.showObjectDescribe = function(name, objectType, namespace) {
        CreateModalFromTemplate('/templates/admin/clusters/k8s-object-describe-modal.html', $scope, null, function(newScope) {
            newScope.uiState = {
                mode: "describe"
            };
            newScope.objectName = name;
            newScope.objectType = objectType;
            newScope.clusterId = $scope.cluster.id;
            newScope.namespace = namespace;
            newScope.describeObject();
        });
    };

    $scope.clusterIsDirty = function() {
        if (!$scope.cluster || !$scope.origCluster) return false;
        return !angular.equals($scope.cluster, $scope.origCluster);
    };

    checkChangesBeforeLeaving($scope, $scope.clusterIsDirty);

    $scope.saveCluster = function() {
        var deferred = $q.defer();
        if (!$scope.clusterIsDirty()) { // for when it's called with a keystroke or from start button
            deferred.resolve("Saved");
            return deferred.promise;
        }
        KubernetesClusterService.sendWt1Event('clusters-action-save', $scope.cluster.architecture, $scope.cluster.type);
        DataikuAPI.admin.clusters.save(angular.copy($scope.cluster)).success(function(data) {
            $scope.cluster = data;
            $scope.origCluster = angular.copy(data);
            deferred.resolve("Saved");
            KubernetesClusterService.sendWt1Event('clusters-status-saved',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        }).error(function (a,b,c) {
            setErrorInScope.bind($scope)(a,b,c);
            deferred.reject("Not saved");
        });
        return deferred.promise;
    };

    $scope.deleteCluster = function() {
        KubernetesClusterService.sendWt1Event('clusters-action-delete',
                                              $scope.cluster.architecture, $scope.cluster.type);
        $scope.openDeleteClusterModal($scope.cluster.id, function() {
            $state.go("admin.clusters.list");
            KubernetesClusterService.sendWt1Event('clusters-status-deleted',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        });
    };

    const doStartCluster = function(){
        Assert.inScope($scope, 'cluster');
        KubernetesClusterService.sendWt1Event('clusters-action-start',
                                              $scope.cluster.architecture, $scope.cluster.type);
        return DataikuAPI.admin.clusters.start($scope.cluster.id)
            .success(
                data => FutureProgressModal.show($scope, data, "Start cluster", undefined, 'static', 'false')
                .then(function(result){
                    if (result) { // undefined in case of abort
                        $scope.refreshItem();
                        KubernetesClusterService.sendWt1Event('clusters-status-started',
                                                            $scope.cluster.architecture, $scope.cluster.type);
                    }
                })
            )
            .error(setErrorInScope.bind($scope));
    };

    const doStopCluster = function(forceStop) {
        Assert.inScope($scope, 'cluster');
        return KubernetesClusterService.stopCluster(forceStop, $scope, $scope.cluster.id, $scope.refreshItem,
                                                    $scope.cluster.architecture, $scope.cluster.type);
    };

    const doLongRunningOperation = function(operation) {
        $scope.isLongClusterOperationRunning = true;
        operation().finally(() => $scope.isLongClusterOperationRunning = false);
    };

    const wrapLongClusterOperationRunning = function(operationName, needsConfirmation, operation) {
        return () => {
            if ($scope.isLongClusterOperationRunning) {
                return;
            }
            if (needsConfirmation) {
                Dialogs.confirm($scope, "Confirm cluster " + operationName, "Are you sure you want to " + operationName + " the cluster?").then(() => {
                    doLongRunningOperation(operation);
                }, () => {});
            } else {
                doLongRunningOperation(operation);
            }
        };
    };

    $scope.startCluster = wrapLongClusterOperationRunning("start", false, function(){
        return $scope.saveCluster().then(doStartCluster);
    });

    $scope.stopCluster = wrapLongClusterOperationRunning("stop", true, function(){
        return doStopCluster(false);
    });

    $scope.forceStopCluster = wrapLongClusterOperationRunning("force stop", true, function() {
        return doStopCluster(true);
    });

    $scope.markStoppedCluster = function(){
        Assert.inScope($scope, 'cluster');

        KubernetesClusterService.sendWt1Event('clusters-action-markstopped',
                                              $scope.cluster.architecture, $scope.cluster.type);
        DataikuAPI.admin.clusters.markStopped($scope.cluster.id).success(function(){
            $scope.refreshItem();
            KubernetesClusterService.sendWt1Event('clusters-status-markstopped',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        }).error(setErrorInScope.bind($scope));
    };

    $scope.currentLogName = null;
    $scope.currentLog = null;
    $scope.fetchLog = function(logName) {
    	DataikuAPI.admin.clusters.getLog($scope.cluster.id, logName).success(function(data) {
            $scope.currentLogName = logName;
            $scope.currentLog = data;
            KubernetesClusterService.sendWt1Event('clusters-logs-opened',
                                                  $scope.cluster.architecture, $scope.cluster.type);
        }).error(setErrorInScope.bind($scope));
    };
    $scope.streamLog = function(logName) {
    	Logs.downloadCluster($scope.cluster.id, logName);
    };

    $scope.downloadClusterDiagnostic = function() {
        ActivityIndicator.success("Preparing cluster diagnosis ...");
        KubernetesClusterService.sendWt1Event('clusters-action-diagdownload',
                                              $scope.cluster.architecture, $scope.cluster.type);
        downloadURL(DataikuAPI.admin.clusters.getDiagnosisURL($scope.cluster.id));
        KubernetesClusterService.sendWt1Event('clusters-status-diagdownloaded',
                                              $scope.cluster.architecture, $scope.cluster.type);
    };
});

app.controller('K8SObjectDescribeController', function($rootScope, $scope, DataikuAPI, FutureWatcher, Dialogs, Logs) {
    $scope.describeObject = function() {
        DataikuAPI.admin.clusters.monitoring.describeObject($scope.clusterId, $scope.objectType.toLowerCase(), $scope.objectName, $scope.namespace).success(function(initialResponse) {
            if (initialResponse.hasResult) {
                $scope.describeResult = initialResponse.result.out;
                $scope.err = initialResponse.result.err;
                $scope.code = initialResponse.result.rv;
            } else {
                $scope.loading = true;
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(function(data) {
                        $scope.loading = false;
                        $scope.describeResult = data.result.out;
                        $scope.err = data.result.err;
                        $scope.code = data.result.rv;
                    }).update(function() {
                        $scope.loading = true;
                    }).error(function(data, status, headers) {
                        $scope.loading = false;
                        setErrorInScope.bind($scope)(data, status, headers);
                    });
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.tailPodLog = function(){
        DataikuAPI.admin.clusters.monitoring.tailPodLog($scope.clusterId, $scope.objectName, $scope.namespace).success(function(data){
            $scope.podLogTail = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.downloadPodLog = function() {
        if ($scope.objectType !== 'Pod') {
            return;
        }

        Logs.downloadPod($scope.clusterId, $scope.objectName, $scope.namespace);
    };

    $scope.switchToDescribe = function(){
        $scope.uiState.mode = "describe";
        $scope.describeObject();
    }

    $scope.switchToLogs = function(){
        $scope.uiState.mode = "logs";
        $scope.tailPodLog();
    }

    $scope.canDeleteObject = function() {
        return ['Pod', 'Job', 'Service', 'Deployment'].includes($scope.objectType);
    }

    $scope.deleteObject = function() {
        if (!$scope.canDeleteObject()) {
            return;
        }

        Dialogs.confirmPositive($scope, "Delete " + $scope.objectType,
                                "Are you sure you want to delete this {0}?".format($scope.objectType.toLowerCase())).then(function() {
            $scope.deleting = true;
            let handleSuccess = (response) => {
                $scope.refreshMonitoring();
                $scope.deleting = false;
                $scope.deleteResult = response.result.out;
                $scope.err = response.result.err;
                $scope.code = response.result.rv;
            };
            DataikuAPI.admin.clusters.monitoring.deleteObject($scope.clusterId, $scope.objectType.toLowerCase(), $scope.objectName, $scope.namespace).success(function(initialResponse) {
                if (initialResponse.hasResult) {
                    handleSuccess(initialResponse);
                } else {
                    FutureWatcher.watchJobId(initialResponse.jobId)
                        .success(function(data) {
                            handleSuccess(data)
                        }).update(function() {
                            $scope.deleting = true;
                        }).error(function(data, status, headers) {
                            $scope.deleting = false;
                            setErrorInScope.bind($scope)(data, status, headers);
                        });
                }
            }).error(setErrorInScope.bind($scope));
        }, function() {
            // Empty function to prevent error in console
        });
    };
});

app.directive('clusterParamsForm', function(Assert, $rootScope, PluginConfigUtils) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/cluster-params-form.html',
        scope: {
            params : '=',
            clusterType : '='
        },
        link: function($scope, element, attrs) {
            $scope.$watch('clusterType', function() {
                if (!$scope.clusterType) return;
                $scope.loadedDesc = $rootScope.appConfig.customPythonPluginClusters.filter(function(x){
                    return x.clusterType == $scope.clusterType;
                })[0];

                Assert.inScope($scope, 'loadedDesc');

                $scope.desc = $scope.loadedDesc.desc;

                // put default values in place
                $scope.params.config = $scope.params.config || {};
                PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.params.config);

                $scope.pluginDesc = $rootScope.appConfig.loadedPlugins.filter(function(x){
                    return x.id == $scope.loadedDesc.ownerPluginId;
                })[0];
            });
        }
    };
});

app.directive('clusterActionsForm', function($rootScope, DataikuAPI, CreateModalFromTemplate, KubernetesClusterService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/cluster-actions-form.html',
        scope: {
            params : '=',
            clusterId : '=',
            clusterType : '=',
            clusterArchitecture: '=',
            clusterState: '='
        },
        link: function($scope) {
            var refreshClusterActions = function() {
                $scope.clusterActions = [];

                const pluginsById = $rootScope.appConfig.loadedPlugins.reduce(function (map, obj) {
                    map[obj.id] = obj;
                    return map;
                }, {});

                $rootScope.appConfig.customRunnables.forEach(function(runnable) {
                    if (!runnable.desc.macroRoles) return;

                    const plugin = pluginsById[runnable.ownerPluginId];
                    if (!plugin || plugin.hideComponents) return; // plugin might have been deleted

                    runnable.desc.macroRoles.forEach(function(macroRole) {
                        if (macroRole.type != 'CLUSTER') return;
                        if (macroRole.limitToSamePlugin) {
                            if (!$scope.loadedDesc || runnable.ownerPluginId != $scope.loadedDesc.ownerPluginId) return;
                        }

                        $scope.clusterActions.push({
                            label: runnable.desc.meta.label || runnable.id,
                            icon: runnable.desc.meta.icon || plugin.icon,
                            roleTarget: macroRole.targetParamsKey || macroRole.targetParamsKeys,
                            runnable: runnable
                        });
                    });
                });

                $scope.isManualOrRunning = () =>  {
                    return $scope.clusterType === 'manual' || $scope.clusterState === 'RUNNING';
                }

                if ($scope.clusterArchitecture === 'KUBERNETES') {

                    $scope.k8sActions = [
                        {
                            label: 'Run kubectl command',
                            description: 'Run an arbitrary kubectl command',
                            icon: 'icon-play',
                            hasFilters: false,
                            customCommand: true,
                            check: () => {return $scope.$parent.isDSSAdmin()}
                        },
                        {
                            label: 'Delete finished pods',
                            description: 'Delete succeeded and failed pods',
                            command: 'delete pods --field-selector=status.phase!=Pending,status.phase!=Running,status.phase!=Unknown',
                            icon: 'icon-trash',
                            hasFilters: true,
                            wt1Event: 'clusters-action-deletefinishedpods',
                            controller: 'RunPredefinedK8sActionController',
                            api: DataikuAPI.admin.clusters.deleteFinishedPods,
                            customCommand: false
                        },
                        {
                            label: 'Delete all pods',
                            description: 'Delete all pods',
                            command: 'delete --all pods',
                            icon: 'icon-trash',
                            hasFilters: true,
                            wt1Event: 'clusters-action-deleteallpods',
                            controller: 'RunPredefinedK8sActionController',
                            api: DataikuAPI.admin.clusters.deleteAllPods,
                            customCommand: false
                        },
                        {
                            label: 'Delete finished jobs',
                            description: 'Delete finished jobs',
                            icon: 'icon-trash',
                            hasFilters: true,
                            wt1Event: 'clusters-action-deletefinishedjobs',
                            controller: 'DeleteFinishedJobsController',
                            customCommand: false
                        }
                    ]

                    $scope.showRunK8sAction = function(clusterId, {command, longDescription, label, icon, hasFilters, controller, wt1Event, api, customCommand}) {
                        KubernetesClusterService.sendWt1Event(wt1Event, $scope.clusterArchitecture, $scope.clusterType);
                        let actionController = controller ? controller : 'RunK8sActionController';
                        CreateModalFromTemplate('/templates/admin/clusters/run-k8s-action-modal.html', $scope, actionController, (newScope) => {
                            newScope.customCommand = customCommand;
                            newScope.longDescription = longDescription;
                            newScope.hasFilters = hasFilters;
                            newScope.clusterId = clusterId;
                            newScope.command = command? command: '';
                            newScope.label = label;
                            newScope.icon = icon;
                            newScope.api = api;
                            newScope.fullCommand = 'kubectl ' + newScope.command;
                        });
                    };

                    $scope.getActionByLabel = label => {
                        return $scope.k8sActions.find(action => action.label === label);
                    }

                    if ($rootScope.$state.params && $rootScope.$state.params.action) {
                        let action = $scope.getActionByLabel($rootScope.$state.params.action);
                        $scope.showRunK8sAction(
                            $scope.clusterId,
                            action
                        );
                    }
                }

                $scope.showCreateRunnable = function(clusterId, {roleTarget, runnable}) {
                    CreateModalFromTemplate('/templates/macros/runnable-modal.html', $scope, null, function(newScope) {
                        newScope.runnable = runnable;
                        newScope.targetKey = roleTarget;
                        newScope.targetValue = clusterId;
                        newScope.cluster = true;
                    });
                };
            };
            $scope.$watch('clusterType', function() {
                if (!$scope.clusterType) return;
                $scope.loadedDesc = $rootScope.appConfig.customPythonPluginClusters.filter(function(x){
                    return x.clusterType == $scope.clusterType;
                })[0];

                refreshClusterActions();
            });
        }
    };
});

app.directive('clusterActionTile', function() {
    return {
        restrict: 'E',
        templateUrl: '/templates/admin/clusters/cluster-action-tile.html',
        scope: {
            action : '<',
            clusterId: '<',
            showCallback: '<',
            description: '<?'
        },
        link: function($scope) {
            if ($scope.description === undefined) {
                // for native action description is stored directly in the action
                $scope.description = $scope.action.description;
            }
        }
    };
});

app.service("KubernetesClusterService", (DataikuAPI, FutureWatcher, FutureProgressModal, WT1) => {
        const svc = {};

        const EKS_CREATE_TYPE = "pycluster_eks-clusters_create-eks-cluster";
        const EKS_ATTACH_TYPE = "pycluster_eks-clusters_attach-eks-cluster";

        const AKS_CREATE_TYPE = "pycluster_aks-clusters_create-aks-cluster";
        const AKS_ATTACH_TYPE = "pycluster_aks-clusters_attach-aks-cluster";

        const GKE_CREATE_TYPE = "pycluster_gke-clusters_create-gke-cluster";
        const GKE_ATTACH_TYPE = "pycluster_gke-clusters_attach-gke-cluster";

        const EMR_CREATE_TYPE = "pycluster_emr-clusters_emr-create-cluster";
        const EMR_ATTACH_TYPE = "pycluster_emr-clusters_emr-attach-cluster";

        svc.getTypeDisplayName = (type) => {
            switch(type) {
                case EKS_CREATE_TYPE:
                    return 'EKS (managed)';
                case EKS_ATTACH_TYPE:
                    return 'EKS (attach)'; /* We say "attach" rather than "attached" to avoid confusion with state */
                case AKS_CREATE_TYPE:
                    return 'AKS (managed)';
                case AKS_ATTACH_TYPE:
                    return 'AKS (attach)';
                case GKE_CREATE_TYPE:
                    return 'GKE (managed)';
                case GKE_ATTACH_TYPE:
                    return 'GKE (attach)';
                case EMR_CREATE_TYPE:
                    return 'EMR (managed)';
                case EMR_ATTACH_TYPE:
                    return 'EMR (attach)';
                default:
                    return type;
            }
        }

        svc.getStartStopButtonDisplayString = (cluster, isStart) => {
            switch(cluster.type) {
                case EKS_CREATE_TYPE:
                case AKS_CREATE_TYPE:
                case GKE_CREATE_TYPE:
                case EMR_CREATE_TYPE:
                    return isStart ? "Start" : "Stop";
                case EKS_ATTACH_TYPE:
                case AKS_ATTACH_TYPE:
                case GKE_ATTACH_TYPE:
                case EMR_ATTACH_TYPE:
                    return isStart ? "Attach" :  "Detach";
                default:
                    return isStart ? "Start/Attach" : "Stop/Detach";
            }
        }

        svc.getStateDisplayString = (cluster) => {
            let displayNames = null;
            switch(cluster.type) {
                case EKS_CREATE_TYPE:
                case AKS_CREATE_TYPE:
                case GKE_CREATE_TYPE:
                case EMR_CREATE_TYPE:
                    displayNames = {NONE:'Stopped', RUNNING:'Running', STARTING:'Starting', STOPPING:'Stopping'};
                    break;
                case EKS_ATTACH_TYPE:
                case AKS_ATTACH_TYPE:
                case GKE_ATTACH_TYPE:
                case EMR_ATTACH_TYPE:
                    displayNames = {NONE:'Detached', RUNNING:'Attached', STARTING:'Attaching', STOPPING:'Detaching'};
                    break;
                default:
                    displayNames = {NONE:'Stopped/Detached', RUNNING:'Running/Attached', STARTING:'Starting/Attaching', STOPPING:'Stopping/Detaching'};
                    break;
            }
            if (cluster.state == null) {
                return 'Unknown';
            }
            return displayNames[cluster.state] || 'unknown';
        }

        svc.getIcon = (architecture, type) => {
            if (architecture === 'KUBERNETES') {
                switch(type) {
                    case EKS_CREATE_TYPE:
                    case EKS_ATTACH_TYPE:
                        return 'icon-amazon-elastic-kubernetes';
                    case AKS_CREATE_TYPE:
                    case AKS_ATTACH_TYPE:
                        return 'icon-azure-kubernetes-services';
                    case GKE_CREATE_TYPE:
                    case GKE_ATTACH_TYPE:
                        return 'icon-gcp-kubernetes-engine';
                    default:
                        // icon for K8S is not available yet using GKE instead
                        return 'icon-gcp-kubernetes-engine';
                }
            } else if (architecture === 'HADOOP') {
                // haven't found the hadoop icon, using the hdfs one
                return 'icon-HDFS'
            }
            // this should never happen, to my knowledge we only support HADOOP AND K8S cluster
            return 'icon-question';
        }

        svc.getTypeForWt1 = function(clusterType) {
            if (['manual',
                 'pycluster_eks-clusters_create-eks-cluster', 'pycluster_eks-clusters_attach-eks-cluster',
                 'pycluster_aks-clusters_create-aks-cluster', 'pycluster_aks-clusters_attach-aks-cluster',
                 'pycluster_gke-clusters_create-gke-cluster', 'pycluster_gke-clusters_attach-gke-cluster'].includes(clusterType)) {
                return clusterType;
            } else {
                return 'custom';
            }
        };

        svc.sendWt1Event = function(eventName, architecture, type) {
            WT1.event(eventName, { architecture: architecture, type: svc.getTypeForWt1(type) });
        };

        svc.stopCluster = (forceStop, scope, clusterId, successCallBack, architecture, type) => {
            svc.sendWt1Event(forceStop ? "clusters-action-forcestop" : "clusters-action-stop", architecture, type);
            return DataikuAPI.admin.clusters.stop(clusterId, false, forceStop)
                .success(
                    data => FutureProgressModal.show(scope, data, "Stop cluster", undefined, 'static', 'false')
                    .then(function(result){
                        if (result) { // undefined in case of abort
                            successCallBack();
                            svc.sendWt1Event("clusters-status-stopped", architecture, type);
                        }
                    })
                )
                .error(setErrorInScope.bind(scope));
        }

        return svc;
    }
);

app.controller("BaseK8sActionController", ($scope, FutureWatcher) => {
    $scope.running =false;
    $scope.dryRun = true;
    $scope.hasFilters = true;
    $scope.hasAdditionalOptions = false;

    $scope.hasCommandDryRunFlag = () => {
        return true;
    }

    $scope.buildCommandArgs = () => {
        throw new Error('Method buildCommandArgs should not be called from the base controller');
    }

    $scope.callApi = () => {
        throw new Error('Method callApi should not be called from the base controller');
    }

    $scope.resetCommandResult = () => {
        $scope.output = '';
        $scope.err = '';
        $scope.code = '';
    }

    $scope.run = () => {
        let parseResponse = (response) => {
            if ($scope.fatalAPIError) {
                delete $scope.fatalAPIError;
            }
            $scope.running = false;
            $scope.output = response.result.out;
            $scope.err = response.result.err;
            $scope.code = response.result.rv;
        }

        let handleError = (data, status, headers, config, statusText, xhrStatus) => {
            $scope.running = false;
            setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
        }

        $scope.resetCommandResult();
        $scope.running = true

        $scope.callApi().success((initialResponse) => {
            if (initialResponse.hasResult) {
                parseResponse(initialResponse);
            } else {
                FutureWatcher.watchJobId(initialResponse.jobId)
                    .success(parseResponse)
                    .error(handleError);
            }
        }).error(handleError);
    };

    $scope.extendedResetSetting = () => {
        // does nothing in the base controller
        return;
    }

    $scope.resetSettings =  () => {
        $scope.extendedResetSetting();
        $scope.namespaceFilter = '';
        $scope.labelFilter = '';
        $scope.commandArgs = '';
        $scope.dryRun = true;
        $scope.resetCommandResult();
        $scope.updateCommand();
    };

    $scope.updateCommand = () => {
        $scope.fullCommand = 'kubectl ' + $scope.buildCommandArgs();
    }
});

app.controller("RunK8sActionController", ($scope, $controller, DataikuAPI) => {
    $controller("BaseK8sActionController", {$scope:$scope});

    const commandsWithDryRun = ["create", "run", "expose", "delete", "apply", "annotate", "autoscale", "label", "patch",
        "replace", "scale", "set", "reconcile", "cordon", "drain", "taint", "uncordon", "undo"];
    const DRY_RUN_FLAG =  " --dry-run=client";
    const EXEC_SEPARATOR = " -- ";

    $scope.hasCommandDryRunFlag = () => {
        let commandArgs = $scope.commandArgs || $scope.command || '';
        let commandParts = commandArgs.split(" ");
        return commandParts.some(el => commandsWithDryRun.includes(el));
    }

    $scope.buildCommandArgs = () => {
        let commandArgs = $scope.commandArgs || $scope.command || '';
        commandArgs += $scope.namespaceFilter && $scope.namespaceFilter.trim() !== '' ? ' --namespace ' +  $scope.namespaceFilter : '';
        if ($scope.labelFilter && $scope.labelFilter.trim() !== '') {
            if (commandArgs.match(/\s--all[\s$]/)) {
                commandArgs = commandArgs.replace(/\s--all[\s$]/, ' ');
            }
            commandArgs += ' -l ' +  $scope.labelFilter;
        }
        if ($scope.dryRun && $scope.hasCommandDryRunFlag()) {
            if (commandArgs.includes(EXEC_SEPARATOR)) {
                commandArgs = commandArgs.replace(EXEC_SEPARATOR, `${DRY_RUN_FLAG} ${EXEC_SEPARATOR}`)
            } else {
                commandArgs += DRY_RUN_FLAG;
            }
        }
        return commandArgs;
    }

    $scope.callApi = () => {
        let commandWithFilters = $scope.buildCommandArgs();
        return DataikuAPI.admin.clusters.runKubectl($scope.clusterId, commandWithFilters);
    }
});

app.controller("RunPredefinedK8sActionController", ($scope, $controller) => {
    $controller("RunK8sActionController", {$scope:$scope});

    $scope.callApi = () => {
        return $scope.api($scope.clusterId, $scope.dryRun, $scope.namespaceFilter, $scope.labelFilter);
    }
});

app.controller("DeleteFinishedJobsController", ($scope, $controller, DataikuAPI) => {
    $controller("BaseK8sActionController", {$scope:$scope});
    const COMMAND = "delete jobs $(kubectl  get job -o go-template='{{range $i, $p := .items}}{{range .status.conditions}}{{if TYPE_FILTER}}{{$p.metadata.name}}{{\" \"}}{{end}}{{end}}{{end}}'LABEL_FILTER NAMESPACE_FILTER)NAMESPACE_FILTER";
    const COMPLETE_JOB_FILTER = "(eq .type \"Complete\")";
    const COMPLETE_AND_FAILED_JOB_FILTER = "or (eq .type \"Failed\") (eq .type \"Complete\")";

    $scope.hasAdditionalOptions = true;

    $scope.deleteFailed = false;
    $scope.fullCommand = 'kubectl ' + $scope.command;

    $scope.buildCommandArgs = () => {
        let commandArgs = COMMAND;
        commandArgs = commandArgs.replace("TYPE_FILTER", $scope.deleteFailed ? COMPLETE_AND_FAILED_JOB_FILTER : COMPLETE_JOB_FILTER);
        let namespaceReplacement = $scope.namespaceFilter && $scope.namespaceFilter.trim() !== '' ? " -n " + $scope.namespaceFilter : "";
        commandArgs = commandArgs.replaceAll("NAMESPACE_FILTER", namespaceReplacement);

        let labelReplacement = $scope.labelFilter && $scope.labelFilter.trim() !== '' ? " -l " + $scope.labelFilter : "";
        commandArgs = commandArgs.replaceAll("LABEL_FILTER", labelReplacement);

        if ($scope.dryRun) {
            commandArgs += " --dry-run=client"
        }
        return commandArgs;
    }

    $scope.callApi = () => {
        return DataikuAPI.admin.clusters.deleteFinishedJobs($scope.clusterId, $scope.dryRun, $scope.deleteFailed, $scope.namespaceFilter, $scope.labelFilter);
    }

    $scope.extendedResetSetting = () => {
        $scope.deleteFailed = false;
    }
});

app.directive('hadoopClusterSettingsBlock', function($rootScope) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/hadoop-cluster-settings-block.html',
        scope: {
            settings : '=',
            mask : '=',
            impersonationEnabled : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
        }
    };
});

app.directive('hiveClusterSettingsBlock', function($rootScope, CodeMirrorSettingService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/hive-cluster-settings-block.html',
        scope: {
            settings : '=',
            hadoopSettings : '=',
            mask : '=',
            impersonationEnabled : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
            $scope.codeMirrorSettingService = CodeMirrorSettingService;

            $scope.copyHadoopSettings = function() {
                if (!$scope.settings) return;
                if (!$scope.hadoopSettings) return;

                var hiveProps = $scope.settings.executionConfigsGenericOverrides;
                var hadoopPropsNames = $scope.hadoopSettings.extraConf.map(function(p) {return p.key;});
                // remove existing properties with the names of those we add (avoid duplicates)
                hadoopPropsNames.forEach(function(k) {
                   var indices = hiveProps.map(function(p, i) {return p.key == k ? i : null;}).filter(function(x) {return x != null;});
                   indices.reverse().forEach(function(i) {hiveProps.splice(i, 1);});
                });
                $scope.hadoopSettings.extraConf.forEach(function(p) {
                    var hp = angular.copy(p);
                    hiveProps.push(hp);
                });
                // to make the list's UI refresh
                $scope.settings.executionConfigsGenericOverrides = [].concat(hiveProps)
            };
        }
    };
});

app.directive('impalaClusterSettingsBlock', function($rootScope) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/impala-cluster-settings-block.html',
        scope: {
            settings : '=',
            mask : '=',
            impersonationEnabled : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
        }
    };
});

app.directive('sparkClusterSettingsBlock', function(DataikuAPI, $rootScope, FutureProgressModal, ActivityIndicator, FeatureFlagsService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/spark-cluster-settings-block.html',
        scope: {
            settings : '=',
            hadoopSettings : '=',
            mask : '=',
            impersonationEnabled : '=',
            clusterId : '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;


            $scope.executionEngines = [
                                   {value:null, label:'Disabled'},
                                   {value:'SPARK_SUBMIT', label:'Enabled'}
                                ];

            $scope.copyHadoopSettings = function() {
                if (!$scope.settings) return;
                if (!$scope.hadoopSettings) return;

                var sparkProps = $scope.settings.executionConfigsGenericOverrides;
                var hadoopPropsNames = $scope.hadoopSettings.extraConf.map(function(p) {return p.key;});
                // remove existing properties with the names of those we add (avoid duplicates)
                hadoopPropsNames.forEach(function(k) {
                   var indices = sparkProps.map(function(p, i) {return p.key == 'spark.hadoop.' + k ? i : null;}).filter(function(x) {return x != null;});
                   indices.reverse().forEach(function(i) {sparkProps.splice(i, 1);});
                });
                $scope.hadoopSettings.extraConf.forEach(function(p) {
                    var sp = angular.copy(p);
                    sp.key = 'spark.hadoop.' + p.key;
                    sparkProps.push(sp);
                });
                // to make the list's UI refresh
                $scope.settings.executionConfigsGenericOverrides = [].concat(sparkProps);
            };

            $scope.preloadYarnClusterFiles = function(yarnClusterSettings) {
                DataikuAPI.admin.clusters.preloadYarnClusterFiles(yarnClusterSettings).success(function(data){
                    FutureProgressModal.show($scope, data, "Preload files on cluster", undefined, 'static', 'false').then(function(result) {
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $scope.sparkVersionsCompatible = function(dssVersion, livyVersion) {
                if (dssVersion == null || dssVersion.length == 0) dssVersion = $rootScope.appConfig.sparkVersion;
                if (dssVersion == null || dssVersion.length == 0) return true;
                if (livyVersion == null || livyVersion.length == 0) return true;
                return dssVersion.substring(0,2) == livyVersion.substring(0,2)
            };
            $scope.shouldUseYarnCluster = function(sparkMaster, deployMode) {
                if (sparkMaster == 'yarn' && deployMode == 'cluster') return true;
                if (sparkMaster == 'yarn-cluster') return true;
                return false;
            };
        }
    };
});

app.directive('containerClusterSettingsBlock', function(DataikuAPI, WT1, $rootScope, FutureProgressModal, Dialogs, CodeMirrorSettingService) {
    return {
        restrict: 'A',
        templateUrl: '/templates/admin/clusters/fragments/container-cluster-settings-block.html',
        scope: {
            settings : '=',
            mask : '=',
            impersonationEnabled : '=',
            clusterId : '=',
            k8sClusters: '=',
            clusterDefinedConfig: '='
        },
        link: function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.addLicInfo = $rootScope.addLicInfo;
            $scope.codeMirrorSettingService = CodeMirrorSettingService;

            $scope.getNewContainerConfig = function() {
                return {
                    type: 'KUBERNETES',
                    usableBy: 'ALL', allowedGroups: [], workloadType: 'ANY',
                    dockerResources: [],
                    kubernetesResources: {
                        memRequestMB: -1, memLimitMB: -1,
                        cpuRequest: -1, cpuLimit: -1,
                        customRequests: [], customLimits: [],
                        hostPathVolumes: []
                    },
                    customSHMValueMB: -1,
                    properties: []
                };
            };

            DataikuAPI.security.listGroups(false)
                .success(data => {
                    if (data) {
                        data.sort();
                    }
                    $scope.allGroups = data;
                })
                .error(setErrorInScope.bind($scope));

            $scope.isBaseImageNameSuspicious = function(baseImage) {
                return /^(?:[\w-_]+\.)+\w+(?::\d+)?\//.test(baseImage);
            };

            var testErrors = {};
            $scope.getTestError = function(config) {
                let s = testErrors[config.name];
                return s != null ? s.fatalAPIError : null;
            };

            $scope.testConf = function(configuration, clusterId) {
                testErrors[configuration.name] = {}; // doesn't need to be a scope for setErrorInScope()
                DataikuAPI.admin.containerExec.testConf(configuration, $scope.clusterDefinedConfig, clusterId || $scope.clusterId, $scope.settings.executionConfigsGenericOverrides).success(function(data){
                    FutureProgressModal.show($scope, data, "Testing container configuration", undefined, 'static', 'false').then(function(result){
                        if (result) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Container test result", result.messages, result.futureLog);
                        }
                    })
                }).error(setErrorInScope.bind(testErrors[configuration.name]));
                WT1.event('container-conf-test');
            }

            $scope.getExtraClusters = function() {
                if (!$scope.k8sClusters) return [];
                return $scope.k8sClusters.filter(function(c) {return (c.id || '__builtin__') != ($scope.clusterId || '__builtin__');});
            };

        }
    };
});

app.directive('clusterSecurityPermissions', function(DataikuAPI, $rootScope, PermissionsService) {
    return {
        restrict : 'A',
        templateUrl : '/templates/admin/clusters/fragments/security-permissions.html',
        replace : true,
        scope : {
                cluster  : '='
        },
        link : function($scope, element, attrs) {
            $scope.appConfig = $rootScope.appConfig;
            $scope.ui = {};

            function makeNewPerm(){
                $scope.newPerm = {
                    update: true,
                    use: true
                }
            }
            makeNewPerm();

            const fixupPermissions = function() {
                if (!$scope.cluster) return;
                /* Handle implied permissions */
                $scope.cluster.permissions.forEach(function(p) {
                    p.$updateDisabled = false;
                    p.$manageUsersDisabled = false;
                    p.$useDisabled = false;

                    if ($scope.cluster.usableByAll) {
                        p.$useDisabled = true;
                    }
                    if (p.update) {
                        p.$useDisabled = true;
                    }
                    if (p.manageUsers) {
                        p.$useDisabled = true;
                        p.$updateDisabled = true;
                    }
                });
            };

            DataikuAPI.security.listGroups(false).success(function(allGroups) {
                if (allGroups) {
                    allGroups.sort();
                }
                $scope.allGroups = allGroups;
                DataikuAPI.security.listUsers().success(function(data) {
                    $scope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                    $scope.allUsersLogin = $scope.allUsers.map(user => '@' + user.login);
                }).error(setErrorInScope.bind($scope));
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
            }).error(setErrorInScope.bind($scope));

            $scope.$watch("cluster.owner", function() {
                $scope.ui.ownerLogin = $scope.cluster.owner;
            });

            $scope.addPermission = function() {
                $scope.cluster.permissions.push($scope.newPerm);
                makeNewPerm();
            };

            $scope.$watch("cluster.usableByAll", function(nv, ov) {
                fixupPermissions();
            })
            $scope.$watch("cluster.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
                fixupPermissions();
            }, true)
            $scope.$watch("cluster.permissions", function(nv, ov) {
                if (!nv) return;
                $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
                fixupPermissions();
            }, false)
            $scope.unassignedGroups = PermissionsService.buildUnassignedGroups($scope.cluster, $scope.allGroups);
            fixupPermissions();

            // Ownership mgmt
            $scope.$watch("ui.ownerLogin", function() {
                PermissionsService.transferOwnership($scope, $scope.cluster, "cluster");
            });


        }
    };
});

})();