(function(){
    'use strict';
    /** @typedef {import('../types').GeneratedSources.DSSVisualizationTheme} DSSVisualizationTheme */

    const app = angular.module('dataiku.dashboards.insights');

    const hasFacetDimension = function(insight) {
        return insight.params && insight.params.def && insight.params.def.facetDimension && insight.params.def.facetDimension.length > 0;
    };

    app.constant('ChartInsightHandler', {
        name: 'Chart',
        desc: 'Visualize data from your source',
        icon: 'icon-dku-nav_dashboard',
        color: 'chart',

        sourceType: 'DATASET',
        getSourceId: function(insight) {
            return insight.params.datasetSmartName;
        },
        hasOptions: (insight) => {
            return insight.params.def; // KPI charts have no axes, legends or tooltips.
        },
        hasEditTab: true,
        goToEditAfterCreation: true,
        getDefaultTileParams: function(insight) {
            return {
                showXAxis: true,
                showYAxis: true,
                showLegend: false,
                showBrush: false,
                inheritLegendPlacement: true,
                showTooltips: true,
                autoPlayAnimation: true,
                legendPlacement: insight.params && insight.params.def.legendPlacement,
                useInsightTheme: false
            };
        },
        getDefaultTileDimensions(insight) {
            if (insight && hasFacetDimension(insight)) {
                return [15, 12];
            }
            return [8, 8];
        }
    });

    app.directive('chartInsightTile', function($controller, ChartRequestComputer, MonoFuture, ChartTypeChangeHandler, ChartFeatures, ChartDataUtils, DashboardUtils, DashboardPageUtils,
        TileLoadingState, TileLoadingBehavior, DashboardFilters, Logger, $stateParams, $q, ChartStoreFactory, DSSVisualizationThemeUtils) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_tile.html',
            scope: {
                insight: '=',
                tile: '=',
                hook: '=',
                filters: '=',
                filtersParams: '<',
                dashboardTheme: '<'
            },
            link: function($scope, element) {
                let originChartTheme;
                let originChartDef;

                $scope.DashboardUtils = DashboardUtils;
                $scope.insight.$filteringStatus = {};
                $scope.originDashboard = {
                    id: $stateParams.dashboardId,
                    name: $stateParams.dashboardName,
                    pageId: $stateParams.pageId,
                    edit: true
                };

                $controller('ChartInsightViewCommon', { $scope });

                $scope.initChartCommonScopeConfig($scope);

                $scope.timing = {
                    directiveLink: new Date().getTime()
                };

                $scope.loading = false;
                $scope.loaded = false;
                $scope.error = null;

                function broadcastResize(event) {
                    if (event.detail && (event.detail.skipInDashboards || event.detail.skipInCharts)) {
                        return;
                    }

                    $scope.$broadcast('resize');
                }
                $(window).on('resize', broadcastResize);

                $scope.loadedCallback = function() {
                    if ($scope.hook.loadStates[$scope.tile.$tileId] != TileLoadingState.COMPLETE) {
                        $scope.timing.loadDone = new Date().getTime();
                        Logger.info('Done loading chart tile', $scope.tile.$tileId,
                            'link->load', $scope.timing.loadStart - $scope.timing.directiveLink,
                            'load->req', $scope.timing.requestReady - $scope.timing.loadStart,
                            'req->resp', $scope.timing.responseReceived - $scope.timing.requestReady,
                            'resp->draw', $scope.timing.drawStart - $scope.timing.responseReceived,
                            'draw->done', $scope.timing.loadDone - $scope.timing.responseReceived,
                            'load->done', $scope.timing.loadDone-$scope.timing.loadStart);
                    }
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.COMPLETE;
                };
                const executePivotRequest = MonoFuture($scope).wrap($scope.getExecutePromise);
                let errorRoutine, loadReject; // Stored on load() in order to be used by applySlideFilters afterwards
                let unregisterLoadingStateWatcher;
                let unregisterVisibilityWatcher;
                let unregisterFiltersWatcher;


                $scope.$on('$destroy', function() {
                    $(window).off('resize', broadcastResize);

                    if (unregisterLoadingStateWatcher != null) {
                        unregisterLoadingStateWatcher();
                    }
                    if (unregisterVisibilityWatcher != null) {
                        unregisterVisibilityWatcher();
                    }
                    if (unregisterFiltersWatcher != null) {
                        unregisterFiltersWatcher();
                    }
                });

                $scope.load = function(resolve, reject) {
                    Logger.info('Start loading chart tile', $scope.tile.$tileId);

                    $scope.timing.loadStart = new Date().getTime();
                    loadReject = reject;
                    errorRoutine = DashboardUtils.setError.bind([$scope, reject]);

                    const unregister = $scope.$watch('chart.summary', nv => {
                        if (!nv) {
                            return;
                        }
                        unregister();
                        if (!$scope.insight.$fixedUp) {
                            $scope.insight.params.summary = nv;
                            const insightTheme = getTheme($scope.insight);
                            ChartTypeChangeHandler.fixupChart($scope.insight.params.def, insightTheme);
                            ChartTypeChangeHandler.fixupSpec($scope.insight.params, insightTheme);

                            if ($scope.origInsight) {
                                $scope.origInsight.params.summary = nv;
                                const origInsightTheme = getTheme($scope.origInsight);
                                ChartTypeChangeHandler.fixupChart($scope.origInsight.params.def, origInsightTheme);
                                ChartTypeChangeHandler.fixupSpec($scope.origInsight.params, origInsightTheme);
                            }
                        }
                    });

                    $scope.loading = true;

                    const loadingData = {
                        successRoutine: DashboardUtils.setLoaded.bind([$scope, resolve]),
                        errorRoutine: errorRoutine,
                        unconfiguredRoutine: DashboardUtils.setUnconfigured.bind([$scope, reject]),
                        resolve: resolve,
                        reject: reject
                    };

                    unregisterFiltersWatcher = $scope.$watch('filters', () => {
                        applySlideFilters(loadingData);
                    });

                    return TileLoadingBehavior.DELAYED_COMPLETE;
                };
                $scope.hook.loadPromises[$scope.tile.$tileId] = $scope.load;
                $scope.hook.reloadPromises[$scope.tile.$tileId] = $scope.load;

                const additionalRequestCallbacks = {
                    successRoutine: () => {
                        DashboardUtils.setLoaded.call([$scope]);
                        $scope.saveChart();
                    },
                    errorRoutine: DashboardUtils.setError.bind([$scope]),
                    unconfiguredRoutine: DashboardUtils.setLoaded.bind([$scope])
                };

                $scope.rebuildSampling = () => {
                    DashboardUtils.setLoading.call([$scope]);
                    $scope.chart.refreshableSelection._refreshTrigger = new Date().getTime();
                    $scope.fetchColumnsSummary().then(() => {
                        executeFiltersRequest($scope.filters, additionalRequestCallbacks);
                    });
                };

                $scope.forceExecuteChart = (requestOptions) => {
                    const { store, id } = ChartStoreFactory.getOrCreate($scope.chart.def.$chartStoreId);
                    $scope.chart.def.$chartStoreId = id;
                    store.setRequestOptions(requestOptions);
                    DashboardUtils.setLoading.call([$scope]);
                    executeFiltersRequest($scope.filters, additionalRequestCallbacks);
                };


                /**
                 * To handle the failure of the loading routine
                 *
                 * @param {Object}          [loadingData]     - Insight loading instructions. Use when loading the insight.
                 * @param {Function}        loadingData.successRoutine
                 * @param {Function}        loadingData.errorRoutine
                 * @param {Function}        loadingData.unconfiguredRoutine
                 * @param {Function}        loadingData.resolve
                 * @param {Function}        loadingData.reject
                 * @param {Object}          [error]           - Caught error
                 * @param {Function}        error.data
                 * @param {Function}        error.status
                 * @param {Function}        error.headers
                 * @param {Function}        error.config
                 * @param {Function}        error.statusText
                 */
                const handleLoadingFailure = ( loadingData, error ) => {
                    const { data, status, headers, config, statusText } = error;
                    if (loadingData) {
                        loadingData.errorRoutine && loadingData.errorRoutine(data, status, headers, config, statusText);
                        if (loadingData.reject && typeof(loadingData.reject) === 'function') {
                            loadingData.reject();
                        }
                    } else {
                        errorRoutine && errorRoutine(data, status, headers, config, statusText);
                        if (loadReject && typeof(loadReject) === 'function') {
                            loadReject();
                        }
                    }
                };

                /**
                 * To handle the success of the loading routine
                 *
                 * @param {Object}          [loadingData]     - Insight loading instructions. Use when loading the insight.
                 * @param {Function}        loadingData.successRoutine
                 * @param {Function}        loadingData.errorRoutine
                 * @param {Function}        loadingData.unconfiguredRoutine
                 * @param {Function}        loadingData.resolve
                 * @param {Function}        loadingData.reject
                 * @param {Function}        [loadedCallback]  - Optional callback to call after loading
                 */
                const handleLoadingSuccess = ( loadingData, loadedCallback ) => {
                    if (loadingData) {
                        loadingData.successRoutine && loadingData.successRoutine();
                        if (loadingData.resolve && typeof(loadingData.resolve) === 'function') {
                            loadingData.resolve();
                        }
                        if(loadedCallback){
                            loadedCallback();
                        }
                    }
                };

                /**
                 * Compute and execute chart's pivot request...
                 *  - With or without additional global filters.
                 *  - With or without insight loading instructions.
                 *
                 * @param {ChartDef.java}   chartDef                - Classical ChartDef
                 * @param {Object}          [loadingData]           - Insight loading instructions. Use when loading the insight.
                 * @param {Function}        loadingData.successRoutine
                 * @param {Function}        loadingData.errorRoutine
                 * @param {Function}        loadingData.unconfiguredRoutine
                 * @param {Function}        loadingData.resolve
                 * @param {Function}        loadingData.reject
                 * @param {Boolean}         hasGlobalFilters                 - Are global filters applied on top of chart's ones
                 */
                const computeAndExecutePivotRequest = (chartDef, loadingData, hasGlobalFilters) => {
                    let request;

                    try {
                        $scope.chartSpecific = {
                            ...$scope.chartSpecific,
                            datasetProjectKey: $scope.getDataSpec().datasetProjectKey,
                            datasetName: $scope.getDataSpec().datasetName,
                            context: $scope.getCurrentChartsContext()
                        };

                        request = ChartRequestComputer.compute(chartDef, element.width(), element.height(), $scope.chartSpecific);
                        const validity = ChartTypeChangeHandler.getValidity({ def: chartDef });
                        if (loadingData && validity.valid === false) {
                            loadingData.errorRoutine({ detailedMessage: 'Invalid configuration, ' + validity.message });
                        }
                    } catch (e) {
                        if (hasGlobalFilters) {
                            const deferred = $q.defer();
                            deferred.reject({ data: e });
                            return deferred.promise;
                        }
                    }

                    if (loadingData && !request) {
                        loadingData.unconfiguredRoutine();
                    } else {
                        $scope.timing.requestReady = new Date().getTime();
                        return executePivotRequest(request).update(data => {
                            $scope.request = request;
                            $scope.response = data;
                        }).success(data => {
                            Logger.info('Got pivot response for chart tile', $scope.tile.$tileId);
                            $scope.timing.responseReceived = new Date().getTime();
                            $scope.request = request;
                            $scope.response = data;
                            $scope.validity = { valid: true };

                            handleLoadingSuccess(loadingData);

                            if (hasGlobalFilters) {
                                $scope.tile.$allFilteredOut = false;
                            }
                            const sampleUIData = ChartDataUtils.getSampleMetadataAndSummaryMessage(data.result.pivotResponse, 'Click to edit the insight');
                            $scope.tile.$sampleMetadata = sampleUIData.sampleMetadata;
                            $scope.tile.$samplingSummaryMessage = sampleUIData.summaryMessage;
                            $scope.tile.$clickableSamplingSummaryMessage = sampleUIData.clickableSummaryMessage;
                            $scope.tile.$recordsMetadata = ChartFeatures.getRecordsMetadata(chartDef, sampleUIData.sampleMetadata, data.result.pivotResponse.beforeFilterRecords, data.result.pivotResponse.afterFilterRecords);
                            $scope.tile.$display0Warning = ChartFeatures.shouldDisplay0Warning($scope.insight.params.def.type, data.result.pivotResponse);
                        }).error(function(data, status, headers, config, statusText) {
                            $scope.tile.$allFilteredOut = false;
                            $scope.tile.$display0Warning = false;
                            if (data.code === 'FILTERED_OUT') {
                                // We still need to display the insight to say it's been filtered out with a proper empty state.
                                handleLoadingSuccess(loadingData, $scope.loadedCallback());
                                $scope.tile.$allFilteredOut = true;
                            } else if (data.code === 'CANNOT_DISPLAY_WITH_EMPTY_AXES') {
                                handleLoadingSuccess(loadingData, $scope.loadedCallback());
                                $scope.tile.$display0Warning = true;
                            } else if (ChartTypeChangeHandler.hasRequestResponseWarning($scope.chart.def, data)) {
                                $scope.validity = ChartTypeChangeHandler.getRequestResponseWarning($scope.chart.def, data);
                                handleLoadingSuccess(loadingData, $scope.loadedCallback());
                            } else {
                                handleLoadingFailure( loadingData, { data, status, headers, config, statusText } );
                            }
                        });
                    }
                };

                function watchLoadingStateChange() {
                    if (unregisterLoadingStateWatcher) {
                        return;
                    }
                    unregisterLoadingStateWatcher = $scope.$watch('loaded', (nv) => {
                        if (nv === true) {
                            unregisterLoadingStateWatcher(); // remove watch
                            applySlideFilters();
                        }
                    });
                }

                function watchVisibilityChange() {
                    if (unregisterVisibilityWatcher) {
                        return;
                    }
                    unregisterVisibilityWatcher = $scope.$watch('hook.visibleStates[tile.$tileId]', function(nv) {
                        if (nv === true) {
                            unregisterVisibilityWatcher(); // remove watch
                            unregisterVisibilityWatcher = null;
                            applySlideFilters();
                        }
                    });
                }

                /**
                 *
                 * @param {Object}      [loadingData]           - Insight loading instructions. Use when loading the insight.
                 * @param {Function}    loadingData.successRoutine
                 * @param {Function}    loadingData.errorRoutine
                 * @param {Function}    loadingData.unconfiguredRoutine
                 * @param {Function}    loadingData.resolve
                 * @param {Function}    loadingData.reject
                 * @param {boolean}     [forceFilterableTilesRefresh=false]
                 */
                function applySlideFilters(loadingData) {
                    /*
                     * If the chart is not yet loaded and no loading instructions are provided,
                     * wait for it to complete before applying latest filter changes (which happened in the meantime)
                     */
                    if (!loadingData && !$scope.loaded && !unregisterLoadingStateWatcher) {
                        watchLoadingStateChange();
                        return;
                    }

                    if (!DashboardPageUtils.isTileVisibleOrAdjacent($scope.tile.$tileId, $scope.hook)) {
                        watchVisibilityChange();
                        return;
                    }

                    DashboardFilters.hasCaseInsensitiveColumnNames($scope.resolvedDataset.projectKey, $scope.resolvedDataset.datasetName, $stateParams.projectKey).then(hasCaseInsensitiveColumnNames => {

                        if ($scope.globalFiltersData.usableColumns == null) {
                            $scope.fetchColumnsSummary($scope.insight.projectKey)
                                .then(() => {
                                    $scope.globalFiltersData.usableColumns = $scope.usableColumns;
                                    return DashboardFilters.applyFilters($scope.filters, $scope.filtersParams, $scope.globalFiltersData, loadingData, hasCaseInsensitiveColumnNames);
                                }).catch(( error ) => {
                                    handleLoadingFailure( loadingData, error );
                                });
                        } else {
                            DashboardFilters.applyFilters($scope.filters, $scope.filtersParams, $scope.globalFiltersData, loadingData, hasCaseInsensitiveColumnNames);
                        }
                    }).catch((error) => {
                        handleLoadingFailure( loadingData, error );
                    });
                };

                function executeFiltersRequest(filters, loadingData) {
                    const chartDef = { ...$scope.insight.params.def };
                    if (filters && filters.length) {
                        chartDef.filters = [
                            ...chartDef.filters,
                            ...angular.copy(filters)
                        ];
                    }
                    const { store, id } = ChartStoreFactory.getOrCreate($scope.chart.def.$chartStoreId);
                    $scope.chart.def.$chartStoreId = id;
                    store.setAppliedDashboardFilters(angular.copy(filters));
                    return computeAndExecutePivotRequest(chartDef, loadingData, true);
                }

                $scope.globalFiltersData = {
                    insightId: $scope.insight.id,
                    usableColumns: $scope.usableColumns,
                    applyFilters: executeFiltersRequest,
                    parseResponse: (data) => data.data.result.pivotResponse,
                    filteringStatus: angular.copy($scope.insight.$filteringStatus),
                    sourceDataset: $scope.insight.params.datasetSmartName,
                    sampleSettings: $scope.insight.params.refreshableSelection,
                    pageId: $stateParams.pageId
                };

                // globalFiltersData.filteringStatus is mutated by applyFilters, and we need to update insight.$filteringStatus reference to trigger Angular's digest cycle.
                $scope.$watch('globalFiltersData.filteringStatus', filteringStatus => {
                    $scope.insight.$filteringStatus = angular.copy(filteringStatus);
                }, true);

                if ($scope.tile.autoLoad) {
                    $scope.hook.loadStates[$scope.tile.$tileId] = TileLoadingState.WAITING;
                }

                // As soon as the chart is available, set its originLegendPlacement property (done only once, needed in chartInsightTileParams).
                const deregisterLegendPlacementWatcher = $scope.$watch('insight.params.def.legendPlacement', () => {
                    $scope.chart.def.originLegendPlacement = $scope.insight.params.def.legendPlacement;
                    deregisterLegendPlacementWatcher();
                });

                const unregisterChartThemeWatcher = $scope.$watch('insight.params.theme', () => {
                    originChartTheme = angular.copy($scope.insight.params.theme);
                    unregisterChartThemeWatcher();
                });

                /*
                 * Before calling the fixUp function we need to make sure summary is defined to have access to the
                 * usableColumns.
                 */
                const unregisterChartSummaryWatcher = $scope.$watch('chart.summary', (nv) => {
                    if (!nv) {
                        return;
                    }
                    unregisterChartSummaryWatcher();

                    /*
                     * We need to wait for the chart summary to be available before fixing up the chart def,
                     * because some fixup operations need the summary (e.g. setting a default color palette
                     * based on the columns types).
                     */
                    const unregisterChartDefWatcher = $scope.$watch('insight.params.def', () => {
                        /*
                         * The goal here is to cache a version of the chart def as it is displayed in the
                         * chart view page. To do so, we need to fix it up.
                         */
                        const originChart = angular.copy($scope.insight.params);
                        ChartTypeChangeHandler.fixupChart(originChart.def, originChart.theme);
                        ChartTypeChangeHandler.fixupSpec(originChart, originChart.theme);
                        originChartDef = originChart.def;

                        unregisterChartDefWatcher();
                    });
                });



                $scope.$watch('dashboardTheme', function(newDashboardTheme) {
                    if ($scope.tile.tileParams.useInsightTheme || !newDashboardTheme) {
                        return;
                    }
                    /*
                     * We create a deep copy to make sure that if the dashboard theme is mutated
                     * (which ideally should not happen), then the chart theme won't be affected.
                     */
                    applyTheme(angular.copy(newDashboardTheme));
                });


                $scope.$watch('tile.tileParams', function(nv, ov) {
                    if (!nv) {
                        return;
                    }
                    const isFirstChange = nv === ov;

                    $scope.noXAxis = !nv.showXAxis;
                    $scope.noYAxis = !nv.showYAxis;
                    $scope.noTooltips = !nv.showTooltips;
                    $scope.autoPlayAnimation = nv.autoPlayAnimation;

                    if ($scope.chart && (isFirstChange || nv.useInsightTheme !== ov.useInsightTheme)) {

                        if (!nv.useInsightTheme) {
                            const newTheme = getTheme();
                            applyTheme(newTheme, { redraw: false });
                        } else {
                            resetChartDef();
                            resetChartTheme();
                            element[0].classList.toggle('chart-insight-no-theme', !!$scope.chart.theme);
                        }
                    }

                    if ($scope.chart) {
                        applyTileChartOptions();
                    }

                    $scope.$broadcast('redraw');
                }, true);

                /**
                 * Apply the given theme to the chart insight and optionally redraw it.
                 * @param {DSSVisualizationTheme | undefined} theme
                 * @param {{ redraw: boolean }} options
                 * @returns {void}
                 */
                function applyTheme(theme, options = { redraw: true }) {
                    if (!theme) {
                        /*
                         * theme can be undefined if:
                         * - both chart and dashboard theme is undefined
                         * - the tile ignores the dashboard theme and the insight theme is undefined
                         * - the tile uses the dashboard theme and the dashboard theme is undefined
                         *
                         * In these cases we need to make sure the chart insight is just displayed
                         * as it is outside of a dashboard.
                         */
                        resetChartDef();
                        // Re-apply the tile chart options on top of the resetted chart def.
                        applyTileChartOptions();
                        element.addClass('chart-insight-no-theme');
                    } else {
                        element.removeClass('chart-insight-no-theme');
                        DSSVisualizationThemeUtils.applyToChart({ chart:$scope.chart.def, theme, formerTheme: $scope.chart.theme });
                    }
                    $scope.chart.theme = theme;
                    if (options.redraw) {
                        $scope.$broadcast('redraw');
                    }
                }

                function applyTileChartOptions() {
                    if (!$scope.tile.tileParams.showLegend) {
                        $scope.chart.def.legendPlacement = 'SIDEBAR';
                    } else {
                        if ($scope.tile.tileParams.inheritLegendPlacement) {
                            $scope.chart.def.legendPlacement = $scope.chart.def.originLegendPlacement;
                        } else {
                            $scope.chart.def.legendPlacement = $scope.tile.tileParams.legendPlacement ? $scope.tile.tileParams.legendPlacement : $scope.chart.def.originLegendPlacement;
                        }
                    }
                    $scope.chart.def.showXAxis = $scope.tile.tileParams.showXAxis;
                    $scope.chart.def.linesZoomOptions.displayBrush = $scope.tile.tileParams.showBrush;
                }

                function resetChartDef() {
                    replaceDeep($scope.chart.def, angular.copy(originChartDef)); // we need to copy originChartDef because some operations mutate $scope.chart.def
                }

                function resetChartTheme() {
                    if (originChartTheme) {
                        replaceDeep($scope.chart.theme, angular.copy(originChartTheme));
                    } else {
                        $scope.chart.theme = undefined;
                    }
                }

                /**
                 * Returns the theme that needs to be used by the insight.
                 * @returns {DSSVisualizationTheme | undefined}
                 */
                function getTheme() {
                    if (!$scope.tile.tileParams.useInsightTheme && $scope.dashboardTheme) {
                        return angular.copy($scope.dashboardTheme);
                    }

                    return originChartTheme;
                }
            }
        };
    });

    app.directive('chartInsightTileParams', function(ChartFeatures){
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_tile_params.html',
            scope: {
                tileParams: '=',
                insight: '='
            },
            link: function($scope){
                $scope.$watch('insight.params.def.type', (type) => {
                    $scope.noAxis = !ChartFeatures.canHideAxes(type);
                    $scope.noTooltips = !ChartFeatures.canShowTooltips(type);
                });

                $scope.$watch('insight.params.def', (def) =>
                    $scope.canDisplayBrush = ChartFeatures.canDisplayBrush(def));

                $scope.$watch('insight.params.def.animationDimension', (animationDimension) =>
                    $scope.noAnimation = (animationDimension || []).length === 0);

                $scope.$watchGroup(['insight.params.def.type', 'insight.params.def.originLegendPlacement'], ([type, legendPlacement]) =>
                    $scope.noLegend = !ChartFeatures.canDisplayLegend(type));
            }
        };
    });

    app.directive('chartInsightCreateForm', function(DataikuAPI, ChartTypeChangeHandler, DatasetChartsUtils, DSSVisualizationThemeUtils) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_create_form.html',
            scope: true,
            link: function($scope, element, attrs) {
                $scope.hook.sourceTypes = ['DATASET'];
                $scope.hook.beforeSave = function(resolve, reject) {
                    DataikuAPI.explores.get($scope.insight.projectKey, $scope.insight.params.datasetSmartName)
                        .success(function(data) {
                            $scope.insight.params.refreshableSelection = DatasetChartsUtils.makeSelectionFromScript(data.script);
                            $scope.insight.params.def = ChartTypeChangeHandler.defaultNewChart(DSSVisualizationThemeUtils.getThemeOrDefault($scope.insight.params.theme));
                            $scope.insight.params.theme = DSSVisualizationThemeUtils.getThemeOrDefault($scope.insight.params.theme);
                            $scope.insight.params.engineType = 'LINO';

                            resolve();
                        })
                        .error(function(data, status, headers, config, statusText){
                            reject(arguments);
                        });
                };

                $scope.hook.defaultName = 'Chart';
                $scope.$watch('hook.sourceObject', function(nv) {
                    if (!nv || !nv.label) {
                        return;
                    }
                    $scope.hook.defaultName = 'Chart on ' + nv.label;
                });
            }
        };
    });

    app.controller('ChartInsightViewCommon', function($scope, $rootScope, DataikuAPI, $stateParams, $controller, ActiveProjectKey, DashboardPageUtils) {
        $controller('ShakerChartsCommonController', { $scope: $scope });

        $scope.isProjectAnalystRW = function() {
            return true;
        };

        $scope.resolvedDataset = resolveDatasetFullName($scope.insight.params.datasetSmartName, ActiveProjectKey.get());

        // Needed by chart directives
        $scope.chart = {
            def: $scope.insight.params.def,
            theme: $scope.insight.params.theme,
            refreshableSelection: $scope.insight.params.refreshableSelection,
            engineType: $scope.insight.params.engineType
        };

        $scope.validity = {}; // Needed by chart redraw

        if ($scope.tile) {
            $scope.chart.def.showLegend = $scope.tile.tileParams.showLegend;
            $scope.chart.def.showXAxis = $scope.tile.tileParams.showXAxis;
        }

        $scope.saveChart = function() {
            DataikuAPI.dashboards.insights.save($scope.insight)
                .error(setErrorInScope.bind($scope))
                .success(function() {});
        };

        $scope.getDataSpec = function() {
            return {
                datasetProjectKey: $scope.resolvedDataset.projectKey,
                datasetName: $scope.resolvedDataset.datasetName,
                copyScriptFromExplore: true,
                copySelectionFromScript: false,
                sampleSettings : $scope.insight.params.refreshableSelection,
                engineType : $scope.insight.params.engineType
            };
        };

        $scope.getExecutePromise = function(request, saveShaker=false, noSpinner=true, requiredSampleId=undefined, dataSpec=$scope.getDataSpec()) {
            if (request) {
                if (requiredSampleId === undefined) {
                    requiredSampleId = $scope.chart.summary == null ? null : $scope.chart.summary.requiredSampleId;
                }
                const projectKey = $scope.insight.projectKey || ActiveProjectKey.get();
                let promise = DataikuAPI.shakers.charts.getPivotResponse(
                    projectKey,
                    dataSpec,
                    request,
                    requiredSampleId
                );
                if (noSpinner === true) {
                    promise = promise.noSpinner();
                }
                return promise;
            }
        };

        $scope.fetchColumnsSummary = function(projectKey){
            // get columns summary
            if (!projectKey) {
                projectKey = ActiveProjectKey.get();
            }
            return DashboardPageUtils.getColumnSummary(projectKey, $scope.getDataSpec())
                .then(data => {
                    $scope.chart.summary = data;
                    $scope.insight.params.summary = data;
                    if ($scope.origInsight) {
                        $scope.origInsight.params.summary = data;
                    }
                    $scope.makeUsableColumns(data);
                });
        };

        function fetchSummaryAndExecute(saveChart = false) {
            $scope.fetchColumnsSummary().then(function() {
                saveChart && $scope.saveChart();
                $scope.forceExecuteChartOrWait();
            }).catch(({ data, status, headers, config, statusText, xhrStatus }) => {
                setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
            });
        }

        $rootScope.$on('chartSamplingChanged', function(event, opts) {
            if (angular.equals($scope.chart, opts.chart)) {
                $scope.chart.summary = null;
                fetchSummaryAndExecute(true);
            }
        });

        // chartHandler options
        $scope.noThumbnail = true;
        $scope.addCustomMeasuresToScopeAndCache($scope.insight.params.customMeasures);
        $scope.addBinnedDimensionToScopeAndCache($scope.insight.params.reusableDimensions);
    });

    app.directive('chartInsightView', function($controller, translate) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_view.html',
            scope: {
                insight: '='
            },
            link: function($scope) {
                $controller('ChartInsightViewCommon', { $scope: $scope });
                $controller('ChartsCommonController', { $scope: $scope });

                $scope.translate = translate;
                $scope.noClickableAxisLabels = true;
                $scope.readOnly = true;
                $scope.bigChart = true;
                $scope.bigChartDisabled = true;
                $scope.showLegends = true;
                $scope.saveChart = function() {};

                $scope.fetchColumnsSummary().then(function(){
                    $scope.forceExecuteChartOrWait();
                }).catch(({ data, status, headers, config, statusText, xhrStatus }) => {
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                });
            }
        };
    });

    app.directive('chartInsightEdit', function($controller, $stateParams, DataikuAPI, $rootScope, ChartTypeChangeHandler, DSSVisualizationThemeUtils, StateUtils, ActivityIndicator) {
        return {
            templateUrl: '/templates/dashboards/insights/chart/chart_edit.html',
            scope: {
                insight: '='
            },
            link: function($scope, element, attrs){
                $controller('ChartInsightViewCommon', { $scope: $scope });
                $controller('ChartsCommonController', { $scope: $scope });

                $scope.currentInsight = $scope.insight;
                $scope.appConfig = $rootScope.appConfig;
                $scope.uiDisplayState = {};

                $scope.bigChart = false;

                $scope.getDefaultNewChart = function() {
                    const defaultTheme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
                    return {
                        theme: defaultTheme,
                        def: ChartTypeChangeHandler.defaultNewChart(defaultTheme)
                    };
                };

                function fetchSummaryAndExecute(){
                    $scope.fetchColumnsSummary().then(function(){
                        $scope.forceExecuteChartOrWait();
                    }).catch(({ data, status, headers, config, statusText, xhrStatus }) => {
                        setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                    });
                }

                $scope.$watch('chart.engineType', function(nv) {
                    if (!nv) {
                        return;
                    }
                    $scope.insight.params.engineType = nv;
                });

                DataikuAPI.datasets.get($scope.resolvedDataset.projectKey, $scope.resolvedDataset.datasetName, $stateParams.projectKey)
                    .success(function(data) {
                        $scope.dataset = data;
                        fetchSummaryAndExecute();
                    }).error(setErrorInScope.bind($scope));

                $scope.$watch('chart.def.name', function(nv, ov) {
                    if ($scope.insight.name == 'Chart on ' + $scope.insight.params.datasetSmartName
                        || $scope.insight.name == ov + ' on ' + $scope.insight.params.datasetSmartName) {
                        $scope.insight.name = nv + ' on ' + $scope.insight.params.datasetSmartName;
                    }
                });

                $scope.overrideFormattingWithTheme = function(theme) {
                    const currentChartCopy = angular.copy($scope.chart);

                    DSSVisualizationThemeUtils.applyToChart({ chart: $scope.chart.def, theme, formerTheme: $scope.chart.theme });
                    $scope.chart.theme = theme;
                    $scope.insight.params.theme = theme;

                    DSSVisualizationThemeUtils.showThemeAppliedSnackbar(
                        $scope.chart,
                        currentChartCopy,
                        () => {
                            $scope.insight.params.theme = currentChartCopy.theme;
                        }
                    );
                };

                $scope.$on('$destroy', () => DSSVisualizationThemeUtils.hideThemeAppliedSnackbar());
            }
        };
    });

})();
