import requests
import dataiku
from dataikuapi.utils import DataikuException

def execute(requesting_user, deployment_id, deployment_report, deployer_client, **kwargs):
    """
    Custom hook function.

    :param str requesting_user: identifier of the DSS user requesting the deployment
    :param str deployment_id: id of the deployment in progress
    :param deployment_report: status of the deployment and messages related if any.
                              In case of a pre deployment hook, this parameter will be None
                              In case of a post deployment hook, it will be a dictionary with:
                                - "status" for the deployment status, can be "SUCCESS", "WARNING" or "ERROR"
                                - "messages" for the list of messages related to the deployment, as strings 
    :type deployment_report: dict or None
    :param deployer_client: an api client to connect to the deployer node with the credentials of the user running hooks
    :type deployer_client: A :class:`dataikuapi.dssclient.DSSClient`

    :returns: The execution status of the hook and a message as string. 
        Use `HookResult.success(message)`, `HookResult.warning(message)` or `HookResult.error(message)`
    """
    adpl = deployer_client.get_apideployer()
    deplyt = adpl.get_deployment(deployment_id)
    if deplyt.get_status().get_health() == "ERROR":
        print("Deployment failed. Not publishing OpenAPI on Swagger portal.")
        return HookResult.success(f"Deployment failed. Not publishing OpenAPI doc.")

    api_name = deplyt.get_settings().get_raw()['publishedServiceId']
    try:
        open_api_content = deplyt.get_open_api()
    except DataikuException as e:
        if ("NotFoundException" in e.args[0]):
            print(f"Service {api_name} does not have OpenAPI documentation")
            return HookResult.success(f"No OpenAPI to publish")
        else:
            raise

    # Define authentication and request url 
    authentication_key = "__TO_BE_DEFINED__"
    owner = "__TO_BE_DEFINED__"
    request_url = f"https://api.swaggerhub.com/apis/{owner}/{api_name}"

    r = requests.post(request_url, 
                      headers={'Authorization': authentication_key},
                      json=open_api_content.get())

    print(f"Pushing OpenAPI to portal using URL {request_url} with result {r.status_code}")
    r.raise_for_status()
    return HookResult.success(f"OpenAPI published on {request_url}")

