-- One action / endpoint can lead to multiple revisions for different entities
CREATE TABLE history_actions (
    history_action_id BIGINT PRIMARY KEY GENERATED ALWAYS AS ((history_action_json ->> 'id')::BIGINT) STORED,
    history_action_timestamp TIMESTAMP WITH TIME ZONE GENERATED ALWAYS AS (TO_TIMESTAMP(((history_action_json ->> 'timestamp')::DOUBLE PRECISION)/1000)) STORED NOT NULL,
    history_action_user TEXT GENERATED ALWAYS AS ((history_action_json ->> 'user')::TEXT) STORED,
    history_action_json JSONB NOT NULL
);
CREATE SEQUENCE history_actions_seq MAXVALUE 9223372036854775807; -- bigint max
CREATE INDEX history_action_timestamp_index ON history_actions (history_action_timestamp);

-- One revision concerns one create/update/delete request (so one entity)
-- This table is used in an abstract way, no need for a PK because it won't be propagated to children tables
CREATE TABLE _history_revisions (
    _revision_id BIGINT GENERATED ALWAYS AS ((_revision_json ->> 'id')::BIGINT) STORED,
    _revision_action_id BIGINT GENERATED ALWAYS AS ((_revision_json ->> 'actionId')::BIGINT) STORED NOT NULL REFERENCES history_actions,
    _revision_timestamp TIMESTAMP WITH TIME ZONE GENERATED ALWAYS AS (TO_TIMESTAMP(((_revision_json ->> 'timestamp')::DOUBLE PRECISION)/1000)) STORED NOT NULL,
    _revision_json JSONB NOT NULL
);
CREATE SEQUENCE history_revisions_seq MAXVALUE 9223372036854775807; -- bigint max

CREATE TABLE govern_configuration_history (
    configuration_item_id TEXT GENERATED ALWAYS AS ((configuration_item_json ->> 'type')::TEXT) STORED NOT NULL,
    configuration_item_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX govern_configuration_history_item_id_index ON govern_configuration_history (configuration_item_id);

CREATE TABLE blueprints_history (
    blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_json ->> 'id')::TEXT) STORED NOT NULL,
    blueprint_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX blueprints_history_blueprint_id_index ON blueprints_history (blueprint_id);

CREATE TABLE blueprint_versions_history (
    blueprint_version_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_version_json -> 'id' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_version_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_json -> 'id' ->> 'versionId')::TEXT) STORED NOT NULL,
    blueprint_version_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX blueprint_versions_history_blueprint_id_version_id_index ON blueprint_versions_history (blueprint_version_blueprint_id, blueprint_version_version_id);

CREATE TABLE blueprint_version_traces_history (
    blueprint_version_trace_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_version_trace_json -> 'blueprintVersionId' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_version_trace_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_trace_json -> 'blueprintVersionId' ->> 'versionId')::TEXT) STORED NOT NULL,
    blueprint_version_trace_origin_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_trace_json ->> 'originVersionId')::TEXT) STORED,
    blueprint_version_trace_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX blueprint_version_traces_history_blueprint_id_version_id_index ON blueprint_version_traces_history (blueprint_version_trace_blueprint_id, blueprint_version_trace_version_id);

CREATE TABLE artifacts_history (
    artifact_id TEXT GENERATED ALWAYS AS ((artifact_json ->> 'id')::TEXT) STORED NOT NULL,
    artifact_blueprint_id TEXT GENERATED ALWAYS AS ((artifact_json -> 'blueprintVersionId' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    artifact_version_id TEXT GENERATED ALWAYS AS ((artifact_json -> 'blueprintVersionId' ->> 'versionId')::TEXT) STORED NOT NULL,
    artifact_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX artifacts_history_artifact_id_index ON artifacts_history (artifact_id);

CREATE TABLE reference_relationships_history (
    reference_relationship_artifact_id_from TEXT GENERATED ALWAYS AS ((reference_relationship_json ->> 'artifactIdFrom')::TEXT) STORED NOT NULL,
    reference_relationship_artifact_id_to TEXT GENERATED ALWAYS AS ((reference_relationship_json ->> 'artifactIdTo')::TEXT) STORED NOT NULL,
    -- extract details key (along with artifactIdFrom, relationshipData object contains all needed information to uniquely identify a relationship, which is needed in the timeback machine request)
    reference_relationship_json_data JSONB GENERATED ALWAYS AS (reference_relationship_json -> 'relationshipData') STORED,
    reference_relationship_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX reference_relationships_history_arti_id_from_json_data_index ON reference_relationships_history (reference_relationship_artifact_id_from, reference_relationship_json_data);

CREATE TABLE uploaded_files_history (
    uploaded_file_id TEXT GENERATED ALWAYS AS ((uploaded_file_json ->> 'id')::TEXT) STORED NOT NULL,
    uploaded_file_json JSONB NOT NULL,
    uploaded_file_data BYTEA,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX uploaded_files_history_uploaded_file_id_index ON uploaded_files_history (uploaded_file_id);

CREATE TABLE uploaded_file_relationships_history (
    uploaded_file_relationship_artifact_id TEXT GENERATED ALWAYS AS ((uploaded_file_relationship_json ->> 'artifactId')::TEXT) STORED NOT NULL,
    uploaded_file_relationship_uploaded_file_id TEXT GENERATED ALWAYS AS ((uploaded_file_relationship_json ->> 'uploadedFileId')::TEXT) STORED NOT NULL,
    -- extract details key (along with artifactId, relationshipData object contains all needed information to uniquely identify a relationship, which is needed in the timeback machine request)
    uploaded_file_relationship_json_data JSONB GENERATED ALWAYS AS (uploaded_file_relationship_json -> 'relationshipData') STORED,
    uploaded_file_relationship_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX uploaded_file_relationships_history_artifact_id_json_data_index ON uploaded_file_relationships_history (uploaded_file_relationship_artifact_id, uploaded_file_relationship_json_data);

CREATE TABLE time_series_history (
    time_series_id TEXT GENERATED ALWAYS AS ((time_series_json ->> 'timeSeriesId')::TEXT) STORED NOT NULL,
    time_series_timestamp TIMESTAMP(3) GENERATED ALWAYS AS (TO_TIMESTAMP(((time_series_json ->> 'timestamp')::DOUBLE PRECISION)/1000)) STORED NOT NULL,
    time_series_json JSONB,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX time_series_history_time_series_id_timestamp_index ON time_series_history (time_series_id, time_series_timestamp);

CREATE TABLE time_series_relationships_history (
    time_series_relationship_artifact_id TEXT GENERATED ALWAYS AS ((time_series_relationship_json ->> 'artifactId')::TEXT) STORED NOT NULL,
    time_series_relationship_time_series_id TEXT GENERATED ALWAYS AS ((time_series_relationship_json ->> 'timeSeriesId')::TEXT) STORED,
    -- extract details key (along with artifactId, relationshipData object contains all needed information to uniquely identify a relationship, which is needed in the timeback machine request)
    time_series_relationship_json_data JSONB GENERATED ALWAYS AS (time_series_relationship_json -> 'relationshipData') STORED,
    time_series_relationship_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX time_series_relationships_history_artifact_id_json_data_index ON time_series_relationships_history (time_series_relationship_artifact_id, time_series_relationship_json_data);

CREATE TABLE custom_pages_history (
    custom_page_id TEXT GENERATED ALWAYS AS ((custom_page_json ->> 'id')::TEXT) STORED NOT NULL,
    custom_page_index INTEGER GENERATED ALWAYS AS ((custom_page_json ->> 'index')::INTEGER) STORED,
    custom_page_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX custom_pages_history_custom_page_id_index ON custom_pages_history (custom_page_id);

CREATE TABLE user_mappings_history (
    user_mapping_artifact_id TEXT GENERATED ALWAYS AS ((user_mapping_json ->> 'artifactId')::TEXT) STORED NOT NULL,
    user_mapping_user_login TEXT GENERATED ALWAYS AS ((user_mapping_json ->> 'userLogin')::TEXT) STORED NOT NULL,
    user_mapping_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX user_mappings_history_artifact_id_user_login_index ON user_mappings_history (user_mapping_artifact_id, user_mapping_user_login);

CREATE TABLE group_mappings_history (
    group_mapping_artifact_id TEXT GENERATED ALWAYS AS ((group_mapping_json ->> 'artifactId')::TEXT) STORED NOT NULL,
    group_mapping_group_name TEXT GENERATED ALWAYS AS ((group_mapping_json ->> 'groupName')::TEXT) STORED NOT NULL,
    group_mapping_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX group_mappings_history_artifact_id_group_name_index ON group_mappings_history (group_mapping_artifact_id, group_mapping_group_name);

CREATE TABLE global_api_key_mappings_history (
    global_api_key_mapping_artifact_id TEXT GENERATED ALWAYS AS ((global_api_key_mapping_json ->> 'artifactId')::TEXT) STORED NOT NULL,
    global_api_key_mapping_api_key_id TEXT GENERATED ALWAYS AS ((global_api_key_mapping_json ->> 'apiKeyId')::TEXT) STORED NOT NULL,
    global_api_key_mapping_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX global_api_key_mappings_history_artifact_id_api_key_id_index ON global_api_key_mappings_history (global_api_key_mapping_artifact_id, global_api_key_mapping_api_key_id);

CREATE TABLE roles_history (
    role_id TEXT GENERATED ALWAYS AS ((role_json ->> 'id')::TEXT) STORED NOT NULL,
    role_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX roles_history_role_id_index ON roles_history (role_id);

CREATE TABLE blueprint_role_assignments_history (
    blueprint_role_assignments_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_role_assignments_json ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_role_assignments_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX blueprint_role_assignments_history_blueprint_id_index ON blueprint_role_assignments_history (blueprint_role_assignments_blueprint_id);

CREATE TABLE blueprint_permissions_history (
    blueprint_permissions_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_permissions_json ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_permissions_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX blueprint_permissions_history_blueprint_id_index ON blueprint_permissions_history (blueprint_permissions_blueprint_id);

CREATE TABLE blueprint_version_signoff_configurations_history (
    blueprint_version_signoff_configurations_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_version_signoff_configurations_json -> 'blueprintVersionId' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_version_signoff_configurations_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_signoff_configurations_json -> 'blueprintVersionId' ->> 'versionId')::TEXT) STORED NOT NULL,
    blueprint_version_signoff_configurations_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX blueprint_version_signoff_configurations_history_bp_i_v_i_index ON blueprint_version_signoff_configurations_history (blueprint_version_signoff_configurations_blueprint_id, blueprint_version_signoff_configurations_version_id);

CREATE TABLE artifact_signoffs_history (
    artifact_signoffs_artifact_id TEXT GENERATED ALWAYS AS ((artifact_signoffs_json ->> 'artifactId')::TEXT) STORED NOT NULL,
    artifact_signoffs_json JSONB NOT NULL,
    PRIMARY KEY (_revision_id)
) INHERITS (_history_revisions);
CREATE INDEX artifact_signoffs_history_artifact_id_index ON artifact_signoffs_history (artifact_signoffs_artifact_id);
