/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.server.services;

import com.dataiku.dip.ApplicationConfigurator;
import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.coremodel.SerializedProject;
import com.dataiku.dip.exceptions.CodedException;
import com.dataiku.dip.exceptions.UnauthorizedException;
import com.dataiku.dip.git.GitInfo;
import com.dataiku.dip.plugins.dev.FolderEditorService;
import com.dataiku.dip.security.AuthCtx;
import com.dataiku.dip.security.UrlRedactionUtils;
import com.dataiku.dip.server.controllers.MergeRequestGitController;
import com.dataiku.dip.server.controllers.NotFoundException;
import com.dataiku.dip.server.notifications.backend.TaggableObjectChangedEvent;
import com.dataiku.dip.server.services.DKUNoGitTransactionService;
import com.dataiku.dip.server.services.PubSubService;
import com.dataiku.dip.server.services.licensing.LimitsStatusComputer;
import com.dataiku.dip.transactions.TransactionContext;
import com.dataiku.dip.transactions.TransactionProvider;
import com.dataiku.dip.transactions.fs.RelFile;
import com.dataiku.dip.transactions.git.DSSTransactionProviderSettings;
import com.dataiku.dip.transactions.git.GitCodes;
import com.dataiku.dip.transactions.git.GitModel;
import com.dataiku.dip.transactions.git.MergeRequest;
import com.dataiku.dip.transactions.git.MergeRequestEvent;
import com.dataiku.dip.transactions.git.MergeRequestInfo;
import com.dataiku.dip.transactions.git.cli.GitRemoteCommands;
import com.dataiku.dip.transactions.git.jgit.DSSSingleRepositoryGitManager;
import com.dataiku.dip.transactions.git.jgit.GitLocalCommands;
import com.dataiku.dip.transactions.git.jgit.ProjectsJGitService;
import com.dataiku.dip.transactions.ifaces.RWTransaction;
import com.dataiku.dip.transactions.ifaces.RWTransactionRef;
import com.dataiku.dip.transactions.ifaces.Transaction;
import com.dataiku.dip.transactions.ifaces.TransactionRef;
import com.dataiku.dip.utils.DKUFileUtils;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.DKUtils;
import com.dataiku.dip.utils.JSON;
import com.google.common.base.Preconditions;
import com.google.gson.JsonSyntaxException;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang.StringUtils;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.RepositoryState;
import org.eclipse.jgit.storage.file.FileBasedConfig;
import org.eclipse.jgit.storage.file.FileRepositoryBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class ProjectGitMergeRequestService {
    public static final String MERGE_REQUESTS_FILE = "merge-requests.json";
    private ProjectsJGitService projectsJGitService;
    private DKUNoGitTransactionService transactionService;
    private FolderEditorService folderEditorService;
    private PubSubService pubSubService;
    private static DKULogger logger = DKULogger.getLogger((String)"dku.git.merge-requests");

    @Autowired
    public ProjectGitMergeRequestService(ProjectsJGitService projectsJGitService, FolderEditorService folderEditorService, PubSubService pubSubService) throws IOException {
        this.projectsJGitService = projectsJGitService;
        this.folderEditorService = folderEditorService;
        this.pubSubService = pubSubService;
        this.transactionService = new DKUNoGitTransactionService();
        pubSubService.subscribe("project-deleted", evt -> DKUFileUtils.deleteDirectory((File)DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"merge_requests", "projects", evt.projectKey})));
    }

    private void saveInitialConflictingFilesIfAny(AuthCtx authCtx, MergeRequest request, Set<String> initialConflictingFiles) throws CodedException, IOException {
        if (!initialConflictingFiles.isEmpty()) {
            try (RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(authCtx);){
                request.initialConflictingFiles = initialConflictingFiles;
                this.saveMergeRequest(request);
                t.commit("Save initial conflicting files");
            }
        }
    }

    public MergeRequestInfo createMergeRequest(AuthCtx authCtx, MergeRequest request, DKUtils.SmartLogTailBuilder logTailBuilder) throws CodedException, IOException, GitAPIException {
        MergeRequestInfo mergeRequestInfo;
        if (request.branchToMerge.startsWith("refs/projects/" + request.projectToMerge)) {
            request.branchToMerge = request.branchToMerge.replace("refs/projects/" + request.projectToMerge + "/", "");
            request.branchToMergeType = MergeRequest.BranchToMergeType.PROJECT;
        } else if (request.branchToMerge.startsWith("origin/")) {
            request.branchToMerge = request.branchToMerge.replace("origin/", "");
            request.branchToMergeType = MergeRequest.BranchToMergeType.REMOTE;
        } else {
            request.branchToMergeType = request.baseProject.equals(request.projectToMerge) ? MergeRequest.BranchToMergeType.LOCAL : MergeRequest.BranchToMergeType.PROJECT;
        }
        try (RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(authCtx);){
            this.createMergeRequestEntry(authCtx, this.getMergeRequestsRefFile(request.baseProject), request);
            t.commit("Create merge request '" + request.id + "'");
        }
        try {
            this.initMergeRequestWorkingDirectory(authCtx, request, logTailBuilder);
        }
        catch (CodedException e) {
            logger.error((Object)"Could not create merge request working directory", (Throwable)e);
            this.deleteMergeRequest(authCtx, request.baseProject, request);
            throw e;
        }
        catch (Exception e) {
            logger.error((Object)"Could not create merge request working directory", (Throwable)e);
            this.deleteMergeRequest(authCtx, request.baseProject, request);
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not create merge request. Could not create working directory", (Throwable)e);
        }
        t = this.transactionService.beginWriteAsLoggedInUser(authCtx);
        try {
            this.saveMergeRequest(request);
            t.commit("Save merge request events");
        }
        finally {
            if (t != null) {
                t.close();
            }
        }
        try (Transaction t = this.transactionService.beginRead();){
            mergeRequestInfo = this.getMergeRequestInfo(request);
        }
        if (mergeRequestInfo.log.logEntries.isEmpty() && !mergeRequestInfo.status.hasUncommittedChanges) {
            logger.error((Object)"No commit and has no uncommited changes");
            this.deleteMergeRequest(authCtx, request.baseProject, request);
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_CREATE_MERGE_REQUEST_NO_CHANGES, "No commit and has no uncommited changes");
        }
        this.saveInitialConflictingFilesIfAny(authCtx, request, mergeRequestInfo.status.conflicting);
        logger.info((Object)"Merge request created successfully");
        logger.info((Object)("Merge request info: " + String.valueOf(mergeRequestInfo)));
        return mergeRequestInfo;
    }

    private void initMergeRequestWorkingDirectory(AuthCtx authCtx, MergeRequest request, DKUtils.SmartLogTailBuilder logTailBuilder) throws IOException, CodedException, UnauthorizedException {
        File baseProjectFolder;
        logger.info((Object)"Initialise merge request working directory for merge request");
        File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(request.baseProject, request.id);
        File branchToMergeProjectFolder = baseProjectFolder = DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"config", "projects", request.baseProject});
        if (request.projectToMerge != null) {
            branchToMergeProjectFolder = DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"config", "projects", request.projectToMerge});
        }
        logger.info((Object)("Create merge request working git directory path '" + mergeRequestWorkingGitDirectory.getPath() + "'"));
        boolean mkdirs = mergeRequestWorkingGitDirectory.mkdirs();
        if (!mkdirs) {
            logger.error((Object)("Couldn't create the merge request working git directory path '" + mergeRequestWorkingGitDirectory.getPath() + "'"));
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not create working directory '" + mergeRequestWorkingGitDirectory.getPath() + "'");
        }
        logger.info((Object)"Do a git init of the directory");
        try {
            Git.init().setBare(false).setDirectory(mergeRequestWorkingGitDirectory).setInitialBranch("master").call();
            Repository repository = ((FileRepositoryBuilder)new FileRepositoryBuilder().setGitDir(new File(mergeRequestWorkingGitDirectory, ".git"))).build();
            FileBasedConfig config = (FileBasedConfig)repository.getConfig();
            config.setString("user", null, "email", authCtx.getUserEmail());
            config.setString("user", null, "name", authCtx.getIdentifier());
            config.save();
        }
        catch (Exception e) {
            logger.error((Object)("Could not git init the directory '" + String.valueOf(mergeRequestWorkingGitDirectory) + "'"), (Throwable)e);
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not init the working directory '" + mergeRequestWorkingGitDirectory.getPath() + "'");
        }
        GitRemoteCommands gitRemoteCommands = new GitRemoteCommands(mergeRequestWorkingGitDirectory);
        logger.info((Object)("Prepare the merge request working git directory by assigning as origin the base project '" + baseProjectFolder.getAbsolutePath() + "'"));
        gitRemoteCommands.addRemoteRepository(authCtx, "origin", baseProjectFolder.getAbsolutePath(), true);
        switch (request.branchToMergeType) {
            case PROJECT: {
                logger.info((Object)("Setting up as fork the duplicated project to merge '" + request.projectToMerge + "'"));
                logger.info((Object)("Configure the 'fork' remote to '" + branchToMergeProjectFolder.getAbsolutePath() + "'"));
                gitRemoteCommands.addRemoteRepository(authCtx, "fork", branchToMergeProjectFolder.getAbsolutePath(), true);
                break;
            }
            case REMOTE: {
                GitRemoteCommands gitRemoteCommandsForBaseBranchRepo = new GitRemoteCommands(baseProjectFolder);
                GitRemoteCommands.GitAuthCtxWithRemoteName ctx = new GitRemoteCommands.GitAuthCtxWithRemoteName(authCtx, "origin");
                try {
                    String remoteUrl = gitRemoteCommandsForBaseBranchRepo.getRemoteURLAndCheckIfItMatchesWhitelist(ctx);
                    logger.info((Object)("Configure the 'fork' remote to '" + UrlRedactionUtils.sanitizeHttpUrls((String)remoteUrl) + "'"));
                    gitRemoteCommands.addRemoteRepository(authCtx, "fork", remoteUrl, true);
                    break;
                }
                catch (InterruptedException e) {
                    logger.error((Object)("Could not init the working directory '" + mergeRequestWorkingGitDirectory.getPath() + "'"), (Throwable)e);
                    Thread.currentThread().interrupt();
                    throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not init the working directory '" + mergeRequestWorkingGitDirectory.getPath() + "'", (Throwable)e);
                }
            }
            default: {
                logger.info((Object)"Setting up as fork the same as origin, as we are actually merging a branch from the same git repo");
                logger.info((Object)("Configure the 'fork' remote to '" + baseProjectFolder.getAbsolutePath() + "'"));
                gitRemoteCommands.addRemoteRepository(authCtx, "fork", baseProjectFolder.getAbsolutePath(), true);
            }
        }
        logger.info((Object)("Pull base branch '" + request.baseBranch + "' into the working git directory"));
        GitRemoteCommands.GitCommandResult resultGitPull = gitRemoteCommands.pullRebase(authCtx, "origin", request.baseBranch, null, logTailBuilder, true);
        if (!resultGitPull.commandSucceeded) {
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not pull branch '" + request.baseBranch + " in working directory '" + mergeRequestWorkingGitDirectory.getPath());
        }
        logger.info((Object)("Merge the branch to merge '" + request.branchToMerge + "' into in the pulled base branch '" + request.baseBranch + "'"));
        GitRemoteCommands.GitCommandResult resultGitMerge = gitRemoteCommands.merge(authCtx, "fork", request.branchToMerge, logTailBuilder);
        try {
            MergeRequestInfo mergeRequestInfo = this.getMergeRequestInfo(request);
            if (mergeRequestInfo.diff != null) {
                try (Transaction t = this.getTransactionProvider(mergeRequestInfo.mergeRequest).beginRead();){
                    Set<String> invalidFiles = this.projectsJGitService.getInvalidFiles(t, new RelFile(new String[0]), mergeRequestInfo.diff);
                    invalidFiles.removeAll(mergeRequestInfo.status.conflicting);
                    if (!invalidFiles.isEmpty()) {
                        logger.warn((Object)("Invalid json files from branch to merge found: " + String.valueOf(invalidFiles)));
                        throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INVALID_FILES, "The branch to merge contains invalid json files: " + String.valueOf(invalidFiles));
                    }
                }
            }
            if (!resultGitMerge.commandSucceeded) {
                logger.info((Object)("The merge of fork/" + request.branchToMerge + " failed but that could simply means there is some conflicts"));
                logger.info((Object)"Checking if there is a conflict");
                if (mergeRequestInfo.diff == null) {
                    logger.warn((Object)"There is no diff, likely to be due to no common commit between the two branches");
                    throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_UNCOMMON_HISTORY, "Could not merge branch '" + request.branchToMerge + " into  branch '" + request.baseBranch + "' in working directory '" + mergeRequestWorkingGitDirectory.getPath() + " likely due to no common history");
                }
                if (mergeRequestInfo.status.conflicting.isEmpty()) {
                    logger.warn((Object)"There is no conflicts, the merge failed for another reason.");
                    throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not merge branch '" + request.branchToMerge + " into  branch '" + request.baseBranch + "' in working directory '" + mergeRequestWorkingGitDirectory.getPath());
                }
                logger.info((Object)("Found merge conflicts: [" + mergeRequestInfo.status.conflicting.stream().map(f -> "'" + f + "'").collect(Collectors.joining(",")) + "]"));
            }
        }
        catch (GitAPIException e) {
            logger.error((Object)("the merge of fork/" + request.branchToMerge + " failed and we could not get the git request info."), (Throwable)e);
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_INIT_MERGE_REQUEST_DIRECTORY, "Could not merge branch '" + request.branchToMerge + " into  branch '" + request.baseBranch + "' in working directory '" + mergeRequestWorkingGitDirectory.getPath());
        }
    }

    public MergeRequestInfo updateMergeRequest(AuthCtx authCtx, String projectKey, MergeRequest requestInput) throws CodedException, IOException, GitAPIException {
        MergeRequest request = this.getMergeRequest(projectKey, requestInput.id);
        logger.info((Object)("Update Merge request with new input: " + String.valueOf(requestInput)));
        try (RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(authCtx);){
            MergeRequest mergeRequest = this.getMergeRequests(projectKey).get(requestInput.id);
            StringBuilder eventMessage = new StringBuilder("changed :\n");
            boolean hasChanges = false;
            if (!mergeRequest.title.equals(requestInput.title)) {
                eventMessage.append("- title from '" + mergeRequest.title + "' to '" + requestInput.title + "'\n");
                mergeRequest.title = requestInput.title;
                hasChanges = true;
            }
            if (!mergeRequest.description.equals(requestInput.description)) {
                eventMessage.append("- description from '" + mergeRequest.description + "' to '" + requestInput.description + "'\n");
                mergeRequest.description = requestInput.description;
                hasChanges = true;
            }
            if (hasChanges) {
                mergeRequest.events.add(new MergeRequestEvent(authCtx, MergeRequestEvent.Type.UPDATE, eventMessage.toString()));
                this.saveMergeRequest(mergeRequest);
            }
            t.commit("Update merge request '" + request.id + "'");
        }
        MergeRequest updatedMergeRequest = this.getMergeRequest(projectKey, requestInput.id);
        return this.getMergeRequestInfo(updatedMergeRequest);
    }

    public void markFileAsResolved(AuthCtx authCtx, String projectKey, MergeRequest request, String conflictFile, MergeRequestGitController.ResolutionStrategy resolutionStrategy) throws IOException, GitAPIException, CodedException {
        File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(projectKey, request.id);
        try (RWTransaction t = this.getTransactionProvider(request).beginWrite(authCtx);){
            this.markFileAsResolved(authCtx, request, mergeRequestWorkingGitDirectory, conflictFile, resolutionStrategy);
            t.commit("Mark file '" + conflictFile + "' as resolved");
        }
        t = this.transactionService.beginWriteAsLoggedInUser(authCtx);
        try {
            this.saveMergeRequest(request);
            t.commit("Save merge request events");
        }
        finally {
            if (t != null) {
                t.close();
            }
        }
    }

    public void markAllFilesAsResolved(AuthCtx authCtx, String projectKey, MergeRequest request, MergeRequestGitController.ResolutionStrategy resolutionStrategy) throws IOException, GitAPIException, CodedException {
        File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(projectKey, request.id);
        MergeRequestInfo mergeRequestInfo = this.getMergeRequestInfo(request);
        if (!mergeRequestInfo.status.conflicting.isEmpty()) {
            try (RWTransaction t = this.getTransactionProvider(request).beginWrite(authCtx);){
                for (String conflictFile : mergeRequestInfo.status.conflicting) {
                    this.markFileAsResolved(authCtx, request, mergeRequestWorkingGitDirectory, conflictFile, resolutionStrategy);
                }
                t.commit(String.format("Mark %d files as resolved", mergeRequestInfo.status.conflicting.size()));
            }
            t = this.transactionService.beginWriteAsLoggedInUser(authCtx);
            try {
                this.saveMergeRequest(request);
                t.commit("Save merge request events");
            }
            finally {
                if (t != null) {
                    t.close();
                }
            }
        }
    }

    public Map<Integer, MergeRequest> getMergeRequests(String projectKey) throws IOException, CodedException {
        try (Transaction t = this.transactionService.retrieveOrBeginRead();){
            Map<Integer, MergeRequest> mergeRequests = this.getMergeRequestsFile((RelFile)this.getMergeRequestsRefFile((String)projectKey)).mergeRequests;
            for (MergeRequest mergeRequest : mergeRequests.values()) {
                mergeRequest.baseProject = projectKey;
            }
            Map<Integer, MergeRequest> map = mergeRequests;
            return map;
        }
    }

    public MergeRequest getMergeRequest(String projectKey, int requestId) throws IOException, CodedException {
        try (Transaction t = this.transactionService.retrieveOrBeginRead();){
            MergeRequest mergeRequest = this.getMergeRequests(projectKey).get(requestId);
            return mergeRequest;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void merge(AuthCtx authCtx, String projectKey, MergeRequest mergeRequest, DKUtils.SmartLogTailBuilder logTailBuilder) throws CodedException, IOException, InterruptedException, GitAPIException, UnauthorizedException, LimitsStatusComputer.LicenseLimitException {
        ProjectsJGitService.ProjectSnapshot snapshot;
        String baseBranchCommitBeforeMerging;
        GitRemoteCommands gitRemoteCommands;
        block18: {
            logger.info((Object)"Merging the merge request");
            File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(projectKey, mergeRequest.id);
            File baseProjectFolder = DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"config", "projects", projectKey});
            MergeRequestInfo mergeRequestInfo = this.getMergeRequestInfo(mergeRequest);
            if (!mergeRequestInfo.status.conflicting.isEmpty()) {
                logger.info((Object)("Merge still paused due to remaining conflict file: '" + String.join((CharSequence)", ", mergeRequestInfo.status.conflicting) + " '"));
                throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_MERGING_STILL_CONFLICTS_UNRESOLVED, "Still some conflicts file to be resolved");
            }
            logger.info((Object)"No remaining conflict files");
            if (mergeRequestInfo.status.hasUncommittedChanges) {
                logger.info((Object)"Completing the merge with a commit");
                this.mergeConflictResolved(authCtx, mergeRequestWorkingGitDirectory, mergeRequest);
            } else {
                logger.info((Object)"Completing the merge with a merge continue");
                this.mergeConflictContinue(authCtx, mergeRequestWorkingGitDirectory, mergeRequest);
            }
            logger.info((Object)("Merging latest changed from base branch '" + mergeRequest.baseBranch + "'"));
            gitRemoteCommands = new GitRemoteCommands(mergeRequestWorkingGitDirectory);
            gitRemoteCommands.fetch(authCtx, "origin", mergeRequest.baseBranch);
            GitRemoteCommands.GitCommandResult mergeOrigin = gitRemoteCommands.merge(authCtx, "origin", mergeRequest.baseBranch, logTailBuilder);
            if (!mergeOrigin.commandSucceeded) {
                mergeRequestInfo = this.getMergeRequestInfo(mergeRequest);
                this.saveInitialConflictingFilesIfAny(authCtx, mergeRequest, mergeRequestInfo.status.conflicting);
                logger.info((Object)"Base branch has drift and new conflicts needs to be resolved.");
                throw new CodedException((InfoMessage.MessageCode)GitCodes.WARNING_GIT_MERGE_MERGING_BASE_CONFLICTS, "Could not merge branch '" + mergeRequest.baseBranch + " in working directory '" + mergeRequestWorkingGitDirectory.getPath());
            }
            baseBranchCommitBeforeMerging = gitRemoteCommands.getHashFor("origin/" + mergeRequest.baseBranch);
            snapshot = this.projectsJGitService.takeProjectSnapshot(mergeRequest.baseProject);
            GitRemoteCommands gitRemoteCommandsForBaseBranchRepo = new GitRemoteCommands(baseProjectFolder);
            if (gitRemoteCommandsForBaseBranchRepo.getCurrentBranch().equals(mergeRequest.baseBranch)) {
                String mergeRequestFinalBranch = "merge-requests/" + mergeRequest.id;
                logger.info((Object)("Push the merge request changes into a dedicated branch '" + mergeRequestFinalBranch + "' back to origin"));
                GitRemoteCommands.GitCommandResult pushToOrigin = gitRemoteCommands.pushWithLogTail(authCtx, "origin", "master:" + mergeRequestFinalBranch, false, logTailBuilder);
                if (!pushToOrigin.commandSucceeded) {
                    logger.error((Object)("Couldn't push branch '" + mergeRequestFinalBranch + "' into origin"));
                    throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_PUSH_TO_ORIGIN_FAILED, "Could not push branch 'master:" + mergeRequestFinalBranch + " in working directory '" + mergeRequestWorkingGitDirectory.getPath());
                }
                try {
                    logger.info((Object)("Merging '" + mergeRequestFinalBranch + "' into the current branch, which is the base branch at this point."));
                    GitRemoteCommands.GitCommandResult finalMergeIntoBaseBranch = gitRemoteCommandsForBaseBranchRepo.merge(authCtx, null, mergeRequestFinalBranch, logTailBuilder);
                    if (!finalMergeIntoBaseBranch.commandSucceeded) {
                        logger.error((Object)("Couldn't merge the branch '" + mergeRequestFinalBranch + "' into base branch '" + mergeRequest.baseBranch + "'"));
                        throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_MERGE_BRANCH_TO_BASE, "Could not merge branch '" + mergeRequestFinalBranch + " in '" + baseProjectFolder.getPath());
                    }
                    break block18;
                }
                finally {
                    logger.info((Object)("Deleting the intermediate branch '" + mergeRequestFinalBranch + "'."));
                    GitRemoteCommands.GitCommandResult deleteBranchResult = gitRemoteCommandsForBaseBranchRepo.deleteBranch(mergeRequestFinalBranch, logTailBuilder);
                    if (!deleteBranchResult.commandSucceeded) {
                        logger.warn((Object)("Deleting the intermediate branch '" + mergeRequestFinalBranch + "' failed."));
                    }
                }
            }
            logger.info((Object)("Push the merge request changes into base branch '" + mergeRequest.baseBranch + "' back to origin"));
            GitRemoteCommands.GitCommandResult pushToOrigin = gitRemoteCommands.pushWithLogTail(authCtx, "origin", mergeRequest.baseBranch, false, logTailBuilder);
            if (!pushToOrigin.commandSucceeded) {
                logger.error((Object)("Couldn't push to branch '" + mergeRequest.baseBranch + "' directly"));
                throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_PUSH_TO_ORIGIN_FAILED, "Could not push branch '" + mergeRequest.baseBranch + " in working directory '" + mergeRequestWorkingGitDirectory.getPath());
            }
        }
        this.projectsJGitService.invalidateProjectCaches(mergeRequest.baseProject);
        TaggableObjectChangedEvent event = this.projectsJGitService.performPostGlobalModificationActions(authCtx, snapshot, TaggableObjectChangedEvent.ActionType.PROJECT_GIT_MERGE, mergeRequest.baseBranch, projectKey);
        event.details.addProperty("request_id", (Number)mergeRequest.id);
        event.details.addProperty("request_title", mergeRequest.title);
        event.details.addProperty("base_branch", mergeRequest.baseBranch);
        event.details.addProperty("project_to_merge", mergeRequest.projectToMerge);
        event.details.addProperty("branch_to_merge", mergeRequest.branchToMerge);
        event.details.addProperty("branch_to_merge_type", mergeRequest.branchToMergeType.name());
        this.pubSubService.publish(event);
        mergeRequest.mergeCommitHash = gitRemoteCommands.getHashFor("master");
        mergeRequest.status = MergeRequest.Status.MERGED;
        mergeRequest.baseBranchCommitBeforeMerging = baseBranchCommitBeforeMerging;
        logger.info((Object)("Mark the merge request '" + mergeRequest.id + "' as merged"));
        mergeRequest.events.add(new MergeRequestEvent(authCtx, MergeRequestEvent.Type.MERGE));
        try (RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(authCtx);){
            this.saveMergeRequest(mergeRequest);
            t.commit("Merge merge request '" + mergeRequest.id + "'");
        }
        logger.info((Object)"Merge request merged successfully");
    }

    public void refreshMergedBranch(AuthCtx authCtx, String projectKey, MergeRequest mergeRequest, DKUtils.SmartLogTailBuilder logTailBuilder) throws CodedException, IOException, InterruptedException, GitAPIException, UnauthorizedException, LimitsStatusComputer.LicenseLimitException {
        logger.infoV("Refreshing branch to merge %s.%s", new Object[]{projectKey, mergeRequest.branchToMerge});
        if (mergeRequest.status != MergeRequest.Status.MERGED) {
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_REFRESH_MERGED_BRANCH_PUSHED_FAILED, "The branch to merge must be merged and not deleted to be refreshed");
        }
        File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(projectKey, mergeRequest.id);
        File baseProjectFolder = DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"config", "projects", projectKey});
        MergeRequestInfo mergeRequestInfo = this.getMergeRequestInfo(mergeRequest);
        GitRemoteCommands gitRemoteCommands = new GitRemoteCommands(mergeRequestWorkingGitDirectory);
        if (mergeRequestInfo.allowUpdatingBranchToMerge) {
            try {
                File branchToMergeRepo = this.getBranchToMergeRepo(mergeRequest);
                mergeRequest.refreshedBranchAfterMerge = true;
                mergeRequest.branchToMergeCommitBeforeMerging = gitRemoteCommands.getHashFor("fork/" + mergeRequest.branchToMerge);
                this.refreshBranch(authCtx, mergeRequest, logTailBuilder, branchToMergeRepo, gitRemoteCommands, mergeRequestWorkingGitDirectory, baseProjectFolder);
            }
            catch (Exception e) {
                logger.warn((Object)("Couldn't refresh the branch to merge, which is not a blocker for the duplicated project branch: " + e.getMessage()));
            }
            try (RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(authCtx);){
                this.saveMergeRequest(mergeRequest);
                t.commitV("Refresh branch to merge %s.%s from merge request %d", new Object[]{mergeRequest.projectToMerge, mergeRequest.branchToMerge, mergeRequest.id});
            }
        }
    }

    private void refreshBranch(AuthCtx authCtx, MergeRequest mergeRequest, DKUtils.SmartLogTailBuilder logTailBuilder, File branchToMergeRepo, GitRemoteCommands gitRemoteCommands, File mergeRequestWorkingGitDirectory, File baseProjectFolder) throws CodedException {
        GitRemoteCommands gitRemoteCommandsForMergedBranch = new GitRemoteCommands(branchToMergeRepo);
        if (branchToMergeRepo != null && gitRemoteCommandsForMergedBranch.getCurrentBranch().equals(mergeRequest.branchToMerge)) {
            String mergeRequestFinalBranch = "merge-requests/" + mergeRequest.id;
            logger.debug((Object)"Using a dedicated branch is necessary as the origin is not in read only and pushing on a potential active branch can be prevented and complexity the flow.");
            logger.info((Object)("Push the merge request changes into a dedicated branch '" + mergeRequestFinalBranch + "' back to fork"));
            GitRemoteCommands.GitCommandResult pushToFork = gitRemoteCommands.pushWithLogTail(authCtx, "fork", "master:" + mergeRequestFinalBranch, false, logTailBuilder);
            if (!pushToFork.commandSucceeded) {
                logger.error((Object)("Could not push branch 'master:" + mergeRequestFinalBranch + " in working directory '" + mergeRequestWorkingGitDirectory.getPath()));
                throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_REFRESH_MERGED_BRANCH_PUSHED_FAILED, "Could not push branch 'master:" + mergeRequestFinalBranch + " in working directory '" + mergeRequestWorkingGitDirectory.getPath());
            }
            if (gitRemoteCommandsForMergedBranch.getHashFor("HEAD").equals(gitRemoteCommandsForMergedBranch.getHashFor(mergeRequestFinalBranch))) {
                logger.info((Object)"Branch already up to date as the previous merge was a fast forward.");
            } else {
                logger.info((Object)("Merging '" + mergeRequestFinalBranch + "' into the current branch, which is the base branch at this point."));
                GitRemoteCommands.GitCommandResult finalMergeIntoBaseBranch = gitRemoteCommandsForMergedBranch.merge(authCtx, null, mergeRequestFinalBranch, logTailBuilder);
                if (!finalMergeIntoBaseBranch.commandSucceeded) {
                    logger.error((Object)("Could not merge branch '" + mergeRequestFinalBranch + " in '" + baseProjectFolder.getPath()));
                    throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_REFRESH_MERGED_BRANCH_PUSHED_FAILED, "Could not merge branch '" + mergeRequestFinalBranch + " in '" + baseProjectFolder.getPath());
                }
            }
            logger.info((Object)("Deleting the intermediate branch '" + mergeRequestFinalBranch + "'."));
            GitRemoteCommands.GitCommandResult deleteBranchResult = gitRemoteCommandsForMergedBranch.deleteBranch(mergeRequestFinalBranch, logTailBuilder);
            if (!deleteBranchResult.commandSucceeded) {
                logger.error((Object)("Could not delete intermediate branch '" + mergeRequestFinalBranch + " in '" + baseProjectFolder.getPath()));
                throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_REFRESH_MERGED_BRANCH_PUSHED_FAILED, "Could not delete intermediate branch '" + mergeRequestFinalBranch + " in '" + baseProjectFolder.getPath());
            }
        } else {
            logger.info((Object)("Push the merge request changes into base branch '" + mergeRequest.baseBranch + "' back to fork"));
            GitRemoteCommands.GitCommandResult pushToFork = gitRemoteCommands.pushWithLogTail(authCtx, "fork", mergeRequest.baseBranch + ":" + mergeRequest.branchToMerge, false, logTailBuilder);
            if (!pushToFork.commandSucceeded) {
                logger.error((Object)("Could not push branch '" + mergeRequest.baseBranch + ":" + mergeRequest.branchToMerge + " in working directory '" + mergeRequestWorkingGitDirectory.getPath()));
                throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_REFRESH_MERGED_BRANCH_PUSHED_FAILED, "Could not push branch '" + mergeRequest.baseBranch + ":" + mergeRequest.branchToMerge + " in working directory '" + mergeRequestWorkingGitDirectory.getPath());
            }
        }
        this.projectsJGitService.invalidateProjectCaches(mergeRequest.projectToMerge);
    }

    public MergeRequestInfo getMergeRequestInfo(MergeRequest mergeRequest) throws IOException, GitAPIException {
        GitInfo gitInfoBaseProject;
        try (Transaction t = this.transactionService.retrieveOrBeginRead();){
            gitInfoBaseProject = this.projectsJGitService.getGitInfo(mergeRequest.baseProject);
        }
        MergeRequestInfo mergeRequestInfo = new MergeRequestInfo();
        mergeRequestInfo.mergeRequest = mergeRequest;
        mergeRequestInfo.allowUpdatingBranchToMerge = !mergeRequest.branchToMerge.equals("master") && !mergeRequest.branchToMerge.equals("main");
        mergeRequestInfo.allowUpdatingBranchToMerge = mergeRequestInfo.allowUpdatingBranchToMerge && (gitInfoBaseProject == null || gitInfoBaseProject.originProjectKey != null && !gitInfoBaseProject.originProjectKey.equals(mergeRequest.projectToMerge));
        File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(mergeRequest);
        if (!mergeRequestWorkingGitDirectory.exists()) {
            logger.warn((Object)"Could not read merge request info as the working git directory is undefined");
            return mergeRequestInfo;
        }
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(mergeRequestWorkingGitDirectory);){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            mergeRequestInfo.log = gitLocalCommands.getLog("", "origin/" + mergeRequest.baseBranch, "fork/" + mergeRequest.branchToMerge);
            mergeRequestInfo.diff = gitLocalCommands.diff("origin/" + mergeRequest.baseBranch, null, false);
            mergeRequestInfo.status = gitLocalCommands.getStatus();
            try {
                mergeRequestInfo.nbConflictsPerFile = this.computeNbConflictPerFile(mergeRequestInfo);
            }
            catch (Exception e) {
                logger.warn((Object)"Couldn't compute the nb of merge conflicts per file. Skipping it as it's just a nice to have information", (Throwable)e);
            }
        }
        return mergeRequestInfo;
    }

    private void deleteMergeRequest(AuthCtx authCtx, String projectKey, MergeRequest request) throws IOException, CodedException {
        logger.info((Object)("Deleting merge request: " + String.valueOf(request)));
        try (RWTransaction t = this.transactionService.beginWriteAsLoggedInUser(authCtx);){
            RelFile mergeRequestsRefsFile = this.getMergeRequestsRefFile(projectKey);
            RWTransactionRef tr = TransactionContext.retrieveWrite();
            MergeRequestsFile mergeRequestsFile = this.getMergeRequestsFile(mergeRequestsRefsFile);
            mergeRequestsFile.mergeRequests.remove(request.id);
            tr.writeObject(mergeRequestsRefsFile, (Object)mergeRequestsFile);
            DKUFileUtils.deleteDirectory((File)this.getMergeRequestWorkingGitDirectory(request));
            t.commit("Delete merge request '" + String.valueOf(request) + "'");
        }
        logger.info((Object)"Merge request deleted");
    }

    public void deleteMergeRequest(AuthCtx authCtx, String projectKey, int requestId) throws IOException, CodedException {
        this.deleteMergeRequest(authCtx, projectKey, this.getMergeRequest(projectKey, requestId));
    }

    private File getBranchToMergeRepo(MergeRequest request) throws IOException {
        switch (request.branchToMergeType) {
            case PROJECT: {
                return DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"config", "projects", request.projectToMerge});
            }
            case LOCAL: {
                return DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"config", "projects", request.baseProject});
            }
        }
        throw new IllegalStateException();
    }

    private File getMergeRequestWorkingGitDirectory(String projectKey, int mergeRequestId) {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)projectKey), (Object)"Project key is not specified");
        Preconditions.checkArgument((mergeRequestId > 0 ? 1 : 0) != 0, (Object)"Merge id is not specified");
        return DKUFileUtils.getWithin((File)ApplicationConfigurator.getBaseFolderF(), (String[])new String[]{"merge_requests", "projects", projectKey, "" + mergeRequestId});
    }

    private File getMergeRequestWorkingGitDirectory(MergeRequest mergeRequest) {
        return this.getMergeRequestWorkingGitDirectory(mergeRequest.baseProject, mergeRequest.id);
    }

    private RelFile getMergeRequestsRefFile(String projectKey) {
        Preconditions.checkArgument((boolean)StringUtils.isNotBlank((String)projectKey), (Object)"Project key is not specified");
        return new RelFile(new String[]{"projects", projectKey, ".dss-meta", MERGE_REQUESTS_FILE});
    }

    public FolderEditorService.FolderContent getFile(MergeRequest mergeRequest, String filePath) throws Exception {
        try (Transaction t = this.getTransactionProvider(mergeRequest).beginRead();){
            RelFile baseFolder = new RelFile(new String[0]);
            FolderEditorService.FolderContent folderContent = this.folderEditorService.getFolderContent(baseFolder, filePath, false, "utf8", t);
            return folderContent;
        }
    }

    public void saveFile(AuthCtx authCtx, MergeRequest mergeRequest, String filePath, String fileContent) throws Exception {
        Transaction t;
        RelFile baseFolder;
        block15: {
            baseFolder = new RelFile(new String[0]);
            t = this.getTransactionProvider(mergeRequest).beginRead();
            try {
                FolderEditorService.FolderContent file = this.getFile(mergeRequest, filePath);
                if (!file.mimeType.endsWith("json")) break block15;
                try {
                    Object parsedContent = JSON.gson().fromJson(fileContent, Object.class);
                    if (!(parsedContent instanceof Map) && !(parsedContent instanceof List)) {
                        throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_ADD_FILE_INVALID_FORMAT, "Invalid JSON format for file '" + filePath + "'. Expected JSON Object or Array.");
                    }
                }
                catch (JsonSyntaxException e) {
                    logger.error((Object)("Invalid JSON format for file '" + filePath + "'"), (Throwable)e);
                    throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_ADD_FILE_INVALID_FORMAT, "Invalid JSON format for file '" + filePath + "'", (Throwable)e);
                }
            }
            finally {
                if (t != null) {
                    t.close();
                }
            }
        }
        t = this.getTransactionProvider(mergeRequest).beginWrite(authCtx);
        try {
            this.folderEditorService.setOrAddFolderContent(baseFolder, filePath, fileContent, (RWTransaction)t);
            t.commitV("Merge request edited file '%s'", new Object[]{filePath});
        }
        finally {
            if (t != null) {
                t.close();
            }
        }
    }

    public GitModel.SingleCommitObjectDiff getDiff(String projectKey, int requestId, String commitId) throws CodedException, IOException, GitAPIException {
        MergeRequest mergeRequest = this.getMergeRequest(projectKey, requestId);
        return this.getDiff(mergeRequest, commitId);
    }

    private GitModel.SingleCommitObjectDiff getDiff(MergeRequest mergeRequest, String commitId) throws IOException, GitAPIException {
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(this.getMergeRequestWorkingGitDirectory(mergeRequest));){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            GitModel.SingleCommitObjectDiff singleCommitObjectDiff = gitLocalCommands.singleCommitDiff(commitId);
            return singleCommitObjectDiff;
        }
    }

    public GitModel.MultiCommitDiff getDiff(String projectKey, int requestId, String commitFrom, String commitTo) throws CodedException, IOException, GitAPIException {
        MergeRequest mergeRequest = this.getMergeRequest(projectKey, requestId);
        return this.getDiff(mergeRequest, commitFrom, commitTo);
    }

    private GitModel.MultiCommitDiff getDiff(MergeRequest mergeRequest, String commitFrom, String commitTo) throws IOException, GitAPIException {
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(this.getMergeRequestWorkingGitDirectory(mergeRequest));){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            GitModel.MultiCommitDiff multiCommitDiff = gitLocalCommands.diff(commitFrom, commitTo, null, false);
            return multiCommitDiff;
        }
    }

    private void createMergeRequestEntry(AuthCtx authCtx, RelFile mergeRequestsRelFile, MergeRequest request) throws IOException, CodedException {
        RWTransactionRef tr = TransactionContext.retrieveWrite();
        MergeRequestsFile mergeRequestsFile = this.getMergeRequestsFile(mergeRequestsRelFile);
        request.id = mergeRequestsFile.mergeRequestIdCounter++;
        mergeRequestsFile.mergeRequests.put(request.id, request);
        logger.info((Object)("Creating merge request in file: " + mergeRequestsRelFile.getFullPath() + " with details: " + String.valueOf(request)));
        request.events.add(new MergeRequestEvent(authCtx, MergeRequestEvent.Type.CREATED));
        tr.writeObject(mergeRequestsRelFile, (Object)mergeRequestsFile);
    }

    private void saveMergeRequest(MergeRequest request) throws IOException, CodedException {
        RelFile mergeRequestsRelFile = this.getMergeRequestsRefFile(request.baseProject);
        RWTransactionRef tr = TransactionContext.retrieveWrite();
        MergeRequestsFile mergeRequestsFile = this.getMergeRequestsFile(mergeRequestsRelFile);
        mergeRequestsFile.mergeRequests.put(request.id, request);
        logger.info((Object)("Saving merge request in file: " + mergeRequestsRelFile.getFullPath() + " with details: " + String.valueOf(request)));
        tr.writeObject(mergeRequestsRelFile, (Object)mergeRequestsFile);
    }

    private void markFileAsResolved(AuthCtx authCtx, MergeRequest mergeRequest, File mergeRequestWorkingGitDirectory, String conflictFile, MergeRequestGitController.ResolutionStrategy resolutionStrategy) throws IOException, GitAPIException, CodedException {
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(mergeRequestWorkingGitDirectory);){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            logger.info((Object)("Marking conflict file as resolved: " + conflictFile));
            switch (resolutionStrategy) {
                case MANUAL: {
                    if (!"params.json".equals(conflictFile)) break;
                    try (Transaction t = this.getTransactionProvider(mergeRequest).retrieveOrBeginRead();){
                        t.readObjectUnsafe(conflictFile, SerializedProject.class);
                        break;
                    }
                    catch (Exception e) {
                        logger.error((Object)"Could not deserialize the params.json", (Throwable)e);
                        throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_GIT_MERGE_ADD_FILE_INVALID_PARAMS_FORMAT, "Could not deserialize the params.json");
                    }
                }
                case MASTER: {
                    gitLocalCommands.checkout("origin/" + mergeRequest.baseBranch, Arrays.asList(conflictFile));
                    break;
                }
                case BRANCH: {
                    gitLocalCommands.checkout("fork/" + mergeRequest.branchToMerge, Arrays.asList(conflictFile));
                }
            }
            gitLocalCommands.addFile(conflictFile);
            mergeRequest.events.add(new MergeRequestEvent(authCtx, MergeRequestEvent.Type.RESOLVED_CONFLICT, "Mark conflict file '" + conflictFile + "' as resolved"));
        }
        MergeRequestInfo mergeRequestInfo = this.getMergeRequestInfo(mergeRequest);
        if (mergeRequestInfo.status.conflicting.isEmpty()) {
            logger.info((Object)"No remaining conflicts");
        } else {
            logger.info((Object)("Files still in conflicts: [" + mergeRequestInfo.status.conflicting.stream().collect(Collectors.joining(",")) + "]"));
        }
    }

    private void mergeConflictResolved(AuthCtx authCtx, File mergeRequestWorkingGitDirectory, MergeRequest request) throws IOException, InterruptedException {
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(mergeRequestWorkingGitDirectory);){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            gitLocalCommands.commit(new GitModel.GitAuthor(authCtx), "Merge branch '" + request.branchToMerge + "' into '" + request.baseBranch + "'");
        }
    }

    private void mergeConflictContinue(AuthCtx authCtx, File mergeRequestWorkingGitDirectory, MergeRequest request) throws IOException, InterruptedException {
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(mergeRequestWorkingGitDirectory);){
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            RepositoryState repositoryState = gitLocalCommands.getRepositoryStatus();
            if (repositoryState == RepositoryState.MERGING_RESOLVED) {
                gitLocalCommands.mergeContinue(new GitModel.GitAuthor(authCtx), "Merge branch '" + request.branchToMerge + "' into '" + request.baseBranch + "'");
            }
        }
    }

    public MergeRequestInfo getMergeRequestInfo(String projectKey, int requestId) throws GitAPIException, IOException, CodedException {
        MergeRequest mr = this.getMergeRequest(projectKey, requestId);
        if (mr == null) {
            throw new NotFoundException("Merge request " + requestId + " not found");
        }
        return this.getMergeRequestInfo(mr);
    }

    private MergeRequestsFile getMergeRequestsFile(RelFile mergeRequestsRelFile) throws IOException, CodedException {
        TransactionRef tr = TransactionContext.retrieveRead();
        if (!tr.isFile(mergeRequestsRelFile)) {
            return new MergeRequestsFile();
        }
        try {
            return (MergeRequestsFile)tr.readObject(mergeRequestsRelFile, MergeRequestsFile.class);
        }
        catch (JsonSyntaxException ex) {
            logger.error((Object)"Error while parsing merge requests file. Make sure that the JSON file is properly formatted.", (Throwable)ex);
            throw new CodedException((InfoMessage.MessageCode)GitCodes.ERR_MALFORMED_MERGE_REQUESTS_JSON, "Error while parsing merge requests file. Make sure that the JSON file is properly formatted.", (Throwable)ex);
        }
    }

    public GitModel.MultiCommitDiff diffNameOnly(MergeRequest mergeRequest) throws IOException, GitAPIException {
        File mergeRequestWorkingGitDirectory = this.getMergeRequestWorkingGitDirectory(mergeRequest);
        if (!mergeRequestWorkingGitDirectory.exists()) {
            logger.warn((Object)"Could not read merge request info as the working git directory is undefined");
        }
        try (DSSSingleRepositoryGitManager gitManager = new DSSSingleRepositoryGitManager(mergeRequestWorkingGitDirectory);){
            GitModel.MultiCommitDiff diff;
            GitLocalCommands gitLocalCommands = new GitLocalCommands(gitManager.getGit());
            String commonCommit = gitLocalCommands.mergeBase("origin/" + mergeRequest.baseBranch, "fork/" + mergeRequest.branchToMerge);
            if (commonCommit != null) {
                diff = gitLocalCommands.diff(commonCommit, "fork/" + mergeRequest.branchToMerge, null, true);
            } else {
                diff = null;
                logger.warn((Object)("No common commit between origin/" + mergeRequest.baseBranch + " and fork/" + mergeRequest.branchToMerge));
            }
            GitModel.MultiCommitDiff multiCommitDiff = diff;
            return multiCommitDiff;
        }
    }

    public synchronized TransactionProvider getTransactionProvider(MergeRequest mergeRequest) throws IOException {
        File mergeRequestBasePath = this.getMergeRequestWorkingGitDirectory(mergeRequest.baseProject, mergeRequest.id);
        if (!mergeRequestBasePath.exists()) {
            throw new RuntimeException("Merge request folder does not exist: " + String.valueOf(mergeRequestBasePath));
        }
        return new TransactionProvider(mergeRequestBasePath, (TransactionProvider.TransactionProviderSettings)new DSSTransactionProviderSettings(false, true, false, true));
    }

    private Map<String, Integer> computeNbConflictPerFile(MergeRequestInfo mergeRequestInfo) throws Exception {
        HashMap<String, Integer> nbConflictPerFiles = new HashMap<String, Integer>();
        for (String fileInConflict : mergeRequestInfo.status.conflicting) {
            nbConflictPerFiles.put(fileInConflict, 0);
            Transaction t = this.getTransactionProvider(mergeRequestInfo.mergeRequest).beginRead();
            try {
                RelFile baseFolder = new RelFile(new String[0]);
                FolderEditorService.FolderContent folderContent = this.folderEditorService.getFolderContent(baseFolder, fileInConflict, false, "utf8", t);
                if (folderContent.hasData && folderContent.data != null) {
                    nbConflictPerFiles.put(fileInConflict, ProjectGitMergeRequestService.countOccurrences(folderContent.data, "<<<<<<<"));
                    continue;
                }
                nbConflictPerFiles.put(fileInConflict, -1);
            }
            finally {
                if (t == null) continue;
                t.close();
            }
        }
        return nbConflictPerFiles;
    }

    private static int countOccurrences(String data, String target) {
        Pattern pattern = Pattern.compile(target);
        Matcher matcher = pattern.matcher(data);
        int count = 0;
        while (matcher.find()) {
            ++count;
        }
        return count;
    }

    public static class MergeRequestsFile {
        public int mergeRequestIdCounter = 1;
        public Map<Integer, MergeRequest> mergeRequests = new HashMap<Integer, MergeRequest>();
    }
}

