(function() {
    "use strict";

    const explanationTabBody = {
        templateUrl: "static/dataiku/ai-explanations/explanation-tab-body/explanation-tab-body.component.html",
        bindings: {
            editorMode: "@",
        },
        controller: function(
            $scope,
            $element,
            $timeout,
            AIExplanationService,
            CodeBasedEditorUtils,
        ) {
            const ctrl = this;

            ctrl.$onInit = function() {
                ctrl.availableExplanationOptions = AIExplanationService.createAvailableOptions();
                ctrl.explanationOptions = AIExplanationService.loadOptions();
                ctrl.language = convertEditorModeToLanguage(ctrl.editorMode);
            };

            ctrl.toggleExplanationOptions = function($event) {
                ctrl.showExplanationOptions = !ctrl.showExplanationOptions;

                if (ctrl.showExplanationOptions) {
                    $timeout(function() {
                        $element.find(".ai-explanation-options-language button").focus();
                    });
                } else {
                    $event.target.blur();
                }
            };

            ctrl.getCode = function() {
                if (ctrl.currentCodeSelection) {
                    return ctrl.currentCodeSelection;
                } else {
                    const script = CodeBasedEditorUtils.getRecipeScript();
                    if (!script || typeof script.data !== "string") {
                        throw new Error("Unexpected recipe script");
                    }
                    return script.data;
                }
            };

            ctrl.explain = function() {
                resetErrorInScope($scope);
                AIExplanationService.explainCode(
                    ctrl.language,
                    ctrl.getCode(),
                    ctrl.explanationOptions,
                    function(initialResponse) {
                        ctrl.lastJobId = initialResponse.jobId;
                    }
                )
                .then(function(data) {
                    if (data.aborted) {
                        return;
                    }

                    ctrl.explanationResponse = data.result;
                    ctrl.explanationRequested = false;
                    ctrl.explanationCodeSelection = ctrl.requestedCodeSelection;
                    AIExplanationService.storeOptions(ctrl.explanationOptions);
                })
                .catch(function(r) {
                    setErrorInScope.bind($scope)(r.data, r.status, r.headers);
                    ctrl.explanationRequested = false;
                });
                ctrl.explanationRequested = true;
                ctrl.lastJobId = null;
                ctrl.requestedCodeSelection = ctrl.currentCodeSelection;
            };

            ctrl.abortExplain = function() {
                AIExplanationService.abortExplain(ctrl.lastJobId)
                .then(function() {
                    ctrl.explanationRequested = false;
                    ctrl.abortRequested = false;
                })
                .catch(function(r) {
                    setErrorInScope.bind($scope)(r.data, r.status, r.headers);
                    ctrl.abortRequested = false;
                });
                ctrl.abortRequested = true;
            };

            ctrl.gotOkResponse = function() {
                return ctrl.explanationResponse && ctrl.explanationResponse.ok && !ctrl.explanationRequested;
            };

            $scope.$on("codeRecipeEditorSelectionChanged", function(_, selection) {
                $scope.$evalAsync(function() {
                    ctrl.currentCodeSelection = selection;
                });
            });

            function convertEditorModeToLanguage(editorMode) {
                switch (editorMode) {
                    case "text/x-python":
                        return "python";
                    case "text/x-sql2":
                        return "sql";
                    default:
                        throw new Error("Unsupported editor mode");
                }
            }
        },
    };

    angular.module("dataiku.aiExplanations").component("explanationTabBody", explanationTabBody);
})();
