(function() {
'use strict';

/**
* Main flow page functionalities
*/
const app = angular.module('dataiku.flow.project', ['dataiku.flow.graph']);

const FLOW_NODE_STATUS = {
    RUNNING: "RUNNING",
    NOT_STARTED: "NOT_STARTED",
    DONE: "DONE",
}


app.directive('flowRightColumn', function(QuickView, TaggableObjectsUtils, FlowGraphSelection, FlowGraph) {
    return {
        scope: true,
        link: function(scope, element, attrs) {
            scope.QuickView = QuickView;

            scope.$watch("rightColumnItem", function() {
                scope.context = "FLOW";
                scope.selection = {
                    selectedObject: scope.rightColumnItem,
                    confirmedItem: scope.rightColumnItem
                };
            });

            scope.getSelectedNodes = function() {
                return scope.rightColumnSelection || [];
            };

            scope.getSelectedTaggableObjectRefs = function() {
                return scope.getSelectedNodes().map(TaggableObjectsUtils.fromNode);
            };

            scope.computeMovingImpact = function() {
                const computedImpact = [];
                const movingItems = FlowGraphSelection.getSelectedTaggableObjectRefs();

                    function addSuccessors(node, original) {
                        if (!['RECIPE', 'LABELING_TASK'].includes(node.nodeType)) return;
                        node.successors.forEach(function(successor) {
                            let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(successor));
                            if (original && successor == original.id
                                || movingItems.some(it => it.id === newTaggableObjectRef.id)
                                || computedImpact.some(it => it.id === newTaggableObjectRef.id)) return;
                            computedImpact.push(newTaggableObjectRef);
                        });
                    }
                    function computeImpact(node) {
                        let predecessor = node.predecessors[0];
                        if (predecessor && !['RECIPE', 'LABELING_TASK'].includes(node.nodeType)) {
                            let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(predecessor));
                            if (computedImpact.some(it => it.id === newTaggableObjectRef.id)) return;
                            if (!movingItems.some(it => it.id === newTaggableObjectRef.id)) {
                                computedImpact.push(newTaggableObjectRef);
                            }
                            addSuccessors(FlowGraph.node(predecessor), node);
                        }

                    addSuccessors(node);
                }

                FlowGraphSelection.getSelectedNodes().forEach(function(node) {
                    let realNode = node.usedByZones.length ? FlowGraph.node(`zone__${node.ownerZone}__${node.realId}`) : node;
                    computeImpact(realNode);
                });
                return computedImpact;
            }
        }
    };
});



// WARNING Keep the switch in sync with other _XXX_MassActionsCallbacks controllers (flow, taggable objects pages, list pages)
app.controller('FlowMassActionsCallbacks', function($scope, $rootScope, FlowTool, FlowGraphSelection, ToolBridgeService, PIPELINEABILITY_ACTIONS, SummaryService) {

    $scope.onAction = function(action) {
        let affectedViews = [];
        switch (action) {
            case 'action-clear':
            case 'action-build':
            case 'action-change-connection':
            case 'action-share':
            case 'action-set-virtualizable':
                reloadGraph();
                break;
            case 'action-delete':
            case 'action-unshare':
                reloadGraph();
                FlowGraphSelection.clearSelection();
                break;
            case 'action-tag':
                affectedViews = ['TAGS'];
                break;
            case 'action-watch':
            case 'action-star':
                affectedViews = ['WATCH']
                $rootScope.$emit('userInterestsUpdated');
                break;
            case 'action-update-status':
                affectedViews = ['COUNT_OF_RECORDS', 'FILESIZE'];
                break;
            case 'action-set-auto-count-of-records':
                affectedViews = ['COUNT_OF_RECORDS'];
                break;
            case 'action-add-to-scenario':
                affectedViews = ['SCHEDULING'];
                break;
            case 'action-change-spark-config':
                affectedViews = ['SPARK_CONFIG'];
                break;
            case PIPELINEABILITY_ACTIONS.changeSpark:
                affectedViews = ['SPARK_PIPELINES'];
                break;
            case PIPELINEABILITY_ACTIONS.changeSQL:
                affectedViews = ['SQL_PIPELINES'];
                break;
            case 'action-change-impala-write-mode':
                affectedViews = ['IMPALA_WRITE_MODE'];
                break;
            case 'action-change-hive-engine':
                affectedViews = ['HIVE_MODE'];
                break;
            case 'action-convert-to-hive':
            case 'action-convert-to-impala':
                reloadGraph();
                break;
            default:
                break;
        }
        FlowTool.refreshFlowStateWhenViewIsActive(affectedViews);
    }

    /*
    * Fetch the whole flow + the view state if any active
    */
    function reloadGraph() {
        $rootScope.$emit('reloadGraph');
    }
});


app.directive('flowEditor', function($stateParams, $timeout, $rootScope, $controller, $filter, translate, AI_EXPLANATION_MODAL_MODES, Debounce, GraphZoomTrackerService,
            Assert, TopNav, CreateModalFromTemplate, uiCustomizationService, DataikuAPI, ClipboardUtils, ContextualMenu, HistoryService, Logger, StateUtils, TaggableObjectsUtils, localStorageService,
            FlowGraphSelection, FlowToolsRegistry, FlowToolsUtils, FlowGraph, FlowGraphFiltering, FlowGraphFolding, executeWithInstantDigest, PageSpecificTourService, OpalsService, OpalsMessageService,
            Notification, $q, MessengerUtils, DatasetRenameService, FlowBuildService, AnyLoc, WatchInterestState, WT1, ZoneService, RecipeRenameService, ToolBridgeService, Dialogs) {

function drawExposedIndicators(svg, nodesGraph) {
    svg.find('.exposed-indicator').remove();

    svg.find('g[data-type=LOCAL_DATASET], g[data-type=LOCAL_SAVEDMODEL], g[data-type=LOCAL_MODELEVALUATIONSTORE], g[data-type=LOCAL_GENAIEVALUATIONSTORE], g[data-type=LOCAL_MANAGED_FOLDER], g[data-type=LOCAL_RETRIEVABLE_KNOWLEDGE]').each(function (index, boxElement) {
        const nodeId = $(boxElement).attr('data-id');
        const node = nodesGraph.nodes[nodeId];
        if (!node) {
            Logger.warn("Graph node not found:", nodeId);
            return;
        }
        if (node.isExposed) {
            const type = {
                LOCAL_DATASET: 'dataset',
                LOCAL_SAVEDMODEL: 'model',
                LOCAL_MODELEVALUATIONSTORE: 'model evaluation store',
                LOCAL_GENAIEVALUATIONSTORE: 'genai evaluation store',
                LOCAL_MANAGED_FOLDER: 'folder',
                LOCAL_RETRIEVABLE_KNOWLEDGE: "knowledge bank"
            }[$(boxElement).data('type')];

            const exposedSVG = $(makeSVG('foreignObject', {
                    x: 2,
                    y: 2,
                    width: 20,
                    height: 20,
                    class: 'exposed-indicator nodeicon-small'+(type == 'dataset' || type == 'knowledge bank' ? '' : '-dark')
                }))
                .append($(`<div><i class="icon-mail-forward" title="This ${type} is exposed in other projects"></i></div>`));
            if (type == 'folder') {
                $(boxElement).find('>g').first().append(exposedSVG);
            } else {
                $(boxElement).find('>g').append(exposedSVG);
            }
        }
    });
}

function drawForbiddenIndicators(svg, nodesGraph) {
    svg.find('.forbidden-indicator').remove();

    svg.find('g[data-type=FOREIGN_DATASET], g[data-type=FOREIGN_SAVEDMODEL], g[data-type=FOREIGN_RETRIEVABLE_KNOWLEDGE], g[data-type=FOREIGN_MODELEVALUATIONSTORE], g[data-type=FOREIGN_GENAIEVALUATIONSTORE], g[data-type=FOREIGN_MANAGED_FOLDER]').each(function (index, boxElement) {
        const nodeId = $(boxElement).attr('data-id');
        const node = nodesGraph.nodes[nodeId];
        if (!node) {
            Logger.warn("Graph node not found:", nodeId);
            return;
        }
        if (node.isForbiddenObject) {
            const type = {
                FOREIGN_DATASET: 'dataset',
                FOREIGN_SAVEDMODEL: 'model',
                FOREIGN_MODELEVALUATIONSTORE: 'model evaluation store',
                FOREIGN_GENAIEVALUATIONSTORE: 'genai evaluation store',
                FOREIGN_MANAGED_FOLDER: 'folder',
                FOREIGN_RETRIEVABLE_KNOWLEDGE: 'knowledge bank'
            }[$(boxElement).data('type')];

            const svgY = {
                FOREIGN_DATASET: 50,
                FOREIGN_SAVEDMODEL: 50,
                FOREIGN_MODELEVALUATIONSTORE: 50,
                FOREIGN_GENAIEVALUATIONSTORE: 50,
                FOREIGN_RETRIEVABLE_KNOWLEDGE: 50,
                FOREIGN_MANAGED_FOLDER: 35
            }[$(boxElement).data('type')];

            const exposedSVG = $(makeSVG('foreignObject', {
                    x: 2,
                    y: svgY,
                    width: 20,
                    height: 20,
                    class: 'exposed-indicator nodeicon-small'+(type == 'dataset' || type == 'knowledge bank' ? '' : '-dark')
                }))
                .append($(`<div><i class="icon-warning-sign" title="You don't have the rights to access this object"></i></div>`));
            if (type == 'folder') {
                $(boxElement).find('>g').first().append(exposedSVG);
            } else {
                $(boxElement).find('>g').append(exposedSVG);
            }
        }
    });
}

function drawBuildInProgressIndicators(svg, nodesGraph) {
    svg.find('.build-indicator').remove();

    svg.find('g[data-type=LOCAL_DATASET], g[data-type=LOCAL_SAVEDMODEL], g[data-type=LOCAL_MODELEVALUATIONSTORE], g[data-type=LOCAL_GENAIEVALUATIONSTORE], g[data-type=LOCAL_MANAGED_FOLDER], g[data-type=LOCAL_RETRIEVABLE_KNOWLEDGE]').each(function (index, boxElement) {
        let nodeId = $(boxElement).attr('data-id');
        let node = nodesGraph.nodes[nodeId];

        if (!node) {
            Logger.warn("Graph node not found:", nodeId)
            return;
        }

        let iconDom = null;
        if (node.beingBuilt) {
            iconDom = $('<div class="icon-being-built"><i class="icon-play" /></div>');
        } else if (node.aboutToBeBuilt) {
            iconDom = $('<div class="icon-about-to-be-built"><i class="icon-spinner"></i></div>');
        }
        if (iconDom) {
            let $pinSvg = $(makeSVG('foreignObject', {
                    x: 75,
                    y: 55,
                    width: 20,
                    height: 20,
                    'class': 'build-indicator'
            })).append(iconDom);

            if ($(boxElement).data('type') == 'LOCAL_MANAGED_FOLDER') {
                $(boxElement).find('>g').first().append($pinSvg);
            } else {
                $(boxElement).find('>g').append($pinSvg);
            }
        }
    });

    svg.find('g[data-type=RECIPE]').each(function (index, boxElement) {
        let nodeId = $(boxElement).attr('data-id');
        let node = nodesGraph.nodes[nodeId];

        if (!node) {
            Logger.warn("Graph node not found:", nodeId)
            return;
        }

        let iconDom = null;
        if (node.continuousActivityDone) {
            iconDom = $('<div class="icon-continuous-activity-done"><i class="icon-warning-sign" /></div>');
        } else if (node.beingBuilt) {
            iconDom = $('<div class="icon-being-built"><i class="icon-play" /></div>');
        }
        if (iconDom) {
            let $pinSvg = $(makeSVG('foreignObject', {
                    x: 55,
                    y: 40,
                    width: 20,
                    height: 20,
                    'class': 'build-indicator',
                    transform: 'scale(1.92 1.92)'  // scale to conteract the 0.52 iconScale
            })).append(iconDom);

            $(boxElement).find('>g').append($pinSvg);
        }
    });
}

return {
    restrict: 'EA',
    scope: true,
    controller: function($scope, $rootScope, SummaryService, PageSpecificTourService, FlowTool) {
        $controller('FlowMassActionsCallbacks', {$scope: $scope});

        TopNav.setLocation(TopNav.TOP_FLOW, TopNav.LEFT_FLOW, TopNav.TABS_NONE, null);
        TopNav.setNoItem();

        uiCustomizationService.getComputeDatasetTypesStatus($scope, $stateParams.projectKey).then(
            (computeStatus) => {
                $scope.canCreateUploadedFilesDataset = computeStatus("UploadedFiles") === uiCustomizationService.datasetTypeStatus.SHOW;
            });

        $scope.projectFlow = true;
        $scope.nodesGraph = {flowFiltersAndSettings : {}};

        $scope.getZoneColor = zoneId => {
            const nodeFound = $scope.nodesGraph.nodes ? $scope.nodesGraph.nodes[`zone_${zoneId}`] : undefined;
            if (nodeFound && nodeFound.customData) {
                return nodeFound.customData.color
            }
            return "#ffffff";
        };

        function updateUserInterests() {
            DataikuAPI.interests.getUserInterests($rootScope.appConfig.login, 0, 10000, {projectKey: $stateParams.projectKey}).success(function(data) {
                // It would be nice to fetch that with the graph but it is a little dangerous to require the database to be functional to see any flow...
                $scope.userInterests = data.interests.filter(x => ['RECIPE', 'DATASET', 'SAVED_MODEL', 'MODEL_EVALUATION_STORE', 'MANAGED_FOLDER', "STREAMING_ENDPOINT", 'LABELING_TASK', 'RETRIEVABLE_KNOWLEDGE'].includes(x.objectType));

                const indexedInterests = {};
                $scope.userInterests.forEach(function(interest) {
                    //TODO @flow using the node ids as keys would be better but we don't generate them in js for now (I think)
                    indexedInterests[interest.objectType+'___'+interest.objectId] = interest;
                });
                $.each($scope.nodesGraph.nodes, function(nodeId, node) {
                    const taggableType = TaggableObjectsUtils.fromNodeType(node.nodeType);
                    const interest = indexedInterests[taggableType+'___'+node.name]
                    if (interest) {
                        node.interest = interest;
                    } else {
                        node.interest = {
                            starred: false,
                            watching: WatchInterestState.values.ENO
                        };
                    }
                });

            }).error(setErrorInScope.bind($scope));
        }
        $scope.setErrorInScopeCallbackForAngular = (error) => setErrorDetailsInScope.bind($scope)(error); // used to put in scope error already handled in Angular
        $scope.isNotInGraph = function(item) {
            return $scope.nodesGraph && $scope.nodesGraph.nodes &&
                ! $scope.nodesGraph.nodes.hasOwnProperty('dataset_' + item.name);
        };

        $scope.processSerializedFilteredGraphResponse = function processSerializedFilteredGraphResponse(serializedFilteredGraph, zoomTo, errorScope) {
            $scope.setGraphData(serializedFilteredGraph.serializedGraph);
            if (typeof zoomTo === 'string') {
                const deregisterListener = $scope.$root.$on("flowDisplayUpdated", function () {
                    deregisterListener();
                    setTimeout(() => {
                        let id = zoomTo;
                        let node = $scope.nodesGraph.nodes[zoomTo];
                        if (!node) {
                            id = graphVizEscape(zoomTo);
                            node = $scope.nodesGraph.nodes[id];
                        }
                        if (!node && $scope.nodesGraph.hasProjectZones) {
                            id = Object.values($scope.nodesGraph.nodes).filter(it => it.realId == id && !it.usedByZones.length)[0].id;
                        }
                        $scope.zoomGraph(id);
                        GraphZoomTrackerService.instantSavePanZoomCtx($scope.panzoom);
                        GraphZoomTrackerService.setFocusItemCtx($scope.nodesGraph.nodes[id]);
                        FlowGraphSelection.onItemClick($scope.nodesGraph.nodes[id]);
                    });
                })
            }

            $rootScope.$emit('drawGraph');
        };

        $scope.processLoadFlowResponse = function processLoadFlowResponse (resp, zoomTo, graphReloaded, resetZoom, errorScope = $scope) {
            Assert.trueish(resp, "Received empty response");
            if (resp.serializedFilteredGraph) {
                $scope.processSerializedFilteredGraphResponse(resp.serializedFilteredGraph, zoomTo, errorScope);
            }
            if (resetZoom) $scope.resetPanZoom();
            $scope.isFlowLoaded = true;
        };

        $scope.updateGraph = function updateGraph(zoomTo, nospinner=false, shouldUpdateUserInterests=true) {
            DataikuAPI.flow.recipes.getGraph($stateParams.projectKey, true, $scope.drawZones.drawZones, $stateParams.zoneId, $scope.collapsedZones, nospinner)
                .success(function(response) {
                        $scope.zonesManualPositioning = response.zonesManualPositioning.projectSettingsValue;
                        $scope.canMoveZones = response.zonesManualPositioning.canMove;

                        $scope.zoneIdLoaded = $stateParams.zoneId;
                        $scope.processLoadFlowResponse(response, zoomTo, true);
                        if (shouldUpdateUserInterests) {
                            updateUserInterests();
                        }
                        $scope.isMoveFlowZonesToggleLoading = false;
                        $scope.isSavingZonePosition = false;
                    }
                )
                .error(setErrorInScope.bind($scope));

            //TODO @flow move to flow_search
            // DataikuAPI.datasets.list($stateParams.projectKey).success(function(data) {
            //     $scope.datasets = data;
            // }).error(setErrorInScope.bind($scope));
            // DataikuAPI.datasets.listHeads($stateParams.projectKey, {}, false).success(function(data) {
            //     $scope.filteredDatasets = data;
            // }).error(setErrorInScope.bind($scope));
        };

        var storageKey = `dku.flow.drawZones.${$stateParams.projectKey}`;
        $scope.drawZones = {
            drawZones: !!$stateParams.zoneId || JSON.parse(localStorageService.get(storageKey) || true)
        }

        var drawZonesSub = ToolBridgeService.drawZonesToggle$.subscribe(() => {
            ToolBridgeService.emitShouldDrawZones(!$scope.drawZones.drawZones);
        });

        // Cleanup when the scope is destroyed
        $scope.$on('$destroy', function() {
            if (drawZonesSub) {
                drawZonesSub.unsubscribe();
            }
        });

        // init call to restore show/hide zone state on page reload (or project navigation)
        ToolBridgeService.emitShouldDrawZones($scope.drawZones.drawZones);

        $scope.redrawZone = function () {
            if (!$scope.inFlowExport) {
                localStorageService.set(storageKey, $scope.drawZones.drawZones);
                $scope.resetPanZoom();
                $scope.updateGraph();
            }
        };

        var collapsedZonesStorageKey = `dku.flow.collapsedZones.${$stateParams.projectKey}`;

        $scope.cleanupCollapsedZones = (collapsedZones = $scope.collapsedZones) => {
            let changed = false;
            [...collapsedZones].forEach(collapsedZone => {
                const zoneFound = FlowGraph.node(`zone_${collapsedZone}`);
                if (!zoneFound) {
                    const index = collapsedZones.indexOf(collapsedZone);
                    if (index !== -1) {
                        collapsedZones.splice(index, 1);
                        changed = true;
                    }
                }
            });
            if (changed) {
                localStorageService.set(collapsedZonesStorageKey, JSON.stringify(collapsedZones));
            }
            return collapsedZones;
        }
        $scope.collapsedZones = localStorageService.get(collapsedZonesStorageKey) || [];

        $scope.toggleZoneCollapse = (collapseItems, multiItemStrategy) => {
            let zoneIds = collapseItems.map(it => it.id);
            zoneIds.forEach(function(zoneId) {
                let index = $scope.collapsedZones.findIndex(it => it === zoneId);
                if (index > -1 && multiItemStrategy !== 'collapseAll') {
                    $scope.collapsedZones.splice(index, 1);
                } else if (index < 0 && multiItemStrategy !== 'expandAll') {
                    $scope.collapsedZones.push(zoneId);
                }
            });
            localStorageService.set(collapsedZonesStorageKey, JSON.stringify($scope.collapsedZones));
            $scope.updateGraph();
        }

        $scope.updateGraph($stateParams.id);

        $scope.nodeSelectorTooltip = function(type, count) {
            const params = { type: $filter('niceTaggableType')(type, count) };
            if (count > 1) {
                return translate('PROJECT.FLOW.GRAPH.SELECT_ALL_OBJECTS', 'Select all {{type}}', params);
            } else {
                return translate('PROJECT.FLOW.GRAPH.SELECT_AN_OBJECT', 'Select a {{type}}', params);
            }
        };

        $scope.copyNameToClipboard = function(name) {
            ClipboardUtils.copyToClipboard(name);
        }

        $scope.renameDataset = function(datasetNode) {
            DatasetRenameService.startRenaming($scope, datasetNode.projectKey, datasetNode.name, datasetNode.datasetType)
        }

        $scope.renameRecipe = function(recipeNode) {
            RecipeRenameService.startRenamingRecipe($scope, recipeNode.projectKey, recipeNode.name);
        }

        const buildModalParams = (hasPredecessors, hasSuccessors) => {
            // hasPredecessors/hasSuccessors is not *exactly* equal to upstreamBuildable / downstreamBuildable, but is a good approximation
            return {
                upstreamBuildable: hasPredecessors,
                downstreamBuildable: hasSuccessors,
            };
        }

        $scope.buildDataset = function(projectKey, name, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "DATASET", AnyLoc.makeLoc(projectKey, name), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.trainModel = function(projectKey, id,  hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "SAVED_MODEL", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.buildManagedFolder = function(projectKey, id, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "MANAGED_FOLDER", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.buildModelEvaluationStore = function(projectKey, id, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "MODEL_EVALUATION_STORE", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };
        $scope.buildRetrievableKnowledge = function(projectKey, id, hasPredecessors, hasSuccessors) {
            FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc($scope, "RETRIEVABLE_KNOWLEDGE", AnyLoc.makeLoc(projectKey, id), buildModalParams(hasPredecessors, hasSuccessors));
        };

        $scope.startCopy = function() {
            $scope.startTool('COPY', {preselectedNodes: FlowGraphSelection.getSelectedNodes().map(n => n.id)});
        };

        const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);

        const TARGET_TYPES_TO_UPDATE = {
            DATASET: "dataset",
            RETRIEVABLE_KNOWLEDGE: "retrievableknowledge"
        }

        function getNodeNameForItem(itemType, itemName) {
            return itemType + "__" + graphVizEscape(itemName);
        }

        function updateNodeStatus(node, state) {
            if (!node) return false;

            switch (state) {
                case FLOW_NODE_STATUS.RUNNING:
                    node.beingBuilt = true;
                    node.aboutToBeBuilt = false;
                    break;
                case FLOW_NODE_STATUS.DONE:
                    node.beingBuilt = false;
                    node.aboutToBeBuilt = false;
                    break;
                case FLOW_NODE_STATUS.NOT_STARTED:
                    node.aboutToBeBuilt = true;
                    node.beingBuilt = false;
                    break;
            }
            return true;
        }

        function updateNodeStatusFlowWithZones(message, state) {
            // selector matching job output datasets realId (can be in multiple zones)
            let refresh = false;

            if (!message.status) return refresh;

            const selector = message.status.targets.filter(target => target.type in TARGET_TYPES_TO_UPDATE).map(target => {
                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                return `svg [data-node-id="${realId}"]`;
            }).join(', ');
            // update node status for each matching output dataset
            if (selector) {
                d3.selectAll(selector).each(function() {
                    const id = this.getAttribute('data-id');
                    if (updateNodeStatus(FlowGraph.node(id), state)) {
                        refresh = true;
                    }
                });
            }
            return refresh;
        }

        function updateNodeStatusFlowWithoutZones(message, state) {
            let refresh = false;
            if (!message.status) return refresh;


            Object.values(message.status.targets).filter(target => target.type in TARGET_TYPES_TO_UPDATE).forEach(target => {
                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                if (updateNodeStatus($scope.nodesGraph.nodes[realId], state)) {
                    refresh = true;
                }
            });
            return refresh;
        }

        function updateAndDrawActivityNodeStatus(message, state) {
            let refreshGraph;

            if ($scope.nodesGraph.hasProjectZones && $scope.drawZones.drawZones) {
                refreshGraph =  updateNodeStatusFlowWithZones(message, state);
            } else {
                refreshGraph = updateNodeStatusFlowWithoutZones(message, state);
            }

            if (refreshGraph) {
                drawBuildInProgressIndicators(FlowGraph.getSvg(), $scope.nodesGraph);
            }
        }

        const jobActivityStartedListener = Notification.registerEvent("job-activity-started", function(evt, message) {
            updateAndDrawActivityNodeStatus(message, FLOW_NODE_STATUS.RUNNING);
        });

        const jobActivityDoneListener = Notification.registerEvent("job-activity-done", function(evt, message) {
            updateAndDrawActivityNodeStatus(message, FLOW_NODE_STATUS.DONE);
        });

        const jobStatusUpdatedListener = Notification.registerEvent("job-status-updated", function(evt, message) {
            let refreshGraph = false;

            function updateAllNodeStatusFlowWithZones() {
                let targets = [];
                Object.values(message.status.activities).forEach(activity => {
                    if (activity.state === "NOT_STARTED" && !activity.skipExplicitOrWriteProtected) {
                        activity.targets.forEach(target => {
                            if (target.type in TARGET_TYPES_TO_UPDATE) {
                                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                                targets.push(`svg [data-node-id="${realId}"]`)
                            }
                        })
                    }
                })

                const selector = targets.join(', ');

                // update node status for each matching output dataset
                if (selector) {
                    d3.selectAll(selector).each(function() {
                        const id = this.getAttribute('data-id');
                        if (updateNodeStatus(FlowGraph.node(id), FLOW_NODE_STATUS.NOT_STARTED)) {
                            refreshGraph = true;
                        }
                    });
                }
            }

            function updateAllNodeStatusFlowWithoutZones() {
                if (!message.status || !message.status.activities) return;
                Object.values(message.status.activities).forEach(activity => {
                    if (activity.state === "NOT_STARTED" && !activity.skipExplicitOrWriteProtected) {
                        activity.targets.forEach(target => {
                            if (target.type in TARGET_TYPES_TO_UPDATE) {
                                const realId = getNodeNameForItem(TARGET_TYPES_TO_UPDATE[target.type], target.id);
                                if ($scope.nodesGraph.nodes && updateNodeStatus($scope.nodesGraph.nodes[realId], FLOW_NODE_STATUS.NOT_STARTED)) {
                                    refreshGraph = true;
                                }
                            }
                        });
                    }
                });
            }

            if ($scope.nodesGraph.hasProjectZones && $scope.drawZones.drawZones) {
                updateAllNodeStatusFlowWithZones()
            } else {
                updateAllNodeStatusFlowWithoutZones();
            }

            if (refreshGraph) {
                drawBuildInProgressIndicators(FlowGraph.getSvg(), $scope.nodesGraph);
            }
        });

        let continuousActivityStateChangeListener = Notification.registerEvent("continuous-activity-state-change", function(evt, message) {

            let refreshGraph = false;


            function isRunning(message) {
                if (message.state=='STOPPED') return false;
                if (message.state=='STARTED') return true;
                return undefined;
            }

            function updateNodeStatus(isRunIcon, node) {
                if (!node) return;

                if (isRunIcon && !node.beingBuilt) {
                    refreshGraph = true;
                    node.beingBuilt = true;
                    node.continuousActivityDone = false;
                    node.aboutToBeBuilt = false;
                }
                else if (node.beingBuilt || node.aboutToBeBuilt) {
                    refreshGraph = true;
                    node.beingBuilt = false;
                    node.continuousActivityDone = false;
                    node.aboutToBeBuilt = false;
                }

            }

            let isRun = isRunning(message);
            if (isRun!==undefined) {
                let nodeName = getNodeNameForItem('recipe', message.continuousActivityId);
                updateNodeStatus(isRun, $scope.nodesGraph.nodes[nodeName]);
            }

            if (refreshGraph) {
                drawBuildInProgressIndicators(FlowGraph.getSvg(), $scope.nodesGraph);
            }
        });

        $scope.$on("$destroy", function() {
            if ($scope.svg) {
                $scope.svg.empty();
                $scope.svg.remove();
                $scope.svg = null;
                jobActivityStartedListener();
                jobActivityDoneListener();
                jobStatusUpdatedListener();
            }
            interestsListener();
            continuousActivityStateChangeListener();
        });

        $scope.unfoldAll = function() {
            FlowGraphFolding.unfoldAll();
        }

        $scope.zoomOnSelection = function(paddingFactor) {
            const selectedNodes = $('.selected', $scope.svg)
            // Only calculate padding factor if not provided
            if (paddingFactor == null) {
                paddingFactor = 1.2
                if (selectedNodes && selectedNodes.length === 1){
                    if (selectedNodes[0].classList.contains('zone_cluster')) {
                        //only one zone, padding is 1.5
                        paddingFactor = 1.5;
                    } else {
                        //only one element, show some context
                        paddingFactor = 3
                    }
                }
            }
            $scope.zoomToBbox(FlowGraphFiltering.getBBoxFromSelector($scope.svg, '.selected'), paddingFactor);
        };

        $scope.exportFlow = function() {
            const graphBBox = $scope.svg.find('g.graph')[0].getBBox();
            CreateModalFromTemplate("/templates/flow-editor/export-flow-modal.html", $scope, "ExportFlowModalController", function(newScope) {
                newScope.init($stateParams.projectKey, graphBBox);
            });
        };

        $scope.generateFlowDocument = function() {
            CreateModalFromTemplate("templates/flow-editor/generate-flow-document-modal.html", $scope, "GenerateFlowDocumentModalController", function(newScope) {
                newScope.init($stateParams.projectKey);
            }, false, 'static');
        };

        // Toolbox used by export-flow.js to prepare the flow to be exported
        $scope.exportToolbox = {
            checkLoading: function() {
                return $scope.httpRequests.length !== 0 || !$scope.isFlowLoaded || ToolBridgeService.isQueryLoading() || ToolBridgeService.isViewLoading();
            },
            removeDecorations: function(drawZones) {
                executeWithInstantDigest(function() {
                    $scope.hideForExport = true;
                    $scope.fullScreen = true;
                    $scope.inFlowExport = true; // Prevent the flow from automatically refreshing when zones are shown/hidden
                }, $scope);
            },
            getGraphBoundaries: function () {
                const graphBBox = $scope.svg.find('g.graph')[0].getBBox();
                return {
                    x: graphBBox.x,
                    y: graphBBox.y,
                    width: graphBBox.width,
                    height: graphBBox.height
                };
            },
            adjustViewBox: function(x, y, width, height) {
                $scope.svg[0].setAttribute('viewBox', [x, y, width, height].join(', '));
            },
            configureZones: function(drawZones, collpasedZones) {
                $scope.drawZones.drawZones = drawZones;
                $scope.collapsedZones = collpasedZones;
                // Reload the flow graph
                $scope.isFlowLoaded = false;
                $scope.updateGraph();
            }
        };

        $scope.explainProject = function() {
            CreateModalFromTemplate(
                "/static/dataiku/ai-explanations/explanation-modal/explanation-modal.html",
                $scope,
                "AIExplanationModalController",
                function(newScope) {
                    newScope.objectType = "PROJECT";
                    newScope.object = $scope.projectSummary;
                    newScope.mode = AI_EXPLANATION_MODAL_MODES.EXPLAIN;
                }
            );
        };

        $scope.zoomOnZone = ZoneService.zoomOnZone;

        $scope.zoomOutOfZone = ZoneService.zoomOutOfZone;

        const hasPreviewKey = `dku.flow.hasPreview`;
        $scope.hasPreview = JSON.parse(localStorageService.get(hasPreviewKey)) || false; // init
        $scope.togglePreview = () => {
            $scope.hasPreview = !$scope.hasPreview;
            localStorageService.set(hasPreviewKey, $scope.hasPreview)
        }

        $scope.isMoveFlowZonesToggleLoading = false;
        $scope.toggleZonesManualPositioning = () => {
            const projectSummary = $rootScope.projectSummary;
            $scope.isMoveFlowZonesToggleLoading = true;
            DataikuAPI.flow.zones.setManualPositioningSetting(projectSummary.projectKey, !projectSummary.zonesManualPositioning)
                .then(() => {
                    projectSummary.zonesManualPositioning = !projectSummary.zonesManualPositioning;
                    WT1.tryEvent("flow-toggle-zones-manual-positioning", () => ({
                        enabled: projectSummary.zonesManualPositioning,
                    }));
                    $scope.$emit('zonesManualPositioningChanged');
                })
                .catch(function(data, status, headers) {
                    $scope.isMoveFlowZonesToggleLoading = false;
                    setErrorInScope.bind($scope)(data, status, headers);
                });
        }

        $scope.autoArrangeFlowZones = () => {
            Dialogs.confirmAlert(
                $scope,
                translate(
                    "PROJECT.FLOW.AUTO_ARRANGE_FLOW_ZONES.TITLE",
                    "Auto-arrange flow zones"
                ),
                translate(
                    "PROJECT.FLOW.AUTO_ARRANGE_FLOW_ZONES.TEXT",
                    "Are you sure you want to continue?",
                ),
                translate(
                    "PROJECT.FLOW.AUTO_ARRANGE_FLOW_ZONES.ALERT",
                    "This will reset all flow zones positions for this project",
                )
                ,
                "WARNING"
            ).then(
                function () {
                    // Confirm
                    $scope.isSavingZonePosition = true;
                    WT1.tryEvent("flow-auto-arrange-zones", () => {});
                    const projectSummary = $rootScope.projectSummary;
                    DataikuAPI.flow.zones.resetPositions(projectSummary.projectKey)
                        .then(() => $scope.$emit('zonesManualPositioningChanged'))
                        .catch(function(data, status, headers) {
                            $scope.isSavingZonePosition = false;
                            setErrorInScope.bind($scope)(data, status, headers);
                        });
                },
                function () {
                    // Cancel: do nothing
                }
            );
        }

        const checkIsFlowActionTool = function () {
            const tool = FlowTool.getCurrent();
            return tool && tool.action;
        };

        $scope.onFlowColor = flowColoring => {
            $scope.flowColoring = flowColoring; // save coloring for later
            if (checkIsFlowActionTool()) {
                ToolBridgeService.queryLoaded();
                return;
            }
            if (flowColoring === undefined) {
                resetDefaultGraphColor();
                ToolBridgeService.queryLoaded();
                return;
            }
            // however sometimes we just want to apply an highlight on a flow that did not get updated
            // yes it can be called two times in a row
            colorFlowNodes(flowColoring.coloring, flowColoring.matchedObjectsId);
            ToolBridgeService.queryLoaded();
        };
        $scope.onSuggestionFocus = flowItemColoring => {
            if (checkIsFlowActionTool()) {
                return;
            }
            if (flowItemColoring === undefined) {
                if ($scope.flowColoring === undefined) {
                    return;
                }
                colorFlowNodes($scope.flowColoring.coloring, $scope.flowColoring.matchedObjectsId);
                return;
            }
            colorFlowNodes(flowItemColoring.coloring, flowItemColoring.matchedObjectsId);
        };

        $scope.onObjectSelect = function(objects) {
            if (objects.length === 0) {
                FlowGraphSelection.clearSelection();
                return;
            }

            FlowGraphSelection.clearSelection();
            FlowGraphSelection.selectMulti(objects);
            if (objects.length === 1) {
                // sc-196337 call zoomGraph to behave like legacy for one object.
                // 65px of offCenterShift to put zoom bottom screen
                const objectsByType = $scope.nodesGraph.includedObjectsByType;
                const hasOnlyFlowZone =
                  objectsByType && Object.keys(objectsByType).length === 1 && objectsByType.hasOwnProperty("FLOW_ZONE");
                if (hasOnlyFlowZone) {
                    $scope.zoomGraph(objects[0].id,3,null, 300);
                } else {
                    $scope.zoomGraph(objects[0].id, 3, null, -65);
                }

            } else {
                $scope.zoomOnSelection();
            }
        };

        let flowViewsSubscriptions = [];
        flowViewsSubscriptions.push(
            SummaryService.selectedObjects$.subscribe(obj => {
                if (obj && obj.length === 0) {
                    FlowGraphSelection.clearSelection();
                    return;
                }
                FlowGraphSelection.clearSelection();
                FlowGraphSelection.selectMulti(obj);
            })
        );

        flowViewsSubscriptions.push(
            SummaryService.zoomOnSelection$.subscribe(paddingFactor => {
                $scope.zoomOnSelection(paddingFactor);
            })
        );

        // when changing some configuration on the flow, for example adding a tag to an element
        // highlighting the nodes will happen before the flow is being rendered
        // so we add a listener to react after the flow display has been updated
        const flowDisplayUpdatedUnsubscribe = $rootScope.$on('flowDisplayUpdated', function(_, shouldResetDefault) {
            if (checkIsFlowActionTool()){
                return;
            }
            if ($scope.flowColoring) {
                colorFlowNodes($scope.flowColoring.coloring, $scope.flowColoring.matchedObjectsId);
            } else if (shouldResetDefault) {
                resetDefaultGraphColor();
            }
        });

        // Cleanup when the scope is destroyed
        $scope.$on('$destroy', function(){
            if (flowViewsSubscriptions) {
                flowViewsSubscriptions.forEach(subs => subs.unsubscribe());
            }
            flowDisplayUpdatedUnsubscribe();
        });

        /**
         * Highlights nodes in a flow based on provided criteria.
         *
         * @param {?Map<string, string | undefined>} coloring - A map where each key corresponds to a node ID and its value corresponds to the color of that node. If the value is `undefined`, the node will not be colored (or greyed out).
         * @param {?Set<string>} matchedObjectsId - A set of string identifiers for nodes that have matched certain criteria and are to be highlighted.

         */
        function colorFlowNodes(coloring, matchedObjectsId) {
            const graphSvg = FlowGraph.getSvg();
            if (!graphSvg) return;
            // Initialize cache if needed
            for (const nodeId of Object.keys($scope.nodesGraph.nodes)) {
                const d3NodeElement = getD3Node(nodeId);
                if (d3NodeElement == null) {
                    continue;
                }
                resetNodeStyle(nodeId, d3NodeElement);
                // Mute all nodes
                if (!matchedObjectsId.has(nodeId)) {
                    d3NodeElement.classed('filter-remove', true);
                }
                // If node is not present in the mapping, it should be colored using its default color
                if (coloring) {
                    let color = coloring.get(nodeId);
                    if (color != null) {
                        colorNode($scope.nodesGraph.nodes[nodeId], d3NodeElement, color);
                        d3NodeElement.classed('focus', true);
                    } else {
                        d3NodeElement.classed('filter-remove', true);
                    }
                }
            }
        }

        function resetNodeStyle(nodeId, d3node) {
            colorNode($scope.nodesGraph.nodes[nodeId], d3node, undefined);
            d3node.classed('focus', false).classed('filter-remove', false);
        }

        /**
         * Color all nodes in the graph with their default co
         */
        function resetDefaultGraphColor() {
            for (const nodeId of Object.keys($scope.nodesGraph.nodes)) {
                const d3NodeElement = getD3Node(nodeId);
                if (d3NodeElement == null) {
                    continue;
                }
                resetNodeStyle(nodeId, d3NodeElement);
            }
        }

        /**
         * Color node by its ID with a given color.
         * @param node the node to be colored
         * @param {D3Node} d3NodeElement D3 node element (a single node in the flow graph, not the whole graph)
         * @param {?string} color color of node. `''` and `undefined` means default node color inferred by its type
         */
        function colorNode(node, d3NodeElement, color) {
            if (node.nodeType === 'ZONE') return;
            FlowToolsUtils.colorNode(node, d3NodeElement, color);
        }

        /**
         * Get the D3 Node for a given node ID
         * @param {string} nodeId
         * @returns {?D3Node}
         */
        function getD3Node(nodeId) {
            const node = FlowGraph.node(nodeId);
            const d3Node = FlowGraph.d3NodeWithIdFromType(
                nodeId,
                node.nodeType
            );
            return d3Node;
        }

        const unregisterFlowTourListener = $rootScope.$on('startFlowTour', function() {
            PageSpecificTourService.startFlowTour({ scope: $scope, fromContext: 'opals' });
        });
        $scope.$on("$destroy", function() {
            unregisterFlowTourListener();
        });

    },
    link: function(scope, element) {
        // Try to find the more recent item for the project and zone
        function getLastItemInHistory(projectKey, zoneId) {
            const items = HistoryService.getRecentlyViewedItems();
            const validItems = items.filter(it => it.type !== 'PROJECT' && it.projectKey === projectKey);
            if (items && items.length) {
                if (!zoneId) {
                    return validItems[0];
                }
                const zoneName = graphVizEscape(`zone_${zoneId}`);
                const zoneContent = Object.keys(scope.nodesGraph.nodes).filter(it => it.startsWith(zoneName)).map(it => scope.nodesGraph.nodes[it]);
                return validItems.find(item => zoneContent.find(it => it.name === item.id));
            }
            return null;
        }

        function getName(item) {
            if (item.type === 'RECIPE' || item.type === 'LABELING_TASK') {
                return item.type.toLowerCase() + graphVizEscape(`_${item.id}`)
            }
            return item.type.toLowerCase().replace('_', '') + graphVizEscape(`_${item.projectKey}.${item.id}`);
        }

        function zoomOnLast() {
            if (!scope.nodesGraph || !scope.nodesGraph.nodes) {
                return; // not ready
            }
            const itemFound = getLastItemInHistory($stateParams.projectKey, $stateParams.zoneId);
            if (itemFound) {
                const id = GraphZoomTrackerService.getZoomedName(FlowGraph, getName(itemFound));
                Logger.info("zooming on " + id + "--> ", scope.nodesGraph.nodes[id]);
                scope.zoomGraph(id);
                FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[id]);
                scope.$apply();
            }
        }

        const lastUsedZoneKey = `dku.flow.lastUsedZone.${$stateParams.projectKey}`;

        scope.moveToFlowZone = (movingItems, forceCreation = false, computedImpact = []) => {
            scope.movingItems = movingItems;
            scope.computedImpact = computedImpact;

            CreateModalFromTemplate("/templates/flow-editor/move-to-zone.html", scope, null, newScope => {
                newScope.uiState = {
                    creationMode: forceCreation ? 'CREATE' : 'SELECT',
                    forceCreation,
                };
                newScope.onClick = () => {
                    let movingTo = newScope.uiState.selectedZone;
                    let promise = null
                    movingItems = movingItems.concat(scope.computedImpact);
                    if (newScope.uiState.creationMode === 'CREATE') {
                        promise = DataikuAPI.flow.zones.create($stateParams.projectKey, newScope.uiState.name, newScope.uiState.color).success(zoneCreated => {
                            movingTo = zoneCreated.id;
                            $rootScope.$emit('zonesListChanged');
                        }).error($q.reject);
                    } else {
                        promise = $q.resolve();
                    }

                    if (movingItems.length > 0) {
                        promise = promise.then(() => DataikuAPI.flow.zones.moveItems($stateParams.projectKey, movingTo, movingItems).error($q.reject));
                    }
                    promise.then(() => {
                        localStorageService.set(lastUsedZoneKey, movingTo);
                        GraphZoomTrackerService.setFocusItemCtx({id: `zone_${movingTo}`}, true);
                        newScope.$emit('reloadGraph');
                        newScope.dismiss();
                    }, setErrorInScope.bind(newScope))
                }
            });
        };

        scope.shareToFlowZone = (sharingItems, forceCreation = false) => {
            CreateModalFromTemplate("/templates/flow-editor/share-to-zone.html", scope, null, newScope => {
                newScope.uiState = {
                    creationMode: forceCreation ? 'CREATE' : 'SELECT',
                    forceCreation,
                };
                newScope.onClick = () => {
                    let sharedTo = newScope.uiState.selectedZone;
                    let promise = null
                    if (newScope.uiState.creationMode === 'CREATE') {
                        promise = DataikuAPI.flow.zones.create($stateParams.projectKey, newScope.uiState.name, newScope.uiState.color).success(zoneCreated => {
                            sharedTo = zoneCreated.id;
                            $rootScope.$emit('zonesListChanged');
                        }).error($q.reject);
                    } else {
                        promise = $q.resolve();
                    }

                    if (sharingItems.length > 0) {
                        promise = promise.then(() => DataikuAPI.flow.zones.shareItems($stateParams.projectKey, sharedTo, sharingItems).error($q.reject));
                    }

                    promise.then(() => {
                        localStorageService.set(lastUsedZoneKey, sharedTo);
                        newScope.$emit('reloadGraph');
                        newScope.dismiss()
                    }, setErrorInScope.bind(newScope));
                }
            });
        };

        scope.unshareToFlowZone = (sharingItems, zoneIds) => {
            if (sharingItems.length > 0) {
                DataikuAPI.flow.zones.unshareItems($stateParams.projectKey, zoneIds, sharingItems).success(scope.$emit('reloadGraph'));
            }
        };

        scope.onItemDblClick = function(item, evt) {
            let destUrl = StateUtils.href.node(item);
            fakeClickOnLink(destUrl, evt);
        };

        scope.onContextualMenu = function(item, evt) {
            let $itemEl = $(evt.target).parents("g[data-type]").first();
            if ($itemEl.length > 0) {
                let x = evt.pageX;
                let y = evt.pageY;
                let ctxMenuScope = scope.$new();
                const selectedNodes = FlowGraphSelection.getSelectedNodes();
                let type = selectedNodes.length > 1 ? 'MULTI' : item.nodeType;

                let controller = {
                    "LOCAL_DATASET": "DatasetContextualMenuController",
                    "FOREIGN_DATASET": "ForeignDatasetContextualMenuController",
                    "LOCAL_STREAMING_ENDPOINT": "StreamingEndpointContextualMenuController",
                    "RECIPE": "RecipeContextualMenuController",
                    "LABELING_TASK": "LabelingTaskContextualMenuController",
                    "LOCAL_SAVEDMODEL": "SavedModelContextualMenuController",
                    "FOREIGN_SAVEDMODEL": "SavedModelContextualMenuController",
                    "LOCAL_MODELEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "FOREIGN_MODELEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "LOCAL_GENAIEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "FOREIGN_GENAIEVALUATIONSTORE": "ModelEvaluationStoreContextualMenuController",
                    "LOCAL_MANAGED_FOLDER": "ManagedFolderContextualMenuController",
                    "FOREIGN_MANAGED_FOLDER": "ManagedFolderContextualMenuController",
                    "LOCAL_RETRIEVABLE_KNOWLEDGE": "KnowledgeBankContextualMenuController",
                    "FOREIGN_RETRIEVABLE_KNOWLEDGE": "KnowledgeBankContextualMenuController",
                    "ZONE": "ZoneContextualMenuController",
                    "MULTI": "MultiContextualMenuController",
                }[type];

                let template = "/templates/flow-editor/" + {
                    "LOCAL_DATASET": "dataset-contextual-menu.html",
                    "FOREIGN_DATASET": "foreign-dataset-contextual-menu.html",
                    "LOCAL_STREAMING_ENDPOINT": "streaming-endpoint-contextual-menu.html",
                    "RECIPE": "recipe-contextual-menu.html",
                    "LABELING_TASK": "labeling-task-contextual-menu.html",
                    "LOCAL_SAVEDMODEL": "savedmodel-contextual-menu.html",
                    "FOREIGN_SAVEDMODEL": "savedmodel-contextual-menu.html",
                    "LOCAL_MODELEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "FOREIGN_MODELEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "LOCAL_GENAIEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "FOREIGN_GENAIEVALUATIONSTORE": "modelevaluationstore-contextual-menu.html",
                    "LOCAL_MANAGED_FOLDER": "managed-folder-contextual-menu.html",
                    "FOREIGN_MANAGED_FOLDER": "managed-folder-contextual-menu.html",
                    "LOCAL_RETRIEVABLE_KNOWLEDGE": "knowledge-bank-contextual-menu.html",
                    "FOREIGN_RETRIEVABLE_KNOWLEDGE": "knowledge-bank-contextual-menu.html",
                    "ZONE": "zone-contextual-menu.html",
                    "MULTI": "multi-contextual-menu.html",
                }[type];

                ctxMenuScope.object = item;
                ctxMenuScope.hasZone = [...selectedNodes, item].find(it => it.nodeType === "ZONE") !== undefined;

                let menu = new ContextualMenu({
                    template: template,
                    scope: ctxMenuScope,
                    contextual: true,
                    controller: controller,
                });
                menu.openAtXY(x, y);
                return false;
            } else {
                ContextualMenu.prototype.closeAny();
                return true;
            }
        };

        FlowGraphSelection.clearSelection();

        scope.flowViews = FlowToolsRegistry.getFlowViews();
        const togglePreviewShortcut = 'shift+p';
        Mousetrap.bind("z", zoomOnLast);

        Mousetrap.bind("left", scope.moveLeft);
        Mousetrap.bind("right", scope.moveRight);
        Mousetrap.bind("up", scope.moveUp);
        Mousetrap.bind("down", scope.moveDown);

        Mousetrap.bind("-", scope.zoomOut);
        Mousetrap.bind("+", scope.zoomIn);
        Mousetrap.bind("=", scope.zoomIn); // For more practicity on qwerty keyboard without numpad
        // Wrap handler in $timeout to trigger a $digest cycle that triggers the watcher on hasPreview
        // Otherwise, the watcher won't be triggered until the next key event (particularly on shift and not on p)
        Mousetrap.bind("shift+p", () => { $timeout(scope.togglePreview); });
        Mousetrap.bind("mod+f", () => { ToolBridgeService.emitOmniboxFocus(); return false; });
        Mousetrap.bind("mod+z", () => { ToolBridgeService.emitOmniboxUndo(); return false; });
        Mousetrap.bind("mod+shift+z", () => { ToolBridgeService.emitOmniboxRedo(); return false; });
        const updateGraphDebounced = Debounce().withDelay(200,200).wrap(scope.updateGraph);
        const updateGraphDebouncedShort = Debounce().withDelay(0,0).withSpinner(false).wrap(scope.updateGraph);

        const deregister1 = $rootScope.$on('datasetsListChangedFromModal', updateGraphDebounced);
        const deregister2 = $rootScope.$on('taggableObjectTagsChanged', updateGraphDebounced);
        const deregister3 = $rootScope.$on('flowItemAddedOrRemoved', updateGraphDebounced);
        const deregister4 = $rootScope.$on('reloadGraph', (event, { zoomTo } = {}) => updateGraphDebounced(zoomTo));
        const deregister5 = $rootScope.$on('objectMetaDataChanged', updateGraphDebounced);
        const deregister6 = $rootScope.$on('discussionCountChanged', updateGraphDebounced);
        //const deregister7 = $rootScope.$on('unreadDiscussionsChanged', updateGraphDebounced); TODO: find a better solution to live-refresh the unread discussions
        const deregister8 = $rootScope.$on('featureGroupStatusChanged', updateGraphDebounced);
        const deregister9 = $rootScope.$on('zonesManualPositioningChanged', (event, { zoomTo } = {}) => updateGraphDebouncedShort(zoomTo, true, false));

        scope.$on("$destroy", function() {
            Mousetrap.unbind("z");
            Mousetrap.unbind("left");
            Mousetrap.unbind("right");
            Mousetrap.unbind("up");
            Mousetrap.unbind("down");
            Mousetrap.unbind("-");
            Mousetrap.unbind("+");
            Mousetrap.unbind("=");
            Mousetrap.unbind(togglePreviewShortcut);
            Mousetrap.unbind("mod+f");
            Mousetrap.unbind("mod+z");
            Mousetrap.unbind("mod+shift+z");
            deregister1();
            deregister2();
            deregister3();
            deregister4();
            deregister5();
            deregister6();
            //deregister7(); TODO: find a better solution to live-refresh the unread discussions
            deregister8();
            deregister9();

        });

        scope.$on('graphRendered', function graphRendered() {
            drawBuildInProgressIndicators(scope.svg, scope.nodesGraph);
            drawExposedIndicators(scope.svg, scope.nodesGraph);
            drawForbiddenIndicators(scope.svg, scope.nodesGraph);
            if (scope.nodesGraph) {

                WT1.tryEvent('project-flow-rendered', () => {
                    let payload = {
                        hasZones: scope.nodesGraph.hasZones,
                        isFlowEmpty: scope.isFlowEmpty,
                        zonesManualPositioning: scope.zonesManualPositioning,
                        canMoveZones: scope.canMoveZones,
                        count_nodes: scope.nodesGraph.nodesOnGraphCount
                    }
                    if (scope.nodesGraph.includedObjectsByType) {
                        for (const [typeName, countForType] of Object.entries(scope.nodesGraph.includedObjectsByType)) {
                            payload[typeName.toLowerCase()] = countForType;
                        }
                    }
                    return payload;
                });
                if (PageSpecificTourService.canStartFlowTour()) {
                    PageSpecificTourService.startFlowTour({ scope: scope, fromContext: 'flow' });
                    OpalsService.sendPageSpecificTourRecommendation(OpalsMessageService.PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.FLOW);
                } else {
                    OpalsService.sendPageSpecificTourRecommendation(null);
                }
            }
        });

        scope.$on('indexNodesDone', function indexNodesDone () {
            scope.cleanupCollapsedZones();
        });
    }
};
});

app.directive('lineageFlowExport', function() {
    return {
        restrict: 'EA',
        scope: true,
        controller: function($scope) {
            // Toolbox used by export-flow.js to prepare the data lineage flow to be exported
            $scope.exportToolbox = {
                checkLoading: function() {
                    return !$scope.svg;
                },
                removeDecorations: function() {}, // Not needed for data lineage as there is a dedicated page for the export
                getGraphBoundaries: function () {
                    const graphBBox = $scope.svg.querySelector("g.graph").getBBox();
                    return {
                        x: graphBBox.x,
                        y: graphBBox.y,
                        width: graphBBox.width,
                        height: graphBBox.height
                    };
                },
                adjustViewBox: function(x, y, width, height) {
                    $scope.svg.setAttribute('viewBox', [x, y, width, height].join(', '));
                },
                configureZones: function() {} // No zone on data lineage graph but it's needed as the generic flow export script use it
            }
        },
        link: function() {}
    }
});

app.directive('flowExportForm', function(GRAPHIC_EXPORT_OPTIONS, WT1, GraphicImportService) {
    return {
        replace: false,
        require: '^form',
        restrict: 'EA',
        scope: {
            params: '=',
            graphBoundaries: '='
        },
        templateUrl: '/templates/flow-editor/export-flow-form.html',
        link: function($scope, element, attrs, formCtrl) {
            WT1.event("flow-export-form-displayed", {});

            $scope.exportFormController = formCtrl;
            // Utilities that give us all the choices possible
            $scope.paperSizeMap = GRAPHIC_EXPORT_OPTIONS.paperSizeMap;
            $scope.orientationMap = GRAPHIC_EXPORT_OPTIONS.orientationMap;
            $scope.ratioMap = GRAPHIC_EXPORT_OPTIONS.ratioMap;
            $scope.paperInchesMap = GRAPHIC_EXPORT_OPTIONS.paperInchesMap;
            $scope.fileTypes = GRAPHIC_EXPORT_OPTIONS.fileTypes;
            $scope.tileScaleModes = GRAPHIC_EXPORT_OPTIONS.tileScaleModes;

            $scope.minResW = 500;
            $scope.minResH = 500;
            $scope.maxResW = 10000;
            $scope.maxResH = 10000;
            $scope.maxDpi = 300;

            let computeTileScale = function (tileScaleProps) {
                if (!tileScaleProps.enabled || tileScaleProps.percentage === undefined) {
                    return 1;
                } else {
                    return Math.max(1, tileScaleProps.percentage / 100)
                }
            };

            let computeBestTileScale = function(width, height) {
                const targetFactor = 1.0; // 1-to-1 between size of graph and exported image
                const xFactor = $scope.graphBoundaries.width / width;
                const yFactor = $scope.graphBoundaries.height / height;
                return Math.max(1, Math.ceil(Math.max(xFactor, yFactor) / targetFactor));
            };

            let capWidth = function(width) {
                return Math.min($scope.maxResW, Math.max($scope.minResW, width));
            };
            let capHeight = function(height) {
                return Math.min($scope.maxResH, Math.max($scope.minResH, height));
            };

            // Given an image width, height and tile scale, compute how many pages
            // will be required to render the whole graph
            let computeTileScaleSheets = function(width, height, tileScale) {
                if (width === undefined || height === undefined || tileScale == undefined) {
                    return {x: 0, y: 0, count: 0};
                }
                const sheetRatio = width / height;
                const graphRatio = $scope.graphBoundaries.width / $scope.graphBoundaries.height;
                let graphSheetWidth;
                let graphSheetHeight;
                if (sheetRatio < graphRatio) {
                    // Dominant width
                    graphSheetWidth = $scope.graphBoundaries.width / tileScale;
                    graphSheetHeight = graphSheetWidth / sheetRatio;
                } else {
                    // Dominant height
                    graphSheetHeight = $scope.graphBoundaries.height / tileScale;
                    graphSheetWidth = graphSheetHeight * sheetRatio;
                }
                const x = Math.max(1, Math.ceil($scope.graphBoundaries.width / graphSheetWidth));
                const y = Math.max(1, Math.ceil($scope.graphBoundaries.height / graphSheetHeight));
                const count = x * y;
                return {x: x, y: y, count: count};
            };

            // Compute the best width, height and tile scale for the exported image
            // for the supplied paper size and orientation.
            let setBestDimensions = function(authorizeTileScaling = true) {
                let exportFormat = $scope.params.exportFormat;

                let width, height;
                const sheetRatio = (exportFormat.orientation == "LANDSCAPE") ?
                    $scope.ratioMap[exportFormat.paperSize] :
                    1 / $scope.ratioMap[exportFormat.paperSize];
                const graphRatio = $scope.graphBoundaries.width / $scope.graphBoundaries.height;
                if (sheetRatio < graphRatio) {
                    // Dominant width
                    width = $scope.graphBoundaries.width;
                    height = width / sheetRatio;
                } else {
                    // Dominant height
                    height = $scope.graphBoundaries.height;
                    width = height * sheetRatio;
                }

                let tileScale = 1;
                let dpi = Math.max(width, height) / $scope.paperInchesMap[exportFormat.paperSize];
                if (authorizeTileScaling && dpi > $scope.maxDpi) {
                    width = (width * $scope.maxDpi) / dpi;
                    height = (height * $scope.maxDpi) / dpi;
                    tileScale = computeBestTileScale(width, height);
                }

                exportFormat.width = capWidth(Math.round(width));
                exportFormat.height = capHeight(Math.round(height));
                exportFormat.tileScale = tileScale;
            };

            // Parameters of the export
            $scope.params.exportFormat = {
                paperSize: "A4",
                orientation: "LANDSCAPE",
                fileType: "PDF",
                width: 1920,
                height: 1358,
                tileScale: 1,
            };
            let exportFormat = $scope.params.exportFormat;

            // Restore values from LocalStorage if they have been saved
            let savedFileType = localStorage.getItem("dku.flow.export.fileType");
            if (savedFileType && $scope.fileTypes.indexOf(savedFileType) >= 0) {
                exportFormat.fileType = savedFileType;
            }
            let savedPaperSize = localStorage.getItem("dku.flow.export.paperSize");
            if (savedPaperSize && $scope.paperSizeMap[savedPaperSize]) {
                exportFormat.paperSize = savedPaperSize;
            }
            if (savedPaperSize == "CUSTOM") {
                let savedWidth = localStorage.getItem("dku.flow.export.width");
                if (savedWidth && !isNaN(Number(savedWidth))) {
                    exportFormat.width = capWidth(Number(savedWidth));
                }
                let savedHeight = localStorage.getItem("dku.flow.export.height");
                if (savedHeight && !isNaN(Number(savedHeight))) {
                    exportFormat.height = capHeight(Number(savedHeight));
                }
            } else {
                let savedOrientation = localStorage.getItem("dku.flow.export.orientation");
                if (savedOrientation && $scope.orientationMap[savedOrientation]) {
                    exportFormat.orientation = savedOrientation;
                }
            }
            if (exportFormat.paperSize != "CUSTOM") {
                // Choose the best width & height and compute the tile scale
                setBestDimensions();
            }
            $scope.tileScale = {};
            $scope.tileScale.enabled = exportFormat.tileScale > 1;
            $scope.tileScale.percentage = exportFormat.tileScale * 100;
            $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);

            let onUpdatePaperSizeOrOrientation = function() {
                setBestDimensions();
                let exportFormat = $scope.params.exportFormat;
                $scope.tileScale.enabled = exportFormat.tileScale > 1;
                $scope.tileScale.percentage = exportFormat.tileScale * 100;
                $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
            };

            $scope.$watch('params.exportFormat.paperSize', function (newVal, oldVal) {
                if (newVal !== oldVal && newVal != 'CUSTOM') {
                    onUpdatePaperSizeOrOrientation();
                }
            });

            $scope.$watch('params.exportFormat.orientation', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    onUpdatePaperSizeOrOrientation();
                }
            });

            $scope.$watch('params.exportFormat.width', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
                }
            });

            $scope.$watch('params.exportFormat.height', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
                }
            });

            $scope.$watch('tileScale.enabled', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    if (newVal == true) {
                        // Try to keep the DPI of exported images around 300 dpi
                        if (exportFormat.paperSize != "CUSTOM") {
                            let dpi = Math.max(exportFormat.width, exportFormat.height) / GRAPHIC_EXPORT_OPTIONS.paperInchesMap[exportFormat.paperSize];
                            if (dpi > $scope.maxDpi) {
                                exportFormat.width = capWidth(Math.round(exportFormat.width * $scope.maxDpi / dpi));
                                exportFormat.height = capHeight(Math.round(exportFormat.height * $scope.maxDpi / dpi));
                            }
                        }
                        $scope.tileScale.percentage = computeBestTileScale(exportFormat.width, exportFormat.height) * 100;
                        exportFormat.tileScale = computeTileScale($scope.tileScale);
                    } else {
                        if (exportFormat.paperSize != "CUSTOM") {
                            setBestDimensions(false);
                        } else {
                            exportFormat.tileScale = 1;
                        }
                    }

                }
            });
            $scope.$watch('tileScale.percentage', function (newVal, oldVal) {
                if (newVal !== oldVal) {
                    let exportFormat = $scope.params.exportFormat;
                    exportFormat.tileScale = computeTileScale($scope.tileScale);
                    $scope.tileScale.sheets = computeTileScaleSheets(exportFormat.width, exportFormat.height, exportFormat.tileScale);
                }
            });
        }
    }
});

app.controller("ExportFlowModalController", function($scope, $stateParams, $rootScope, DataikuAPI, ActivityIndicator, FutureProgressModal, WT1, ViewsLocalStorage) {
    $scope.init = function (projectKey, graphBoundaries) {
        $scope.params = {};
        $scope.modalTitle = "Export Flow graph";
        $scope.projectKey = projectKey;
        $scope.graphBoundaries = graphBoundaries;
    };

    $scope.doExportFlow = function() {
        WT1.event("flow-exported", {});

        // Duplicate export format and add the zones export information
        let exportFormat = JSON.parse(JSON.stringify($scope.params.exportFormat));
        exportFormat.drawZones = $scope.drawZones.drawZones;
        exportFormat.collapsedZones = $scope.collapsedZones;
        const currentSearchQuery = ViewsLocalStorage.getQueryForProject(
            $rootScope.appConfig.login,
            $stateParams.projectKey
        );
        if (currentSearchQuery && currentSearchQuery.trim().length !== 0) {
            exportFormat.searchQuery = currentSearchQuery;
        }

        // Save options into LocalStorage to use them again for next export
        localStorage.setItem("dku.flow.export.fileType", exportFormat.fileType);
        localStorage.setItem("dku.flow.export.paperSize", exportFormat.paperSize);
        if (exportFormat.paperSize === "CUSTOM") {
            localStorage.setItem("dku.flow.export.width", exportFormat.width);
            localStorage.setItem("dku.flow.export.height", exportFormat.height);
        } else {
            localStorage.setItem("dku.flow.export.orientation", exportFormat.orientation);
        }
        // Starting with Puppeteer 13.7.0, it's no longer possible to use tiling with PDF output
        if (exportFormat.fileType === 'PDF') {
            exportFormat.tileScale = 1;
        }

        // Export the flow
        DataikuAPI.flow.export($scope.projectKey, exportFormat)
            .error(setErrorInScope.bind($scope))
            .success(function (resp) {
                FutureProgressModal.show($scope, resp, "Export Flow graph").then(function (result) {
                    if (result) { // undefined in case of abort
                        downloadURL(DataikuAPI.flow.getExportURL(result.projectKey, result.exportId));
                        ActivityIndicator.success("Flow graph export downloaded!", 5000);
                    } else {
                        ActivityIndicator.error("Export Flow failed", 5000);
                    }
                    $scope.resolveModal();
                });
            });
    }
});

app.controller("ExportDataLineageModalController", function($scope, DataikuAPI, ActivityIndicator, FutureProgressModal, WT1) {
    $scope.init = function (projectKey, graphBoundaries, datasetName, columnName) {
        $scope.params = {};
        $scope.modalTitle = "Export Lineage graph";
        $scope.exportButtonText = "Export";
        $scope.graphBoundaries = graphBoundaries;
        $scope.datasetName = datasetName;
        $scope.columnName = columnName;
        $scope.projectKey = projectKey;
    };

    $scope.doExportFlow = function() {
        WT1.tryEvent("data-lineage-flow-exported", () => ({}));

        // Duplicate export format and add export information
        let exportFormat = JSON.parse(JSON.stringify($scope.params.exportFormat));

        // Save options into LocalStorage to use them again for next export
        localStorage.setItem("dku.datalineage.export.fileType", exportFormat.fileType);
        localStorage.setItem("dku.datalineage.export.paperSize", exportFormat.paperSize);
        if (exportFormat.paperSize === "CUSTOM") {
            localStorage.setItem("dku.datalineage.export.width", exportFormat.width);
            localStorage.setItem("dku.datalineage.export.height", exportFormat.height);
        } else {
            localStorage.setItem("dku.datalineage.export.orientation", exportFormat.orientation);
        }
        // Starting with Puppeteer 13.7.0, it's no longer possible to use tiling with PDF output
        if (exportFormat.fileType === 'PDF') {
            exportFormat.tileScale = 1;
        }

        // Export the data lineage flow
        DataikuAPI.datalineage.export($scope.projectKey, $scope.datasetName, $scope.columnName, exportFormat)
            .error(setErrorInScope.bind($scope))
            .success(function (resp) {
                FutureProgressModal.show($scope, resp, "Export Data Lineage graph").then(function (result) {
                    if (result) { // undefined in case of abort
                        downloadURL(DataikuAPI.datalineage.getExportURL(result.projectKey, result.exportId));
                        ActivityIndicator.success("Data Lineage graph export downloaded!", 5000);
                    } else {
                        ActivityIndicator.error("Export Data Lineage failed", 5000);
                    }
                    $scope.resolveModal();
                });
            });
    }
});

app.controller('GenerateFlowDocumentModalController', function($scope, DataikuAPI, WT1, FutureWatcher, ProgressStackMessageBuilder) {
    $scope.init = function (projectKey) {
        $scope.projectKey = projectKey;
        $scope.template = { type: "DEFAULT" };
        $scope.newTemplate = {};
        $scope.state = "WAITING";
    };

    $scope.export = function() {
        if ($scope.template.type == "DEFAULT") {
            WT1.event("render-flow-documentation", {type: "default"});
            DataikuAPI.flow.docGenDefault($scope.projectKey)
            .success(watchFuture)
            .error(setErrorInScope.bind($scope));
        } else {
            WT1.event("render-flow-documentation", {type: "custom"});
            DataikuAPI.flow.docGenCustom($scope.newTemplate.file, $scope.projectKey, (e) => {
                // Unlikely to upload big files so no need to track progress
            })
            .then(watchFuture)
            .catch((error) => { setErrorInScope2.call($scope, error); });
        }

        function watchFuture(initialResponse) {
            $scope.initialResponse = angular.fromJson(initialResponse);
            $scope.data = undefined;
            $scope.state = "LOADING";

            FutureWatcher.watchJobId($scope.initialResponse.jobId)
            .success(function(response) {
                let exportId = response.result.exportId;
                $scope.data = response.result.data;
                $scope.state = "READY";

                $scope.text = "The flow documentation is ready.";
                $scope.errorOccurred = false;
                if ($scope.data.maxSeverity === 'WARNING') {
                    $scope.text += " Be aware that the placeholders which couldn't be resolved are not shown in the flow documentation.";
                } else if ($scope.data.maxSeverity === 'ERROR') {
                    $scope.text = "";
                    $scope.errorOccurred = true;
                }

                if (!$scope.errorOccurred) {
                    $scope.flowDocumentationURL = DataikuAPI.flow.getFlowDocumentationExportURL(exportId);
                }

            }).update(function(response) {
                $scope.futureResponse = response;
                $scope.percentage =  ProgressStackMessageBuilder.getPercentage(response.progress);
                $scope.stateLabels = ProgressStackMessageBuilder.build(response.progress, true);
            }).error(function(response, status, headers) {
               setErrorInScope.bind($scope)(response, status, headers);
            });
        }
    }

    $scope.download = function() {
        downloadURL($scope.flowDocumentationURL);
        WT1.event("download-flow-documentation");
        $scope.state = "DOWNLOADED";
    };

    $scope.abort = function() {
        DataikuAPI.futures.abort($scope.initialResponse.jobId).error(setErrorInScope.bind($scope));
        $scope.dismiss();
        WT1.event("abort-flow-documentation-rendering");
    }
});

app.directive('facetFilterableList', function ($filter) {
    return {
        scope: {items: '=', model: '=facetFilterableList', showAllItems: '=?', orderBy:'@'},
        transclude: true,
        link: function (scope, element, attr) {
            if (attr.filterFunction) {
                scope.filterFunction = scope.$parent.$eval(attr.filterFunction);
            } else {
                scope.filterFunction = $filter('filter');
            }
            scope.model = scope.model || [];
            scope.onFacetSearchKeyDown = function (e) {
                if (e.keyCode === 27) { // ESC key
                    e.target.blur();
                    angular.element(e.target).scope().$parent.showInput = false;
                    angular.element(e.target).scope().$parent.facetValueSearch = '';
                }
            };
        },
        templateUrl: '/templates/flow-editor/facet-filterable-list.html'
    }
});


app.directive('multiItemsRightColumnSummary', function($controller, $rootScope, $stateParams,
    DataikuAPI, Fn, TaggableObjectsUtils, RecipeDescService, CodeEnvsService, SavedModelsService,
    FlowGraphSelection, SelectablePluginsService, WatchInterestState, FlowGraph, SubFlowCopyService, WT1, translate, PluginCategoryService) {

    return {
        templateUrl:'/templates/flow-editor/multi-items-right-column-summary.html',

        link: function(scope, element, attrs) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            const getType = attrs.singleType ? () => attrs.singleType : item => item.nodeType;
            const getSelectedItems = attrs.selectedItems ? () => scope.$eval(attrs.selectedItems) : FlowGraphSelection.getSelectedNodes;
            const newItemsWatch = attrs.selectedItems ? () => scope.$eval(attrs.selectedItems) : 'rightColumnItem';

            scope.projectsWithManageExposedElementsPrivilege = null;
            if (!scope.canWriteProject()) {
                // No need to load this information unless the user doesn't have write privilege
                DataikuAPI.projects.listHeads("MANAGE_EXPOSED_ELEMENTS").success(function(projects) {
                    scope.projectsWithManageExposedElementsPrivilege = projects;
                }).error(setErrorInScope.bind(scope));
            }

            function getCountByNodeType(selectedNodes) {
                let ret = {};
                selectedNodes.forEach(function(item) {
                    const type = getType(item);
                    ret[type] = (ret[type] || 0) + 1;
                });
                return ret;
            }
            function getCountByTaggableType(selectedNodes) {
                let ret = {};
                selectedNodes.forEach(function(item) {
                    const taggableType = TaggableObjectsUtils.fromNodeType(getType(item));
                    ret[taggableType] = (ret[taggableType] || 0) + 1;
                });
                return ret;
            }

            scope.getTaggableTypeMap = function () {
                let ret = {};
                scope.getSelectedNodes().forEach(function (item) {
                    let type = TaggableObjectsUtils.fromNodeType(getType(item));
                    if (ret.hasOwnProperty(type)) {
                        ret[type].push(getSmartName(item));
                    } else {
                        ret[type] = [getSmartName(item)];
                    }
                })
                return ret;
            }

            function count(nodeType) {
                return scope.selection.countByNodeType[nodeType] || 0;
            }

            function selectedNodes() {
                return scope.selection.selectedObjects;
            }
            scope.getSelectedNodes = selectedNodes;

            function isAll(nodeTypes) {
                return function() {
                    const total = scope.selection.selectedObjects.length;
                    return total > 0 && nodeTypes.map(count).reduce(Fn.SUM) == total;
                };
            }
            function containsNodeType(nodeTypes) {
                return () => scope.selection.selectedObjects.length > 0 && nodeTypes.map(count).reduce(Fn.SUM) > 0;
            }
            function allHaveFlag(propName) {
                return function() {
                    const total = scope.selection.selectedObjects.length;
                    return total > 0 && scope.selection.selectedObjects.filter(Fn.prop(propName)).length == total
                };
            }
            // TODO @labeling right panel : multi items
            scope.isAllRecipes = isAll(['RECIPE']);
            scope.containsRecipes = containsNodeType(['RECIPE']);
            scope.isAllContinuousRecipes = allHaveFlag("continuous");
            scope.isAllDatasets = isAll(['LOCAL_DATASET', 'FOREIGN_DATASET']);
            scope.containsDatasets = containsNodeType(['LOCAL_DATASET', 'FOREIGN_DATASET']);
            scope.isAllFolders = isAll(['LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER']);
            scope.isAllStreamingEndpoints = isAll(['LOCAL_STREAMING_ENDPOINT']);
            scope.isAllModels = isAll(['LOCAL_SAVEDMODEL', 'FOREIGN_SAVEDMODEL']);
            scope.isAllEvaluationStores = isAll(['LOCAL_MODELEVALUATIONSTORE', 'FOREIGN_MODELEVALUATIONSTORE', 'LOCAL_GENAIEVALUATIONSTORE', 'FOREIGN_GENAIEVALUATIONSTORE']);
            scope.isAllCodeStudios = isAll(['CODE_STUDIO']);
            scope.isAllZones = isAll(['ZONE']);
            scope.isAllProjects = isAll(['PROJECT']);
            scope.isAllLocal = isAll(['RECIPE', 'LOCAL_DATASET', 'LOCAL_MANAGED_FOLDER', 'LOCAL_SAVEDMODEL', 'LOCAL_MODELEVALUATIONSTORE', 'LOCAL_GENAIEVALUATIONSTORE', 'LOCAL_RETRIEVABLE_KNOWLEDGE']);
            scope.isAllForeign = isAll(['FOREIGN_DATASET', 'FOREIGN_MANAGED_FOLDER', 'FOREIGN_SAVEDMODEL', 'FOREIGN_MODELEVALUATIONSTORE', 'FOREIGN_GENAIEVALUATIONSTORE', 'FOREIGN_RETRIEVABLE_KNOWLEDGE']);
            scope.isAllComputables = isAll(['LOCAL_DATASET', 'FOREIGN_DATASET', 'LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER', 'LOCAL_SAVEDMODEL', 'FOREIGN_SAVEDMODEL', 'LOCAL_MODELEVALUATIONSTORE', 'FOREIGN_MODELEVALUATIONSTORE', 'LOCAL_GENAIEVALUATIONSTORE', 'FOREIGN_GENAIEVALUATIONSTORE', 'LOCAL_RETRIEVABLE_KNOWLEDGE', 'FOREIGN_RETRIEVABLE_KNOWLEDGE']);
            scope.isAllDatasetsAndFolders = isAll(['LOCAL_DATASET', 'FOREIGN_DATASET', 'LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER']);
            scope.isAllSharable = isAll(['LOCAL_DATASET', 'DASHBOARD', 'WEB_APP']);
            scope.hasVisualSection = scope.isAllDatasets || scope.isAllFolders || scope.isDatasetAndFolder;

            scope.getSingleSelectedDataset = function() {
                const candidates = selectedNodes().filter(({nodeType}) => ['LOCAL_DATASET', 'FOREIGN_DATASET'].includes(nodeType));
                return candidates.length == 1 ? candidates[0] : null;
            }

            scope.getSingleSelectedModel = function() {
                const candidates = selectedNodes().filter(({nodeType}) => ['LOCAL_SAVEDMODEL', 'FOREIGN_SAVEDMODEL'].includes(nodeType));
                return candidates.length == 1 ? candidates[0] : null;
            }

            scope.getSingleSelectedFolder = function() {
                const candidates = selectedNodes().filter(({nodeType}) => ['LOCAL_MANAGED_FOLDER', 'FOREIGN_MANAGED_FOLDER'].includes(nodeType));
                return candidates.length == 1 ? candidates[0] : null;
            }

            scope.isPredictionModel = function() {
                return scope.singleSelectedModelInfos
                    && scope.singleSelectedModelInfos.model.miniTask
                    && scope.singleSelectedModelInfos.model.miniTask.taskType == 'PREDICTION'
                    && scope.singleSelectedModelInfos.model.miniTask.backendType !== 'VERTICA';
            }

            scope.isClusteringModel = function() {
                return scope.singleSelectedModelInfos
                    && scope.singleSelectedModelInfos.model.miniTask
                    && scope.singleSelectedModelInfos.model.miniTask.taskType == 'CLUSTERING'
                    && scope.singleSelectedModelInfos.model.miniTask.backendType !== 'VERTICA';
            }

            scope.isLLM = function() {
                return scope.singleSelectedModelInfos
                    && (SavedModelsService.isAgent(scope.singleSelectedModelInfos.model)
                        || SavedModelsService.isRetrievalAugmentedLLM(scope.singleSelectedModelInfos.model)
                        || SavedModelsService.isLLMGeneric(scope.singleSelectedModelInfos.model));
            };

            scope.isDatasetAndModel = function() {
                return selectedNodes().length == 2 && scope.getSingleSelectedDataset() && scope.getSingleSelectedModel();
            }

            scope.isDatasetAndFolder = function() {
                return selectedNodes().length == 2 && scope.getSingleSelectedDataset() && scope.getSingleSelectedFolder();
            }

            scope.isTwoDatasets = function() {
                return selectedNodes().length == 2 && scope.isAllDatasets();
            }

            scope.isAllMetastoreAware = function() {
                const total = selectedNodes().length;
                const hiveRecipes = selectedNodes().filter(n => TaggableObjectsUtils.isHDFSAbleType(n.datasetType)).length;
                return total > 0 && hiveRecipes == total;
            };
            scope.isAllImpalaRecipes = function() {
                const total = selectedNodes().length;
                const impalaRecipes = selectedNodes().filter(n => (n.recipeType||n.type) == 'impala').length;
                return total > 0 && impalaRecipes == total;
            };
            scope.isAllPythonCodeEnvSelectableRecipes = function() {
                const total = selectedNodes().length;
                const codeEnvSelectableRecipes = selectedNodes().filter(n => (n.recipeType||n.type) && CodeEnvsService.canPythonCodeEnv(n)).length;
                return total > 0 && codeEnvSelectableRecipes == total;
            };
            scope.isAllRCodeEnvSelectableRecipes = function() {
                const total = selectedNodes().length;
                const codeEnvSelectableRecipes = selectedNodes().filter(n => (n.recipeType||n.type) && CodeEnvsService.canRCodeEnv(n)).length;
                return total > 0 && codeEnvSelectableRecipes == total;
            };
            scope.isAllHiveRecipes = function() {
                const total = selectedNodes().length;
                const hiveRecipes = selectedNodes().filter(n => (n.recipeType||n.type) == 'hive').length;
                return total > 0 && hiveRecipes == total;
            };

            scope.isAllManaged = function() {
                const total = selectedNodes().length;
                const managed = selectedNodes().filter(n => n.managed).length;
                return total > 0 && managed == total;
            };
            scope.isAllWatched = function() {
                const total = selectedNodes().length;
                const watched = selectedNodes().filter(n => n.interest && WatchInterestState.isWatching(n.interest.watching)).length;
                return total > 0 && watched == total;
            };
            scope.isAllStarred = function() {
                const total = selectedNodes().length;
                const starred = selectedNodes().filter(n => n.interest && n.interest.starred).length;
                return total > 0 && starred == total;
            };
            scope.isAllVirtualizable = function() {
                return selectedNodes().map(x => !!x.virtualizable).reduce((a,b) => a && b, true);
            };
            scope.canManageExposedElementsOnAllOriginalProjects = function() {
                if (scope.projectsWithManageExposedElementsPrivilege == null) {
                    return false;
                }

                const projectKeysWithManageExposedElementsPrivilege = scope.projectsWithManageExposedElementsPrivilege.map(p => p.projectKey);
                const originalProjectKeys = new Set(selectedNodes().map(n => n.projectKey));
                return [...originalProjectKeys].every(key => projectKeysWithManageExposedElementsPrivilege.includes(key));
            };

            scope.anyPipelineTypeEnabled = function() {
                return $rootScope.projectSummary.sparkPipelinesEnabled || $rootScope.projectSummary.sqlPipelinesEnabled;
            };

            function showVirtualizationAction(showDeactivate) {
                return function() {
                    return scope.isProjectAnalystRW()
                        && scope.isAllDatasets()
                        && scope.isAllLocal()
                        && showDeactivate === scope.isAllVirtualizable();
                }
            }
            scope.showAllowVirtualizationAction = showVirtualizationAction(false);
            scope.showStopVirtualizationAction = showVirtualizationAction(true);


            scope.anyMultiEngineRecipe = function() {
                function isMultiEngine(recipeType) {
                    const desc = RecipeDescService.getDescriptor(recipeType);
                    return !!desc && desc.isMultiEngine;
                }
                return !!selectedNodes().filter(node => isMultiEngine(node.recipeType||node.type)).length;
            };

            scope.anyImpala = function() {
                return !!selectedNodes().filter(n => (n.recipeType||n.type) == 'impala').length;
            };

            scope.anyHive = function() {
                return !!selectedNodes().filter(n => (n.recipeType||n.type) == 'hive').length;
            };

            scope.anyCanSpark = function() {
                return !!selectedNodes().filter(node => scope.canSpark(node)).length;
            };

            scope.allAreSparkNotSQLRecipes = function() {
                return selectedNodes().every(node => ['spark_scala','pyspark','sparkr'].indexOf(node.recipeType||node.type) >= 0);
            };

            scope.anyCanSparkPipeline = function() {
                return selectedNodes().some(node => scope.canSparkPipeline(node));
            };

            scope.anyCanSqlPipeline = function() {
                return selectedNodes().some(node => scope.canSqlPipeline(node));
            };

            scope.allAutoTriggersDisabled = function() {
                return scope.getAutoTriggerDisablingReason($rootScope.appConfig, $rootScope.projectSummary);
            };

            scope.autoTriggersObjects = function(autoTriggerStatus, objects) {
                objects.forEach(function(object){
                    object.active = autoTriggerStatus;
                    scope.toggleActive(object);
                })
            };

            scope.isAllUnshareable = function() {
                const total = selectedNodes().length;
                const unshareables = selectedNodes().filter(n => n.usedByZones && n.usedByZones.length && !n.successors.length).length;
                return total > 0 && unshareables == total;
            }

            scope.canPublishAllToDataCollection = function() {
                // we already know isAllDataset is true, nodes are either 'LOCAL_DATASET' or 'FOREIGN_DATASET'
                // we only check global auth & local auth, as foreign would require to much info (it will trigger a warning on modal open if there is an issue)
                return $rootScope.appConfig.globalPermissions.mayPublishToDataCollections &&
                    (count('LOCAL_DATASET') === 0 || $rootScope.projectSummary.canPublishToDataCollections);
            }

            scope.getSelectedObjectsZones = function() {
                return getSelectedItems().map(n => n.usedByZones[0]);
            }

            scope.getCommonZone = function () {
                const nodesSelected = selectedNodes();
                return nodesSelected.length ? nodesSelected[0].ownerZone : null;
            };

            function getSmartName(it) {
                return it.projectKey == $stateParams.projectKey ? it.name : it.projectKey+'.'+it.name;
            }
            scope.getSmartNames = function () {
                return selectedNodes().map(getSmartName);
            };

            scope.clearSelection = function() {
                FlowGraphSelection.clearSelection();
            };

            scope.refreshData = function() {
                let selectedNodes = getSelectedItems();
                scope.selection = {
                    selectedObjects: selectedNodes,
                    taggableType: TaggableObjectsUtils.getCommonType(selectedNodes, node => TaggableObjectsUtils.fromNodeType(getType(node))),
                    countByNodeType: getCountByNodeType(selectedNodes),
                    countByTaggableType: getCountByTaggableType(selectedNodes)
                };
                scope.usability = scope.computeActionsUsability();
                scope.selectablePlugins =  scope.isAllComputables(selectedNodes) ? SelectablePluginsService.listSelectablePlugins(scope.selection.countByTaggableType) : [];
                let availableCategories = []
                if (scope.hasVisualSection) {
                    availableCategories.push('visual')
                }
                if (scope.isAllComputables) {
                    availableCategories.push('code')
                }
                scope.noRecipesCategoryPlugin = PluginCategoryService.standardCategoryPlugins(scope.selectablePlugins, availableCategories);

                scope.singleSelectedDataset = scope.getSingleSelectedDataset();
                scope.singleSelectedFolder = scope.getSingleSelectedFolder();
                scope.selectedFolderIsSharepoint = scope.singleSelectedFolder?.folderType === "SharePointOnline";
                scope.refreshSingleSelectedModelInfos();
            };

            scope.refreshSingleSelectedModelInfos = function() {
                if(!scope.getSingleSelectedModel()) {
                    scope.singleSelectedModelInfos = null;
                    scope.singleSelectedModel = null;
                }
                if(scope.getSingleSelectedModel() == scope.singleSelectedModel) {
                    return;
                }
                scope.singleSelectedModel = scope.getSingleSelectedModel();
                scope.singleSelectedModelInfos = null;
                let projectKey = scope.singleSelectedModel.projectKey;
                let name = scope.singleSelectedModel.name;
                DataikuAPI.savedmodels.getFullInfo($stateParams.projectKey, getSmartName(scope.singleSelectedModel)).success(data => {
                    if (!scope.singleSelectedModel || scope.singleSelectedModel.projectKey != projectKey || scope.singleSelectedModel.name != name) {
                        return; // too late, the selected model has changed in the meantime
                    }
                    scope.singleSelectedModelInfos = data;
                    scope.singleSelectedModelInfos.zone = (scope.singleSelectedModel.usedByZones || [])[0] || scope.singleSelectedModel.ownerZone;
                }).error(setErrorInScope.bind(scope));
            };

            scope.filterSelection = function(taggableType) {
                FlowGraphSelection.filterByTaggableType(taggableType);
                scope.refreshData();
            }

            scope.refreshData();
            scope.$watch(newItemsWatch, scope.refreshData);

            scope.collapseSelectedZones = () => {
                scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'collapseAll');
            }

            scope.expandSelectedZones = () => {
                scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'expandAll');
            }

            scope.isAllZonesExpanded = function() {
                const allZones = scope.selection.selectedObjects.filter(so => getType(so) === 'ZONE');
                return allZones.filter(z => z.customData.isCollapsed === false).length === allZones.length;
            };

            scope.isAllZonesCollapsed = function() {
                const allZones = scope.selection.selectedObjects.filter(so => getType(so) === 'ZONE');
                return allZones.filter(z => z.customData.isCollapsed === true).length === allZones.length;
            };

            scope.copyZone = () => {
                const startCopyTool = scope.startTool('COPY',  {preselectedNodes: FlowGraphSelection.getSelectedNodes().map(n => n.id)});
                startCopyTool.then(() => {
                    const selectedTaggableObjectRefs = FlowGraphSelection.getSelectedTaggableObjectRefs();
                    const itemsByZones = FlowGraph.nodesByZones((node) => TaggableObjectsUtils.fromNode(node));
                    SubFlowCopyService.start(selectedTaggableObjectRefs, itemsByZones, scope.stopAction);
                });
            };

            scope.insertRecipeBetween = function() {
                scope.showInsertRecipeModal(scope.selectedDataset, scope.selectedRecipes)
                    .then(selectedRecipe => WT1.event('rightpanelrecipe_actions_insertrecipe', {position: 'between', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
            };

            scope.canInsertRecipeBetween = function() {
                const selectedNodes = getSelectedItems();
                const selectedDatasets = [];
                const selectedRecipes =  [];
                for (const node of selectedNodes) {
                    if (node.nodeType === "LOCAL_DATASET" || node.nodeType === "FOREIGN_DATASET") {
                        selectedDatasets.push(node);
                    } else if (node.nodeType === "RECIPE") {
                        selectedRecipes.push(node);
                    }
                }
                scope.selectedDataset = selectedDatasets[0];    // Because we know for sure only one dataset is selected
                scope.selectedRecipes = selectedRecipes;
                return FlowGraphSelection.canInsertRecipeBetweenSelection(selectedNodes, selectedDatasets, selectedRecipes);
            }

            scope.getInsertRecipeBetweenTooltip = function() {
                if (!scope.canWriteProject()) return translate('PROJECT.PERMISSIONS.WRITE_ERROR', 'You don\'t have the permissions to write to this project');
                if (!scope.canInsertRecipeBetween()) return translate('PROJECT.FLOW.RIGHT_PANEL.INSERT_RECIPE.ERROR', 'The current selection does not allow to insert a new recipe. Select a dataset and a specific downstream recipe to insert a recipe in between.');
                return translate('PROJECT.FLOW.RIGHT_PANEL.INSERT_RECIPE.TOOLTIP', 'Insert recipe (and corresponding output dataset) between selected items');
            }
        }
    }
});


app.controller('_FlowContextMenus', function($scope, $state, $stateParams, $controller, WT1, GlobalProjectActions, FlowGraph, FlowGraphSelection, FlowGraphFolding, TaggableObjectsUtils, Logger) {

    WT1.event("flow-context-menu-open");

    $controller('_TaggableObjectsCapabilities', {$scope: $scope});
    $controller('_TaggableObjectsMassActions', {$scope: $scope});

    $scope.toggleTab = tabName => {
        FlowGraphSelection.clearSelection();
        FlowGraphSelection.onItemClick($scope.object);

       $scope.standardizedSidePanel.toggleTab(tabName);
    };

    $scope.getSelectedTaggableObjectRefs = function() {
        return [TaggableObjectsUtils.fromNode($scope.object)];
    };

    $scope.computeMovingImpact = function() {
        let realNode = $scope.object.usedByZones.length ? FlowGraph.node(`zone__${$scope.object.ownerZone}__${$scope.object.realId}`) : $scope.object;
        var computedImpact = [];
        function addSuccessors(node) {
            if (node.nodeType != "RECIPE") return;
            let successors = node.successors;
            successors.forEach(function(successor) {
                if (successor == realNode.id) return;
                computedImpact.push(TaggableObjectsUtils.fromNode(FlowGraph.node(successor)));
            });
        }

        let predecessor = realNode.predecessors[0];
        if (predecessor && realNode.nodeType != "RECIPE" && !realNode.isHiddenLinkTarget) {
            computedImpact.push(TaggableObjectsUtils.fromNode(FlowGraph.node(predecessor)));
            addSuccessors(FlowGraph.node(predecessor));
        }

        addSuccessors(realNode);
        return computedImpact;
    }

    $scope.zoomToOtherZoneNode = function(zoneId) {
        const otherNodeId = $scope.object.id.replace(/zone__.+?__/, "zone__" + zoneId + "__");
        if ($stateParams.zoneId) {
            $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId, id: graphVizUnescape(otherNodeId) }));
        }
        else {
            $scope.zoomGraph(otherNodeId);
            FlowGraphSelection.clearSelection();
            FlowGraphSelection.onItemClick($scope.nodesGraph.nodes[otherNodeId]);
        }
    }

    $scope.$state = $state;
    $scope.$stateParams = $stateParams;
    $scope.othersZones = FlowGraph.nodeSharedBetweenZones($scope.object) ? Array.from(FlowGraph.nodeSharedBetweenZones($scope.object)) : null;

    $scope.startPropagateToolFromRecipe = function(node) {
        const predecessorNodeId = node.predecessors[0];
        const predecessorNode = FlowGraph.get().nodes[predecessorNodeId];
        $scope.startTool('PROPAGATE_SCHEMA', {projectKey: predecessorNode.projectKey, datasetName: predecessorNode.name});
    }

    $scope.selectSuccessors = function() {
        WT1.event("flow-context-menu-select-successors");
        FlowGraphSelection.selectSuccessors($scope.object);
    };

    $scope.selectPredecessors = function() {
        WT1.event("flow-context-menu-select-predecessors");
        FlowGraphSelection.selectPredecessors($scope.object);
    };

    $scope.hasPredecessorsInOtherZone = function(object) {
        return !$stateParams.zoneId && FlowGraphSelection.hasPredecessorsInOtherZone(object);
    }

    $scope.hasSuccessorsInOtherZone = function(object) {
        return !$stateParams.zoneId && FlowGraphSelection.hasSuccessorsInOtherZone(object);
    }

    $scope.foldSuccessors = function() {
        WT1.event("flow-context-menu-fold", {direction:'successors'});
        FlowGraphFolding.foldSuccessors($scope.object);
    };

    $scope.foldPredecessors = function() {
        WT1.event("flow-context-menu-fold", {direction: 'predecessors'});
        FlowGraphFolding.foldPredecessors($scope.object);
    };

    $scope.previewSelectSuccessors = function(object) {
        FlowGraphFolding.previewSelect($scope.object, "successors");
    };

    $scope.previewSelectPredecessors = function(object) {
        FlowGraphFolding.previewSelect($scope.object, "predecessors");
    };

    $scope.previewFoldSuccessors = function(object) {
        FlowGraphFolding.previewFold($scope.object, "successors");
    };

    $scope.previewFoldPredecessors = function(object) {
        FlowGraphFolding.previewFold($scope.object, "predecessors");
    };

    $scope.endPreviewBranch = function() {
        FlowGraphFolding.endPreviewBranch();
    };

    $scope.deleteFlowItem = function() {
        WT1.event('flow-context-menu-delete');

        const type = TaggableObjectsUtils.fromNodeType($scope.object.nodeType);
        const id = $scope.object.name;
        const displayName = $scope.object.description;
        GlobalProjectActions.deleteTaggableObject($scope, type, id, displayName)
            .then(FlowGraphSelection.clearSelection);
    };

    $scope.canInsertRecipeAfter = function() {
        return $scope.object.successors && $scope.object.successors.length && $scope.object.successors.every(successor => FlowGraph.node(successor).nodeType === 'RECIPE');
    }

    $scope.insertRecipeAfter = function() {
        $scope.showInsertRecipeModal($scope.object)
            .then(selectedRecipe => WT1.event('flow-context-menu-insertrecipe', {position: 'after', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
    }

    $scope.isDeleteAndReconnectAllowed = false;

    // Call async function to set isDeleteAndReconnectAllowed appropriately (do not use directly from html)
    $scope.canDeleteAndReconnectObject($scope.object).then(function(result) {
        $scope.isDeleteAndReconnectAllowed = result;
    }).catch(function(error) {
        Logger.warn(`Problem determining whether to show delete & reconnect option for ${$scope.object.id}:`, error);
    });

    $scope.deleteAndReconnectRecipe = function(wt1_event_name) {
        $scope.doDeleteAndReconnectRecipe($scope.object, wt1_event_name);
    }
});


app.controller("SavedModelContextualMenuController", function($scope, $state, $controller, WT1, SavedModelRenameService) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.renameSavedModel = function(savedModel) {
        SavedModelRenameService.renameSavedModel({
            scope: $scope,
            state: $state,
            projectKey: savedModel.projectKey,
            savedModelId: savedModel.name,
            savedModelName: savedModel.description
        });
    }

    $scope.trainThisModel = function() {
        WT1.event('flow-context-menu-train');
        $scope.trainModel($scope.object.projectKey, $scope.object.name);
    };
});

app.controller("ModelEvaluationStoreContextualMenuController", function ($scope, $controller, $state, ModelEvaluationStoreRenameService) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.renameMES = function(modelEvaluationStore) {
        ModelEvaluationStoreRenameService.renameModelEvaluationStore({
            scope: $scope,
            state: $state,
            projectKey: modelEvaluationStore.projectKey,
            modelEvaluationStoreId: modelEvaluationStore.name,
            modelEvaluationStoreName: modelEvaluationStore.description
        });
    };
});

app.controller("ManagedFolderContextualMenuController", function($scope, $controller, $state, WT1, ManagedFolderRenameService) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.renameManagedFolder = function (managedFolder) {
        const projectKey = managedFolder.projectKey;
        const managedFolderCurrentName = managedFolder.description;
        const scopeWithObjectType = $scope.$new();
        scopeWithObjectType.objectType = "MANAGED_FOLDER";

        ManagedFolderRenameService.renameManagedFolder({
            scope: scopeWithObjectType,
            state: $state,
            projectKey,
            managedFolderId: managedFolder.name,
            managedFolderCurrentName
        });
    };

    $scope.buildThis = function() {
        WT1.event('flow-context-menu-build');
        $scope.buildManagedFolder($scope.object.projectKey, $scope.object.id);
    };
});

app.controller("KnowledgeBankContextualMenuController", function($scope, $controller, WT1, DatasetsService) {
    $controller('_FlowContextMenus', {$scope: $scope});
});

app.controller("ZoneContextualMenuController", function($scope, $rootScope, $controller, DataikuAPI, TaggableObjectsUtils, $stateParams, CreateModalFromTemplate, FlowGraph, FlowGraphSelection) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.deleteZone = () => {
        DataikuAPI.flow.zones.delete($stateParams.projectKey, $scope.object.name).success(() => {
            if ($stateParams.zoneId) {
                $scope.zoomOutOfZone();
            } else {
                $scope.$emit('reloadGraph');
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.openZone = items => {
        const zoneToOpen = items.map(ref => ref.id)[0];
        $scope.zoomOnZone(zoneToOpen);
    }

    $scope.collapseAllZones = () => {
        const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
        $scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'collapseAll');
    }

    $scope.expandAllZones = () => {
        const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
        $scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'expandAll');
    }

    $scope.collapseSelectedZones = () => {
        $scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'collapseAll');
    }

    $scope.expandSelectedZones = () => {
        $scope.toggleZoneCollapse(FlowGraphSelection.getSelectedTaggableObjectRefs(), 'expandAll');
    }

    $scope.editZone = () => {
        CreateModalFromTemplate("/templates/zones/edit-zone-modal.html", $scope, null, function(newScope){
            newScope.uiState = {
                color: $scope.object.customData.color,
                name: $scope.object.description
            };

            newScope.go = function(){
                DataikuAPI.flow.zones.edit($stateParams.projectKey, $scope.object.name, newScope.uiState.name, newScope.uiState.color).success(function () {
                    $scope.$emit('reloadGraph');
                    if ($stateParams.zoneId) {
                        $rootScope.$emit("zonesListChanged", newScope.uiState.name);
                    }
                    newScope.dismiss()
                }).error(setErrorInScope.bind(newScope));
            }
        });
    }
});


app.controller("DatasetContextualMenuController", function($scope, $rootScope, $controller, WT1, DataikuAPI, TaggableObjectsUtils) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.buildThisDataset = function() {
        WT1.event('flow-context-menu-build');
        $scope.buildDataset($scope.object.projectKey, $scope.object.name, $scope.object.predecessors.length, $scope.object.successors.length);
    };

    $scope.reloadSchema = function() {
        WT1.event('flow-context-menu-dataset-reload-schema');
        DataikuAPI.datasets.reloadSchema($scope.object.projectKey, $scope.object.name).error(setErrorInScope.bind($scope));
    };

    $scope.markAsBuilt = function() {
        WT1.event('flow-context-menu-mark-as-built');
        DataikuAPI.datasets.markAsBuilt([TaggableObjectsUtils.fromNode($scope.object)]).then(function() {
            $rootScope.$emit('reloadGraph');
        }, setErrorInScope.bind($scope));
    };

});


app.controller("ForeignDatasetContextualMenuController", function($scope, $controller, $state, WT1, DataikuAPI) {
    $controller('_FlowContextMenus', {$scope: $scope});
});


app.controller("StreamingEndpointContextualMenuController", function($scope, $rootScope, $controller, WT1, DataikuAPI, TaggableObjectsUtils) {
    $controller('_FlowContextMenus', {$scope: $scope});
});

app.controller("RecipeContextualMenuController", function($scope, $controller, $stateParams, WT1, ComputableSchemaRecipeSave) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $scope.propagateSchema = function() {
        WT1.event('flow-context-menu-propagate-schema');
        ComputableSchemaRecipeSave.handleSchemaUpdateFromAnywhere($scope, $stateParams.projectKey, $scope.object.name)
    }
});

app.controller("LabelingTaskContextualMenuController", function($scope, $controller, $stateParams, WT1, ComputableSchemaRecipeSave) {
    $controller('_FlowContextMenus', {$scope: $scope});
    // TODO @labeling implement propagateSchema
});


app.controller("MultiContextualMenuController", function($scope, $controller, WT1, FlowGraphSelection, FlowGraphFolding, TaggableObjectsService, TaggableObjectsUtils, FlowGraph) {
    $controller('_FlowContextMenus', {$scope: $scope});

    $controller('_TaggableObjectsMassActions', {$scope: $scope});
    $controller('_TaggableObjectsCapabilities', {$scope: $scope});

    $scope.getSelectedTaggableObjectRefs = FlowGraphSelection.getSelectedTaggableObjectRefs;

    $scope.computeMovingImpact = function() {
        const computedImpact = [];
        const movingItems = FlowGraphSelection.getSelectedTaggableObjectRefs();

        function addSuccessors(node, original) {
            if (!['RECIPE', 'LABELING_TASK'].includes(node.nodeType)) return;
            node.successors.forEach(function(successor) {
                let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(successor));
                if (original && successor === original.id
                    || movingItems.some(it => it.id === newTaggableObjectRef.id)
                    || computedImpact.some(it => it.id === newTaggableObjectRef.id)
                ) return;
                computedImpact.push(newTaggableObjectRef);
            });
        }
        function computeImpact(node) {
            let predecessor = node.predecessors[0];
            if (predecessor && !['RECIPE', 'LOCAL_SAVEDMODEL', 'LABELING_TASK'].includes(node.nodeType)) {
                let newTaggableObjectRef = TaggableObjectsUtils.fromNode(FlowGraph.node(predecessor));
                if (computedImpact.some(it => it.id === newTaggableObjectRef.id)) return;
                if (!movingItems.some(it => it.id === newTaggableObjectRef.id)) {
                    computedImpact.push(newTaggableObjectRef);
                }
                addSuccessors(FlowGraph.node(predecessor), node);
            }

            addSuccessors(node);
        }

        FlowGraphSelection.getSelectedNodes().forEach(function(node) {
            let realNode = node.usedByZones.length ? FlowGraph.node(`zone__${node.ownerZone}__${node.realId}`) : node;
            computeImpact(realNode);
        });
        return computedImpact;
    }

    $scope.selectedObjectsZones = FlowGraphSelection.getSelectedNodes().map(n => n.usedByZones[0]);

    $scope.deleteFlowItems = function() {
        WT1.event('flow-context-menu-delete-multi');

        TaggableObjectsService.delete(FlowGraphSelection.getSelectedTaggableObjectRefs())
            .then(FlowGraphSelection.clearSelection);
    };

    $scope.selectSuccessors = function() {
        WT1.event("flow-context-menu-select-successors-multi");
        FlowGraphSelection.getSelectedNodes().forEach(FlowGraphSelection.selectSuccessors);
    };

    $scope.selectPredecessors = function() {
        WT1.event("flow-context-menu-select-predecessors-multi");
        FlowGraphSelection.getSelectedNodes().forEach(FlowGraphSelection.selectPredecessors);
    };

    $scope.startPropagate = function() {
        const items = FlowGraphSelection.getSelectedTaggableObjectRefs().map(r => {
            return {
                projectKey: r.projectKey,
                id: r.id
            }
        });
        $scope.startTool("PROPAGATE_SCHEMA", {"sources": items})
    }

    $scope.havePredecessors = false;
    $scope.haveSuccessors = false;
    $scope.anyLocalDataset = false;
    $scope.anyLocalFolder = false;
    $scope.anyLocalComputable = false;
    $scope.anyRecipe = false;
    $scope.anyNonVirtualizable = false;
    $scope.anyCanSpark = false;
    $scope.anyCanChangeConnection = false;
    $scope.allShareable = true;
    $scope.allUnshareable = true;
    $scope.isAllZonesCollapsed = true;
    $scope.isAllZonesExpanded = true;
    $scope.selectedDatasets = [];
    $scope.selectedRecipes = [];
    $scope.selectedNodes = [];

    $scope.selectedNodes = FlowGraphSelection.getSelectedNodes();

    $scope.selectedNodes.forEach(function(node) {
        if (node.nodeType.startsWith('LOCAL')) {
            $scope.anyLocalComputable = true;
        }
        if (node.nodeType == 'LOCAL_DATASET') {
            $scope.anyLocalDataset = true;
            if (!node.virtualizable) {
                $scope.anyNonVirtualizable = true;
            }
        }
        if (node.nodeType === 'LOCAL_DATASET' || node.nodeType === 'FOREIGN_DATASET') {
            $scope.selectedDatasets.push(node);
        }
        if (node.nodeType == 'LOCAL_MANAGED_FOLDER') {
            $scope.anyLocalFolder = true;
        }
        if (node.nodeType == 'RECIPE') {
            $scope.anyRecipe = true;
            if ($scope.canSpark(node)) {
                $scope.anyCanSpark = true;
            }
            $scope.selectedRecipes.push(node);
        }
        if (node.predecessors.length) {
            $scope.havePredecessors = true;
        }
        if (node.successors.length) {
            $scope.haveSuccessors = true;
        }
        if (["ZONE","RECIPE"].includes(node.nodeType)) {
            $scope.allShareable = false;
        }
        if (!node.usedByZones.length || node.successors.length) {
            $scope.allUnshareable = false;
        }
        if (node.nodeType == "ZONE" && !node.customData.isCollapsed) {
            $scope.isAllZonesCollapsed = false;
        }
        if (node.nodeType == "ZONE" && node.customData.isCollapsed) {
            $scope.isAllZonesExpanded = false;
        }

        $scope.anyCanChangeConnection = $scope.anyCanChangeConnection || $scope.canChangeConnection(node);
    })

    $scope.canInsertRecipeBetween = function() {
        return FlowGraphSelection.canInsertRecipeBetweenSelection($scope.selectedNodes, $scope.selectedDatasets, $scope.selectedRecipes);
    }

    $scope.insertRecipeBetween = function() {
        const selectedDataset = $scope.selectedDatasets[0];
        const selectedRecipes = $scope.selectedRecipes;

        $scope.showInsertRecipeModal(selectedDataset, selectedRecipes)
            .then(selectedRecipe => WT1.event('flow-context-menu-insertrecipe', {position: 'between', recipe: selectedRecipe && selectedRecipe.type || 'Unknown'}));
    }
});

app.service('FlowFilterQueryService', function() {
    const svc = this;

    this.escapeStr = function (string) {
        if (string.includes(' ') || string.includes('"') || string.includes(':')) {
            return `"${string.replace(/"/g, '\\"')}"`
        }
        return string;
    };

    function uiFilterArrayToQueryClause(elements, key) {
        if (!elements) return;
        const resultString = elements.map(el => key + svc.escapeStr(el)).join(' OR ');
        return elements.length > 1 ? `(${resultString})` : resultString;
    }

    this.pickerFormat = "YYYY-MM-DD HH:mm";

    const queryClauseOrNull = (types, type) => types && types.includes(type) ? uiFilterArrayToQueryClause([type], "type:"): null;

    this.uiFilterToQuery = function(structuredFlowObjectFilter) {

        function formatDate(date) {
            return moment(date).format(svc.pickerFormat);
        }

        const creationDate = structuredFlowObjectFilter.customCreationDateRange;
        const modificationDate = structuredFlowObjectFilter.customModificationDateRange;

        let createdRangeClause;
        let modifiedRangeClause;
        if (structuredFlowObjectFilter.creationDateRange) {
            if (structuredFlowObjectFilter.creationDateRange === 'CUSTOM') {
                createdRangeClause = creationDate && creationDate.from && creationDate.to ? `createdBetween:${formatDate(creationDate.from)} / ${formatDate(creationDate.to)}` : null;
            } else {
                createdRangeClause = `created:${structuredFlowObjectFilter.creationDateRange}`;
            }
        }
        if (structuredFlowObjectFilter.modificationDateRange) {
            if (structuredFlowObjectFilter.modificationDateRange === 'CUSTOM') {
                modifiedRangeClause = modificationDate && modificationDate.from && modificationDate.to ? `modifiedBetween:${formatDate(modificationDate.from)} / ${formatDate(modificationDate.to)}` : null;
            } else {
                modifiedRangeClause = `modified:${structuredFlowObjectFilter.modificationDateRange}`;
            }
        }
        const datasetTypeClause = uiFilterArrayToQueryClause(structuredFlowObjectFilter.datasetTypes, "datasetType:");
        const recipeTypeClause = uiFilterArrayToQueryClause(structuredFlowObjectFilter.recipeTypes, "recipeType:");

        const recipeClauseArr = [queryClauseOrNull(structuredFlowObjectFilter.types, 'RECIPE'), recipeTypeClause].filter(e=>e);
        const recipeClause = recipeClauseArr.length > 1 ? `(${recipeClauseArr.join(' AND ')})` : recipeClauseArr.join(' AND ');
        const datasetClauseArr = [queryClauseOrNull(structuredFlowObjectFilter.types, 'DATASET'), datasetTypeClause].filter(e=>e);
        const datasetClause = datasetClauseArr.length > 1 ? `(${datasetClauseArr.join(' AND ')})` : datasetClauseArr.join(' AND ');

        const typeClauses = structuredFlowObjectFilter.types.filter(e => (e !== 'RECIPE' && e !== 'DATASET')).map(e => uiFilterArrayToQueryClause([e], "type:"));
        const typeWithRefinements = [...typeClauses,recipeClause, datasetClause].filter(e=>e);

        let typeWithRefinementClause = typeWithRefinements.join(' OR ');
        if (typeWithRefinements.length > 1){
            typeWithRefinementClause = `(${typeWithRefinementClause})`
        }

        return [
            uiFilterArrayToQueryClause(structuredFlowObjectFilter.tags, "tag:"),
            uiFilterArrayToQueryClause(structuredFlowObjectFilter.creator, "user:"),
            typeWithRefinementClause,
            createdRangeClause,
            modifiedRangeClause
        ].filter(e => e).join(' AND ');
    }
});

})();
