/* jshint loopfunc: true*/
(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/scatter.js
    app.factory('DensityHeatMapChart', function(ChartFormatting, ChartUADimension, _MapCharts, ChartLabels, UaChartsCommon, ChartLegendUtils, ChartColorScales, DKU_PALETTE_NAMES) {
        return function($container, chartDef, data, chartHandler) {

            const elt = _MapCharts.getOrCreateMapContainer($container);
            // Handle the scatter map diverging color palette after transition to density map
            chartHandler.legendsWrapper.deleteLegends();

            chartDef.colorOptions.paletteType = 'CONTINUOUS';

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container, ChartColorScales).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                // Create leaflet layer
                const layerGroup = L.layerGroup();

                // Get map
                const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                _MapCharts.repositionMap(map, elt, data);

                // Remove the existing layer to avoid multiple layers
                if (elt.data('leaflet-data-layer')) {
                    map.removeLayer(elt.data('leaflet-data-layer'));
                }


                // Remove the existing heatmap is there is one
                let existingHeatMapLayer;
                map.eachLayer(function(layer) {
                    if (layer.options && layer.options.id) {
                        if (layer.options.id === 'heatmap') {
                            existingHeatMapLayer = layer;
                        }
                    }
                });
                if (existingHeatMapLayer) {
                    map.removeLayer(existingHeatMapLayer);
                }

                // Get the gradient for leaflet heatmap
                const colorOptions = {
                    colorPalette: chartDef.colorOptions.colorPalette,
                    transparency: 1,
                    customPalette: chartDef.colorOptions.customPalette
                };
                let scale = ChartColorScales.continuousColorScale(colorOptions, 0, 1, undefined, undefined, chartHandler.getChartTheme());
                if (colorOptions.colorPalette === DKU_PALETTE_NAMES.CUSTOM && colorOptions.customPalette.colors.length === 0) {
                    scale = (bin) => 'rgba(0, 0, 0, 1)'; // avoid errors when colorOptions.customPalette.colors is empty
                }
                const gradient = {};
                for (let i = 0; i <= 9; i++) {
                    gradient[i / 10] = scale(i / 10);
                }

                const uaLFn = ChartLabels.getUaLabel;
                const hasUAColor = false;

                // Intermediate operation for the scale computation in the scatter plot
                const makeIntensityScale = function(chartDef, data) {
                    if (ChartUADimension.isTrueNumerical(chartDef.uaSize[0])) {
                        return d3.scale.sqrt().range([0, 1])
                            .domain([data.values.size.num.min, data.values.size.num.max]);
                    } else if (ChartUADimension.isDateRange(chartDef.uaSize[0])) {
                        return d3.scale.sqrt().range([0, 1])
                            .domain([data.values.size.ts.min, data.values.size.ts.max]);
                    } else {
                        throw new ChartIAE('Cannot use ALPHANUM as size scale');
                    }
                };

                // If a column is given as a size in the front bar, create the helper function to get the right weight
                const hasUASize = UaChartsCommon.hasUASize(chartDef);
                let intensityScale;
                let getScaleWeight;
                if (hasUASize) {
                    intensityScale = makeIntensityScale(chartDef, data);
                    getScaleWeight = function(chartDef, data, i, sizeScale) {
                        if (chartDef.uaSize.length) {
                            let sizeValue;
                            if (ChartUADimension.isTrueNumerical(chartDef.uaSize[0])) {
                                sizeValue = data.values.size.num.data[i];
                            } else if (ChartUADimension.isDateRange(chartDef.uaSize[0])) {
                                sizeValue = data.values.size.ts.data[i];
                            }
                            return sizeScale(sizeValue);
                        } else {
                            return 1;
                        }
                    };
                }

                // Tuning values for the visual parameters
                const radiusRangeMultiplier = 40;

                // Create the core data that will be displayed by Leaflet.heat
                const geopoints = [];
                data.xAxis.num.data.forEach(function(x, i) {
                    const y = data.yAxis.num.data[i];
                    let r;
                    if (hasUASize) {
                        r = getScaleWeight(chartDef, data, i, intensityScale);
                    } else {
                        r = 1;
                    }
                    geopoints.push([y, x, r * chartDef.colorOptions.heatDensityMapIntensity]);
                });

                // Create the heatmap and add it as a layer
                if (chartDef.colorOptions.heatDensityMapRadius !== 0) {
                    const heatMapLayer = L.heatLayer(geopoints, { radius: chartDef.colorOptions.heatDensityMapRadius * radiusRangeMultiplier, id: 'heatmap', gradient: gradient });
                    heatMapLayer.addTo(map);
                }

                // Options of the Leaflet CircleMarker
                const options = {
                    stroke: false,
                    color: 'rgb(0,0,0)',
                    opacity: 1,
                    fill: false,
                    fillColor: 'rgb(255,0,0)',
                    fillOpacity: 1,
                    radius: 5
                };

                // Create tooltip
                data.xAxis.num.data.forEach(function(x, i) {

                    const y = data.yAxis.num.data[i];

                    const pointLayer = L.circleMarker([y, x], options);

                    let html = '';
                    html += 'Lon: <strong>' + ChartFormatting.getForIsolatedNumber()(x) + '</strong><br />';
                    html += 'Lat: <strong>' + ChartFormatting.getForIsolatedNumber()(y) + '</strong><br />';

                    if (hasUASize && (!hasUAColor || (chartDef.uaSize[0].column !== chartDef.uaColor[0].column || chartDef.uaColor[0].dateMode !== 'RANGE'))) {
                        html += uaLFn(chartDef.uaSize[0]) + ': <strong>' +
                        UaChartsCommon.formattedSizeVal(chartDef, data, i) + '</strong><br />';
                    }
                    if (chartDef.uaTooltip.length > 0) {
                        html += '<hr/>';
                    }
                    chartDef.uaTooltip.forEach(function(ua, j) {
                        html += uaLFn(ua) + ': <strong>' + UaChartsCommon.formattedVal(data.values['tooltip_' + j], ua, i) + '</strong><br/>';
                    });

                    //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                    pointLayer.bindPopup(html, { className: 'qa-chart-tooltip' });
                    pointLayer.on('mouseover', function(e) {
                        this.setStyle({
                            stroke: true,
                            fill: true
                        });
                        this.openPopup();
                    });
                    pointLayer.on('mouseout', function(e) {
                        this.setStyle({
                            stroke: false,
                            fill: false
                        });
                        this.closePopup();
                    });
                    layerGroup.addLayer(pointLayer);
                });

                // Add layer to map
                layerGroup.addTo(map);
                elt.data('leaflet-data-layer', layerGroup);

            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof (chartHandler.loadedCallback) === 'function') {
                    chartHandler.loadedCallback();
                }
            });
        };
    });

})();
