from typing import List, Optional
from dataiku.eda.types import Literal

from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.filtering.filter import Filter
from dataiku.eda.types import AnumFilterModel


class AnumFilter(Filter):
    def __init__(self, column: str, values: List[str], name: Optional[str] = None):
        self.column = column
        self.values = values
        self.name = name

    @staticmethod
    def get_type() -> Literal["anum"]:
        return "anum"

    @staticmethod
    def build(params: AnumFilterModel) -> 'AnumFilter':
        return AnumFilter(params["column"], params["values"], params.get("name"))

    def apply(self, idf: ImmutableDataFrame, inverse: bool = False) -> ImmutableDataFrame:
        mask = idf.text_col(self.column).isin(self.values)
        if inverse:
            mask = ~mask
        return idf[mask]

    def serialize(self) -> AnumFilterModel:
        return {
            "type": self.get_type(),
            "values": self.values,
            "column": self.column,
            "name": self.name
        }
