import logging
import pandas as pd

from dataiku.base.utils import safe_unicode_str
from dataiku.doctor.prediction.decisions_and_cuts import DecisionsAndCuts
from dataiku.doctor.prediction.regression_scoring import save_regression_statistics
from dataiku.modelevaluation.data_types import cast_as_string
from dataikuscoring.mlflow import mlflow_classification_predict_to_scoring_data, \
    mlflow_regression_predict_to_scoring_data
from dataiku.core import doctor_constants as constants
from dataiku.doctor.prediction.classification_scoring import BinaryClassificationModelScorer, MulticlassModelScorer, \
    save_classification_statistics
from dataiku.doctor.evaluation.base import add_evaluation_columns, RegressionModelScorer, \
    handle_percentiles_and_cond_outputs

logger = logging.getLogger(__name__)


def process_input_df(
    part_df_and_model_params_generator, mlflow_imported_model, partition_dispatch,modeling_params, with_sample_weight,
    recipe_desc, cond_outputs, evaluation_store_folder_context, dont_compute_performance
):
    prediction_type = mlflow_imported_model.get("predictionType")
    treat_perf_metrics_failure_as_error = recipe_desc.get("treatPerfMetricsFailureAsError", True)

    # MLflow Pyfunc models cannot be partitioned, but we keep the whole logic
    # of multi-partitioning from the regular evaluation recipe for ease of comparison
    part_dfs = {"pred": [], "target": [], "weight": [], "unprocessed": []}
    for part_df, part_params in part_df_and_model_params_generator:

        model_folder_context, mlflow_model = part_params

        logger.info("Predicting partition df to evaluate")

        target_name = mlflow_imported_model["targetColumnName"]
        if not dont_compute_performance:
            if target_name not in part_df:
                raise RuntimeError("""Could not find the column %s in the evaluation dataset.
                It appears you are trying to compute performance metrics without having the ground truth.
                Please use a dataset with the ground truth, or don't ask to compute performance metrics.""" % target_name)
            part_df = part_df.dropna(subset=[target_name])

        if target_name in part_df:
            target = part_df[target_name]
            part_df = part_df.drop(target_name, axis=1)

        if prediction_type == constants.BINARY_CLASSIFICATION:
            # Override threshold
            if recipe_desc.get("overrideModelSpecifiedThreshold"):
                modeling_params["forcedClassifierThreshold"] = recipe_desc.get("forcedClassifierThreshold")

            scoring_data = mlflow_classification_predict_to_scoring_data(
                mlflow_model,
                mlflow_imported_model,
                part_df,
                modeling_params["forcedClassifierThreshold"]
            )
            # labelToIntMap is filled by mlflow_classification_predict_to_scoring_data
            target_map = mlflow_imported_model["labelToIntMap"]

            pred_df = scoring_data.pred_and_proba_df

            # Probability percentile & Conditional outputs
            handle_percentiles_and_cond_outputs(pred_df, recipe_desc, cond_outputs, model_folder_context, target_map)

            if not dont_compute_performance:
                logger.info("TARGET IS: %s" % target)
                target_as_int = cast_as_string(target).replace(mlflow_imported_model["labelToIntMap"])
                target_as_int = target_as_int.astype(int)
                part_dfs["target"].append(target_as_int)

            if evaluation_store_folder_context is not None and not dont_compute_performance:
                if pred_df.shape[0] == 0:
                    logger.error("No predictions were made")
                    return

                # For mlflow models in DSS, preds = probas_one > cut. (not possible to apply overrides)
                if not scoring_data.prediction_result.has_probas():
                    decisions_and_cuts = DecisionsAndCuts.from_unmapped_preds(scoring_data.prediction_result.unmapped_preds_not_declined,
                                                                              target_map)
                else:
                    decisions_and_cuts = DecisionsAndCuts.from_probas(scoring_data.prediction_result.probas, target_map)

                binary_classif_scorer = BinaryClassificationModelScorer(
                    modeling_params,
                    evaluation_store_folder_context,
                    decisions_and_cuts,
                    target_as_int,
                    target_map,
                    test_unprocessed=None,  # No custom metric support for MLflow Pyfunc models
                    test_X=None,  # Not dumping on disk predicted_df
                    test_df_index=None,  # Not dumping on disk predicted_df
                    test_sample_weight=None,
                    assertions=None)

                binary_classif_scorer.score(with_assertions=False, treat_metrics_failure_as_error=treat_perf_metrics_failure_as_error)
                binary_classif_scorer.save()

                # We must create a iperf.json with the proba-awareness
                iperf = {
                    "probaAware": scoring_data.probas_df is not None
                }
                evaluation_store_folder_context.write_json("iperf.json", iperf)

            if evaluation_store_folder_context is not None and dont_compute_performance: # Only save statistics if we don't run the scoring
                save_classification_statistics(pred_df.iloc[:, 0],
                                               evaluation_store_folder_context,
                                               probas=scoring_data.probas_df.values if scoring_data.probas_df is not None else None,
                                               sample_weight=None,
                                               target_map=target_map)

        elif prediction_type == constants.MULTICLASS:

            scoring_data = mlflow_classification_predict_to_scoring_data(mlflow_model, mlflow_imported_model, part_df)
            # labelToIntMap is filled by mlflow_classification_predict_to_scoring_data
            target_map = mlflow_imported_model["labelToIntMap"]

            pred_df = scoring_data.pred_and_proba_df
            if not dont_compute_performance:
                target_as_int = target.astype(str).replace(mlflow_imported_model["labelToIntMap"]).astype(int)
                part_dfs["target"].append(target_as_int)

            if evaluation_store_folder_context is not None and not dont_compute_performance:

                if pred_df.shape[0] == 0:
                    logger.error("No predictions were made")
                    return

                multiclass_scorer = MulticlassModelScorer(
                    modeling_params,
                    evaluation_store_folder_context,
                    scoring_data.prediction_result,
                    target_as_int,
                    mlflow_imported_model["labelToIntMap"],
                    test_unprocessed=None,  # No custom metric support for MLflow Pyfunc models
                    test_X=None,
                    test_df_index=None,  # Not dumping on disk predicted_df
                    test_sample_weight=None,
                    assertions=None)

                multiclass_scorer.score(with_assertions=False, treat_metrics_failure_as_error=treat_perf_metrics_failure_as_error)
                multiclass_scorer.save()

                # We must create a iperf.json with the proba-awareness
                iperf = {
                    "probaAware": scoring_data.probas_df is not None
                }
                evaluation_store_folder_context.write_json("iperf.json", iperf)

            if evaluation_store_folder_context is not None and dont_compute_performance: # Only save statistics if we don't run the scoring
                save_classification_statistics(pred_df.iloc[:, 0],
                                               evaluation_store_folder_context,
                                               probas=scoring_data.probas_df.values if scoring_data.probas_df is not None else None,
                                               sample_weight=None,
                                               target_map=target_map)

        elif prediction_type == constants.REGRESSION:
            scoring_data = mlflow_regression_predict_to_scoring_data(mlflow_model, mlflow_imported_model, part_df)
            pred_df = scoring_data.preds_df

            if not dont_compute_performance:
                part_dfs["target"].append(target)

            if evaluation_store_folder_context is not None and not dont_compute_performance:
                if pred_df.shape[0] == 0:
                    logger.error("No predictions were made")
                    return

                regression_scorer = RegressionModelScorer(modeling_params,
                                                          scoring_data.prediction_result,
                                                          target,
                                                          evaluation_store_folder_context,
                                                          test_unprocessed=None,  # No custom metric support for MLflow Pyfunc models
                                                          test_X=None,  # Not dumping on disk predicted_df
                                                          test_df_index=None,  # Not dumping on disk predicted_df
                                                          test_sample_weight=None,
                                                          assertions=None)

                regression_scorer.score(with_assertions=False, treat_metrics_failure_as_error=treat_perf_metrics_failure_as_error)
                regression_scorer.save()

                # We must create a iperf.json with the proba-awareness
                iperf = {
                    "probaAware": None
                }
                evaluation_store_folder_context.write_json("iperf.json", iperf)

            if evaluation_store_folder_context is not None and dont_compute_performance:  # Only save statistics if we don't run the scoring
                save_regression_statistics(pred_df.iloc[:, 0], evaluation_store_folder_context)

        else:
            raise ValueError("bad prediction type %s" % prediction_type)

        part_dfs["pred"].append(pred_df)
        part_dfs["unprocessed"].append(part_df)

    # Re-patch partitions together --> This part onwards is entirely copied from reg_evaluation_recipe and should be made 
    # generic
    if False and partition_dispatch:
        if len(part_dfs["pred"]) > 0:
            pred_df = pd.concat(part_dfs["pred"], axis=0)
            if dont_compute_performance:
                y = None
            else:
                y = pd.concat(part_dfs["target"], axis=0)
            unprocessed = pd.concat(part_dfs["unprocessed"], axis=0)
            sample_weight = pd.concat(part_dfs["weight"], axis=0) if with_sample_weight else None
        else:
            raise Exception("All partitions found in dataset are unknown to the model, cannot evaluate it")
    else:
        pred_df = part_dfs["pred"][0]
        if dont_compute_performance:
            y = None
        else:
            y = part_dfs["target"][0]
        unprocessed = part_dfs["unprocessed"][0]
        sample_weight = part_dfs["weight"][0] if with_sample_weight else None

    # add error information to pred_df
    target_mapping = {}
    if prediction_type in [constants.BINARY_CLASSIFICATION, constants.MULTICLASS]:
        target_mapping = {
            label: int(class_id)
            for label, class_id in target_map.items()
        }
        if not recipe_desc["outputProbabilities"]:  # was only for conditional outputs
            classes = [class_label for (class_label, _) in sorted(target_map.items())]
            proba_cols = [u"proba_{}".format(safe_unicode_str(c)) for c in classes]
            to_drop = set(proba_cols).intersection(pred_df.columns)
            if to_drop:
                pred_df.drop(to_drop, axis=1, inplace=True)
    if y is not None:
        pred_df = add_evaluation_columns(prediction_type, pred_df, y, recipe_desc["outputs"], target_mapping)

    return pred_df, y, unprocessed, sample_weight, modeling_params, target_mapping
