<#
.SYNOPSIS
    Boostraps or upgrades a Python virtual environment for Dataiku Science Studio.
.DESCRIPTION
    Boostraps or upgrades a Python virtual environment for Dataiku Science Studio.
.PARAMETER Upgrade
    (Optional) Upgrade existing DSS Python environment (default: create new).
.PARAMETER ForceRebuildPyenv
    (Optional) Rebuild the Python environment even if it already exists.
.PARAMETER RebuildIfNeeded
    (Optional) Rebuilds the python environment if needed.
.PARAMETER PythonBin
    Base Python binary to use when creating new environment.
.PARAMETER DATADIR
    (Required) Path to the DSS data directory (also called data home).
#>
param(
    [Parameter(Mandatory=$false)] [switch] $Upgrade = $false,
    [Parameter(Mandatory=$false)] [switch] $ForceRebuildPyenv = $false,
    [Parameter(Mandatory=$false)] [switch] $RebuildIfNeeded = $false,
    [Parameter(Mandatory=$false)] [string] $PythonBin,
    [Parameter(Mandatory=$true)] [string] $DATADIR
)

$MyDir = $PSScriptRoot
$InstallDir = (Get-Item $MyDir).parent.FullName

# Version of the python binary
function GetPythonVersion()
{
    param(
        [Parameter(Mandatory=$true)] [string] $PythonBin
    )

    & $PythonBin -c "import sysconfig;print(sysconfig.get_python_version())"
}

<#
.SYNOPSIS
    Returns version and additional info of the python binary
.DESCRIPTION
    Returns version and additional info of the python binary
#>
function GetPythonFullVersion()
{
    param(
        [Parameter(Mandatory=$true)] [string] $PythonBin
    )

    & $PythonBin -c "import sys;print(sys.version)"
}

function DeleteIfExists()
{
    param(
        [string[]] $Items
    )

    foreach ($item in $items)
    {
        if (Test-Path $item)
        {
            Remove-Item -Force -Recurse -Path $item
        }
    }
}

$DIP_HOME = $DATADIR

# Determine the python binary to use
if (!$Upgrade -or $ForceRebuildPyenv)
{
    # Fresh install or force rebuild: create a new pyenv
    if (!$PythonBin)
    {
        $PythonBin = (Get-Command python).Path
        Write-Output "+ Using Python from PATH: $PythonBin"
    }
    else
    {
        Write-Output "+ Using specified base Python binary: $PythonBin"
    }
}
else
{
    # Upgrade existing environment
    if ($RebuildIfNeeded)
    {
        # Get current version of Python
        try {
            $existingPyenvBin = "$DIP_HOME/pyenv/Scripts/python.exe"
            $existingPyenvVersion = GetPythonVersion -PythonBin $existingPyenvBin
            $existingPyenvFullVersion = GetPythonFullVersion -PythonBin $existingPyenvBin
        } catch {
            Write-Error "! Failed to get existing Python version"
        }
        Write-Output "+ Current DSS Python version: '$existingPyenvVersion'"

        # Get base Python version
        if ($PythonBin)
        {
            $candidatePythonBin = $PythonBin
        }
        else
        {
            $candidatePythonBin = (Get-Command python).Path
        }

        $candidatePythonVersion = GetPythonVersion -PythonBin $candidatePythonBin
        $candidatePythonFullVersion = GetPythonFullVersion -PythonBin $candidatePythonBin
        Write-Output "+ Base Python version: '$candidatePythonVersion'"

        if ($existingPyenvFullVersion -ne $candidatePythonFullVersion)
        {
            Write-Output "! Version mismatch, forcing Python environment rebuild"
            $PythonBin = $candidatePythonBin
            $ForceRebuildPyenv = $true
        }
        else
        {
            Write-Output "+ Versions match, no rebuild necessary"
            $PythonBin = $existingPyenvBin
        }
    }
    elseif ($PythonBin)
    {
        # This actually is possible when specifying the -f or -r options but we don't want
        # users to specify these themselves.
        Write-Error "[-] Error : cannot specify base Python binary when upgrading environment"
        exit 1
    }
    else
    {
        # Reuse existing virtual environment
        $PythonBin = "$DIP_HOME/pyenv/Scripts/python.exe"
    }
}

# TODO: check if the python interpreter is compatible with DSS standard packages?
# We do in the linux script.

if ($ForceRebuildPyenv -and (Test-Path -Path "$DIP_HOME/pyenv" -PathType Container))
{
    # Remove the virtualenv, keeping backup
    $envBackup = "$DIP_HOME/pyenv.backup.$PID"
    Write-Output "+ Backing up $DIP_HOME/pyenv to $envBackup"
    DeleteIfExists $envBackup
    Move-Item "$DIP_HOME/pyenv" $envBackup
}
else
{
    $envBackup = $null
}

if (!$Upgrade -or $ForceRebuildPyenv)
{
    # Initialize virtualenv
    New-Item -Path "$DIP_HOME/pyenv" -ItemType Directory

    & "$InstallDir/scripts/_create-virtualenv.ps1" $pythonBin "--no-download" "$DIP_HOME/pyenv"
    if ($LastExitCode) # failed
    {
        Write-Error "[-] Error: could not initialize Python virtualenv in $DIP_HOME/pyenv"
        Remove-Item -Recurse -Force -Path "$DIP_HOME/pyenv"
        if ($envBackup)
        {
            Write-Error "[-] Recovering backed up Python virtualenv from $envBackup"
            Move-Item "$envBackup" "$DIP_HOME/pyenv"
        }

        exit 1
    }
}

# Link standard packages from kit
$filesToDelete = "py", "pyx" | ForEach-Object { "$DIP_HOME/pyenv/lib/python$pythonVersion/sitecustomize.$_" }
$filesToDelete += "$DIP_HOME/pyenv/lib/python$pythonVersion/__pycache__/sitecustomize.*.pyc"  # migration from DSS versions <= 8
DeleteIfExists $filesToDelete
New-Item -ItemType Directory -Force -Path "$DIP_HOME/pyenv/lib/python$pythonVersion/site-packages"
Set-Content -Path "$DIP_HOME/pyenv/lib/site-packages/_dss_packages.pth" -Value "import site; site.addsitedir(r`"$InstallDir/pythonwin.packages`")"
$pythonBin = "$DIP_HOME/pyenv/Scripts/python.exe"

if ($envBackup)
{
    Write-Output "+ Removing backed up environment $envBackup"
    Remove-Item -Recurse -Force -Path $envBackup
}