/* CODE STUDIO RUNTIME STANDALONE PAGE */

window.DKU = (() => {
    "use strict";

    return (config) => {
        let syncZones = [], exposedUrl;
        
        function stopEvent(e) {
            e.stopPropagation();
            e.preventDefault();
        }

        /* COOKIES */
        
        const cookies = document.cookie.split(';').reduce((cookies, c) => { 
            const [k, v] = c.split('=');
            cookies[k.trim()] = (v || '').trim();
            return cookies;
        }, {});
      
        /* API */

        const API = {
            getJSON: (url, data) => $.ajax({
                dataType: "json",
                url: url,
                data: data,
                headers: {
                    'Content-Type': 'application/x-www-form-urlencoded;charset=utf-8',
                    'X-XSRF-TOKEN': cookies[config.XSRFCookieName]
                }
            }),
            getFullInfo: () => API.getJSON(
                '/dip/api/code-studio-objects/get-full-info',
                {
                    codeStudioObjectId: config.codeStudio.id,
                    projectKey: config.projectKey
                }
            ),
            pull: (zone, force=true) => API.getJSON(
                `/dip/api/code-studio-objects/pull/${zone.id}`, 
                {
                    codeStudioObjectId: config.codeStudio.id,
                    projectKey: config.projectKey,
                    force: force
                }
            ),
            pulls: (zones=syncZones, force=true) => $.when(...zones.map(z => API.pull(z, force))),
            checkConflicts: (zones=syncZones) => API.getJSON(
                '/dip/api/code-studio-objects/check-conflicts',
                {
                    codeStudioObjectId: config.codeStudio.id,
                    projectKey: config.projectKey,
                    zones: JSON.stringify(zones.map(z => z.id))
                }
            )
        };

        /* TOOLBAR */

        const 
            TOOLBAR_STORAGE_KEY = 'DKU.code-studio-standalone.toolbar',
            DROPZONE_ANCHOR_SIZE_PX = 50, 
            FREE_TOOLBAR_WIDTH = 30, 
            FREE_TOOLBAR_HEIGHT = 60,
            drag = {x:0, y:0, dx:0, dy:0},
            $dropzone = document.getElementById('dropzone'),
            $toolbar = document.getElementById('code-studio-standalone-toolbar'),
            $toolbarName = $toolbar.querySelector('.code-studio-standalone__name'),
            $toolbarInfo = $toolbar.querySelector('.code-studio-standalone__info'),
            toolbar = TOOLBAR_STORAGE_KEY in localStorage 
                ? JSON.parse(localStorage.getItem(TOOLBAR_STORAGE_KEY)) 
                : {mode:'right', x:0, y:0}; //available modes: free/top/left/right/bottom
        
        function setupToolbar() {
            if( toolbar.mode === 'free' ){
                //adjust position (in case of position saved on a bigger screen)
                toolbar.x = Math.min(toolbar.x, window.innerWidth - FREE_TOOLBAR_WIDTH);
                toolbar.y = Math.min(toolbar.y, window.innerHeight - FREE_TOOLBAR_HEIGHT);
                $toolbar.style.top = toolbar.y + 'px';
                $toolbar.style.left = toolbar.x + 'px';
            } else {
                //grid mode, back to css default
                $toolbar.style.top = '';
                $toolbar.style.left = '';
            }
            localStorage.setItem(TOOLBAR_STORAGE_KEY, JSON.stringify(toolbar));
            document.body.className = 'code-studio-standalone toolbar-' + toolbar.mode;
        }
        setupToolbar();

        function dragToolbar(e) {
            drag.x = e.pageX;
            drag.y = e.pageY;
            if( drag.y < DROPZONE_ANCHOR_SIZE_PX ){
                toolbar.mode = 'top';
            } else if( drag.y > window.innerHeight - DROPZONE_ANCHOR_SIZE_PX ) {
                toolbar.mode = 'bottom';
            } else if( drag.x < DROPZONE_ANCHOR_SIZE_PX ) {
                toolbar.mode = 'left';
            } else if( drag.x > window.innerWidth - DROPZONE_ANCHOR_SIZE_PX ) {
                toolbar.mode = 'right';
            } else {
                toolbar.mode = 'free';
                toolbar.x = drag.x - drag.dx;
                toolbar.y = drag.y - drag.dy;
            }
            setupToolbar();
        }

        //connot use the drag events
        //coz of Firefox bug https://bugzilla.mozilla.org/show_bug.cgi?id=505521

        function clickable(e, depth=666) {
            if( depth > 0 && e ) {
                const tag = e.tagName;
                return (tag === 'BUTTON' || tag === 'INPUT' || tag === 'A')
                    ? true
                    : clickable(e.parentElement, depth-1);
            }
            return false;
        }

        let dragging = false;
        $toolbar.onmousedown = (e) => {
            if( !clickable(e.target, 2) ) {
                stopEvent(e);
                dragging = true;
                drag.x = e.pageX;
                drag.y = e.pageY;
                if( toolbar.mode === 'free' ){
                    drag.dx = e.pageX - toolbar.x;
                    drag.dy = e.pageY - toolbar.y;
                } else {
                    drag.dx = FREE_TOOLBAR_WIDTH / 2;
                    drag.dy = 5;
                }            
                $dropzone.style.display = 'block';
            }
        };

        $dropzone.onmouseleave =
        $dropzone.onmouseup = (e) => {
            if( dragging ) {
                stopEvent(e);
                dragToolbar(e);
                dragging = false;
                $dropzone.style.display = 'none';
            }
        };

        $dropzone.onmousemove = (e) => {
            if( dragging ) dragToolbar(e);
        };

        /* OVERLAY */

        const $overlayPanel = document.getElementById('overlay-panel');
        const overlay = {
            show: (txt) => {
                $overlayPanel.innerHTML = '<span>' + txt + '</span>';
                $overlayPanel.style.display = '';
            },
            hide: () => $overlayPanel.style.display = 'none',
            error: (args, {timeout, timeoutFn}) => {
                let txt;
                if( Array.isArray(args) ){
                    const [xhr, status, error] = args;
                    txt = xhr && xhr.message ? xhr.message : error || status;
                } else {
                    txt = '' + args;
                }
                overlay.show('<i class="icon-warning-sign"></i> ' + (txt || 'unknown error'));
                if( timeoutFn ) 
                    setTimeout(timeoutFn, timeout || 3000);
                else if( timeout ) 
                    setTimeout(overlay.hide, timeout);
            }
        };

        /* MODAL */

        function modal({icon, title="title", body="", buttons=[]}) {
            return new Promise((resolve, reject) => {
                const 
                    $modal = document.createElement('div'),
                    closeModalFn = (action) => () => {
                        document.removeEventListener('keydown', escape);
                        document.body.removeChild($modal);
                        resolve(action);
                    },
                    escape = (e) => {
                        if( e.key.toUpperCase() === 'ESCAPE' )
                            closeModalFn('cancel')();
                    };
                $modal.className = 'modal-container';
                $modal.innerHTML = `
                    <div class="modal modal3">
                        <div class="modal-header">
                            ${icon?`<i class="modal-header__totem ${icon}"></i>`:''}
                            <h4 class="modal-header__title">${title}</h4>
                            <div class="modal-header__button-wrapper">
                                <button type="button" data-action="cancel"
                                    class="btn btn--text btn--dku-icon"
                                >
                                    <i class="dku-icon-dismiss-20"></i>
                                </button>
                        </div>
                        </div>
                        <div class="modal-body">${body}</div>
                        <div class="modal-footer modal-footer-std-buttons">
                            <div class="pull-right">
                                ${buttons.map(({action,icon,text,css}) => 
                                    `<button class="btn btn--text ${css}" data-action="${action}">
                                        <i class="${icon}"></i> ${text}
                                    </button>`
                                ).join('')}
                            </div>
                        </div>
                    </div>
                `;
                $modal.querySelectorAll('button[data-action]')
                    .forEach(b => b.onclick = closeModalFn(b.getAttribute('data-action')));
                document.body.appendChild($modal);
                document.addEventListener('keydown', escape);
            });
        }

        /* SYNC BUTTON */

        const 
            $butSync = document.getElementById('but-sync'), 
            $butSyncIcon = document.getElementById('but-sync-icon');
            
        function setSyncState(icon='icon-exchange', disabled=false, temporary=false){
            $butSyncIcon.className = icon;
            $butSync.disabled = disabled;
            if( temporary )
                setTimeout(() => setSyncState('icon-exchange', !disabled), 3000);
        }

        function pullAllZones(){
            API.pulls()
                .done(() => setSyncState('icon-ok-sign text-success', true, true))
                .fail(() => setSyncState('icon-warning-sign text-error', true, true));
        }

        $butSync.onclick = (e) => {
            stopEvent(e);
            setSyncState('icon-spinner icon-spin', true);
            API.checkConflicts()
                .done(data => {
                    const conflictZones = Object.keys(data)
                        .filter(zid => data[zid].added.length + data[zid].deleted.length + data[zid].modified.length > 0)
                        .map(zid => {
                            return {
                                ...syncZones.find(z => z.id === zid),
                                ...data[zid]
                            };
                        });
                    if( conflictZones.length > 0 ){
                        modal({
                            icon: 'icon-warning-sign', 
                            title: 'A conflict occurred',
                            body: 'You have files that were modified in the Code Studio, and are attempting to save them back into the DSS instance, but some files were modified concurrently in the Code Studio and in the DSS instance',
                            buttons: [
                                {action: 'pull', text: 'Save anyway', icon: 'icon-hdd', css: 'btn--primary'},
                                {action: 'dss', text: 'Resolve in DSS', icon: 'icon-dss', css: 'btn--secondary'},
                                {action: 'cancel', text: 'Decide later', icon: 'icon-time', css: 'btn--secondary'}
                            ]
                        }).then((action) => {
                            switch(action) {
                                case 'pull':
                                    pullAllZones();
                                    break;
                                case 'dss':
                                    window.open(`/projects/${config.projectKey}/code-studios/${config.codeStudio.id}/view`);
                                default:
                                    setSyncState();
                            }
                        });
                    } else {
                        pullAllZones();
                    }
                })
                .fail(() => setSyncState('icon-warning-sign text-error', true, true));
        };

        /* CHECK STATE */
        
        const CHECKDELAY = 60 * 1000; //check every 60s
        let runOutdatedTemplate = false;
        
        function checkState(uiState){
            if( uiState.state === 'RUNNING' ) {
                overlay.hide();
                if( !runOutdatedTemplate && uiState.lastTemplateBuilt > uiState.lastStateChange ){
                    runOutdatedTemplate = true;
                    $toolbarInfo.innerHTML = '<i class="icon-info-sign"></i><span>This Code Studio is not on the latest template, restart to use it</span>';
                    modal({
                        icon: 'icon-info-sign', 
                        title: 'Outdated runtime',
                        body: 'This Code Studio is not using the latest template, you should restart to use it',
                        buttons: [
                            {text: 'Go back to DSS', action: 'dss', css: 'btn--primary'},
                            {text: 'Close', css: 'btn--secondary'}]
                    }).then((action) => {
                        if( action === 'dss' ){
                            window.location = `/projects/${config.projectKey}/code-studios/`;
                        }
                    });
                } else if( runOutdatedTemplate && !uiState.runOutdatedTemplate ){
                    runOutdatedTemplate = false;
                    $toolbarInfo.innerHTML = '';
                }
            } else {
                overlay.error('Code Studio is ' + uiState.state.toLowerCase());
            }
        }

        function refreshState(uiState){
            API.getFullInfo()
                .done(data => {
                    checkState(data.uiState);
                    if( !runOutdatedTemplate ) setTimeout(refreshState, CHECKDELAY);
                });
        }

        /* STARTUP */

        function loadInfos(){
            API.getFullInfo()
                .done(data => {
                    const exposed = data.uiState.exposed.find(e => e.exposedPort == config.codeStudio.exposedPort);
                    if( exposed ){
                        syncZones = data.uiState.syncedZones;
                        if( exposedUrl != exposed.url ){
                            exposedUrl = exposed.url;
                            const $iframe = document.getElementsByTagName('iframe')[0];
                            $iframe.onload = overlay.hide;
                            $iframe.src = exposedUrl;
                        } else {
                            overlay.hide();
                        }
                        $toolbarName.innerText = data.codeStudioObject.name;
                        document.title = `${data.codeStudioObject.name} - Code Studio | Dataiku`;
                        $butSync.disabled = false;
                        checkState(data.uiState);
                    } else {
                        overlay.error('Code Studio is not running', {timeoutFn: loadInfos});
                    }
                    setTimeout(refreshState, CHECKDELAY);
                })
                .fail((...args) => {
                    overlay.error(args, {timeoutFn: loadInfos});
                    setTimeout(loadInfos, 5*1000); //retry in 5s
                });
            
        }
        loadInfos();
    };
})();