(function(){
    'use strict';

    var widgets = angular.module('dataiku.directives.widgets');

    widgets.directive('computedColumnStep', function(Logger, $timeout, translate) {
        return {
            templateUrl: '/templates/recipes/visual-recipes-fragments/inline-computed-columns.html',
            restrict: 'EA',
            scope: true,
            link: function(scope, element, attrs) {
                scope.translate = translate;
                scope.computedColumnStep = {};

                scope.getComputedColumnsStatus = function(recipeStatus, computedColumnIndex) {
                    return ((recipeStatus || {}).messages || []).filter(function(msg) {
                        return msg.line === computedColumnIndex;
                    });
                };

                scope.addNewComputedColumn = function() {
                    scope.computedColumnStep.editingNewComputedColumn = true;
                    scope.computedColumnListDesc.push({name:'', type: 'double', expr: '', mode: 'GREL'});
                    // focus the newly added computed column's name input
                    $timeout(function() {
                        element.find('[computed-column-step-editor]').last().find('input.computed-column-name').focus();
                    });
                };

                // --- init parameters given in attributes:
                var defaults = {
                    computedColumnListUpdateCallback: null,
                    dataset: null,
                    schema: null,
                    computedColumnListDesc: null
                };

                var regenInternalFields = function() {
                    $.each(defaults, function(param, value) {
                        scope[param] = scope.$eval(attrs[param]) || value;
                    });
                    if (!scope.computedColumnListDesc) {
                        Logger.error('"computed-column-list-desc" attribute is required by directive computed-column-step', attrs);
                    }
                    if (!scope.schema && !scope.dataset) {
                        Logger.error('computedColumnStep must have either a dataset or a schema', attrs);
                    }
                };

                regenInternalFields();

                if (scope.computedColumnListUpdateCallback) {
                    var computedColumnListDescOldValue;

                    var computedColumnListDescChanged = function(nv, ov){
                        // check if the newly added computed column's expression is set,
                        // if so compute status for this news computed column aswell
                        if (scope.computedColumnStep.editingNewComputedColumn && nv[nv.length-1] && nv[nv.length-1].expr) {
                            scope.computedColumnStep.editingNewComputedColumn = false;
                        }
                        var oldValue = computedColumnListDescOldValue;
                        var newValue = nv;
                        // ignoring the last element of the array if it has been recently added
                        if (scope.computedColumnStep.editingNewComputedColumn) {
                            newValue = newValue.slice(0, -1);
                        }
                        if(angular.equals(oldValue, newValue)) {
                            return;
                        }
                        computedColumnListDescOldValue = angular.copy(newValue);
                        scope.computedColumnListUpdateCallback(newValue);
                    };

                    scope.$on('$destroy', function() {
                        scope.computedColumnStep.editingNewComputedColumn = false;
                        computedColumnListDescChanged(scope.computedColumnListDesc, computedColumnListDescOldValue);
                    });

                    scope.$watch('computedColumnListDesc', computedColumnListDescChanged, true);
                }

                scope.$watch('recipeStatus', function() {
                    // in case these computed columns are in a visual recipe, make sure to update the internal
                    // flags when the engine or something else changes
                    regenInternalFields();
                },true);
            }
        };
    });

    widgets.directive('computedColumnStepEditor', function(Logger, InfoMessagesUtils, ColumnTypeConstants, CodeMirrorSettingService, translate) {
        return {
            templateUrl: '/templates/recipes/fragments/computed-column-step-editor.html',
            restrict: 'EA',
            scope: true,
            link: function(scope, element, attrs) {
                scope.translate = translate;
                scope.getColumns = function() {
                    return scope.schema && scope.schema.columns || [];
                };

                scope.clickedOnColumnName = function(colName) {
                    scope.addFormulaElement(colName.match(/^[a-z0-9_]+$/i) ? colName : `val('${colName}')`);
                };

                scope.clickedOnVariableName = function (varName) {
                    scope.addFormulaElement(`variables['${varName}']`);
                };

                scope.addFormulaElement = function(code) {
                    // replace selection and focuses editor
                    var cm = $('.CodeMirror', element).get(0).CodeMirror;
                    cm.replaceSelection(code);
                    cm.focus();
                };

                scope.InfoMessagesUtils = InfoMessagesUtils;
                scope.ColumnTypeConstants = ColumnTypeConstants;

                // --- init parameters given in attributes:
                var defaults = {
                    mustRunInDatabase: false,
                    computedColumnUpdateCallback: null,
                    dataset: null,
                    schema: null,
                    computedColumnDesc: null,
                    recipeStatusMessages: null
                };

                var regenInternalFields = function() {
                    $.each(defaults, function(param, value) {
                        scope[param] = scope.$eval(attrs[param]) || value;
                    });
                    if (!scope.computedColumnDesc) {
                        Logger.error('"computed-column-desc" attribute is required by directive computed-column-step-editor', attrs);
                    }
                    if (!scope.schema && !scope.dataset) {
                        Logger.error('computedColumnStepEditor must have either a dataset or a schema', attrs);
                    }
                };

                regenInternalFields();

                scope.sqlEditorOptions = CodeMirrorSettingService.get('text/x-sql');

                if (scope.computedColumnUpdateCallback) {
                    scope.$watch('computedColumnDesc',
                        function(nv, ov){
                            if(angular.equals(ov, nv)) {
                                return;
                            }
                            scope.computedColumnUpdateCallback(scope.computedColumnDesc);
                        },
                        true
                    );
                }

                scope.$watch('recipeStatus', function() {
                    // in case these computed columns are in a visual recipe, make sure to update the internal
                    // flags when the engine or something else changes
                    regenInternalFields();
                },true);
            }
        };
    });

    widgets.component('computedColumnPreview', {
        templateUrl:  '/templates/recipes/widgets/computed-column-preview.html',
        bindings: {
            dataset: "<",
            outputName: "<",
            outputType: "<",
            inputColumns: "<",
            expr: "<",
            errors: "<",
            showLastPreviewOnError: "<?"
        },
        controller: function($scope, $stateParams, DataikuAPI, Debounce, MonoFuture, PrettyPrintDoubleService) {
            this.hasError = () => this.errors !== undefined && this.errors.length !== 0;
            this.$onInit = function () {
                let lastExprUsedForQuery = undefined;
                let validateExpressionMonoFuture = MonoFuture($scope).wrap(DataikuAPI.shakers.validateExpression);
                const refreshPreviewData = () => {
                    if(this.hasError() || !this.expr || !this.expr.trim()) {
                        if(!this.showLastPreviewOnError) {
                            this.preview = undefined;
                            lastExprUsedForQuery = undefined; // next valid input must trigger update
                        }
                        return;
                    }
    
                    // when expr is changed and it triggers a change of error status, it could trigger a double update. Let's just ignore update if expr is unchanged
                    if(this.expr === lastExprUsedForQuery) return;
                    lastExprUsedForQuery = this.expr;
                    this.inProgress = true;
    
                    // handle the case where the input dataset is foreign
                    const { projectKey: datasetProjectKey, datasetName: notSmartDatasetName} = resolveDatasetFullName(this.dataset, $stateParams.projectKey);
    
                    validateExpressionMonoFuture(
                        $stateParams.projectKey,
                        datasetProjectKey,
                        notSmartDatasetName,
                        null, // Use the default shaker script (the one from the Explore view)
                        null, // Not setting the sampleId will let the backend calculate it
                        this.expr,
                        0,
                        0,
                        0,
                        true
                    ).success(({ result }) => {
                        this.inProgress=false;
                        if (result.ok) {
                            this.errorMessage = "";
                            PrettyPrintDoubleService.patchFormulaPreview(result.table, this.outputType, this.inputColumns);
                            this.preview = result.table;
                        } else {
                            this.preview = undefined;
                            lastExprUsedForQuery = undefined; // next valid input must trigger update
                            this.errorMessage = result.message;
                        }
                    }).error(() => {
                        this.inProgress = false;
                    });
                };
                refreshPreviewData();
    
                this.hasError = () => this.errors !== undefined && this.errors.length !== 0;
    
                $scope.$watch(() => this.expr, Debounce().withDelay(700, 700).withSpinner(false).wrapWatch(refreshPreviewData));
                $scope.$watch(this.hasError, refreshPreviewData);
            }
            
        }
    });

    widgets.component('columnPreview', {
        templateUrl:  '/templates/recipes/widgets/column-preview.html',
        bindings: {
            dataset: "<",
            column: "<",
            columnType: "<"
        },
        controller: function($scope, $stateParams, DataikuAPI, Debounce, MonoFuture, PrettyPrintDoubleService) {
            let previewColumnMonoFuture = MonoFuture($scope).wrap(DataikuAPI.shakers.previewColumn);

            let lastColumnUsedForQuery = undefined;
            const emptyPreview = () => {
                let result = { colNames: [this.column], rows: [] };
                for (let i = 0; i < 10; i++) {
                    result.rows.push("");
                }
                return result;
            }
            const refreshPreviewData = () => {
                if (this.column === lastColumnUsedForQuery) {
                    return;
                }
                this.preview = emptyPreview();
                this.errorMessage = "";
                if (!this.column || !this.column.trim()) {
                    lastColumnUsedForQuery = undefined; // next valid input must trigger update
                    return;
                }
                lastColumnUsedForQuery = this.column;
                this.inProgress = true;
                previewColumnMonoFuture($stateParams.projectKey, this.dataset, this.column, true).success(({ result }) => {
                    this.inProgress = false;
                    if (result.ok) {
                        PrettyPrintDoubleService.patchFormulaPreview(result.table, this.columnType, []);
                        this.preview = result.table;
                        this.errorMessage = "";
                    } else {
                        this.errorMessage = result.message;
                        lastColumnUsedForQuery = undefined; // next valid input must trigger update
                    }
                }).error(() => {
                    this.inProgress = false;
                    lastColumnUsedForQuery = undefined; // next valid input must trigger update
                });
            };

            const debouncedRefreshPreviewData = Debounce().withDelay(10, 10).withSpinner(false).wrap(refreshPreviewData);

            this.$onChanges = (changes) => {
                if(changes.column || changes.columnType) {
                    debouncedRefreshPreviewData();
                }
            }
            this.$onInit = function (){
                this.preview = emptyPreview();
            }
        }
    });

    widgets.directive('inputComputedColumnsBlock', function (CreateModalFromTemplate, $timeout, Logger, translate) {
        return {
            templateUrl : '/templates/recipes/fragments/computed-columns-management-block.html',
            restrict: 'AE',
            scope: {
                inputIndex: '=',
                computedColumnListDesc: '=',
                dataset: '=',
                schema: '=',
                recipeStatus: '=',
                onChange: '&',
                recipeVariables: '='
            },
            link : function(scope) {
                scope.translate = translate;
                scope.computedColumnListDesc = scope.computedColumnListDesc || [];
                scope.translate = translate;

                scope.getLabelMode = function(computedColumnDesc) {
                    return computedColumnDesc.mode === "SQL" ? translate("RECIPE.COMPUTED_COLUMNS.SQL_EXPRESSION", "SQL Expression") : translate("RECIPE.COMPUTED_COLUMNS.DSS_FORMULA", "DSS Formula")
                };

                const inputComputedColumnErrorsMemo = {
                    lastMessagesValue: undefined,
                    cache: {}
                };
                scope.getInputComputedColumnsStatus = function(inputIndex, computedColumnIndex) {
                    const messages = ((scope.recipeStatus || {}).inputComputedColumns || {}).messages;

                    if(messages !== inputComputedColumnErrorsMemo.lastMessagesValue) {
                        // status has changed, reset cache.
                        inputComputedColumnErrorsMemo.cache = {};
                        inputComputedColumnErrorsMemo.lastMessagesValue = messages;
                    }

                    const key = `${inputIndex}-${computedColumnIndex}`;
                    if(inputComputedColumnErrorsMemo.cache[key] === undefined) {
                        inputComputedColumnErrorsMemo.cache[key] = (messages || []).filter(function(msg) {
                            return msg.column === inputIndex && msg.line === computedColumnIndex;
                        });
                    }
                    
                    return inputComputedColumnErrorsMemo.cache[key];
                };


                var getSchema = function() {
                    if (scope.schema) {
                        return scope.schema;
                    } else {
                        Logger.error("schema not found for computed column")
                        return {columns: {}};
                    }
                };

                scope.getColumns = function() {
                    var schema = getSchema();
                    return schema && schema.columns || [];
                };

                scope.showComputedColumnsModal = function(computedColumnDescIdx) {
                    var computedColumnDesc = scope.computedColumnListDesc[computedColumnDescIdx];

                    var newScope = scope.$new();

                    newScope.computedColumnListDesc = scope.computedColumnListDesc;
                    newScope.dataset = scope.dataset;
                    newScope.schema = scope.schema;
                    newScope.recipeVariables = scope.recipeVariables;

                    newScope.newComputedColumn = computedColumnDescIdx === undefined || !computedColumnDesc;

                    newScope.computedColumnDesc = newScope.newComputedColumn ? {name:'', type: 'double', expr: '', mode: 'GREL'} : computedColumnDesc;
                    newScope.computedColumnDescIdx = newScope.newComputedColumn ? scope.computedColumnListDesc.length : computedColumnDescIdx;

                    CreateModalFromTemplate('/templates/recipes/fragments/computed-columns-modal.html', newScope, null, function(newScope) {
                        $timeout(function() {
                            $('#computed-columns-modal input.computed-column-name').focus();
                        });
                        if (newScope.newComputedColumn) {
                            newScope.$watch('computedColumnDesc.expr', function(nv) {
                                if (newScope.newComputedColumn && !!nv) {
                                    newScope.newComputedColumn = false;
                                    newScope.computedColumnListDesc.push(newScope.computedColumnDesc);
                                }
                            });

                            newScope.$on('$destroy', function() {
                                if (newScope.newComputedColumn) {
                                    newScope.newComputedColumn = false;
                                    newScope.computedColumnListDesc.push(newScope.computedColumnDesc);
                                }
                            });
                        }
                    });
                };

                scope.addNewComputedColumn = function() {
                    scope.showComputedColumnsModal();
                };

                scope.ok = function() {
                    if (scope.onChange) {
                        scope.onChange();
                    }
                }
            }
        };
    });
})();
