(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This directive previously was in static/dataiku/js/simple_report/config_ui.js
    app.directive('contextualMenu', function($rootScope, $window, $compile, $timeout, CHART_FILTERS, ChartStoreFactory, ChartFeatures, ChartAxesUtils, ChartsStaticData) {
        let isCurrentlyEdited = false;

        function getContextualMenuClassName(axis) {
            return `${axis}-axis-contextual-menu qa_charts_${axis}-axis-contextual-menu`;
        }

        function targetIsSelectInput(e) {
            return e.target.closest('.bootstrap-select') !== null;
        }

        function targetShouldNotCloseInput(e) {
            return e.target.closest('.no-global-contextual-menu-close') !== null;
        }

        $($window).on('click', function(e) {
            if (targetIsSelectInput(e) || targetShouldNotCloseInput(e) || isCurrentlyEdited) {
                e.stopPropagation();
                isCurrentlyEdited = false;
            } else if (!e.isDefaultPrevented() && !e.target.hasAttribute('no-global-contextual-menu-close')) {
                $rootScope.globallyOpenContextualMenu = undefined;
                $rootScope.$apply();
            }
        });
        return {
            scope: true,
            compile: function(element, attrs) {
                const popoverTemplate = element.find('.contextualMenu').detach();
                return function($scope, element, attrs) {
                    let popover = null;
                    let popoverScope = null;
                    $scope.contextualMenu = false;
                    $scope.ChartFeatures = ChartFeatures;
                    $scope.getXAxisTitle = ChartAxesUtils.getXAxisTitle;
                    $scope.getYAxisTitle = ChartAxesUtils.getYAxisTitle;
                    $scope.COLUMN_TYPES = CHART_FILTERS.COLUMN_TYPES;

                    $scope.getXAxisHint = () => {
                        return ChartFeatures.hasMultipleXDim($scope.chart.def.type) ? '(settings will be common to all X dimensions)' : '';
                    };

                    function hide() {
                        if (popover) {
                            if (popoverScope) {
                                popoverScope.$destroy();
                                popoverScope = null;
                            }
                            popover.hide().remove();
                            popover = null;
                        }
                    }
                    function show() {
                        if (popover === null) {
                            /*
                             * Since Angular 1.6, in a <select>, ng-model is set to null when the corresponding <option> is removed.
                             *
                             * Here is what happens when a contextualMenu containing a select is removed (using hide()):
                             * - The selected <option> is removed from DOM (like the others) triggering its $destroy callback.
                             * - This callback removes the value from the optionsMap and set a digest's call back (ie: $$postDigest function).
                             * - $$postDigest is triggered after angular's digest and checks if its scope (popoverScope in our case) is destroyed.
                             * - If yes it does nothing (return)
                             * - If not, $$postDigest set the select's ngModel to null, because its current value is no longer in optionsMap
                             *
                             * popoverScope prevents any nested <select>'s ngModel to get set to null when a contextualMenu is closed.
                             * This fix work because we destroy popoverScope (where the select lives), before deleting the DOM containing it (along with the <option> elements).
                             * So when $$postDigest positively checks if its scope is $destroyed, it just returns without setting the select's ngModel to null.
                             */
                            popoverScope = $scope.$new();
                            /*
                             * We may need the original scope in some context, e.g. modals opened from a contextualMenu
                             * because clicking on the modal will close the menu and destroyed its scope
                             */
                            popoverScope.$contextScope = $scope;
                            popover = $compile(popoverTemplate.get(0).cloneNode(true))(popoverScope);
                        }
                        popover.appendTo('body');

                        const position = attrs.cepPosition || 'align-left-bottom';
                        const mainZone = element;
                        const mzOff = element.offset();

                        /* Fairly ugly ... */
                        if (element.parent().parent().hasClass('chartdef-dropped')) {
                            mzOff.top -= 4;
                            mzOff.left -= 10;
                        }

                        switch (position) {
                            case 'align-left-bottom':
                                popover.css({ left: mzOff.left, top: mzOff.top + mainZone.height() });
                                break;
                            case 'align-right-bottom':
                                popover.css({
                                    top: mzOff.top + mainZone.height(),
                                    left: mzOff.left + mainZone.innerWidth() - popover.innerWidth()
                                });
                                break;
                            case 'align-right-top':
                                popover.css({
                                    top: mzOff.top,
                                    left: mzOff.left + mainZone.innerWidth()
                                });
                                break;
                            case 'smart':
                                var offset = { left: 'auto', right: 'auto', top: 'auto', bottom: 'auto' };
                                if (mzOff.left * 2 < window.innerWidth) {
                                    offset.left = mzOff.left;
                                } else {
                                    offset.right = window.innerWidth - mzOff.left - mainZone.innerWidth();
                                }
                                if (mzOff.top * 2 < window.innerHeight) {
                                    offset.top = mzOff.top + mainZone.height();
                                } else {
                                    offset.bottom = window.innerHeight - mzOff.top;
                                }
                                popover.css(offset);
                                break;
                            case 'smart-left-bottom':
                                $timeout(function() {
                                    // Left-bottom position, except if the menu would overflow the window, then left-top
                                    const offset = { left: mzOff.left, right: 'auto', top: 'auto', bottom: 'auto' };

                                    if (mzOff.top + mainZone.height() + popover.outerHeight() > window.innerHeight) {
                                        offset.bottom = window.innerHeight - mzOff.top;
                                    } else {
                                        offset.top = mzOff.top + mainZone.height();
                                    }
                                    popover.css(offset);
                                });
                                break;
                        }
                        if (attrs.cepWidth === 'fit-main') {
                            popover.css('width', mainZone.innerWidth());
                        }
                        popover.show();

                        popover.on('click', function(e) {
                            isCurrentlyEdited = false;
                            if (!targetIsSelectInput(e)) {
                                e.stopPropagation();
                            }
                        });

                        popover.on('mouseleave', function(e) {
                            // eslint-disable-next-line no-undef
                            if (isLeftClickPressed(e)) {
                                isCurrentlyEdited = true;
                            }
                        });
                    }

                    $scope.$watch('contextualMenu', function(nv, ov) {
                        if (nv) {
                            show();
                        } else {
                            hide();
                        }
                    });

                    $scope.toggleContextualMenu = function(e) {
                        if ($scope.globallyOpenContextualMenu && $scope.globallyOpenContextualMenu[0] === element[0]) {
                            $rootScope.globallyOpenContextualMenu = undefined;
                        } else {
                            $rootScope.globallyOpenContextualMenu = element;
                        }
                        e.preventDefault();
                    };

                    $scope.getMeasureClassName = function(chartType) {
                        const measureAxis = chartType === 'stacked_bars' ? 'x' : 'y';
                        return getContextualMenuClassName(measureAxis);
                    };

                    $scope.getDimensionClassName = function(chartType) {
                        const dimensionAxis = chartType === 'stacked_bars' ? 'y' : 'x';
                        return getContextualMenuClassName(dimensionAxis);
                    };

                    $scope.$on('$destroy', function() {
                        hide();
                    });

                    const globallyOpenContextualMenuWatcherUnsubscribe = $rootScope.$watch('globallyOpenContextualMenu', function(nv, ov) {
                        $scope.contextualMenu = ($rootScope.globallyOpenContextualMenu && $rootScope.globallyOpenContextualMenu[0] === element[0]);
                    });
                    $scope.$on('$destroy', globallyOpenContextualMenuWatcherUnsubscribe);

                    $scope.getYAxisFormatting = function(formatting, id) {
                        return ChartAxesUtils.getFormattingForYAxis(formatting, id);
                    };

                    if ($scope.chart) {
                        const { id, store } = ChartStoreFactory.getOrCreate($scope.chart.def.$chartStoreId);
                        $scope.chart.def.$chartStoreId = id;
                        $scope.$watch(() => store.get('axisSpecs'), (axisSpecs, oldAxisSpecs) => {
                            if (axisSpecs != null && (!angular.equals(axisSpecs, oldAxisSpecs) || _.isNil($scope.xAxisSpec) || _.isNil($scope.yAxisSpec))) {
                                const isXAxisPercentScale = !!(axisSpecs.xSpec && axisSpecs.xSpec.isPercentScale);
                                const isYAxisPercentScale = !!(axisSpecs.ySpec && axisSpecs.ySpec.isPercentScale);
                                $scope.xAxisComputeMode = isXAxisPercentScale ? 'PERCENTAGE' : 'NORMAL';
                                $scope.yAxisComputeMode = isYAxisPercentScale ? 'PERCENTAGE' : 'NORMAL';
                                $scope.ySpecId = attrs.ySpecId;
                                $scope.xAxisSpec = store.getAxisSpec('x');
                                $scope.yAxisSpec = store.getAxisSpec($scope.ySpecId || ChartsStaticData.LEFT_AXIS_ID);
                                if ($scope.yAxisSpec) {
                                    $scope.yAxisFormatting = $scope.getYAxisFormatting($scope.chart.def.yAxesFormatting, $scope.yAxisSpec.id);
                                }
                            }
                        });
                    }
                };
            }
        };
    });
})();
