from dataiku.core import intercom, snowflake_utils
import pandas as pd
import logging

class DataikuReadFastPathNotCompatibleException(Exception):
    """Exception launched by the Dataiku API clients when an error occurs"""

def try_read_dataframe(dataset, columns=None, pandas_read_kwargs=None):
    #client = intercom._api_client()
    #dataset_obj = client.get_project(self.project_key).get_dataset(self.short_name)

    if pandas_read_kwargs is None:
        pandas_read_kwargs = {}

    settings = intercom.jek_or_backend_json_call("datasets/get-settings/", data={
                "fullDatasetName": dataset.full_name
            }, err_msg="Failed to get the dataset settings")

    if settings["type"] == "S3":
        fmt_type = settings.get("formatType", "unknown")

        if fmt_type == "parquet":
            _check_s3fs_package()

            storage_options = _get_s3_storage_options(dataset, settings)
            path = dataset.get_location_info()["info"]["path"]

            return pd.read_parquet(path, columns=columns, storage_options=storage_options, **pandas_read_kwargs)

        else:
            raise DataikuReadFastPathNotCompatibleException("fast-path not compatible with this format on S3: %s" % fmt_type)

    elif settings["type"] == "Snowflake":
        _check_snowflake_package()

        selector = None
        if columns is not None:
            selector = snowflake_utils.get_quoted_columns_list(columns)
        else:
            selector = "*"

        conn_details = _get_connection_details(settings)

        sf_conn_params = snowflake_utils.get_snowflake_connection_params(settings["params"]["connection"], conn_details)

        with snowflake_utils.get_snowflake_connection(sf_conn_params) as sf_conn:

            quoted_table_name = dataset.get_location_info()["info"]["quotedResolvedTableName"]

            query = "SELECT %s FROM %s" % (selector, quoted_table_name)

            with sf_conn.cursor() as cur:
                cur.execute(query)
                return cur.fetch_pandas_all()

    else:
        raise DataikuReadFastPathNotCompatibleException("fast-path not compatible with this dataset type: %s" % settings["type"])

def _check_s3fs_package():
    try:
        import s3fs
    except Exception as e:
        raise DataikuReadFastPathNotCompatibleException("s3fs package is required for fast-path read on S3") from e
 
def _check_snowflake_package():
    try:
        import snowflake.connector
    except Exception as e:
        raise DataikuReadFastPathNotCompatibleException("snowflake-connector-python[pandas] package is required for fast-path read on Snowflake") from e


def _get_connection_details(settings):
    client = intercom._api_client()

    try:
        return client.get_connection(settings["params"]["connection"]).get_info()
    except Exception  as e:
        logging.exception("Failure while reading connection details")
        raise DataikuReadFastPathNotCompatibleException("Cannot read connection details, cannot use fast-path") from e


def _get_s3_storage_options(dataset, settings):
    client = intercom._api_client()

    conn_details = _get_connection_details(settings)

    if not "resolvedAWSCredential" in conn_details:
        raise DataikuReadFastPathNotCompatibleException("Cannot read credentials, cannot use fast-path")    
                
    cred = conn_details["resolvedAWSCredential"]
    storage_options = {
        "key": cred["accessKey"],
        "secret": cred["secretKey"]
    }
    if "sessionToken" in cred:
        storage_options["token"] = cred["sessionToken"]

    return storage_options