import os, json, re, logging
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid

from block_utils import LibLocationPathReplacer, generate_python_codenv

from distutils.version import LooseVersion

class GradioCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config

    _ENTRYPOINT_FILE = "gradio-entrypoint.sh"
    _RESTART_SCRIPT = "gradio-restart.sh"

    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, self._ENTRYPOINT_FILE)
        return entrypoint_path

    def _get_restart_script_path(self):
        restart_script_path = os.path.join("/opt/dataiku", self._RESTART_SCRIPT)
        return restart_script_path

    def _get_port(self):
        return self.config.get("port", 8282)

    def build_spec(self, spec, env, template):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        start_file = self.config.get("startFile", "__CODE_STUDIO_VERSIONED__/gradio/app.py")

        # replace the lib locations in settings_path and open_in_path
        replacer = LibLocationPathReplacer(spec)
        start_file = replacer.replace_variable_by_path(start_file)

        # get code env stuff

        default_packages = "gradio==3.48.0 urllib3==1.26.9"
        generate_codenv, pyenv_path = generate_python_codenv("GRADIO", self.config, template, default_packages, "/opt/dataiku/python-code-envs/pyenv-gradio", "python3.9", env.get("globalCodeEnvsExtraSettings"))

        python_bin = os.path.join(pyenv_path, "bin/python")

        # add the entrypoint script in the buildir
        entrypoint_script = """
#! /bin/bash

USER=dataiku
HOME=/home/dataiku

# CREATE THE START APP.PY IF NEEDED
if [ ! -f {start_file} ]; then
    mkdir -p $(dirname {start_file})
    cat << EOF > {start_file}
import gradio as gr
import os
import re

browser_path = os.getenv("DKU_CODE_STUDIO_BROWSER_PATH_{port}")
# replacing env var keys in browser_path with their values
env_var_pattern = re.compile(r'(\\\\\\${{(.*)}})')
env_vars = env_var_pattern.findall(browser_path)
for env_var in env_vars:
    browser_path = browser_path.replace(env_var[0], os.getenv(env_var[1], ''))

def greet(name):
    return "Hello " + name + "!"

demo = gr.Interface(
    fn=greet,
    inputs=gr.Textbox(lines=2, placeholder="Name Here..."),
    outputs="text"
)

# WARNING: make sure to use the same params as the ones defined below when calling the launch method,
# otherwise you app might not be responding!
demo.launch(server_port={port}, root_path=browser_path)
EOF
fi

# START GRADIO
LC_ALL=en_US.utf8 {python_bin} {start_file}
""".format(python_bin=python_bin, start_file=start_file, port=port)
        with open(os.path.join(env["buildDir"], self._ENTRYPOINT_FILE), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))

        # add the restart script in the buildir
        restart_script = """
#! /bin/bash

USER=dataiku
HOME=/home/dataiku

#! /bin/bash

PID=`pgrep -f {start_file}`
[  -z "$PID" ] && echo "Did not found any running Gradio app." || kill $PID

LC_ALL=en_US.utf8 {python_bin} {start_file}
""".format(python_bin=python_bin, start_file=start_file, port=port)
        with open(os.path.join(env["buildDir"], self._RESTART_SCRIPT), "wb") as f:
            f.write(restart_script.encode("utf8"))

        # the dockerfile addition
        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### GRADIO BLOCK #####

USER root
WORKDIR /opt/dataiku

{generate_codenv}

# entrypoint.sh
COPY {entrypoint_file} {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}

# restart_script
COPY {restart_script_file} {restart_script_path}
RUN chown dataiku:root {restart_script_path} && chmod +x {restart_script_path}

# USER dataiku
USER {uid}
WORKDIR /home/dataiku
""".format(entrypoint_path=entrypoint_path, entrypoint_file=self._ENTRYPOINT_FILE,
           restart_script_path=self._get_restart_script_path(), restart_script_file=self._RESTART_SCRIPT,
           generate_codenv=generate_codenv,
           uid=get_dataiku_user_uid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) + "${baseUrlPort" + str(port) + "}/info" # baseUrlPort should be replaced by actual url in BlockBasedCodeStudioMeta/buildYaml
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {
            "label": "Gradio",
            "proxiedUrlSuffix": "/",
            "exposeHtml": True,
            "port": port,
            "readinessProbeUrl": readiness_probe_url
        }
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        return spec

    def build_creation(self, spec, env):
        return spec
        
