(function () {
    "use strict";

    const app = angular.module("dataiku.coloring");

    app.component("conditionalFormattingEditor", {
        bindings: {
            coloring: "=",
            paneState: "=",
            refreshTable: "<",
            saveAndRefresh: "<",
            savePromise: "<",
            setScaleMinMaxFromAnalysis: "<",
        },
        templateUrl: "/static/dataiku/coloring/conditional-formatting/conditional-formatting-editor.component.html",
        controller: function conditionalFormattingEditorController(
            $scope,
            $timeout,
            ATSurveyService,
            ConditionalFormattingEditorService,
            Dialogs,
            Logger,
            openDkuPopin,
            translate,
            WT1
        ) {
            /*
            Initializations
            */
            const ctrl = this;
            ctrl.$onInit = function () {
                // Read attributes and save them into scope
                const defaults = {
                    coloring: {},
                    paneState: {},
                    refreshTable: null,
                    saveAndRefresh: null,
                    savePromise: null,
                    setScaleMinMaxFromAnalysis: null,
                };
                for (const [param, value] of Object.entries(defaults)) {
                    if (param in ctrl) {
                        $scope[param] = ctrl[param] || value;
                    } else {
                        $scope[param] = value;
                    }
                }
                $scope.ConditionalFormattingEditorService = ConditionalFormattingEditorService;
                $scope.groups = [];
                $scope.previousExpandedStates = [];
                $scope.allColumnsBasedOnAnotherColumnGroup = null;
                $scope.focusedGroup = null;
                $scope.pendingMigration = false;

                let isPreviousSchemeColoringGroups = $scope.coloring && $scope.coloring.scheme === "COLORING_GROUPS";

                // Checks if a migration is needed and needs to be saved
                if (
                    ConditionalFormattingEditorService.migrateToColoringGroups(
                        $scope.coloring,
                        $scope.paneState.tableHeaders
                    )
                ) {
                    $scope.pendingMigration = true;

                    // Waits that the migration was saved before trying to initialize the coloring groups
                    $scope
                        .savePromise()
                        .catch(error => {
                            Logger.error("Could not save after migration on conditional formatting", { error });
                        })
                        .finally(() => {
                            // Continues setting up the right pane even if the save had an issue
                            // Refreshes the table without waiting for it to finish
                            $scope.refreshTable(false);

                            init();
                        });
                } else {
                    // A refresh is needed if no migration happened/needed to be saved, but the conditional formatting has no coloring groups.
                    // This is in order to display a "neutral" coloring of the table when coming from MEANING_AND_STATUS or ALL_COLUMNS_VALUES.
                    if ($scope.coloring.coloringGroups.length === 0 && !isPreviousSchemeColoringGroups) {
                        // Refreshes the table without waiting for it to finish
                        $scope.refreshTable(false);
                    }

                    init();
                }
            };

            function init() {
                initUiGroups();

                addOrExpandColumnHeaderGroup();

                if ($scope.isAllColumnsBasedOnAnotherColumnGroupEnabled()) {
                    setEnabledOnAllGroups(false);
                    $scope.allColumnsBasedOnAnotherColumnGroup.coloringGroup.enabled = true;
                    // Update all title tooltips to explain why some groups are disabled
                    updateAllGroupTitleTooltips();
                }

                $scope.pendingMigration = false;
            }

            $scope.isAllColumnsBasedOnAnotherColumnGroupEnabled = function () {
                return (
                    $scope.allColumnsBasedOnAnotherColumnGroup &&
                    $scope.allColumnsBasedOnAnotherColumnGroup.coloringGroup.enabled
                );
            };

            /**
             * Initializes the UI representations of the coloring groups
             */
            function initUiGroups() {
                $scope.groups = [];
                let needToSave = false;

                for (let i = 0; i < $scope.coloring.coloringGroups.length; i++) {
                    needToSave |= addUiGroup(
                        true,
                        $scope.coloring.coloringGroups[i],
                        $scope.previousExpandedStates.length > i ? $scope.previousExpandedStates[i] : false
                    );
                }

                $scope.previousExpandedStates = [];
                $scope.groupsInitialized = true;

                if (needToSave) {
                    $scope.applyChanges();
                }
            }

            /**
             * Adds the associated UI data of a group to the list of UI coloring groups (not the model itself)
             *
             * @param {boolean} highPrio if the group should be placed at the top of the list (true) or bottom (false)
             * @param {ColoringGroup} coloringGroup the coloring group model
             * @param {boolean} expanded
             * @returns {boolean} true if it needs to save otherwise false
             */
            function addUiGroup(highPrio, coloringGroup, expanded) {
                let needToSave = false;
                const title = getUiGroupTitle(coloringGroup);
                const group = {
                    coloringGroup: coloringGroup,
                    expanded: expanded,
                    focused: false,
                    isAllColumnsBasedOnAnotherColumn:
                        ConditionalFormattingEditorService.isAllColumnsBasedOnAnotherColumn(coloringGroup),
                    title: title,
                    titleTooltip: title,
                };

                // If the added group is a duplication, rules are already present
                if (!group.rules) {
                    group.rules = [];
                    const filterDescs = coloringGroup.rulesGroup.filterDescs;
                    for (let i = 0; i < filterDescs.length; i++) {
                        ConditionalFormattingEditorService.addUiRuleToGroup(group, filterDescs[i], true);
                    }
                }

                // Stores the group applied to whole rows in the scope
                if (group.isAllColumnsBasedOnAnotherColumn) {
                    $scope.allColumnsBasedOnAnotherColumnGroup = group;
                }

                // Init color scale definition
                if (ConditionalFormattingEditorService.isSchemeScale(coloringGroup)) {
                    const targetedColumns = group.isAllColumnsBasedOnAnotherColumn
                        ? [coloringGroup.basedOnColumnName]
                        : coloringGroup.targetedColumnNames;
                    if (!coloringGroup.colorScaleDef || !coloringGroup.colorScaleDef.colorPalette) {
                        coloringGroup.colorScaleDef = ConditionalFormattingEditorService.newColorScaleDef(
                            targetedColumns,
                            $scope.paneState.tableHeaders,
                            true
                        );
                        $scope.initScaleMinMax(targetedColumns, group);
                        needToSave = true;
                    } else {
                        group.scaleColumnType = ConditionalFormattingEditorService.getColumnType(
                            targetedColumns,
                            $scope.paneState.tableHeaders
                        );
                    }
                    const sample = ConditionalFormattingEditorService.getColorScaleDefSample(
                        coloringGroup.colorScaleDef
                    );
                    group.scaleBorderGradient = ConditionalFormattingEditorService.getScaleBorderGradient(sample);
                }

                if (highPrio) {
                    $scope.groups.unshift(group);
                } else {
                    $scope.groups.push(group);
                }

                return needToSave;
            }

            $scope.initScaleMinMax = function (targetedColumns, group) {
                let resetMinMax = true;
                if (targetedColumns && targetedColumns.length !== 0 && group && group.coloringGroup) {
                    group.scaleColumnType = ConditionalFormattingEditorService.getColumnType(
                        targetedColumns,
                        $scope.paneState.tableHeaders
                    );

                    if (group.scaleColumnType === "NUM" && $scope.setScaleMinMaxFromAnalysis) {
                        $scope.setScaleMinMaxFromAnalysis(targetedColumns[0], group.coloringGroup.colorScaleDef);
                        resetMinMax = false;
                    }
                }

                if (resetMinMax && group.coloringGroup.colorScaleDef) {
                    // Sets max lower than min to be not computed by the backend
                    group.coloringGroup.colorScaleDef.max = -1;
                    group.coloringGroup.colorScaleDef.min = 0;
                }
            };

            function getGroupTitleNoColumnsSelected(index) {
                return translate(
                    "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.TITLE_NO_COLUMNS",
                    "Rule " + index + " (no columns selected)",
                    { index: index }
                );
            }

            /**
             * Returns the column name or the column name + message if the column does not exist
             */
            function getColumnName(columnName) {
                if (!$scope.paneState.columns.find(obj => obj.name === columnName)) {
                    return translate(
                        "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.COLUMN_UNKNOWN",
                        columnName + " (no longer exists)",
                        { columnName: columnName }
                    );
                }
                return columnName;
            }

            /**
             * Get the title of a coloring group to display on the UI
             *
             * @param {ColoringGroup} coloringGroup the coloring group model
             * @returns {string} the title of the given group
             */
            function getUiGroupTitle(coloringGroup) {
                if (coloringGroup) {
                    if ($scope.ConditionalFormattingEditorService.isAllColumnsBasedOnAnotherColumn(coloringGroup)) {
                        if (coloringGroup.basedOnColumnName) {
                            return getColumnName(coloringGroup.basedOnColumnName);
                        }
                    } else if (coloringGroup.targetedColumnNames && coloringGroup.targetedColumnNames.length) {
                        let title = getColumnName(coloringGroup.targetedColumnNames[0]);
                        for (let i = 1; i < coloringGroup.targetedColumnNames.length; i++) {
                            title += ", " + getColumnName(coloringGroup.targetedColumnNames[i]);
                        }
                        return title;
                    }
                    // Default group title with id
                    for (let i = 0; i < $scope.coloring.coloringGroups.length; i++) {
                        if (coloringGroup === $scope.coloring.coloringGroups[i]) {
                            return getGroupTitleNoColumnsSelected($scope.coloring.coloringGroups.length - i);
                        }
                    }
                }
                // Should never happen
                return "Rule XXX";
            }

            function findFirstGroupTargetingThisColumn(columnName) {
                // Searches the first group targeting columnName only or targeting multi columns containing columnName and enabled
                for (let i = 0; i < $scope.groups.length; i++) {
                    const group = $scope.groups[i];
                    const coloringGroup = group.coloringGroup;

                    // ColumnName includes in targetedColumnNames?
                    if (
                        !group.isAllColumnsBasedOnAnotherColumn &&
                        coloringGroup.targetedColumnNames.includes(columnName) &&
                        (coloringGroup.targetedColumnNames.length === 1 || coloringGroup.enabled)
                    ) {
                        return group;
                    }
                }
                return null;
            }

            function focusOnGroup(group) {
                // Expands the group if it is not expanded
                if (!group.expanded) {
                    $scope.toggleGroup(group);
                }
                $scope.setFocusOnGroup(group);

                setScrollbarToSeeGroupEntirely(group);
            }

            function addOrExpandColumnHeaderGroup() {
                if ($scope.paneState.openedFromColumnHeader && $scope.groupsInitialized) {
                    if ($scope.isAllColumnsBasedOnAnotherColumnGroupEnabled()) {
                        if (
                            $scope.allColumnsBasedOnAnotherColumnGroup.coloringGroup.basedOnColumnName ===
                            $scope.paneState.openedFromColumnHeader
                        ) {
                            focusOnGroup($scope.allColumnsBasedOnAnotherColumnGroup);
                        } else {
                            const groupFound = findFirstGroupTargetingThisColumn(
                                $scope.paneState.openedFromColumnHeader
                            );
                            if (groupFound) {
                                $scope.enableGroup(groupFound, true);
                            } else {
                                // Adds a new group targeting $scope.paneState.openedFromColumnHeader after confirming the modal
                                $scope.addGroup([$scope.paneState.openedFromColumnHeader]);
                            }
                        }
                    } else {
                        const groupFound = findFirstGroupTargetingThisColumn($scope.paneState.openedFromColumnHeader);
                        if (groupFound) {
                            focusOnGroup(groupFound);
                            if (!groupFound.coloringGroup.enabled) {
                                groupFound.coloringGroup.enabled = true;

                                $scope.applyChanges();
                            }
                        } else {
                            // Adds a new group targeting $scope.paneState.openedFromColumnHeader
                            addNewGroup([$scope.paneState.openedFromColumnHeader]);

                            $scope.applyChanges();
                        }
                    }

                    // Removes $scope.paneState.openedFromColumnHeader to avoid opening pane with this column group next time
                    delete $scope.paneState.openedFromColumnHeader;
                }
            }

            function setScrollbarToSeeGroupEntirely(group) {
                const groupIndex = getGroupIndex(group);
                if (groupIndex === -1) {
                    // Should never happen
                    return;
                }

                // Uses $timeout for waiting the next digest cycle to have good bounding client rect values
                $timeout(function () {
                    const paneDiv = document.getElementById("cf-pane");
                    const paneRect = paneDiv.getBoundingClientRect();

                    const allDivGroups = document.getElementsByName("cf-group");
                    if (groupIndex >= allDivGroups.length) {
                        // Should never happen
                        return;
                    }
                    const groupRect = allDivGroups[groupIndex].getBoundingClientRect();

                    // Sets the scrollbar if the top part of the group is not fully visible
                    // Case for the first groups with the scrollbar on bottom
                    if (groupRect.top < paneRect.top) {
                        paneDiv.scrollTop += groupRect.top - paneRect.top;
                    }
                    // Case for the latest groups with the scrollbar on top
                    else if (groupRect.bottom > paneRect.bottom) {
                        let delta = groupRect.bottom - paneRect.bottom;
                        if (groupRect.height > paneRect.height) {
                            delta -= groupRect.height - paneRect.height;
                        }
                        paneDiv.scrollTop += delta;
                    }
                }, 100);
            }

            function setEnabledOnAllGroups(enabled) {
                for (let i = 0; i < $scope.groups.length; i++) {
                    $scope.groups[i].coloringGroup.enabled = enabled;
                }
            }

            function updateAllDefaultGroupTitles() {
                for (let i = 0; i < $scope.groups.length; i++) {
                    const coloringGroup = $scope.groups[i].coloringGroup;
                    const isAllColumnsBasedOnAnotherColumn = $scope.groups[i].isAllColumnsBasedOnAnotherColumn;
                    if (
                        (isAllColumnsBasedOnAnotherColumn && !coloringGroup.basedOnColumnName) ||
                        (!isAllColumnsBasedOnAnotherColumn &&
                            (!coloringGroup.targetedColumnNames || !coloringGroup.targetedColumnNames.length))
                    ) {
                        $scope.groups[i].title = getGroupTitleNoColumnsSelected(i + 1);
                    }
                }
            }

            function updateAllGroupTitleTooltips() {
                if ($scope.isAllColumnsBasedOnAnotherColumnGroupEnabled()) {
                    // This message to explain why the group is disabled when "Apply to whole rows" is enabled
                    const disabledMessage = translate(
                        "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.TITLE_TOOLTIP_DISABLED",
                        "This rule is disabled because the rule '" +
                            $scope.allColumnsBasedOnAnotherColumnGroup.title +
                            "' is applied to whole rows",
                        { title: $scope.allColumnsBasedOnAnotherColumnGroup.title }
                    );

                    for (let i = 0; i < $scope.groups.length; i++) {
                        if (!$scope.groups[i].isAllColumnsBasedOnAnotherColumn) {
                            $scope.groups[i].titleTooltip = disabledMessage;
                        } else {
                            $scope.groups[i].titleTooltip = $scope.groups[i].title;
                        }
                    }
                } else {
                    for (let i = 0; i < $scope.groups.length; i++) {
                        $scope.groups[i].titleTooltip = $scope.groups[i].title;
                    }
                }
            }

            $scope.addGroup = function (targetedColumnNames = []) {
                if ($scope.isAllColumnsBasedOnAnotherColumnGroupEnabled()) {
                    // Need to confirm via a modal to add a rule
                    Dialogs.confirm(
                        $scope,
                        translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.MODAL_ADD_TITLE", "Add a rule"),
                        translate(
                            "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.MODAL_ADD_TEXT",
                            "Adding a rule will remove applying the rule <strong>'" +
                                $scope.allColumnsBasedOnAnotherColumnGroup.title +
                                "'</strong> to whole rows.<br><br>Are you sure you want to continue?",
                            { title: $scope.allColumnsBasedOnAnotherColumnGroup.title }
                        )
                    ).then(
                        function () {
                            // Confirm
                            $scope.removeColoringToWholeRows(false);
                            addNewGroup(targetedColumnNames);

                            $scope.applyChanges();
                        },
                        function () {
                            // Empty function to prevent error in console
                        }
                    );
                } else {
                    addNewGroup();
                }
            };

            function addNewGroup(targetedColumnNames = []) {
                const coloringGroup = $scope.ConditionalFormattingEditorService.newColoringGroup({
                    scope: "COLUMNS",
                    scheme: "RULES",
                    targetedColumnNames,
                });

                coloringGroup.rulesGroup.filterDescs.push(
                    $scope.ConditionalFormattingEditorService.getDefaultFilterDesc()
                );
                // Add new group at the beginning of the array to be low priority
                $scope.coloring.coloringGroups.unshift(coloringGroup);
                addUiGroup(false, coloringGroup, false);
                // Collapses all groups and expands the new one
                const lastGroup = $scope.groups[$scope.groups.length - 1];
                $scope.toggleGroup(lastGroup);

                ATSurveyService.updateCounter('ConditionalFormattingRuleAdd');
                WT1.event("conditional-formatting-rule-add", {
                    nbRules: $scope.coloring.coloringGroups.length,
                });

                setScrollbarToSeeGroupEntirely(lastGroup);

                updateAllDefaultGroupTitles();
            }

            $scope.setFocusOnGroup = function (group) {
                if ($scope.focusedGroup) {
                    $scope.focusedGroup.focused = false;
                }
                $scope.focusedGroup = group;
                group.focused = true;
            };

            $scope.toggleGroup = function (group) {
                group.expanded = !group.expanded;
                $scope.setFocusOnGroup(group);
            };

            /**
             * Finds the index of the group.
             * Returns the index found or -1 if not found
             */
            function getGroupIndex(group) {
                return $scope.groups.indexOf(group);
            }

            $scope.removeGroup = function (group) {
                const index = getGroupIndex(group);
                if (index === -1) {
                    // Should never happen
                    return;
                }

                if (group.isAllColumnsBasedOnAnotherColumn) {
                    $scope.removeColoringToWholeRows(true, false);
                }

                if (index < $scope.groups.length) {
                    const coloringModeToDelete = $scope.groups[index].coloringGroup.scheme;
                    $scope.coloring.coloringGroups.splice($scope.groups.length - (index + 1), 1);
                    $scope.groups.splice(index, 1);

                    WT1.event("conditional-formatting-rule-delete", {
                        coloringModeDeleted: coloringModeToDelete,
                        nbRules: $scope.coloring.coloringGroups.length,
                    });

                    updateAllDefaultGroupTitles();

                    $scope.applyChanges();
                }
            };

            $scope.updateGroupTitle = function (group) {
                group.title = getUiGroupTitle(group.coloringGroup);
                group.titleTooltip = group.title;

                if (group.isAllColumnsBasedOnAnotherColumn) {
                    updateAllGroupTitleTooltips();
                }
            };

            function updateAllGroupTitles() {
                for (let i = 0; i < $scope.groups.length; i++) {
                    $scope.updateGroupTitle($scope.groups[i]);
                }
            }

            var applyChangesTimeout = null;
            $scope.applyChanges = function () {
                if (applyChangesTimeout !== null) {
                    clearTimeout(applyChangesTimeout);
                }
                applyChangesTimeout = setTimeout(function () {
                    $scope.saveAndRefresh();
                }, 1000);
            };

            $scope.enableGroup = function (group, needToFocusOnGroup = false) {
                // Case where the input group is applied to whole rows
                if (group.isAllColumnsBasedOnAnotherColumn) {
                    if (group.coloringGroup.enabled) {
                        // Re-applies to whole rows
                        $scope.applyColoringToWholeRows(group);
                    } else {
                        updateAllGroupTitleTooltips();

                        $scope.applyChanges();
                    }
                    return;
                }

                // Case where the input group is not applied to whole rows
                if ($scope.isAllColumnsBasedOnAnotherColumnGroupEnabled()) {
                    // Need to confirm via a modal to remove "Apply to whole rows" feature
                    Dialogs.confirm(
                        $scope,
                        translate(
                            "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.MODAL_ENABLE_TITLE",
                            "Enable the rule '" + group.title + "'",
                            { title: group.title }
                        ),
                        translate(
                            "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.MODAL_ENABLE_TEXT",
                            "Enabling this rule will remove applying the rule <strong>'" +
                                $scope.allColumnsBasedOnAnotherColumnGroup.title +
                                "'</strong> to whole rows.<br><br>Are you sure you want to continue?",
                            { title: $scope.allColumnsBasedOnAnotherColumnGroup.title }
                        )
                    ).then(
                        function () {
                            // Confirm
                            group.coloringGroup.enabled = true;
                            $scope.removeColoringToWholeRows(false);

                            if (needToFocusOnGroup) {
                                focusOnGroup(group);
                            }

                            $scope.applyChanges();
                        },
                        function () {
                            // Cancel
                            group.coloringGroup.enabled = false;
                        }
                    );
                } else {
                    $scope.applyChanges();
                }
            };

            $scope.btnEnableGroupTitle = function (group) {
                if (group.coloringGroup.enabled) {
                    return translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.DISABLE", "Disable this rule");
                }

                return translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.ENABLE", "Enable this rule");
            };

            /*
             * Options menu
             */
            $scope.btnOptionsTitle = function () {
                return translate("SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.OPTIONS", "More options");
            };

            $scope.openGroupMenu = function ($event, group) {
                // dismiss existing menu
                if (typeof $scope.activeGroupMenu === "function") {
                    $scope.activeGroupMenu();
                }

                function isElsewhere() {
                    return true;
                }

                const newScope = $scope.$new();
                newScope.group = group;
                newScope.isAllColumnsBasedOnAnotherColumnDisabled =
                    !group.isAllColumnsBasedOnAnotherColumn && group.coloringGroup.targetedColumnNames.length !== 1;
                newScope.isAllColumnsBasedOnAnotherColumnDisabledTooltip = function () {
                    if (!group.isAllColumnsBasedOnAnotherColumn) {
                        if (group.coloringGroup.targetedColumnNames.length !== 1) {
                            return translate(
                                "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.OPTIONS.APPLY_COLORING_TO_WHOLE_ROWS.INVALID_COLUMN",
                                "Disabled because no or several columns are selected"
                            );
                        }
                    }
                    return "";
                };

                const template = `
                    <ul class="dropdown-menu z-index-10000">
                        <li title="{{isAllColumnsBasedOnAnotherColumnDisabledTooltip()}}">
                            <a ng-if="!group.isAllColumnsBasedOnAnotherColumn || !group.coloringGroup.enabled" ng-click="applyColoringToWholeRows(group);" ng-disabled="isAllColumnsBasedOnAnotherColumnDisabled">
                                <i class="dku-icon-arrow-double-horizontal-16 dibvab"/>
                                <span translate="SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.OPTIONS.APPLY_COLORING_TO_WHOLE_ROWS">Apply this rule to whole rows</span>
                            </a>
                            <a ng-if="group.isAllColumnsBasedOnAnotherColumn && group.coloringGroup.enabled" ng-click="removeColoringToWholeRows(true);" title="{{removeColoringToWholeRowsTooltip()}}">
                                <i class="dku-icon-arrow-double-horizontal-16 dibvab"/>
                                <span translate="SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.OPTIONS.REMOVE_COLORING_TO_WHOLE_ROWS">Remove apply this rule to whole rows</span>
                            </a>
                        </li>
                        <li>
                            <a ng-click="duplicateGroup(group);" ng-disabled="group.isAllColumnsBasedOnAnotherColumn">
                                <i class="dku-icon-copy-16 dibvab"/>
                                <span translate="SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.OPTIONS.DUPLICATE">Duplicate this rule</span>
                            </a>
                        </li>
                        <li>
                            <a ng-click="removeGroup(group);" class="text-error">
                                <i class="dku-icon-trash-16 dibvab"/>
                                <span translate="SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.OPTIONS.DELETE">Delete this rule</span>
                            </a>
                        </li>
                    </ul>
                `;

                const dkuPopinOptions = {
                    template: template,
                    isElsewhere: isElsewhere,
                    callback: null,
                    popinPosition: "CLICK",
                    onDismiss: () => {
                        $scope.activeGroupMenu = null;
                    },
                };

                $scope.activeGroupMenu = openDkuPopin(newScope, $event, dkuPopinOptions);
            };

            $scope.applyColoringToWholeRows = function (group) {
                if ($scope.allColumnsBasedOnAnotherColumnGroup) {
                    // Removes old "Apply to whole rows" group
                    $scope.removeColoringToWholeRows(true, false);
                }

                const indexGroupWholeRows = getGroupIndex(group);
                if (indexGroupWholeRows === -1) {
                    // Should never happen
                    return;
                }
                WT1.event("conditional-formatting-whole-rows-apply", {
                    indexRuleApplied: indexGroupWholeRows,
                });

                $scope.allColumnsBasedOnAnotherColumnGroup = group;
                group.isAllColumnsBasedOnAnotherColumn = true;
                const coloringGroup = group.coloringGroup;

                // Sets basedOnColumnName with the first column of targetedColumnNames
                if (coloringGroup.targetedColumnNames.length) {
                    coloringGroup.basedOnColumnName = coloringGroup.targetedColumnNames[0];
                    coloringGroup.targetedColumnNames = [];
                }
                coloringGroup.scope = "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN";

                setEnabledOnAllGroups(false);
                coloringGroup.enabled = true;
                updateAllGroupTitleTooltips();

                $scope.applyChanges();
            };

            $scope.removeColoringToWholeRows = function (enableAll, applyChanges = true) {
                const indexGroupWholeRows = getGroupIndex($scope.allColumnsBasedOnAnotherColumnGroup);
                if (indexGroupWholeRows === -1) {
                    // Should never happen
                    return;
                }
                WT1.event("conditional-formatting-whole-rows-remove", {
                    indexRuleApplied: indexGroupWholeRows,
                });

                const group = $scope.allColumnsBasedOnAnotherColumnGroup;
                $scope.allColumnsBasedOnAnotherColumnGroup = null;
                group.isAllColumnsBasedOnAnotherColumn = false;
                const coloringGroup = group.coloringGroup;

                // Resets targetedColumnNames and adds basedOnColumnName in targetedColumnNames
                coloringGroup.targetedColumnNames = [];
                if (coloringGroup.basedOnColumnName) {
                    coloringGroup.targetedColumnNames.push(coloringGroup.basedOnColumnName);
                    coloringGroup.basedOnColumnName = null;
                }
                coloringGroup.scope = "COLUMNS";

                if (enableAll) {
                    setEnabledOnAllGroups(true);
                }
                updateAllGroupTitleTooltips();

                if (applyChanges) {
                    $scope.applyChanges();
                }
            };

            $scope.removeColoringToWholeRowsTooltip = function () {
                return translate(
                    "SHAKER.VIEW.CONDITIONAL_FORMATTING.GROUP.APPLIED_TO_WHOLE_ROWS.REMOVE.TOOLTIP",
                    "This will enable all the rules"
                );
            };

            $scope.duplicateGroup = function (group) {
                // Finds the index of the group
                const index = getGroupIndex(group);
                if (index === -1) {
                    // Should never happen
                    return;
                }

                const newGroup = angular.copy(group);
                if (newGroup.colorScalePaletteSelectorId) {
                    // To force the creation of new palette
                    delete newGroup.colorScalePaletteSelectorId;
                }
                $scope.setFocusOnGroup(newGroup);

                $scope.coloring.coloringGroups.splice($scope.groups.length - (index + 1), 0, newGroup.coloringGroup);
                $scope.groups.splice(index + 1, 0, newGroup);

                WT1.event("conditional-formatting-rule-duplicate", {
                    nbRules: $scope.coloring.coloringGroups.length,
                });

                updateAllGroupTitles();

                $scope.applyChanges(true);
            };

            /*
             * Reordering Groups
             */
            function afterGroupMove(event) {
                const destIndex = event.dest.index;
                const srcIndex = event.source.index;

                if (destIndex !== srcIndex) {
                    const destIndexReversed = $scope.coloring.coloringGroups.length - (destIndex + 1);
                    const srcIndexReversed = $scope.coloring.coloringGroups.length - (srcIndex + 1);
                    const coloringGroupSrc = $scope.coloring.coloringGroups[srcIndexReversed];
                    $scope.coloring.coloringGroups.splice(srcIndexReversed, 1);
                    $scope.coloring.coloringGroups.splice(destIndexReversed, 0, coloringGroupSrc);
                    updateAllDefaultGroupTitles();

                    $scope.applyChanges();
                }
            }

            $scope.childNodesCount = function () {
                return $scope.groups.length;
            };

            $scope.treeOptions = {
                dropped: afterGroupMove,
                accept: function (sourceNodeScope, destNodesScope, destIndex) {
                    // Authorizes drag and drop in same parent
                    return sourceNodeScope.$parent.$id === destNodesScope.$id;
                },
            };

            /*
            Watchers
            */
            $scope.$watch(
                "paneState.openedFromColumnHeader",
                function (newVal, oldVal) {
                    addOrExpandColumnHeaderGroup();
                },
                false
            );
            $scope.$watch(
                "paneState.columns",
                function (newVal, oldVal) {
                    // If columns have changed we need to update the pane
                    if (newVal !== oldVal) {
                        // Stores expanded states to refresh the pane with old states
                        $scope.previousExpandedStates = [];
                        for (let i = 0; i < $scope.groups.length; i++) {
                            $scope.previousExpandedStates.unshift($scope.groups[i].expanded);
                        }

                        // Reinitializes the pane
                        init();
                    }
                },
                true // Compare for object equality using angular.equals instead of comparing for reference equality
            );
        },
    });
})();
