/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.connections;

import com.dataiku.dip.connections.CassandraConnection;
import com.dataiku.dip.connections.ConnectionWithBasicCredential;
import com.dataiku.dip.connections.ConnectionsDAO;
import com.dataiku.dip.coremodel.Schema;
import com.dataiku.dip.coremodel.SchemaColumn;
import com.dataiku.dip.datasets.Type;
import com.dataiku.dip.exceptions.DKUSecurityException;
import com.dataiku.dip.security.AuthCtx;
import com.dataiku.dip.security.model.ICredentialsService;
import com.dataiku.dip.server.SpringUtils;
import com.dataiku.dip.shaker.types.AnyTemporal;
import com.dataiku.dip.shaker.types.DateOnly;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.DKUtils;
import com.dataiku.dip.utils.ErrorContext;
import com.dataiku.dip.utils.JSON;
import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.Cluster;
import com.datastax.driver.core.CodecRegistry;
import com.datastax.driver.core.ColumnMetadata;
import com.datastax.driver.core.DataType;
import com.datastax.driver.core.KeyspaceMetadata;
import com.datastax.driver.core.LocalDate;
import com.datastax.driver.core.Metadata;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ProtocolVersion;
import com.datastax.driver.core.Row;
import com.datastax.driver.core.Session;
import com.datastax.driver.core.SocketOptions;
import com.datastax.driver.core.TableMetadata;
import com.google.common.net.InetAddresses;
import com.google.common.reflect.TypeToken;
import java.io.Closeable;
import java.io.IOException;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.regex.Pattern;
import org.apache.commons.lang.NotImplementedException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

public class CassandraUtil {
    private static DateOnly dateOnlyMeaning = new DateOnly();
    private static AnyTemporal temporalMeaning = new AnyTemporal();
    private static final ConnectionCache connectionCache = new ConnectionCache();
    private static final Pattern lowercaseId = Pattern.compile("[a-z][a-z0-9_]*");
    public static final String DSS_COLUMN_PREFIX = "_dss";
    public static final String ROWID_COLUMN = "_dssId";
    public static final String SPREAD_COLUMN = "_dssSpread";
    protected static DKULogger logger = DKULogger.getLogger((String)"dku.cassandra");

    public static ClusterConnection acquireConnection(AuthCtx authCtx, CassandraConnection conn) throws DKUSecurityException {
        return connectionCache.acquire(authCtx, conn, conn.params);
    }

    public static ClusterConnection acquireConnection(AuthCtx authCtx, String connectionName) throws IOException, DKUSecurityException {
        CassandraConnection conn = ((ConnectionsDAO)SpringUtils.getBean(ConnectionsDAO.class)).getMandatoryConnectionAs(authCtx, connectionName, CassandraConnection.class);
        return CassandraUtil.acquireConnection(authCtx, conn);
    }

    public static void releaseConnection(ClusterConnection conn) {
        connectionCache.release(conn);
    }

    public static DataType getCassandraType(SchemaColumn sc) {
        switch (sc.getType()) {
            case TINYINT: 
            case SMALLINT: 
            case INT: {
                return DataType.cint();
            }
            case BIGINT: {
                return DataType.bigint();
            }
            case FLOAT: {
                return DataType.cfloat();
            }
            case DOUBLE: {
                return DataType.cdouble();
            }
            case STRING: {
                return DataType.text();
            }
            case BOOLEAN: {
                return DataType.cboolean();
            }
            case DATE: {
                return DataType.timestamp();
            }
            case DATETIMENOTZ: {
                return DataType.timestamp();
            }
            case DATEONLY: {
                return DataType.date();
            }
            case MAP: {
                return DataType.map((DataType)DataType.text(), (DataType)DataType.text());
            }
            case ARRAY: {
                return DataType.list((DataType)DataType.text());
            }
        }
        throw new Error("unreachable");
    }

    public static void checkManagedTableSchema(TableMetadata tableMeta, Schema schema) {
        int n = 0;
        for (ColumnMetadata column : tableMeta.getColumns()) {
            String colName = column.getName();
            if (colName.startsWith(DSS_COLUMN_PREFIX)) continue;
            SchemaColumn sc = schema.getColumn(colName);
            if (sc == null) {
                throw ErrorContext.iaef((String)"table column %s not found in schema", (Object)colName, (Object[])new Object[0]);
            }
            if (!column.getType().equals(CassandraUtil.getCassandraType(sc))) {
                throw ErrorContext.iaef((String)"type mismatch for table column %s : %s should be %s", (Object)colName, (Object[])new Object[]{column.getType(), CassandraUtil.getCassandraType(sc)});
            }
            ++n;
        }
        if (n != schema.getColumns().size()) {
            for (SchemaColumn sc : schema.getColumns()) {
                if (tableMeta.getColumn(CassandraUtil.forceQuote(sc.getName())) != null) continue;
                throw ErrorContext.iaef((String)"schema column %s not found in table", (Object)sc.getName(), (Object[])new Object[0]);
            }
            throw ErrorContext.iae((String)"Unexpected table schema mismatch");
        }
    }

    public static Type getDSSType(DataType.Name cassandraType) {
        switch (cassandraType) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: 
            case TIMEUUID: 
            case UUID: 
            case INET: 
            case DECIMAL: 
            case VARINT: {
                return Type.STRING;
            }
            case BOOLEAN: {
                return Type.BOOLEAN;
            }
            case INT: {
                return Type.INT;
            }
            case BIGINT: 
            case COUNTER: {
                return Type.BIGINT;
            }
            case FLOAT: {
                return Type.FLOAT;
            }
            case DOUBLE: {
                return Type.DOUBLE;
            }
            case TIMESTAMP: {
                return Type.DATE;
            }
            case DATE: {
                return Type.DATEONLY;
            }
            case LIST: 
            case SET: {
                return Type.ARRAY;
            }
            case MAP: {
                return Type.MAP;
            }
        }
        return null;
    }

    public static Schema getSchemaFromTable(TableMetadata tableMeta) {
        Schema ret = new Schema();
        for (ColumnMetadata column : tableMeta.getColumns()) {
            Type dssType;
            String colName = column.getName();
            if (colName.startsWith(DSS_COLUMN_PREFIX) || (dssType = CassandraUtil.getDSSType(column.getType().getName())) == null) continue;
            ret.getColumns().add(new SchemaColumn(colName, dssType));
        }
        return ret;
    }

    public static void checkExternalTableSchema(TableMetadata tableMeta, Schema schema) {
        int n = 0;
        for (ColumnMetadata column : tableMeta.getColumns()) {
            Type dssType;
            String colName = column.getName();
            if (colName.startsWith(DSS_COLUMN_PREFIX) || (dssType = CassandraUtil.getDSSType(column.getType().getName())) == null) continue;
            SchemaColumn sc = schema.getColumn(colName);
            if (sc == null) {
                throw ErrorContext.iaef((String)"table column %s not found in schema", (Object)colName, (Object[])new Object[0]);
            }
            if (sc.getType() != dssType) {
                throw ErrorContext.iaef((String)"schema mismatch for column %s : %s should be %s", (Object)colName, (Object[])new Object[]{sc.getType().name(), dssType.name()});
            }
            ++n;
        }
        if (n != schema.getColumns().size()) {
            for (SchemaColumn sc : schema.getColumns()) {
                if (tableMeta.getColumn(CassandraUtil.forceQuote(sc.getName())) != null) continue;
                throw ErrorContext.iaef((String)"schema column %s not found in table", (Object)sc.getName(), (Object[])new Object[0]);
            }
            throw ErrorContext.iae((String)"Unexpected table schema mismatch");
        }
    }

    public static Object getCassandraValue(Object o, DataType.Name type) {
        switch (type) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: 
            case TIMEUUID: 
            case UUID: {
                return o.toString();
            }
            case DECIMAL: 
            case VARINT: 
            case BOOLEAN: 
            case INT: 
            case BIGINT: 
            case COUNTER: 
            case FLOAT: 
            case DOUBLE: {
                return o;
            }
            case TIMESTAMP: {
                return DKUtils.isoFormatReadableByDateFormat((long)((Date)o).getTime());
            }
            case DATE: {
                return DateOnly.FORMATTER.print(((Date)o).getTime());
            }
            case INET: {
                return ((InetAddress)o).getHostAddress();
            }
            case LIST: 
            case SET: 
            case MAP: {
                return null;
            }
        }
        return null;
    }

    public static String getCassandraValue(Row dbRow, int i) {
        switch (dbRow.getColumnDefinitions().getType(i).getName()) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return dbRow.getString(i);
            }
            case BOOLEAN: {
                return Boolean.toString(dbRow.getBool(i));
            }
            case INT: {
                return Integer.toString(dbRow.getInt(i));
            }
            case BIGINT: 
            case COUNTER: {
                return Long.toString(dbRow.getLong(i));
            }
            case FLOAT: {
                return Float.toString(dbRow.getFloat(i));
            }
            case DOUBLE: {
                return Double.toString(dbRow.getDouble(i));
            }
            case TIMESTAMP: {
                Date d = dbRow.getTimestamp(i);
                return d == null ? null : DKUtils.isoFormatReadableByDateFormat((long)d.getTime());
            }
            case DATE: {
                LocalDate d = dbRow.getDate(i);
                return d == null ? null : d.toString();
            }
            case DECIMAL: {
                return dbRow.getDecimal(i).toString();
            }
            case VARINT: {
                return dbRow.getVarint(i).toString();
            }
            case TIMEUUID: 
            case UUID: {
                return dbRow.getUUID(i).toString();
            }
            case INET: {
                return dbRow.getInet(i).getHostAddress();
            }
            case LIST: {
                DataType type = (DataType)dbRow.getColumnDefinitions().getType(i).getTypeArguments().get(0);
                TypeToken javaType = CodecRegistry.DEFAULT_INSTANCE.codecFor(type).getJavaType();
                ArrayList<Object> result = new ArrayList<Object>();
                for (Object o : dbRow.getList(i, javaType)) {
                    result.add(CassandraUtil.getCassandraValue(o, type.getName()));
                }
                return JSON.json(result);
            }
            case MAP: {
                DataType keyType = (DataType)dbRow.getColumnDefinitions().getType(i).getTypeArguments().get(0);
                DataType valueType = (DataType)dbRow.getColumnDefinitions().getType(i).getTypeArguments().get(1);
                TypeToken keyJavaType = CodecRegistry.DEFAULT_INSTANCE.codecFor(keyType).getJavaType();
                TypeToken valueJavaType = CodecRegistry.DEFAULT_INSTANCE.codecFor(valueType).getJavaType();
                HashMap<String, Object> result = new HashMap<String, Object>();
                for (Map.Entry entry : dbRow.getMap(i, keyJavaType, valueJavaType).entrySet()) {
                    Object key = CassandraUtil.getCassandraValue(entry.getKey(), keyType.getName());
                    if (key == null) continue;
                    Object value = CassandraUtil.getCassandraValue(entry.getValue(), valueType.getName());
                    result.put(key.toString(), value);
                }
                return JSON.json(result);
            }
            case SET: {
                DataType type = (DataType)dbRow.getColumnDefinitions().getType(i).getTypeArguments().get(0);
                TypeToken javaType = CodecRegistry.DEFAULT_INSTANCE.codecFor(type).getJavaType();
                HashSet<Object> result = new HashSet<Object>();
                for (Object o : dbRow.getSet(i, javaType)) {
                    result.add(CassandraUtil.getCassandraValue(o, type.getName()));
                }
                return JSON.json(result);
            }
        }
        return null;
    }

    public static Object toCassandraValue(String val, DataType.Name type) {
        switch (type) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                return val;
            }
            case BOOLEAN: {
                return Boolean.parseBoolean(val);
            }
            case INT: {
                return Integer.parseInt(val);
            }
            case BIGINT: 
            case COUNTER: {
                return Long.parseLong(val);
            }
            case FLOAT: {
                return Float.valueOf(Float.parseFloat(val));
            }
            case DOUBLE: {
                return Double.parseDouble(val);
            }
            case TIMESTAMP: {
                long timestamp = temporalMeaning.msSinceEpoch(val);
                if (timestamp == Long.MAX_VALUE) {
                    throw new IllegalArgumentException("invalid temporal value: " + val);
                }
                return new Date(timestamp);
            }
            case DATE: {
                long timestamp = dateOnlyMeaning.msSinceEpoch(val);
                if (timestamp == Long.MAX_VALUE) {
                    throw new IllegalArgumentException("invalid timestamp value: " + val);
                }
                return LocalDate.fromMillisSinceEpoch((long)timestamp);
            }
            case DECIMAL: {
                return new BigDecimal(val);
            }
            case VARINT: {
                return new BigInteger(val);
            }
            case TIMEUUID: 
            case UUID: {
                return UUID.fromString(val);
            }
            case INET: {
                return InetAddresses.forString((String)val);
            }
            case LIST: 
            case SET: 
            case MAP: {
                throw new NotImplementedException("nested collection type not supported: " + val);
            }
        }
        return null;
    }

    public static void setCassandraValue(BoundStatement st2, int i, String val) {
        switch (st2.preparedStatement().getVariables().getType(i).getName()) {
            case ASCII: 
            case TEXT: 
            case VARCHAR: {
                st2.setString(i, val);
                break;
            }
            case BOOLEAN: {
                st2.setBool(i, Boolean.parseBoolean(val));
                break;
            }
            case INT: {
                st2.setInt(i, Integer.parseInt(val));
                break;
            }
            case BIGINT: 
            case COUNTER: {
                st2.setLong(i, Long.parseLong(val));
                break;
            }
            case FLOAT: {
                st2.setFloat(i, Float.parseFloat(val));
                break;
            }
            case DOUBLE: {
                st2.setDouble(i, Double.parseDouble(val));
                break;
            }
            case TIMESTAMP: {
                long timestamp = temporalMeaning.msSinceEpoch(val);
                if (timestamp == Long.MAX_VALUE) {
                    throw new IllegalArgumentException("invalid temporal value: " + val);
                }
                st2.setTimestamp(i, new Date(timestamp));
                break;
            }
            case DATE: {
                long timestamp = dateOnlyMeaning.msSinceEpoch(val);
                if (timestamp == Long.MAX_VALUE) {
                    throw new IllegalArgumentException("invalid timestamp value: " + val);
                }
                st2.setDate(i, LocalDate.fromMillisSinceEpoch((long)timestamp));
                break;
            }
            case DECIMAL: {
                st2.setDecimal(i, new BigDecimal(val));
                break;
            }
            case VARINT: {
                st2.setVarint(i, new BigInteger(val));
                break;
            }
            case TIMEUUID: 
            case UUID: {
                st2.setUUID(i, UUID.fromString(val));
                break;
            }
            case INET: {
                st2.setInet(i, InetAddresses.forString((String)val));
                break;
            }
            case LIST: {
                try {
                    DataType type = (DataType)st2.preparedStatement().getVariables().getType(i).getTypeArguments().get(0);
                    JSONArray obj = new JSONArray(val);
                    ArrayList<Object> list = new ArrayList<Object>();
                    for (int j = 0; j < obj.length(); ++j) {
                        Object so = obj.get(i);
                        String objAsString = so == null ? null : so.toString();
                        list.add(CassandraUtil.toCassandraValue(objAsString, type.getName()));
                    }
                    st2.setList(i, list);
                    break;
                }
                catch (JSONException e) {
                    throw new IllegalArgumentException("invalid JSON value", e);
                }
            }
            case MAP: {
                try {
                    DataType keyType = (DataType)st2.preparedStatement().getVariables().getType(i).getTypeArguments().get(0);
                    DataType valueType = (DataType)st2.preparedStatement().getVariables().getType(i).getTypeArguments().get(1);
                    JSONObject obj = new JSONObject(val);
                    HashMap<Object, Object> map = new HashMap<Object, Object>();
                    Iterator it = obj.keys();
                    while (it.hasNext()) {
                        String k = (String)it.next();
                        Object ov = obj.get(k);
                        String v = ov == null ? null : ov.toString();
                        Object key = CassandraUtil.toCassandraValue(k, keyType.getName());
                        if (key == null) {
                            throw new IllegalStateException("Unsupported type: " + String.valueOf(keyType.getName()));
                        }
                        map.put(key, CassandraUtil.toCassandraValue(v, valueType.getName()));
                    }
                    st2.setMap(i, map);
                    break;
                }
                catch (JSONException e) {
                    throw new IllegalArgumentException("invalid JSON value", e);
                }
            }
            case SET: {
                try {
                    DataType type = (DataType)st2.preparedStatement().getVariables().getType(i).getTypeArguments().get(0);
                    JSONArray obj = new JSONArray(val);
                    HashSet<Object> set = new HashSet<Object>();
                    for (int j = 0; j < obj.length(); ++j) {
                        Object so = obj.get(i);
                        String objAsString = so == null ? null : so.toString();
                        set.add(CassandraUtil.toCassandraValue(objAsString, type.getName()));
                    }
                    st2.setSet(i, set);
                    break;
                }
                catch (JSONException e) {
                    throw new IllegalArgumentException("invalid JSON value", e);
                }
            }
        }
    }

    public static String forceQuote(String id) {
        StringBuilder sb = new StringBuilder();
        sb.append('\"');
        for (int i = 0; i < id.length(); ++i) {
            char c2 = id.charAt(i);
            if (c2 == '\"') {
                sb.append(c2);
            }
            sb.append(c2);
        }
        return sb.append('\"').toString();
    }

    public static String quote(String id) {
        return lowercaseId.matcher(id).matches() ? id : CassandraUtil.forceQuote(id);
    }

    public static String partitionFilterTerm(String partitioningColumn, int partitionSpread) {
        StringBuilder sb = new StringBuilder(CassandraUtil.quote(partitioningColumn) + " = ?");
        if (partitionSpread > 0) {
            sb.append(" AND ").append(CassandraUtil.quote(SPREAD_COLUMN)).append(" in (0");
            for (int i = 1; i < partitionSpread; ++i) {
                sb.append(',');
                sb.append(Integer.toString(i));
            }
            sb.append(')');
        }
        return sb.toString();
    }

    private CassandraUtil() {
    }

    private static class ConnectionCache {
        private final Map<CassandraConnection.CassandraConnectionParams, ClusterConnection> connections = new HashMap<CassandraConnection.CassandraConnectionParams, ClusterConnection>();
        private static final long expirationDelay = 5000L;
        private final Timer reaperTimer = new Timer("Cassandra connection reaper", true);

        private ConnectionCache() {
        }

        private void scheduleReap() {
            this.reaperTimer.schedule(new TimerTask(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    ConnectionCache connectionCache = this;
                    synchronized (connectionCache) {
                        long now = System.currentTimeMillis();
                        Iterator<Map.Entry<CassandraConnection.CassandraConnectionParams, ClusterConnection>> iterator = connections.entrySet().iterator();
                        while (iterator.hasNext()) {
                            Map.Entry<CassandraConnection.CassandraConnectionParams, ClusterConnection> entry = iterator.next();
                            ClusterConnection conn = entry.getValue();
                            if (conn.refCount != 0 || conn.lastUsed + 5000L >= now) continue;
                            try {
                                conn.close();
                            }
                            catch (IOException iOException) {
                                // empty catch block
                            }
                            iterator.remove();
                        }
                    }
                }
            }, 7000L);
        }

        synchronized ClusterConnection acquire(AuthCtx authCtx, CassandraConnection dssConn, CassandraConnection.CassandraConnectionParams ccp) throws DKUSecurityException {
            ClusterConnection conn = this.connections.get(ccp);
            if (conn == null) {
                conn = new ClusterConnection(authCtx, dssConn, ccp);
                this.connections.put(ccp, conn);
            }
            ++conn.refCount;
            return conn;
        }

        synchronized void release(ClusterConnection conn) {
            --conn.refCount;
            if (conn.refCount == 0) {
                conn.lastUsed = System.currentTimeMillis();
                this.scheduleReap();
            }
        }
    }

    public static class ClusterConnection
    implements Closeable {
        private Cluster cluster;
        private Session session;
        private ProtocolVersion protocolVersion;
        private String keyspace;
        private int refCount;
        private long lastUsed;
        private ConcurrentMap<String, PreparedStatement> preparedStatements;

        public ClusterConnection(AuthCtx authCtx, CassandraConnection conn, CassandraConnection.CassandraConnectionParams ccp) throws DKUSecurityException {
            ccp.check();
            Cluster.Builder builder = Cluster.builder();
            for (String h : ccp.hosts.split(",")) {
                builder.addContactPoint(h.trim());
            }
            if (ccp.port >= 0) {
                builder.withPort(ccp.port);
            }
            try {
                ICredentialsService.BasicCredential cred = conn.getFullyResolvedCredentials_fsLike(new ConnectionWithBasicCredential.CredentialResolutionContext(authCtx, null), ICredentialsService.BasicCredential.class);
                if (cred != null) {
                    builder.withCredentials(cred.user, cred.password);
                }
            }
            catch (IOException e) {
                throw new DKUSecurityException("Failed to read credentials", (Throwable)e);
            }
            if (ccp.ssl) {
                builder.withSSL();
            }
            if (ccp.protocolVersion != null) {
                builder.withProtocolVersion(ccp.protocolVersion);
            }
            if (ccp.readTimeout >= 0) {
                builder.withSocketOptions(new SocketOptions().setReadTimeoutMillis(ccp.readTimeout));
            }
            logger.info((Object)("connecting to cluster: hosts=" + ccp.hosts));
            this.cluster = builder.build();
            this.keyspace = ccp.keyspace;
            try {
                this.session = this.cluster.connect(this.keyspace);
                this.protocolVersion = this.cluster.getConfiguration().getProtocolOptions().getProtocolVersion();
            }
            catch (Throwable e) {
                this.cluster.close();
                throw e;
            }
            this.preparedStatements = new ConcurrentHashMap<String, PreparedStatement>();
        }

        @Override
        public void close() throws IOException {
            logger.info((Object)"Closing cluster connection");
            this.cluster.close();
        }

        public ProtocolVersion getProtocolVersion() {
            return this.protocolVersion;
        }

        public Session getSession() {
            return this.session;
        }

        public KeyspaceMetadata getKeyspaceMetadata() {
            return this.cluster.getMetadata().getKeyspace(this.keyspace);
        }

        public PreparedStatement prepare(String query) {
            PreparedStatement ps2 = (PreparedStatement)this.preparedStatements.get(query);
            if (ps2 == null) {
                ps2 = this.session.prepare(query);
                this.preparedStatements.put(query, ps2);
            }
            return ps2;
        }

        public String test() {
            Metadata meta = this.cluster.getMetadata();
            String clusterName = meta.getClusterName();
            String clusterVersion = meta.getAllHosts().stream().findFirst().map(h -> h.getCassandraVersion().toString()).orElse("<unknown>");
            return "cluster name = " + clusterName + "; server version = " + clusterVersion + "; protocol version = " + this.protocolVersion.toInt();
        }
    }
}

