import logging
import os.path
import threading
import time
from pathlib import Path

from dataiku.core import default_project_key
from dataiku.core import intercom
from dataiku.core.intercom import has_ticket_auth_mode

logger = logging.getLogger(__name__)

_lock = threading.Lock()
_ticket_folder_path = None


def get_ticket_local_folder_path(context_project_key=None):
    if not has_ticket_auth_mode():
        raise ValueError("Ticket folders require ticket-based authentication")

    if not context_project_key:
        context_project_key = default_project_key()

    global _ticket_folder_path
    with _lock:
        if not _ticket_folder_path:
            _ticket_folder_path = _create_ticket_local_folder(context_project_key)

        return _ticket_folder_path


def _create_ticket_local_folder(context_project_key):
    api_resp = intercom.backend_json_call("ticket-folder/get", data={
        "projectKey": context_project_key
    })

    folder_path = api_resp["path"]
    if os.path.exists(folder_path):
        logger.debug("start keeping alive ticket folder {}".format(folder_path))
        threading.Thread(target=_keep_alive_worker, daemon=True, args=(folder_path, )).start()
        return folder_path

    # the folder does not exist on this fs -> must be because of remote execution
    folder_path = os.path.join(os.getcwd(), "ticket-folder")
    if not os.path.exists(folder_path):
        os.mkdir(folder_path)

    return folder_path


def _keep_alive_worker(folder_path):
    try:
        keep_file_path = os.path.join(folder_path, "keep.txt")
        while True:
            if os.path.exists(keep_file_path):
                Path(keep_file_path).touch()
            else:
                logger.warning("exit keep-alive worker because file no longer exists: {}".format(keep_file_path))
                return

            time.sleep(30)  # every 30s

    except Exception as e:
        logger.warning("cannot keep alive ticket folder: {}".format(str(e)))
