import os, os.path

from nbformat import sign
from traitlets.config.configurable import LoggingConfigurable
from jupyter_core.application import JupyterApp
from six.moves.urllib.parse import quote
from traitlets import (
    Unicode,
    default
)

from notebook.dataiku.handlers import handler_context_var
from notebook.dataiku.dipproperties import DipProperties, PER_USER_TRUSTSTORE_DIP_PROPERTY

PER_USER_STORE_TYPE = "PER_USER"
UNSAFE_STORE_TYPE = "UNSAFE"


class DKUUserSpecificSignatureStore(sign.SignatureStore, LoggingConfigurable):
    stores = dict()
    single_user_store = None

    data_dir = Unicode()
    @default('data_dir')
    def _data_dir_default(self):
        return JupyterApp.instance().data_dir

    store_type = Unicode()
    @default('store_type')
    def _store_type_default(self):
        try:
            return UNSAFE_STORE_TYPE if DipProperties().get(PER_USER_TRUSTSTORE_DIP_PROPERTY) == "false" else PER_USER_STORE_TYPE
        except Exception as exc:
            self.log.error(exc)
            return PER_USER_STORE_TYPE

    def _get_store(self):
        if self.store_type == PER_USER_STORE_TYPE:
            return self._get_user_store()
        return self._get_main_store()

    def _get_main_store(self):
        if self.single_user_store is None:
            self.log.info("DKUUSS: Creating main store")
            self.single_user_store = sign.SQLiteSignatureStore(db_file=os.path.join(self.data_dir, "nbsignatures.db"))
        return self.single_user_store

    def _get_user_store(self):
        handler = handler_context_var.get()
        user = handler.current_user

        if user not in self.stores:
            user_dir_path = os.path.join(self.data_dir,  "dku_user_nbsignatures")
            if not os.path.exists(user_dir_path):
                os.makedirs(user_dir_path)
            encoded_user = quote(user, safe='')
            self.stores[user] = sign.SQLiteSignatureStore(db_file=os.path.join(user_dir_path, "nbsign__%s.db" % encoded_user))
            self.log.info("DKUUSS: Creating store %s for %s" % (self.stores[user].db_file, user))
        self.log.info("DKUUSS: Using store %s for user %s" % (self.stores[user].db_file, user))
        return self.stores[user]

    def store_signature(self, digest, algorithm):
        self.log.info("DKUUSS: Store signature")
        return self._get_store().store_signature(digest, algorithm)

    def check_signature(self, digest, algorithm):
        self.log.info("DKUUSS: Check signature")
        return self._get_store().check_signature(digest, algorithm)

    def remove_signature(self, digest, algorithm):
        self.log.info("DKUUSS: Remove signature")
        return self._get_store().remove_signature(digest, algorithm)

    def close(self):
        self.log.info("DKUUSS: Close")
        for (user, store) in self.store.items():
            store.close()
        self.log.fin("DKUUSS: Close done")

def create_dku_store():
    return DKUUserSpecificSignatureStore()
