(function(){
'use strict';

/**
 * Services related to sampling
 */

var app = angular.module('dataiku.common.sampling', []);

app.service("SamplingData", function($filter, translate) {

	const longSmartNumber = $filter('longSmartNumber');

	var svc = {

		memSamplingMethods : [
			["HEAD_SEQUENTIAL", translate("SHAKER.SAMPLING.METHODS.HEAD_SEQUENTIAL", "First records")],
			["RANDOM_FIXED_NB_EXACT", translate("SHAKER.SAMPLING.METHODS.RANDOM_FIXED_NB_EXACT", "Random (nb. records)")],
			["RANDOM_FIXED_RATIO", translate("SHAKER.SAMPLING.METHODS.RANDOM_FIXED_RATIO", "Random (approx. ratio)")],
			/*["RANDOM_FIXED_RATIO_EXACT", "Random (target ratio of data)"],*/
			["COLUMN_BASED", translate("SHAKER.SAMPLING.METHODS.COLUMN_BASED", "Column values subset (approx. nb. records)")],
			["STRATIFIED_TARGET_NB_EXACT", translate("SHAKER.SAMPLING.METHODS.STRATIFIED_TARGET_NB_EXACT", "Stratified (nb. records)")],
			["STRATIFIED_TARGET_RATIO_EXACT", translate("SHAKER.SAMPLING.METHODS.STRATIFIED_TARGET_RATIO_EXACT", "Stratified (ratio)")],
			["RANDOM_FIXED_NB", translate("SHAKER.SAMPLING.METHODS.RANDOM_FIXED_NB", "Random (approx. nb. records)")],
			["CLASS_REBALANCE_TARGET_NB_APPROX", translate("SHAKER.SAMPLING.METHODS.CLASS_REBALANCE_TARGET_NB_APPROX", "Class rebalance (approx. nb. records)")],
			["CLASS_REBALANCE_TARGET_RATIO_APPROX", translate("SHAKER.SAMPLING.METHODS.CLASS_REBALANCE_TARGET_RATIO_APPROX", "Class rebalance (approx. ratio)")],
			["TAIL_SEQUENTIAL", translate("SHAKER.SAMPLING.METHODS.TAIL_SEQUENTIAL", "Last records")]
	    ],

	    memSamplingMethodsDesc : [
	        translate('SHAKER.SAMPLING.METHODS.HEAD_SEQUENTIAL.HELP', "Takes the first N rows of the dataset. Very fast (only reads N rows) but may result in a very biased view of the dataset."),
			translate('SHAKER.SAMPLING.METHODS.RANDOM_FIXED_NB_EXACT.HELP', "Randomly selects N rows. Requires a full pass reading the data."),
			translate('SHAKER.SAMPLING.METHODS.RANDOM_FIXED_RATIO.HELP', "Randomly selects approximately X% of the rows. Requires a full pass reading the data."),
			/* "Randomly selects X% of the rows. Requires 2 full passes reading the data.",*/
			translate('SHAKER.SAMPLING.METHODS.COLUMN_BASED.HELP', "Randomly selects a subset of values and chooses all rows with these values, in order to obtain approximately N rows. This is useful for selecting a subset of customers, for example. Requires 2 full passes."),
			translate('SHAKER.SAMPLING.METHODS.STRATIFIED_TARGET_NB_EXACT.HELP', "Randomly selects N rows, ensuring that the repartition of values in a column is respected in the sampling. Ensures that all modalities of the column appear in the output. May return a few more than N rows. Requires 2 full passes reading the data."),
			translate('SHAKER.SAMPLING.METHODS.STRATIFIED_TARGET_RATIO_EXACT.HELP', "Randomly selects X% of the rows, ensuring that the repartition of values in a column is respected in the sampling. Ensures that all modalities of the column appear in the output. May return a bit more than X% rows. Requires 2 full passes reading the data."),
			translate('SHAKER.SAMPLING.METHODS.RANDOM_FIXED_NB.HELP', "Randomly selects approximately N rows. Requires 2 full passes reading the data. "),
			translate('SHAKER.SAMPLING.METHODS.CLASS_REBALANCE_TARGET_NB_APPROX.HELP', "Randomly selects approximately N rows, trying to rebalance equally all modalities of a column. Does not oversample, only undersample (so some rare modalities may remain under-represented). Rebalancing is not exact. Requires 2 full passes."),
			translate('SHAKER.SAMPLING.METHODS.CLASS_REBALANCE_TARGET_RATIO_APPROX.HELP', "Randomly selects approximately X% of the rows, trying to rebalance equally all modalities of a column. Does not oversample, only undersample  (so some rare modalities may remain under-represented). Rebalancing is not exact. Requires 2 full passes."),
			translate('SHAKER.SAMPLING.METHODS.TAIL_SEQUENTIAL.HELP', "Takes the last N rows of the dataset. Requires a full pass.")
	    ],

	    streamSamplingMethods : [
			["FULL", translate("SAMPLING.METHOD.FULL", "No sampling (whole data)")],
			["HEAD_SEQUENTIAL", translate("SAMPLING.METHOD.HEAD_SEQUENTIAL", "First records")],
			["RANDOM_FIXED_RATIO", translate("SAMPLING.METHOD.RANDOM_FIXED_RATIO", "Random (approx. ratio)")],
			["RANDOM_FIXED_NB", translate("SAMPLING.METHOD.RANDOM_FIXED_NB", "Random (approx. nb. records)")],
			["COLUMN_BASED", translate("SAMPLING.METHOD.COLUMN_BASED", "Column values subset (approx. nb. records)")],
			["CLASS_REBALANCE_TARGET_NB_APPROX", translate("SAMPLING.METHOD.CLASS_REBALANCE_TARGET_NB_APPROX", "Class rebalance (approx. nb. records)")],
			["CLASS_REBALANCE_TARGET_RATIO_APPROX", translate("SAMPLING.METHOD.CLASS_REBALANCE_TARGET_RATIO_APPROX", "Class rebalance (approx. ratio)")],
			["COLUMN_ORDERED", translate("SAMPLING.METHOD.COLUMN_ORDERED", "First records sorted by a column")],
	    ],

	    streamSamplingMethodsDesc : [
			translate("SAMPLING.METHOD.DESC.FULL", "Takes the whole data"),
			translate("SAMPLING.METHOD.DESC.HEAD_SEQUENTIAL", "Takes the first N rows of the dataset. Very fast (only reads N rows) but may result in a very biased view of the dataset."),
			translate("SAMPLING.METHOD.DESC.RANDOM_FIXED_RATIO", "Randomly selects approximately X% of the rows. Requires a full pass reading the data."),
			translate("SAMPLING.METHOD.DESC.RANDOM_FIXED_NB", "Randomly selects approximately N rows. Requires 2 full passes."),
			translate("SAMPLING.METHOD.DESC.COLUMN_BASED", "Randomly selects a subset of values and chooses all rows with these values, in order to obtain approximately N rows. This is useful for selecting a subset of customers, for example. Requires 2 full passes."),
			translate("SAMPLING.METHOD.DESC.CLASS_REBALANCE_TARGET_NB_APPROX", "Randomly selects approximately N rows, trying to rebalance equally all modalities of a column. Does not oversample, only undersample (so some rare modalities may remain under-represented). Rebalancing is not exact. Requires 2 full passes."),
			translate("SAMPLING.METHOD.DESC.CLASS_REBALANCE_TARGET_RATIO_APPROX", "Randomly selects approximately X% of the rows, trying to rebalance equally all modalities of a column. Does not oversample, only undersample  (so some rare modalities may remain under-represented). Rebalancing is not exact. Requires 2 full passes."),
			translate("SAMPLING.METHOD.DESC.COLUMN_ORDERED", "Retrieve the first N values (sorted by a column, ascending or descending) from the dataset. <strong><i class='icon-warning-sign'></i> Requires to write all data on disk for sorting.</strong>")
	    ],

	    partitionSelectionMethods : [
	        ["ALL", "All partitions"],
    	    ["SELECTED", "Select partitions"],
    	    ["LATEST_N", "Latest partitions"]
    	],

    	partitionSelectionMethodsDesc : [
    		"Use all partitions of the dataset",
    		"Use an explicitly selected list of partitions",
    		"Use the 'latest' partitions currently available in the dataset. This option is only defined for single-dimension time-based partitioning. This method is slower because the list of partitions needs to be recomputed often."
    	],


    	needsColumn : function(method) {
    		return [
    			"STRATIFIED_TARGET_NB_EXACT", "STRATIFIED_TARGET_RATIO_EXACT",
    			"CLASS_REBALANCE_TARGET_NB_APPROX", "CLASS_REBALANCE_TARGET_RATIO_APPROX",
    			"COLUMN_BASED", "COLUMN_ORDERED"].indexOf(method) >= 0;
    	},

    	needsAscending : function(method) {
    		return method === "COLUMN_ORDERED";
    	},

    	needsRatio : function(method) {
    		return [
    			"RANDOM_FIXED_RATIO",
    			"RANDOM_FIXED_RATIO_EXACT",
    			"STRATIFIED_TARGET_RATIO_EXACT",
    			"CLASS_REBALANCE_TARGET_RATIO_APPROX",
    			].indexOf(method) >= 0;
    	},

    	needsMaxRecords : function(method) {
    		return [
    			"HEAD_SEQUENTIAL",
    			"TAIL_SEQUENTIAL",
    			"RANDOM_FIXED_NB",
    			"RANDOM_FIXED_NB_EXACT",
    			"STRATIFIED_TARGET_NB_EXACT",
    			"CLASS_REBALANCE_TARGET_NB_APPROX",
    			"COLUMN_BASED",
    			"COLUMN_ORDERED"].indexOf(method) >= 0;
    	},

    	needsRandomSeed : function(method) {
    		return [
    			"RANDOM_FIXED_NB",
    			"RANDOM_FIXED_RATIO",
    			"RANDOM_FIXED_NB_EXACT",
    			"RANDOM_FIXED_RATIO_EXACT",
    			"STRATIFIED_TARGET_NB_EXACT",
    			"STRATIFIED_TARGET_RATIO_EXACT",
    			"CLASS_REBALANCE_TARGET_NB_APPROX",
    			"CLASS_REBALANCE_TARGET_RATIO_APPROX"].indexOf(method) >= 0;
    	},

    	// Warning: must be kept in sync with Java code in PivotTablesService
    	makeStreamableFromMem : function(memSelection) {
    		var ret = angular.copy(memSelection);
            if (ret.samplingMethod == "RANDOM_FIXED_NB_EXACT") {
                ret.samplingMethod = "RANDOM_FIXED_NB";
            } else if (ret.samplingMethod == "STRATIFIED_TARGET_NB_EXACT") {
                ret.samplingMethod = "RANDOM_FIXED_NB";
            } else if (ret.samplingMethod == "STRATIFIED_TARGET_RATIO_EXACT") {
                ret.samplingMethod = "RANDOM_FIXED_RATIO";
            } else if (ret.samplingMethod == "TAIL_SEQUENTIAL") {
            	ret.samplingMethod = "HEAD_SEQUENTIAL";
            }
            return ret;
    	},

        getSamplingMethodForDocumentation: function (selection) {
            let samplingMethodDoc = 'No sampling (whole data)';

            switch (selection.samplingMethod) {
                case 'HEAD_SEQUENTIAL':
                    samplingMethodDoc = `First ${selection.maxRecords || 0} records`;
                break;
                case 'RANDOM_FIXED_NB_EXACT':
                    samplingMethodDoc = `Random (${selection.maxRecords || 0} records)`;
                break;
                case 'RANDOM_FIXED_RATIO':
                    samplingMethodDoc = `Random (approximately ${Math.round(selection.targetRatio * 100) || 0}%)`;
                break;
                case 'COLUMN_BASED':
                    samplingMethodDoc = `Column values subset (approximately ${selection.maxRecords || 0} records)`;
                break;
                case 'STRATIFIED_TARGET_NB_EXACT':
                    samplingMethodDoc = `Stratified (${selection.maxRecords || 0} records)`;
                break;
                case 'STRATIFIED_TARGET_RATIO_EXACT':
                    samplingMethodDoc = `Stratified (${Math.round(selection.targetRatio * 100) || 0}%)`;
                break;
                case 'RANDOM_FIXED_NB':
                    samplingMethodDoc = `Random (approximately ${selection.maxRecords || 0} records)`;
                break;
                case 'CLASS_REBALANCE_TARGET_NB_APPROX':
                    samplingMethodDoc = `Class rebalance (approximately ${selection.maxRecords || 0} records)`;
                    break;
                case 'CLASS_REBALANCE_TARGET_RATIO_APPROX':
                    samplingMethodDoc = `Class rebalance (approximately ${Math.round(selection.targetRatio * 100) || 0}%)`;
                break;
                case 'TAIL_SEQUENTIAL':
                    samplingMethodDoc = `Last ${selection.maxRecords || 0} records`;
                break;
                case 'COLUMN_ORDERED':
                    samplingMethodDoc = `First ${selection.maxRecords || 0} records sorted by ${selection.column || "???"}`;
                break;
            }

            return samplingMethodDoc;
        },

		formatSamplingConfig: (samplingMethod, nbRows, ratio) => {
			const params = {
				rowCount: nbRows,
				smartRowCount: $filter('longSmartNumber')(nbRows),
			};
			switch (samplingMethod) {
				case "HEAD_SEQUENTIAL":
					return translate("SHAKER.SAMPLING.CONFIG.HEAD_SEQUENTIAL", "First <strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}}", params);
				case "TAIL_SEQUENTIAL":
					return translate("SHAKER.SAMPLING.CONFIG.TAIL_SEQUENTIAL", "Last <strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}}", params);
				case "RANDOM_FIXED_NB":
				case "COLUMN_BASED":
				case "STRATIFIED_TARGET_NB_EXACT":
				case "CLASS_REBALANCE_TARGET_NB_APPROX":
				case "RANDOM_FIXED_NB_EXACT":
					return translate("SHAKER.SAMPLING.CONFIG.RANDOM", "<strong>{{smartRowCount}}</strong> random {{rowCount == 1 ? 'row': 'rows'}}", params);
				case "RANDOM_FIXED_RATIO":
				case "STRATIFIED_TARGET_RATIO_EXACT":
				case "CLASS_REBALANCE_TARGET_RATIO_APPROX":
				case "RANDOM_FIXED_RATIO_EXACT":
					if (ratio !== undefined) {
						return translate("SHAKER.SAMPLING.CONFIG.RANDOM.RATIO", "<strong>{{ratio}}</strong>% random rows", {
							ratio: Math.round(ratio * 100),
						});
					} else {
						return translate("SHAKER.SAMPLING.CONFIG.RANDOM", "<strong>{{smartRowCount}}</strong> random {{rowCount == 1 ? 'row': 'rows'}}", params);
					}
				case "FULL":
				default:
					return translate("SHAKER.SAMPLING.CONFIG.FULL", "<strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}}", params);
			}
		}
    }

	return svc;
})

app.directive("datasetSelectionSamplingDetailsFields", function(SamplingData){
	return {
		templateUrl : "/templates/widgets/dataset-selection-sampling-details-fields.html",
		scope : {
			selection : '=',
			hideRandomSeed : '<?',
		},
		link : function($scope) {
			$scope.SamplingData = SamplingData;
		}
	}
})

app.directive("datasetSelectionSorting", function(SamplingData){
	return {
		templateUrl : "/templates/widgets/dataset-selection-sampling-details-fields.html",
		scope : {
			selection : '=',
		},
		link : function($scope) {
			$scope.SamplingData = SamplingData;
		}
	}
})

app.directive("datasetSelectionSamplingDetailsControlgroups", function(SamplingData, translate){
	return {
		templateUrl : "/templates/widgets/dataset-selection-sampling-details-controlgroups.html",
		scope : {
			selection : '=',
			datasetSchema : '<',
		},
		link : function($scope) {
			$scope.needsMaxRecords = SamplingData.needsMaxRecords;
			$scope.needsRatio = SamplingData.needsRatio;
			$scope.needsColumn = SamplingData.needsColumn;
			$scope.needsAscending = SamplingData.needsAscending;
			$scope.needsRandomSeed = SamplingData.needsRandomSeed;

			$scope.getSamplingColumnNames = function(datasetSchema) {
				if (datasetSchema && datasetSchema.columns) return datasetSchema.columns.map(a => a.name);
			};

			$scope.getSamplingColumnPlaceholder = function(samplingMethod) {
				return SamplingData.needsAscending(samplingMethod) ? translate("SAMPLING.METHOD.SETTINGS.SELECT_COLUMN_SORTING", "Select column for sorting") : translate("SAMPLING.METHOD.SETTINGS.SELECT_COLUMN", "Select column");
			}

			$scope.getSamplingColumnLabel = function(samplingMethod) {
				return SamplingData.needsAscending(samplingMethod) ? translate("SAMPLING.METHOD.SETTINGS.SORTED_BY", "Sorted by") : translate("SAMPLING.METHOD.SETTINGS.COLUMN", "Column");
			}
		}
	}
})


app.directive("datasetSelectionOrderingFields", function(){
	return {
		templateUrl : "/templates/widgets/dataset-selection-ordering-fields.html",
		scope : {
			selection : '=',
			datasetSupportsReadOrdering: '=',
			shakerState : '='
		},
		link : function($scope) {
		}
	}
})


app.directive("datasetSelectionPartitionsSelectionFields", function(SamplingData){
	return {
		templateUrl : "/templates/widgets/dataset-selection-partitions-selection-fields.html",
		scope : {
			selection : '=',
			partitioned : '=',
			getPartitionsList: '=',
		},
		link : function($scope) {
			$scope.SamplingData = SamplingData;

			$scope.loadPartitionsList = function(){
				$scope.getPartitionsList().then(function(data){
					$scope.partitionsList = data.sort(function(a,b){
						if ($.isNumeric(a) && $.isNumeric(b)) {return b - a}
						else {return a === b ? 0 : (a < b ? 1 : -1)}
					});
				})
			}
		}
	}
})
app.directive("datasetSelectionPartitionsSelectionControlgroups", function(SamplingData, $timeout){
	return {
		templateUrl : "/templates/widgets/dataset-selection-partitions-selection-controlgroups.html",
		scope : {
            selection: '=',
            partitioned: '=',
            getPartitionsList: '=',
        },
		link : function($scope) {
			$scope.SamplingData = SamplingData;
            $scope.$on('datasetChange', function () {
                $scope.partitionsList = null;
            });
			$scope.loadPartitionsList = function(){
                $scope.getPartitionsList().then(function (data) {
                    $scope.partitionsList = data.sort(function (a, b) {
                        if ($.isNumeric(a) && $.isNumeric(b)) {
                            return b - a
                        }
                        else {
                            return a === b ? 0 : (a < b ? 1 : -1)
                        }
                    });
                });
			}
		}
	}
})
})();
