(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('BarChartUtils', BarChartUtils);

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/bars.js
     */
    function BarChartUtils(ChartDimension, SVGUtils, ValuesInChartOverlappingStrategy) {

        return {
            getTranslateFunctions: function(axisName, axis, labels, thickness) {
                const dimension = axis.dimension;
                const hasOneTickPerBin = ChartDimension.hasOneTickPerBin(dimension);
                let translate;
                if (!hasOneTickPerBin && ChartDimension.isTimeline(dimension)) {
                    translate = (i) => axis.scale()((labels[i].min + labels[i].max) / 2) - thickness / 2;
                } else if (!hasOneTickPerBin && ChartDimension.isTrueNumerical(dimension)) {
                    translate = (i) => axis.scale()(labels[i].sortValue) - thickness / 2;
                } else {
                    translate = (i) => axis.ordinalScale(i);
                }
                switch (axisName) {
                    case 'x':
                        return {
                            x: (d, i) => translate(i),
                            y: () => 0
                        };
                    case 'y':
                        return {
                            x: () => 0,
                            y: (d, i) => translate(i)
                        };
                    default:
                        return null;
                }
            },

            /** Returns a translation function for the specified axis */
            translate: function(axisName, axis, labels, thickness) {
                if (['x', 'y'].includes(axisName)) {
                    const transformations = this.getTranslateFunctions(axisName, axis, labels, thickness);
                    return (d, i) => `translate(${transformations.x(d, i)}, ${transformations.y(d, i)})`;
                }

                return null;
            },



            drawLabels(drawContext) {
                drawContext.transform = this.translate(drawContext.axisName, drawContext.axis, drawContext.labels, drawContext.thickness);
                // we need these transformations to compute labels bounding boxes in absolute coordinates
                drawContext.transformations = this.getTranslateFunctions(drawContext.axisName, drawContext.axis, drawContext.labels, drawContext.thickness);

                SVGUtils.drawLabels(drawContext, 'bars');
            },



            getExtraData(d, i, d3Context, chartBase, mainLabelDetectionHandler, labelDetectionHandlersByColor, colorScaleIndex, getLabelXPosition, getLabelYPosition, overlappingStrategy, defaultFontSize, onHoverMode) {
                const hideOverlaps = overlappingStrategy === ValuesInChartOverlappingStrategy.AUTO;

                const labelPosition = {
                    x: getLabelXPosition(d, i),
                    y: getLabelYPosition(d)
                };

                const overlappingKey = onHoverMode ? 'isOverlapAlt' : 'isOverlap';
                let overlapStatus = false;
                const getCurrentColorLabelCollisionDetectionHandler = () => {
                    if (!labelDetectionHandlersByColor[colorScaleIndex]) {
                        // we need a separate label collision detection handler for each color scale
                        labelDetectionHandlersByColor[colorScaleIndex] = SVGUtils.initLabelCollisionDetection(chartBase);
                    }
                    return labelDetectionHandlersByColor[colorScaleIndex];
                };
                const labelDetectionHandler = onHoverMode ? getCurrentColorLabelCollisionDetectionHandler() : mainLabelDetectionHandler;

                let translate = { x: 0, y: 0 };
                // retrieve the translations applied to the parent node, to compute bounding box in absolute coordinates
                if (d3Context && d3Context.parentNode) {
                    const parentNode = d3Context.parentNode;
                    const parentData = d3.select(parentNode).datum();
                    if (parentData.translate) {
                        translate = parentData.translate;
                    }
                }

                const labelRectangles = SVGUtils.getRectanglesFromPosition(labelPosition, d, defaultFontSize).map(rectangle => ({
                    ...rectangle,
                    x: rectangle.x + translate.x,
                    y: rectangle.y + translate.y
                }));

                //  even if we display all values, we need to add the bounding box to quadtree, in order to avoid overlapping with lines in mix chart
                if ((!hideOverlaps || !labelDetectionHandler.checkOverlaps(labelRectangles))) {
                    const isDisplayed = !!d.valuesInChartDisplayOptions?.displayValues;
                    if (isDisplayed) {
                        labelDetectionHandler.addBoundingBoxesToQuadTree(labelRectangles);
                    }
                } else {
                    overlapStatus = true;
                }
                d[overlappingKey] = overlapStatus;

                return { labelPosition };
            },

            shouldDisplayBarLabel(binCount, barValue) {
                // for most cases binCount !== 0 is sufficient, but for cumulative/differential compute modes, we might have both binCount === 0 and a barValue !== 0, but we want to display the value in that case
                return binCount !== 0 || barValue !== 0;
            }
        };
    }
})();
