(function() {
    'use strict';

    angular.module("dataiku.shared").factory("PromptChatService", PromptChatService);

    function PromptChatService($dkuSanitize, Logger, CreateModalFromTemplate, CodeMirrorSettingService, ClipboardUtils, ConnectionsService, PromptUtils) {
        const svc = {};

        svc.buildCurrentChatBranch = function(messages, lastMessageId) {
            let currentBranch = [];
            if (!messages || Object.keys(messages).length === 0) {
                return currentBranch;
            }
            let currentMessage = messages[lastMessageId];
            while (currentMessage && currentMessage.parentId) {
                currentBranch.push(currentMessage);
                try {
                    if (currentMessage.message.content){
                        currentMessage.message.$markdownContent = $dkuSanitize(marked(currentMessage.message.content, {
                            breaks: true
                        }));
                    }
                } catch(e) {
                    Logger.error('Error parsing markdown HTML, switching to plain text', e);
                    currentMessage.message.$markdownContent = null;
                }

                if (currentMessage.error && currentMessage.llmError) {
                    currentMessage.credentialsError = ConnectionsService.getCredentialsError(currentMessage.llmError);
                }

                currentMessage = messages[currentMessage.parentId];
            }
            let previousAssistantMessage;
            currentBranch =  currentBranch.reverse();
            Object.entries(currentBranch).forEach(([key, message]) => {
                if (message.message && message.message.role === 'assistant'){
                    if (previousAssistantMessage){
                        message.$changedSettings = compareLlmSettings(message, previousAssistantMessage);
                    }
                    previousAssistantMessage = message;
                }
            });
            return currentBranch;
        };

        svc.enrichChatMessages = function(messages) {
            if (!messages) return {};

            const enrichedMessages = angular.copy(messages);
            Object.entries(enrichedMessages).forEach(([key, message]) => {
                message.statusClass = getChatOutputClass(message);
                const parentId = message.parentId;
                if (parentId) {
                    if (!enrichedMessages[parentId].childrenIds) {
                        enrichedMessages[parentId].childrenIds = [];
                    }
                    enrichedMessages[parentId].childrenIds[message.version] = message.id;
                }
            });
            return enrichedMessages;
        };

        svc.openConversationHistoryModal = function($scope, chatMessages, lastMessageId, llmSettings, chatSystemMessage) {
            CreateModalFromTemplate("/templates/promptstudios/conversation-history-modal.html", $scope, null, function (modalScope) {
                modalScope.uiState = {
                    activeTab: "json",
                };

                modalScope.pythonReadOnlyOptions = CodeMirrorSettingService.get('text/x-python');
                modalScope.pythonReadOnlyOptions["readOnly"] = "nocursor";
                modalScope.jsonReadOnlyOptions = CodeMirrorSettingService.get('application/json');
                modalScope.jsonReadOnlyOptions["readOnly"] = "nocursor";

                function getJsonMessages() {
                    let messages = svc.buildCurrentChatBranch(chatMessages, lastMessageId)
                        .filter(item => !item.error)
                        .map(item => {
                            const { $markdownContent, ...rest } = item.message;
                            return rest;
                        });
                    if (chatSystemMessage) {
                        let systemMessage = {
                            "role": "system",
                            "content": chatSystemMessage
                        };
                        messages.unshift(systemMessage);
                    }
                    return messages;
                };

                function getPythonMessages() {
                    let messages = getJsonMessages();

                    let pythonMessages = "import dataiku\n";
                    pythonMessages += `LLM_ID = "${$scope.activeLLM.id}"\n`;
                    pythonMessages += `llm = dataiku.api_client().get_default_project().get_llm(LLM_ID)\n`;
                    pythonMessages += `# Create and run a completion query with your chat history\n`;
                    pythonMessages += `chat = llm.new_completion()\n\n`;

                    if (llmSettings['reasoningEffort'] && PromptUtils.isReasoningModel($scope.activeLLM)) {
                        pythonMessages += `chat.settings["reasoningEffort"] = "${llmSettings['reasoningEffort']}"\n`
                        if (llmSettings['reasoningEffort'] === "CUSTOM" && llmSettings['customReasoningEffort']) {
                            pythonMessages += `chat.settings["customReasoningEffort"] = "${llmSettings['customReasoningEffort']}"\n`
                        }
                    }
                    if (llmSettings['temperature']) {
                        pythonMessages += `chat.settings["temperature"] = ${llmSettings['temperature']}\n`
                    }
                    if (llmSettings['topK']) {
                        pythonMessages += `chat.settings["topK"] = ${llmSettings['topK']}\n`
                    }
                    if (llmSettings['topP']) {
                        pythonMessages += `chat.settings["topP"] = ${llmSettings['topP']}\n`
                    }
                    if (llmSettings['maxOutputTokens']) {
                        pythonMessages += `chat.settings["maxOutputTokens"] = ${llmSettings['maxOutputTokens']}\n`
                    }
                    if (llmSettings['stopSequences'] && llmSettings['stopSequences'].length > 0) {
                        pythonMessages += `chat.settings["stopSequences"] = ${llmSettings['stopSequences']}\n`
                    }
                    if (llmSettings['presencePenalty']) {
                        pythonMessages += `chat.settings["presencePenalty"] = ${llmSettings['presencePenalty']}\n`
                    }
                    if (llmSettings['frequencyPenalty']) {
                        pythonMessages += `chat.settings["frequencyPenalty"] = ${llmSettings['frequencyPenalty']}\n`
                    }
                    if (llmSettings['responseFormat']) {
                        pythonMessages += `chat.with_json_output()\n`
                    }

                    pythonMessages += "\n";
                    messages.forEach((message) => {
                        pythonMessages += `chat.with_message(${JSON.stringify(message["content"])}, role=${JSON.stringify(message["role"])})\n`;
                    });

                    pythonMessages += "\n";
                    pythonMessages += `response = chat.execute()`;
                    return pythonMessages;
                }

                modalScope.jsonMessages = getJsonMessages();
                modalScope.pythonMessages = getPythonMessages();

                modalScope.copyConversationHistory = function () {
                    if (modalScope.uiState.activeTab === "json") {
                        ClipboardUtils.copyToClipboard(JSON.stringify(modalScope.jsonMessages, null, 2), "Conversation copied to clipboard");
                    }
                    if (modalScope.uiState.activeTab === "python") {
                        ClipboardUtils.copyToClipboard(modalScope.pythonMessages, "Conversation copied to clipboard");
                    }
                }
            });
        };

        const logs = {};
        svc.getLog = function(sessionId) {
            return logs[sessionId];
        };

        svc.setLog = (sessionId, log) => {
            logs[sessionId] = log;
        };

        svc.clearLogs = () => {
            Object.keys(logs).forEach(sessionId => delete logs[sessionId]);
        };

        return svc;

        function compareLlmSettings(newMessage, oldMessage){
            let changes = [];
            const compareAndPush = (newVal, oldVal, label) => {
                if (newVal !== oldVal) {
                    if(["System message", "Response Format"].includes(label)){
                        changes.push(`${label} has changed`);
                    }
                    else{
                        changes.push(`${label} has changed from ${oldVal ?? "not defined"} to ${newVal ?? "not defined"}`);
                    }
                }
            };

            const compareAndPushReasoning = (newSettings, oldSettings) => {
                if (newSettings["reasoningEffort"] !== oldSettings["reasoningEffort"]) {
                    if (newSettings["reasoningEffort"] === "CUSTOM") {
                        changes.push(`Reasoning Effort has changed from ${oldSettings["reasoningEffort"]} to custom value ${newSettings["customReasoningEffort"]}`);
                    } else if (oldSettings["reasoningEffort"] === "CUSTOM") {
                        changes.push(`Reasoning Effort has changed from custom value ${oldSettings["customReasoningEffort"]} to ${newSettings["reasoningEffort"]}`);
                    } else {
                        changes.push(`Reasoning Effort has changed from ${oldSettings["reasoningEffort"]} to custom value ${newSettings["reasoningEffort"]}`);
                    }
                }
                if (newSettings["reasoningEffort"] === "CUSTOM" && oldSettings["reasoningEffort"] === "CUSTOM"
                    && oldSettings["customReasoningEffort"] !== newSettings["customReasoningEffort"]) {
                    changes.push(`Reasoning Effort has changed from custom value ${oldSettings["customReasoningEffort"]} to custom value ${newSettings["customReasoningEffort"]}`);
                }
            }

            compareAndPush(newMessage.llmStructuredRef.id, oldMessage.llmStructuredRef.id, "Model");
            compareAndPush(newMessage.systemMessage, oldMessage.systemMessage, "System message");
            const settings = {
                "temperature": "Temperature",
                "topK": "TopK",
                "topP": "TopP",
                "maxOutputTokens": "Max Output Tokens",
                "frequencyPenalty": "Frequency Penalty",
                "presencePenalty": "Presence Penalty"
            };

            Object.entries(settings).forEach(([key, label]) => {
                compareAndPush(newMessage.completionSettings?.[key], oldMessage.completionSettings?.[key], label);
            });
            compareAndPush(newMessage.completionSettings.responseFormat?.type, oldMessage.completionSettings.responseFormat?.type, "Response Format");
            compareAndPushReasoning(newMessage.completionSettings, oldMessage.completionSettings);
            return changes;
        }

        function getChatOutputClass(message) {
            let status = '';
            if (message.error) {
                status = 'invalid'
            } else {
                status = 'valid';
            }
            return 'prompt-chat__assistant-message--' + status;
        }
    }
})();
