from abc import ABC, abstractmethod
from typing import Dict, Generic, Iterator, Type, TypeVar
from dataiku.eda.types import Final

# Grouping defines how to construct the groups (for instance: group by value of column "gender")
from dataiku.eda.computations.immutable_data_frame import ImmutableDataFrame
from dataiku.eda.exceptions import UnknownObjectType
from dataiku.eda.types import GroupingModel, GroupingResultModel, GroupingTypeLiteral


GroupingModelType = TypeVar("GroupingModelType", bound=GroupingModel)


class Grouping(ABC, Generic[GroupingModelType]):
    REGISTRY: Final[Dict[GroupingTypeLiteral, Type['Grouping']]] = {}

    @staticmethod
    @abstractmethod
    def get_type() -> GroupingTypeLiteral:
        raise NotImplementedError

    def describe(self) -> str:
        return self.__class__.__name__

    @staticmethod
    @abstractmethod
    def build(params: GroupingModelType) -> 'Grouping':
        try:
            grouping_class = Grouping.REGISTRY[params["type"]]
        except KeyError:
            raise UnknownObjectType("Unknown grouping type: %s" % params.get("type"))
        return grouping_class.build(params)

    @staticmethod
    def define(computation_class: Type['Grouping']) -> None:
        Grouping.REGISTRY[computation_class.get_type()] = computation_class

    # Compute actual groups (for instance: [gender:Male, gender:Female]) and returns a GroupingResult
    @abstractmethod
    def compute_groups(self, idf: ImmutableDataFrame) -> 'GroupingResult':
        raise NotImplementedError


class GroupingResult(ABC):
    @abstractmethod
    def serialize(self) -> GroupingResultModel:
        raise NotImplementedError

    # Generator producing a ImmutableDataFrame for each group
    @abstractmethod
    def iter_groups(self) -> Iterator[ImmutableDataFrame]:
        raise NotImplementedError

    # Return the number of groups
    def __len__(self) -> int:
        # TODO: implement this for each subclass would probably be more efficient
        return len(list(self.iter_groups()))
