import json
import logging
import os
import re
import subprocess
from abc import ABC, abstractmethod
from collections import defaultdict

from dateutil import parser
from six.moves import configparser

from cleaning import format_size
from dataiku.runnables import ResultTable, Runnable

TIMEOUT_DURATION = 60

def execute_pre_auth_script(script_name, *args):
    dku_install_directory = os.environ["DKUINSTALLDIR"]
    scripts_dir = os.path.join(dku_install_directory, "resources", "container-exec", "kubernetes")
    cmd = [os.path.join(scripts_dir, script_name)] + list(args)
    p = execute_command(cmd)
    logging.info(f"Pre-auth script stdout: {p.stdout}")

def execute_command_with_exception(cmd):
    logging.info("Executing: " + " ".join(cmd))
    p = subprocess.run(cmd, text=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, timeout=TIMEOUT_DURATION)
    if p.returncode != 0:
        raise Exception(f"Failed execute. Error code:{p.returncode}. Error message: {p.stderr}")
    return p.stdout

def execute_command(cmd):
    cmd_str = ' '.join(cmd)
    logging.info(f"Executing: {cmd_str}")
    try:
        p = subprocess.run(cmd, text=True, stdout=subprocess.PIPE, stderr=subprocess.PIPE, timeout=TIMEOUT_DURATION)
        if p.returncode != 0:
            logging.error(f"Failed execute. Command: {cmd_str} Error code:{p.returncode}. Error message: {p.stderr}")
        return p
    except subprocess.TimeoutExpired:
        logging.error(f"Timeout while executing {cmd_str}")
        return type('', (), {'returncode': 408, 'stdout': 'ERROR', 'stderr': f"Timeout while calling {cmd_str}"})


class AbstractCleaner(ABC):

    def requires_install_id(self):
        return True

    @abstractmethod
    def get_registry_url(self):
        pass

    @abstractmethod
    def list_repositories(self):
        pass

    @abstractmethod
    def delete_images(self, images_marked_for_deletion, force=False):
        pass

    @abstractmethod
    def list_images_in_repository(self, repository):
        pass

class AbstractClient(ABC):
    @abstractmethod
    def list_images_in_repository(self, repository_name):
        pass

    @abstractmethod
    def list_repositories(self):
        pass

    @abstractmethod
    def delete_image(self, repository, image_to_delete):
        pass


class LocalRegistryCleaner(AbstractCleaner):
    # Local docker returns golang time, we need to process it to have uniformized output.
    # Also parser.parse does not know how to handle both timezone abbreviation and time offset, so let's remove the last part (timezone abbreviation)
    def parse_golang_datetime(self, golang_time: str):
        # Docker images returns time like "2025-06-17 17:10:05 +0200 CEST"
        sanitized_time = golang_time.split(" ")[:3]
        return parser.parse(" ".join(sanitized_time))

    def __init__(self, config):
        self.config = config
        self.images_dict = self.build_images_dict()

    def requires_install_id(self):
        return False

    def get_registry_url(self):
        return "local.docker"

    def _get_docker_cmd(self, *args):
        if self.config['use_custom_host']:
            return ['docker', '--host', self.config['custom_docker_host']] + list(args)
        else:
            return ['docker'] + list(args)

    def _list_images_with_docker(self):
        cmd = self._get_docker_cmd('images', '--format', 'json')
        return execute_command_with_exception(cmd).splitlines()

    def build_images_dict(self):
        r = {}
        for line in self._list_images_with_docker():
            image = json.loads(line)
            if "Repository" not in image and "Tag" not in image:
                continue

            creation_date = self.parse_golang_datetime(image["CreatedAt"])
            r.setdefault(image["Repository"], []).append(
                {"tag": image["Tag"], "id": image["ID"], 'createdAt': creation_date.replace(microsecond=0).isoformat(), "size": image["Size"]})
        return r

    def list_repositories(self):
        return self.images_dict.keys()

    def list_images_in_repository(self, repository):
        return self.images_dict.get(repository, [])

    def delete_images(self, images_marked_for_deletion, force=False):
        for elt in images_marked_for_deletion:
            cmd_args = ["rmi"]
            if force:
                cmd_args.append("--force")
            # if we use the identifier (id) all images will be removed (thus using it in case of force delete),
            # if we use <repo>:<tag>, only the given tag will be removed.
            cmd_args.append(elt['id'] if self.config['force_rm'] or elt['tag'] == '<none>' else elt["repository"] + ':' + elt['tag'])
            cmd = self._get_docker_cmd(*cmd_args)
            result = execute_command(cmd)
            elt["status"] = "Success" if result.returncode == 0 else "Failed"
            elt["msg"] = result.stdout.strip() if result.returncode == 0 else result.stderr.strip()
        return images_marked_for_deletion

class GcrCleaner(AbstractCleaner):
    def __init__(self, registry_url: str, prefix: str):
        # No need to authenticate as this as must be done previously. Following the documentation
        # https://cloud.google.com/docs/authentication
        # Must use Workload Identity Federation or Service Account Key - gcloud commands will work out of the box afterwards.

        # Extract project_id from prefix
        prefix_split = prefix.split("/")
        project_id = prefix_split[0]
        self.registry_url = self._adapt_legacy_gcr(registry_url, project_id)
        self.prefix = "/".join(prefix_split[1:]) if len(prefix_split) > 1 else None
        self.images_dict = self.build_images_dict()

    def get_registry_url(self):
        return self.registry_url

    def _adapt_legacy_gcr(self, registry: str, project_id: str) -> str:
        if registry.endswith("us.gcr.io"):
            return "us-docker.pkg.dev/" + project_id + "/us.gcr.io"
        elif registry.endswith("eu.gcr.io"):
            return "europe-docker.pkg.dev/" + project_id + "/eu.gcr.io"
        elif registry.endswith("asia.gcr.io"):
            return "asia-docker.pkg.dev/" + project_id + "/asia.gcr.io"
        elif registry.endswith("gcr.io"):
            return "us-docker.pkg.dev/" + project_id + "/gcr.io"
        else:
            return registry + "/" + project_id

    def _list_images_with_gcloud(self):
        url = self.registry_url
        if self.prefix:
            url += "/" + self.prefix
        return json.loads(execute_command_with_exception(
            ["gcloud", "--format", "json", "artifacts", "docker", "images", "list", url, "--include-tags"])
        )

    def build_images_dict(self):
        images = self._list_images_with_gcloud()
        images_with_tags = {}
        for image in images:
            if "tags" not in image:
                images_with_tags.setdefault(image["package"], []).append({
                    "id": image["version"],
                    "tag": "<none>",
                    "createdAt": parser.parse(image["createTime"]).replace(microsecond=0).isoformat(),
                    "size": format_size(image["metadata"]["imageSizeBytes"])
                })
            else:
                for tag in image["tags"]:
                    images_with_tags.setdefault(image["package"], []).append({
                        "id": image["version"],
                        "tag": tag,
                        "createdAt": parser.parse(image["createTime"]).replace(microsecond=0).isoformat(),
                        "size": format_size(image["metadata"]["imageSizeBytes"])
                    })
        return images_with_tags

    def list_repositories(self):
        return self.images_dict.keys()

    def list_images_in_repository(self, repository):
        return self.images_dict.get(repository, [])

    def delete_images(self, images_marked_for_deletion, force=False):
        for image in images_marked_for_deletion:
            cmd_args = ["gcloud", "artifacts", "docker", "images", "delete"]
            image_identifier = image["repository"] + "@" + image["id"]
            cmd_args.append(image_identifier)
            cmd_args.append("--delete-tags")
            cmd_args.append("--quiet")
            result = execute_command(cmd_args)
            image["status"] = "Success" if result.returncode == 0 else "Failed"
            image["msg"] = result.stdout.strip() if result.returncode == 0 else result.stderr.strip()
        return images_marked_for_deletion


class AcrClient(AbstractClient):
    def __init__(self, registry_url, prefix=None):
        # Need a prepush hook with ACR
        self.registry_url = registry_url
        self.prefix = prefix
        self.authenticate()

    def authenticate(self):
        execute_pre_auth_script("azure-acr-prepush.sh", self.registry_url)

    def list_repositories(self):
        repositories = json.loads(execute_command_with_exception(
            ["az", "acr", "repository", "list", "--name", self.registry_url]
        ))

        if self.prefix:
            return [repository for repository in repositories if repository.startswith(self.prefix)]
        else:
            return repositories

    def list_images_in_repository(self, repository):
        return json.loads(execute_command_with_exception(
            ["az", "acr", "manifest", "list-metadata", "--registry", self.registry_url, "--name", repository]
        ))

    def delete_image(self, repository, image_to_delete):
        return execute_command(["az", "acr", "repository", "delete", "--name", self.registry_url, "--image", f"{repository}@{image_to_delete}", "--yes"])

class AcrCleaner(AbstractCleaner):
    def __init__(self, client: AcrClient):
        self.client = client

    def get_registry_url(self):
        return self.client.registry_url

    def list_repositories(self):
        return self.client.list_repositories()

    def delete_images(self, images_marked_for_deletion, force=False):
        # All delete in ACR will delete all tags associated to the image.
        for image in images_marked_for_deletion:
            result = self.client.delete_image(image["repository"], image["id"])
            image["status"] = "Success" if result.returncode == 0 else "Failed"
            image["msg"] = result.stdout.strip() if result.returncode == 0 else result.stderr.strip()
        return images_marked_for_deletion

    def list_images_in_repository(self, repository):
        images = self.client.list_images_in_repository(repository)
        image_with_tags = list()
        for image in images:
            if "tags" not in image or not image["tags"]:
                image_with_tags.append({
                    "id": image.get("digest"),
                    "createdAt": parser.parse(image.get("createdTime")).replace(microsecond=0).isoformat(),
                    "tag": "<none>",
                    "size": format_size(image.get("imageSize"))
                })
            else:
                for tag in image["tags"]:
                    image_with_tags.append({
                        "id": image["digest"],
                        "createdAt": parser.parse(image["createdTime"]).replace(microsecond=0).isoformat(),
                        "tag": tag,
                        "size": format_size(image["imageSize"])
                    })
        return image_with_tags

class EcrClient(AbstractClient):
    # ECR should already be configured to use the CLI. As we are not using docker, there is no need to configure docker neither
    def __init__(self, prefix=None):
        self.prefix = prefix

    def list_images_in_repository(self, repository_name):
        return json.loads(execute_command_with_exception(["aws", "ecr", "describe-images", "--repository", repository_name]))

    def list_repositories(self):
        cmd = ["aws", "ecr", "describe-repositories", "--query"]
        if self.prefix:
            cmd.append(f"repositories[?starts_with(repositoryName, `{self.prefix}`)].repositoryName")
        else:
            cmd.append(f"repositories[].repositoryName")
        return json.loads(execute_command_with_exception(cmd))

    def delete_image(self, repository, image_to_delete):
        return execute_command(["aws", "ecr", "batch-delete-image", "--repository-name", repository, "--image-ids", image_to_delete])

class EcrCleaner(AbstractCleaner):
    def __init__(self, registry_url, prefix=None):
        self.registry_url = registry_url
        self.ecr_client = EcrClient(prefix)

    def get_registry_url(self):
        return self.registry_url

    def list_images_in_repository(self, repository_name):
        images = self.ecr_client.list_images_in_repository(repository_name).get("imageDetails", [])
        images_with_tags = list()
        for image in images:

            if "imageTags" not in image or not image["imageTags"]:
                images_with_tags.append({
                    "id": image["imageDigest"],
                    "createdAt": parser.parse(image["imagePushedAt"]).replace(microsecond=0).isoformat(),
                    "size": format_size(image["imageSizeInBytes"]),
                    "tag": "<none>"
                })
            else:
                for tag in image["imageTags"]:
                    images_with_tags.append({
                        "id": image["imageDigest"],
                        "createdAt": parser.parse(image["imagePushedAt"]).replace(microsecond=0).isoformat(),
                        "size": format_size(image["imageSizeInBytes"]),
                        "tag": tag
                    })
        return images_with_tags

    def list_repositories(self):
        return self.ecr_client.list_repositories()

    def delete_images(self, images_marked_for_deletion, force=False):
        for image in images_marked_for_deletion:
            if image["tag"] == "<none>" or force:
                image_identifier = f"imageDigest={image['id']}"
            else:
                image_identifier = f"imageTag={image['tag']}"

            result = self.ecr_client.delete_image(image["repository"], image_identifier)

            image["status"] = "Success" if result.returncode == 0 else "Failed"
            image["msg"] = result.stdout.strip() if result.returncode == 0 else result.stderr.strip()
        return images_marked_for_deletion

class DeleteOldContainerImages(Runnable):
    def __init__(self, project_key, raw_config, plugin_config):
        super().__init__(project_key, raw_config, plugin_config)
        self.cleaners = list()
        logging.info(plugin_config)
        self.config = self._get_config(self.config)
        self.init_registry_cleaners()

    def init_registry_cleaners(self):
        self.cleaners.append(LocalRegistryCleaner(self.config))
        if self.config['with_remote_registries']:
            self.cleaners += self._get_remote_registries_cleaner()
        logging.info(f"Repositories to clean: {[c.get_registry_url() for c in self.cleaners]}")

    def get_progress_target(self):
        return (len(self.cleaners), 'NONE')

    def _get_install_id(self):
        config = configparser.RawConfigParser()
        config.read(os.path.join(os.environ['DIP_HOME'], 'install.ini'))

        if config.has_option('general', 'installid'):
            return config.get('general', 'installid').lower()

        return 'notattributed'

    def _is_automation_node(self):
        config = configparser.RawConfigParser()
        config.read(os.path.join(os.environ['DIP_HOME'], 'install.ini'))

        if config.has_option('general', 'nodetype'):
            return config.get('general', 'nodetype').lower() == 'automation'

        return False

    def _get_dss_version(self):
        dip_home = os.environ['DIP_HOME']
        with open(os.path.join(dip_home, 'dss-version.json'), 'r') as f:
            version = json.load(f)
            if 'product_version' in version:
                return version['product_version'].replace('/', '_').replace('.', '\\.').lower()

            return 'dev_doesnotmatter'

    def _get_config(self, raw_config):
        config = {
            'perform_deletion': bool(raw_config.get('perform_deletion', False)),
            'force_rm': bool(raw_config.get('force_rm', False)),
            'with_remote_registries': bool(raw_config.get('with_remote_registries', False))
        }

        for opt in ['rm_none_images', 'use_custom_host', 'dont_guess_image_name', 'container-exec', 'spark', 'api_deployer', 'cde', 'code_envs', 'code_studio']:
            config[opt] = bool(raw_config.get(opt, True))

        if config['dont_guess_image_name']:
            config['base_image_name'] = raw_config.get('custom_image_name', '')
            if not config['base_image_name']:
                raise ValueError('You should input a custom base image name that is not empty.')
        else:
            config['base_image_name'] = 'dku-exec-base-' + self._get_install_id()
        logging.info(f"Base image name: {config['base_image_name']}")
        config['custom_docker_host'] = raw_config.get('custom_docker_host', '')
        config['is_automation_node'] = self._is_automation_node()

        return config

    def _automation_node_images_in_use(self):
        automation_node_used_images = {}
        code_env_root_dir = os.path.join(os.environ['DIP_HOME'], 'acode-envs')
        code_env_dirs = (os.path.join(code_env_root_dir, 'R'), os.path.join(code_env_root_dir, 'python'))
        link_targets = set()
        for dir in code_env_dirs:
            if not os.path.isdir(dir):
                continue

            with os.scandir(dir) as files:
                for file in files:
                    links_path = os.path.join(dir, file, 'links')
                    if not os.path.isdir(links_path):
                        continue

                    link_targets.update(
                        os.readlink(link.path)
                        for link in os.scandir(links_path)
                        if os.path.islink(link.path)
                    )

        for version in link_targets:
            container_exec_path = os.path.join(version, 'desc', 'container-exec.json')
            if not os.path.isfile(container_exec_path):
                continue

            with open(container_exec_path, 'r') as file:
                img_versions = json.load(file).get('versionPerBaseImage', {})

            for repo, tag in img_versions.items():
                normalized_repo = repo.replace('|', '' if repo.startswith('|') else '/')
                automation_node_used_images.setdefault(normalized_repo, set()).add(tag)

        return automation_node_used_images

    def _build_precomputed_patterns(self, install_id, dss_version, requires_install_id=False):

        types = {
            "container-exec": [
                ('(^|.+/)dataiku-dss-container-exec-base$', True),
                ("(^|.+/){base_image_name}$", True)
            ],
            "spark": [
                ("(^|.+/)dataiku-dss-spark-exec-base$", True),
                ("(^|.+/)dku-spark-base-{install_id}$", True),
                ("(^|.+/)dku-spark-{install_id}-dss-(?!.*(?:pyenv|renv)){dss_version}$", False),
                ("(^|.+/)dku-spark-base-{install_id}-dss-(?!.*(?:pyenv|renv)){dss_version}$", False),
                ("(^|.+/)dku-spark-{install_id}-dss-{dss_version}-(?=.*(?:pyenv|renv))(?!.*pyenv.*renv|.*renv.*pyenv).*$", False),
                ("(^|.+/)dku-spark-base-{install_id}-dss-{dss_version}-(?=.*(?:pyenv|renv))(?!.*pyenv.*renv|.*renv.*pyenv).*$", False)
            ],
            "api_deployer": [
                ("(^|.+/)dataiku-dss-apideployer-base$", True),
                ("(^|.+/)dku-apideployer-apinode-base$", True),
                ("(^|.+/)dataiku-mad/apimodel-.+$", False),
                ("(^|.+/)dataiku-mad/apicodeenv.+{dss_version}$", False)
            ],
            "cde": [
                ("(^|.+/)dataiku-dss-cde-base$", True),
                ("(^|.+/)dku-cde-base-{install_id}$", True),
                ("(^|.+/)dku-cde-plugins-{install_id}-dss-{dss_version}$", False)
            ],
            "code_studio": [
                ("(^|.+/)dku-kub-.+$", False)
            ],
            'code_envs': [
                ("(^|.+/)dku-exec-{install_id}-dss-(?!.*(?:pyenv|renv)){dss_version}$", False),
                ("(^|.+/)dku-exec-base-{install_id}-dss-(?!.*(?:pyenv|renv)){dss_version}$", False),
                ("(^|.+/)dku-exec-{install_id}-dss-{dss_version}-(?=.*(?:pyenv|renv))(?!.*pyenv.*renv|.*renv.*pyenv).*$", False),
                ("(^|.+/)dku-exec-base-{install_id}-dss-{dss_version}-(?=.*(?:pyenv|renv))(?!.*pyenv.*renv|.*renv.*pyenv).*$", False)
            ]
        }

        precomputed_patterns = []
        for image_type in types:
            if self.config[image_type]:
                for img_name_reg, is_base in types[image_type]:
                    if not requires_install_id or "{install_id}" in img_name_reg or "{base_image_name}" in img_name_reg:
                        search_string = img_name_reg.format(
                            install_id=install_id,
                            dss_version=dss_version,
                            base_image_name=self.config['base_image_name']
                        )
                        search_string_without_version = img_name_reg.format(
                            install_id=install_id,
                            dss_version='.+',
                            base_image_name=self.config['base_image_name']
                        )

                        pattern_with_version = re.compile(search_string)
                        pattern_without_version = re.compile(search_string_without_version)
                        precomputed_patterns.append((pattern_with_version, pattern_without_version, is_base))
                        logging.debug(f"pattern_with_version: {pattern_with_version}, pattern_without_version: {pattern_without_version}, is_base: {is_base}")
        return precomputed_patterns

    def list_images_for_deletion(self, registry_cleaner: AbstractCleaner):
        install_id = self._get_install_id()
        dss_version = self._get_dss_version()

        img_to_delete = []
        precomputed_patterns = self._build_precomputed_patterns(install_id, dss_version, registry_cleaner.requires_install_id())
        repositories = registry_cleaner.list_repositories()
        for repo in repositories:
            logging.info("Look for repository: '" + repo + "'")
            managedByDSS = False
            for pattern_with_version, pattern_without_version, is_base in precomputed_patterns:
                if pattern_without_version.match(repo):
                    managedByDSS = True
                    images = registry_cleaner.list_images_in_repository(repo)
                    if not pattern_with_version.match(repo):
                        logging.info(f"Repository created by an older DSS version. Deleting all images.")
                        for img in images:
                            logging.info(f"Marking image for deletion: {img}")
                            img_to_delete.append({"repository": repo, **img})
                    else:

                        grouped_images = defaultdict(list)
                        for img in images:
                            if re.search(r'^(renv|pyenv).+-r-[\d-]{23}', img['tag']):
                                base_name = img['tag'].split('-r-', 1)[0]
                            else:
                                base_name = 'old-tagging-format'
                            grouped_images[base_name].append(img)

                        for base_name, images in grouped_images.items():
                            images.sort(key=lambda x: (x['createdAt'], x['tag']), reverse=True)
                            logging.info(f"Processing base name: {base_name}, found {len(images)} images")

                            start = 0 if self.config['is_automation_node']  and not is_base else 1 # Add the first image in an automation node, will be filtered later
                            for img in images[start:]:
                                logging.info(f"Marking image for deletion: {img}")
                                img_to_delete.append({"repository": repo, **img})

            if not managedByDSS:
                logging.info(f"Repository not managed by current DSS instance: {repo}")

        # Dangling images, that could be wiped with `docker image prune` (but would need the docker daemon to be up-to-date)
        if self.config['rm_none_images'] and isinstance(registry_cleaner, LocalRegistryCleaner):
            for key, value in registry_cleaner.build_images_dict().items():
                for elt in value:
                    if key == "<none>" or elt["tag"] == "<none>":
                        img_to_delete.append({"repository": key, **elt})

        if self.config['is_automation_node']:
            automation_node_used_images = self._automation_node_images_in_use()
            img_to_delete = [x for x in img_to_delete if x['tag'] not in automation_node_used_images.get(x["repository"], set())]

        return img_to_delete

    def _find_execution_configs(self, d):
        execution_configs = list()
        if isinstance(d, dict):
            if "executionConfigs" in d:
                execution_configs.append(d["executionConfigs"])
            for k, v in d.items():
                execution_configs += self._find_execution_configs(v)
        if isinstance(d, list):
            for l in d:
                execution_configs += self._find_execution_configs(l)
        return execution_configs

    def _find_registries(self, d):
        registries = list()
        if isinstance(d, dict):
            if "repositoryURL" in d:
                registries.append(d["repositoryURL"])

            for k, v in d.items():
                tmp = self._find_registries(v)
                if tmp:
                    registries += tmp
        if isinstance(d, list):
            for l in d:
                tmp = self._find_registries(l)
                if tmp:
                    registries += tmp
        return registries

    def _get_remote_registries_cleaner(self):
        remote_cleaner = list()
        dip_home = os.environ['DIP_HOME']
        with open(os.path.join(dip_home, 'config', 'general-settings.json')) as f:
            general_settings = json.load(f)
            execution_configs = self._find_execution_configs(general_settings)
            registries = set(self._find_registries(execution_configs))
            for registry in registries:
                registry_split = registry.split("/")

                registry_url = registry_split[0]
                prefix = "/".join(registry_split[1:]) if len(registry_split) > 1 else None

                if registry_url.endswith('amazonaws.com'):
                    remote_cleaner.append(EcrCleaner(registry_url, prefix))
                elif registry_url.endswith('azurecr.io'):
                    remote_cleaner.append(AcrCleaner(AcrClient(registry_url, prefix)))
                elif registry_url.endswith(".pkg.dev") or registry_url.endswith("gcr.io"):
                    if not prefix:
                        logging.warn("Invalid format for GCR repository, a valid docker repository should be LOCATION-docker.DOMAIN/PROJECT-ID/REPOSITORY-ID")
                        continue
                    remote_cleaner.append(GcrCleaner(registry_url, prefix))
        return remote_cleaner

    def run_cleaner(self, registry: AbstractCleaner):
        deletion_status = list()
        try:
            logging.info(f"Cleaning up registry: {registry.get_registry_url()}")
            logging.info(f"List images to delete")
            img_to_delete = self.list_images_for_deletion(registry)

            images_deleted = img_to_delete
            if self.config['perform_deletion']:
                logging.info(f"Perform image deletion")
                images_deleted = registry.delete_images(img_to_delete, self.config['force_rm'])
            else:
                logging.info(f"Dry run, do not perform image deletion")

            for elt in images_deleted:
                record = [registry.get_registry_url(), elt["repository"], elt['tag'], elt['id'], elt['createdAt'], elt['size']]
                if self.config['perform_deletion']:
                    record = record + [elt.get('status', 'Missing status'), elt.get('msg', '')]
                deletion_status.append(record)
        except Exception as e:
            logging.exception(f"Failed to list images to delete for {registry.get_registry_url()}", e)
        return deletion_status

    def run(self, progress_callback):
        rt = ResultTable()
        rt.set_name("Removed containers")
        rt.add_column("registry", "Registry", "STRING")
        rt.add_column("repo", "Repository", "STRING")
        rt.add_column("tag", "Tag", "STRING")
        rt.add_column("id", "Identifier", "STRING")
        rt.add_column("createdAt", "Created at", "STRING")
        rt.add_column("imageSize", "Image Size", "STRING")
        if self.config['perform_deletion']:
            rt.add_column("status", "Status", "STRING")
            rt.add_column("msg", "Message", "STRING")

        progress_status = 0
        for cleaner in self.cleaners:
            progress_callback(progress_status)
            progress_status += 1
            for image_delete in self.run_cleaner(cleaner):
                rt.add_record(image_delete)
        return rt
