from __future__ import print_function
import subprocess, sys
import os
import os.path as osp
import java
import base

def get_apinode_envdefault(install_dir, javabin, installConfig):
    base_port = installConfig.getServerPort()
    node_type = installConfig.getNodeType()
    jvm = java.get_jvm(javabin)

    # Common Java options
    javaOpts = java.get_common_java_opts(jvm)
    # Per-process Java options
    apimainJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "apimain", gcDefault="auto_lowpause")
    dkuJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "dku", addGCLogging=False, gcDefault="parallel")
    apimainJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "apimain")
    dkuJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "dku")

    return """# This file is automatically updated at install time.
# Do not edit this file. To customize the Dataiku DSS
# environment, edit the bin/env-site.sh file
export DKUINSTALLDIR="%s"
export DKU_NODE_TYPE="%s"
export DKU_BASE_PORT="%d"
export DKU_NGINX_PORT="%d"
export DKU_APIMAIN_PORT="%d"
if test -z "$DKU_JAVA_TMP_DIR"; then
  export DKU_JAVA_TMP_DIR="$DIP_HOME/tmp"
fi
export DKUJAVABIN="%s"
export DKU_JAVA_OPTS="%s"
export DKU_APIMAIN_JAVA_OPTS="%s"
export DKU_DKU_JAVA_OPTS="%s"
export DKU_APIMAIN_JAVA_LIBRARY_PATH="%s"
export DKU_DKU_JAVA_LIBRARY_PATH="%s"
""" % (install_dir, node_type, base_port, base_port, base_port + 1, javabin, javaOpts, apimainJavaOpts, dkuJavaOpts, apimainJavaLibraryPath, dkuJavaLibraryPath)

def get_designnode_envdefault(install_dir, javabin, installConfig):
    node_type = installConfig.getNodeType()
    if base.is_os_windows() and node_type != "design":
        raise Exception("Unsupported node type on Windows: " % node_type)
    base_port = installConfig.getServerPort()
    node_type = installConfig.getNodeType()
    jvm = java.get_jvm(javabin)
    
    # Common Java options
    javaOpts = java.get_common_java_opts(jvm)
    
    # Per-process Java options
    backendJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "backend", gcDefault="auto_lowpause")
    fekJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "fek", gcDefault="parallel")
    hproxyJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "hproxy", gcDefault="parallel")
    jekJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "jek", gcDefault="parallel")
    cakJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "cak", gcDefault="parallel")
    eventServerJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "eventserver", gcDefault="auto_lowpause")
    
    dkuXmx = installConfig.getPrefixedOption("javaopts", "dku", "xmx", "4g")
    dkuJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "dku", xmxDefault=dkuXmx, addGCLogging=False, gcDefault="parallel") # No logging GC stuff for dku, it's already spitting too much on the standard output

    backendJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "backend")
    fekJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "fek")
    hproxyJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "hproxy")
    jekJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "jek")
    cakJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "cak")
    eventServerJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "eventserver")
    dkuJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "dku")

    encrypted_rpc = installConfig.getBoolOption("rpc", "encrypted", False)

    return """# This file is automatically updated at install time.
# Do not edit this file. To customize the Dataiku DSS
# environment, edit the bin/env-site.sh file
export DKUINSTALLDIR="%s"
export DKU_NODE_TYPE="%s"
export DKU_BASE_PORT="%d"
export DKU_NGINX_PORT="%d"
export DKU_BACKEND_PORT="%d"
export DKU_IPYTHON_PORT="%d"
export DKU_HPROXY_PORT="%d"
export DKU_EVENTSERVER_PORT="%d"
export DKU_STORIES_PORT="%d"
%s
if test -z "$DKU_JAVA_TMP_DIR"; then
  export DKU_JAVA_TMP_DIR="$DIP_HOME/tmp"
fi
export DKUJAVABIN="%s"
export DKU_JAVA_OPTS="%s"
export DKU_BACKEND_JAVA_OPTS="%s"
export DKU_FEK_JAVA_OPTS="%s"
export DKU_HPROXY_JAVA_OPTS="%s"
export DKU_JEK_JAVA_OPTS="%s"
export DKU_CAK_JAVA_OPTS="%s"
export DKU_EVENTSERVER_JAVA_OPTS="%s"
export DKU_DKU_JAVA_OPTS="%s"
export DKU_BACKEND_JAVA_LIBRARY_PATH="%s"
export DKU_FEK_JAVA_LIBRARY_PATH="%s"
export DKU_HPROXY_JAVA_LIBRARY_PATH="%s"
export DKU_JEK_JAVA_LIBRARY_PATH="%s"
export DKU_CAK_JAVA_LIBRARY_PATH="%s"
export DKU_EVENTSERVER_JAVA_LIBRARY_PATH="%s"
export DKU_DKU_JAVA_LIBRARY_PATH="%s"

""" % (install_dir, node_type, base_port,
    base_port, base_port + 1, base_port + 2, base_port + 3, base_port + 4, base_port + 6,
    "export DKU_BACKEND_RPC_PORT=\"%s\"" % (base_port+5) if encrypted_rpc else "",
    javabin, javaOpts,
    backendJavaOpts, fekJavaOpts, hproxyJavaOpts, jekJavaOpts, cakJavaOpts, eventServerJavaOpts, dkuJavaOpts,
    backendJavaLibraryPath, fekJavaLibraryPath, hproxyJavaLibraryPath, jekJavaLibraryPath, cakJavaLibraryPath,
    eventServerJavaLibraryPath, dkuJavaLibraryPath)

def get_designnode_envdefault_windows(install_dir, javabin, installConfig):
    base_port = installConfig.getServerPort()
    node_type = installConfig.getNodeType()
    jvm = java.get_jvm(javabin)

    # Common Java options
    javaOpts = java.get_common_java_opts(jvm)

    # Per-process Java options
    backendJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "backend", gcDefault="auto_lowpause")
    fekJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "fek", gcDefault="parallel")
    hproxyJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "hproxy", gcDefault="parallel")
    jekJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "jek", gcDefault="parallel")
    cakJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "cak", gcDefault="parallel")
    eventServerJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "eventserver", gcDefault="auto_lowpause")
    dkuJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "dku", addGCLogging=False, gcDefault="parallel") # No logging GC stuff for dku, it's already spitting too much on the standard output

    backendJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "backend")
    fekJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "fek")
    hproxyJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "hproxy")
    jekJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "jek")
    cakJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "cak")
    eventServerJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "eventserver")
    dkuJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "dku")

    return """# This file is automatically updated at install time.
# Do not edit this file. To customize the Dataiku DSS
# environment, edit the bin/env-site.ps1 file
$env:DKUINSTALLDIR="%s"
$env:DKU_NODE_TYPE="%s"
$env:DKU_BASE_PORT="%d"
$env:DKU_NGINX_PORT="%d"
$env:DKU_BACKEND_PORT="%d"
$env:DKU_IPYTHON_PORT="%d"
$env:DKU_HPROXY_PORT="%d"
$env:DKU_EVENTSERVER_PORT="%d"
$env:DKU_STORIES_PORT="%d"
if (!$env:DKU_JAVA_TMP_DIR)
{
    $env:DKU_JAVA_TMP_DIR = "$env:DIP_HOME/tmp"
}
$env:DKU_SUPERVISORCTL_PORT="%d"
$env:DKUJAVABIN="%s"
$env:DKU_JAVA_OPTS="%s"
$env:DKU_BACKEND_JAVA_OPTS="%s"
$env:DKU_FEK_JAVA_OPTS="%s"
$env:DKU_HPROXY_JAVA_OPTS="%s"
$env:DKU_JEK_JAVA_OPTS="%s"
$env:DKU_CAK_JAVA_OPTS="%s"
$env:DKU_EVENTSERVER_JAVA_OPTS="%s"
$env:DKU_DKU_JAVA_OPTS="%s"
$env:DKU_BACKEND_JAVA_LIBRARY_PATH="%s"
$env:DKU_FEK_JAVA_LIBRARY_PATH="%s"
$env:DKU_HPROXY_JAVA_LIBRARY_PATH="%s"
$env:DKU_JEK_JAVA_LIBRARY_PATH="%s"
$env:DKU_CAK_JAVA_LIBRARY_PATH="%s"
$env:DKU_EVENTSERVER_JAVA_LIBRARY_PATH="%s"
$env:DKU_DKU_JAVA_LIBRARY_PATH="%s"
$env:PYTHONUTF8=1

""" % (install_dir, node_type, base_port,
    base_port, base_port + 1, base_port + 2, base_port + 3, base_port + 4, base_port + 5, base_port + 6,
    javabin, javaOpts,
    backendJavaOpts, fekJavaOpts, hproxyJavaOpts, jekJavaOpts, cakJavaOpts, eventServerJavaOpts, dkuJavaOpts,
    backendJavaLibraryPath, fekJavaLibraryPath, hproxyJavaLibraryPath, jekJavaLibraryPath, cakJavaLibraryPath,
    eventServerJavaLibraryPath, dkuJavaLibraryPath)

def get_fm_envdefault(install_dir, javabin, installConfig):
    base_port = installConfig.getServerPort()
    node_type = installConfig.getNodeType()
    jvm = java.get_jvm(javabin)

    # Common Java options
    javaOpts = java.get_common_java_opts(jvm)
    # Per-process Java options
    fmmainJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "fmmain", gcDefault="auto_lowpause")
    fmcliJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "fmcli", addGCLogging=False, gcDefault="parallel")
    # Per-process Java libs
    fmmainJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "fmmain")
    fmcliJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "fmcli")

    return """# This file is automatically updated at install time.
# Do not edit this file. To customize the Dataiku FM
# environment, edit the bin/env-site.sh file
export DKUINSTALLDIR="%s"
export DKU_NODE_TYPE="%s"
export DKU_BASE_PORT="%d"
export DKU_NGINX_PORT="%d"
export DKU_FMMAIN_PORT="%d"
if test -z "$DKU_JAVA_TMP_DIR"; then
  export DKU_JAVA_TMP_DIR="$DIP_HOME/tmp"
fi
export DKUJAVABIN="%s"
export DKU_JAVA_OPTS="%s"
export DKU_FMMAIN_JAVA_OPTS="%s"
export DKU_FMCLI_JAVA_OPTS="%s"
export DKU_FMMAIN_JAVA_LIBRARY_PATH="%s"
export DKU_FMCLI_JAVA_LIBRARY_PATH="%s"
""" % (install_dir, node_type, base_port, base_port, base_port + 1, javabin, javaOpts, fmmainJavaOpts, fmcliJavaOpts, fmmainJavaLibraryPath, fmcliJavaLibraryPath)

def get_govern_envdefault(install_dir, javabin, installConfig):
    base_port = installConfig.getServerPort()
    node_type = installConfig.getNodeType()
    jvm = java.get_jvm(javabin)
    
    # Common Java options
    javaOpts = java.get_common_java_opts(jvm)
    
    # Per-process Java options
    governserverJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "governserver", gcDefault="auto_lowpause")
    
    dkugovernXmx = installConfig.getPrefixedOption("javaopts", "dkugovern", "xmx", "4g")
    dkugovernJavaOpts = java.get_perprocess_java_opts(installConfig, jvm, "dkugovern", xmxDefault=dkugovernXmx, addGCLogging=False, gcDefault="parallel") # No logging GC stuff for dku, it's already spitting too much on the standard output

    governserverJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "governserver")
    dkugovernJavaLibraryPath = java.get_perprocess_java_library_path(installConfig, jvm, "dkugovern")

    return """# This file is automatically updated at install time.
# Do not edit this file. To customize the Dataiku Govern
# environment, edit the bin/env-site.sh file
export DKUINSTALLDIR="%s"
export DKU_NODE_TYPE="%s"
export DKU_BASE_PORT="%d"
export DKU_NGINX_PORT="%d"
export DKU_GOVERNSERVER_PORT="%d"
if test -z "$DKU_JAVA_TMP_DIR"; then
  export DKU_JAVA_TMP_DIR="$DIP_HOME/tmp"
fi
export DKUJAVABIN="%s"
export DKU_JAVA_OPTS="%s"
export DKU_GOVERNSERVER_JAVA_OPTS="%s"
export DKU_DKUGOVERN_JAVA_OPTS="%s"
export DKU_GOVERNSERVER_JAVA_LIBRARY_PATH="%s"
export DKU_DKUGOVERN_JAVA_LIBRARY_PATH="%s"
""" % (install_dir, node_type, base_port, base_port, base_port + 1, javabin, javaOpts, governserverJavaOpts, dkugovernJavaOpts, governserverJavaLibraryPath, dkugovernJavaLibraryPath)

defaultenvs_funcs = {
    ('design', 'unix'): get_designnode_envdefault,
    ('automation', 'unix'): get_designnode_envdefault,
    ('api', 'unix'): get_apinode_envdefault,
    ('fm', 'unix'): get_fm_envdefault,
    ('govern', 'unix'): get_govern_envdefault,
    ('design', 'windows'): get_designnode_envdefault_windows,
}

def write_envdefault(dip_home):
    installConfig = dip_home.get_install_config()
    node_type = installConfig.getNodeType()
    osname = 'windows' if base.is_os_windows() else 'unix'
    if not (node_type, osname) in defaultenvs_funcs:
        raise Exception("node type not supported: " % node_type)

    get_envdefault = defaultenvs_funcs[(node_type, osname)]
    content = get_envdefault(os.environ["DKUINSTALLDIR"], os.environ["DKUJAVABIN"], installConfig)
    env_filename = "env-default.ps1" if base.is_os_windows() else "env-default.sh"
    with open(osp.join(dip_home.path, "bin", env_filename), "w") as f:
        f.write(content)

#
# Regenerates the env-default file in $DIP_HOME
# Usage: envfiles.py -save
#
if __name__ == "__main__":
    if len(sys.argv) == 2 and sys.argv[1] == '-save':
        write_envdefault(base.DipHome(os.environ["DIP_HOME"]))
    else:
        print("Usage: %s -save" % sys.argv[0], file=sys.stderr)
