(function () {
    "use strict";

    angular.module("dataiku.aiSqlGeneration").factory("AISqlGenerationService", AISqlGenerationService);
    function AISqlGenerationService(DKUSQLFormatter) {
        var service = {
            isGenerateButtonDisabled: isGenerateButtonDisabled,
            getGenerateButtonDisabledMessage: getGenerateButtonDisabledMessage,
            getAIQueryToInsert: getAIQueryToInsert,
            processSQLGenerationResponse: processSQLGenerationResponse,
            buildWT1SqlGenerationParams: buildWT1SqlGenerationParams,
            buildWT1SqlGenerationResponseParams: buildWT1SqlGenerationResponseParams,
            buildWT1SqlInsertQueryParams: buildWT1SqlInsertQueryParams,
            buildWT1SqlErrorResponseParams: buildWT1SqlErrorResponseParams
        };
        return service;

        /**
         * @param {boolean} canWriteProject - Whether the user has WRITE permission on the current project
         * @param {string} userPrompt - User prompt to be sent the the AI server
         * @returns {boolean}
         */
        function isGenerateButtonDisabled(canWriteProject, userPrompt) {
            return !canWriteProject || !userPrompt || userPrompt.trim() === "";
        }

        function getGenerateButtonDisabledMessage(canWriteProject) {
            if (!canWriteProject) {
                return "You do not have write permissions for this project"
            }
    
            return "";
        }
    
        /**
         * @param {{ reasoning: string, query: string }} aiQuery - AI server response, contains the generated SQL query and the reasoning behind it.
         * @returns {string} - SQL query to be inserted in the code editor
         */
        function getAIQueryToInsert(aiQuery) {
            if (!aiQuery.query.trim()) {
                return;
            }
            if (!aiQuery.reasoning || aiQuery.reasoning.trim() === "") {
                return `${aiQuery.query}`;
            }
            // splits the `reasoning` string into chunks of up to 100 characters, ensuring chunks end at a word boundary (either a space or end of string).
            const reasoningComments = aiQuery.reasoning.match(RegExp(".{1,100}(\\s|$)|\\S+?(\\s|$)", 'g')).join("\n-- ");
            return `-- ${reasoningComments}\n${aiQuery.query}`;
        }
    
        function processSQLGenerationResponse(queryName, sqlQuery, reasoning, messages) {
            const aiQueryResponse = {};
            if (queryName) {
                aiQueryResponse.name = queryName;
            }
            if (sqlQuery) {
                aiQueryResponse.query = DKUSQLFormatter.format(sqlQuery, 4);
            }
            if (reasoning) {
                aiQueryResponse.reasoning = reasoning;
            }
            if (messages) {
                aiQueryResponse.errors = messages.filter(m => m.level === "ERROR");
                aiQueryResponse.warnings = messages.filter(m => m.level === "WARNING");
            }
            aiQueryResponse.loaded = true;
            return aiQueryResponse;
        }

        function buildWT1SqlGenerationParams(requestOrigin, nbTables, dialect) {
            return {
                origin: requestOrigin,
                nbTables,
                dialect
            }
        }

        function buildWT1SqlGenerationResponseParams(requestOrigin, dialect, requestId, warningMessages, errorMessages) {
            return {
                origin: requestOrigin,
                dialect,
                requestId,
                hasWarning: (warningMessages && warningMessages.length > 0) || false,
                hasError: (errorMessages && errorMessages.length > 0) || false
            }
        }

        function buildWT1SqlInsertQueryParams(requestOrigin, dialect, requestId) {
            return {
                origin: requestOrigin,
                dialect,
                requestId
            }
        }

        function buildWT1SqlErrorResponseParams(requestOrigin, dialect) {
            return {
                origin: requestOrigin,
                dialect
            }
        }
    }
})();
