(function(){
    'use strict';

    const app = angular.module('dataiku.dashboards');

    /**
     * Was previously located here: src/main/platypus/static/dataiku/js/dashboards/view.js
     */
    app.directive('dashboardPage', function($timeout, CreateModalFromTemplate, AnyLoc, $rootScope, ChartFormattingPaneSections, DetectUtils,
        $state, $stateParams, executeWithInstantDigest,WT1, DashboardFilters, DashboardUtils, TileUtils, FiltersPanelPosition, FiltersPanelDirection, $location, DashboardPageUtils, translate) {
        return {
            restrict: 'EA',
            replace: true,
            templateUrl: '/templates/dashboards/page.html',
            scope: {
                page: '=',
                insightsMap: '=',
                accessMap: '=',
                editable: '=',
                selectedTile: '=',
                showGrid: '=',
                pageIdx: '=',
                dashboard: '=',
                uiState: '=dkuUiState',
                canEditDashboard: '=',
                canModerateDashboards: '=',
                exportDatasetFn: '=',
                canExportDatasets: '='
            },
            link: function(scope) {
                scope.translate = translate;

                function goToPage(pageIdx) {
                    if (pageIdx < 0) {
                        return Promise.reject('Invalid page index');
                    }
                    if (scope.uiState.currentPageIdx === pageIdx) {
                        return Promise.resolve();
                    }

                    executeWithInstantDigest(() => {
                        scope.isLoading = true;
                        scope.uiState.currentPageIdx = pageIdx;
                    }, scope);

                    return new Promise((resolve) => {
                        const unlisten = $rootScope.$on('dashboardUrlUpdateDone', () => {
                            unlisten();
                            resolve();
                        });
                    });
                }

                /*
                 * Dashboard Export Toolbox
                 */
                scope.dashboardExportToolbox = {

                    replaceWebGLCanvas: function() {
                        document.querySelectorAll('d3fc-group > d3fc-canvas.webgl-plot-area').forEach(webglCanvas => {
                            webglCanvas.style.display = 'none';
                        });
                        // Iterate over all 2D canvas and remove the display so we can see them
                        document.querySelectorAll('d3fc-group > canvas').forEach(canvasNode => {
                            canvasNode.style.display = 'block';
                        });
                    },

                    checkLoading: function() {
                        return scope.isLoading || scope.isLoadingFilters;
                    },

                    getPagesToExport: function() {
                        const indexes = [];
                        scope.dashboard.pages.forEach((page, index) => {
                            if (page.show) {
                                indexes.push(index);
                            }
                        });
                        return indexes;
                    },

                    getFirstVisiblePageIdx: function() {
                        return scope.dashboard.pages.findIndex(page => page.show);
                    },

                    getVerticalBoxesCount: function(pageIndex) {
                        let total = 0;
                        for (const tile of scope.dashboard.pages[pageIndex].grid.tiles) {
                            if (tile.box.top + tile.box.height > total) {
                                total = tile.box.top + tile.box.height;
                            }
                        }
                        return total;
                    },

                    getTilesCount: function(pageIndex) {
                        const pages = scope.dashboard.pages;
                        return (pageIndex >= 0 && pageIndex < pages.length) ? pages[pageIndex].grid.tiles.length : 0;
                    },

                    scroll: function(scrolledHeight) {
                        return document.querySelector('.dashboard-export-page-wrapper').scrollTop = scrolledHeight;
                    },

                    getTitleBoundaries: function() {
                        return document.querySelector('.dashboard-export-title').getBoundingClientRect();
                    },

                    getGridBoundaries: function() {
                        return document.querySelector('.dashboard-export-grid').getBoundingClientRect();
                    },

                    clearDashboard: function() {
                        executeWithInstantDigest(() => scope.uiState.hideForExport = true, scope);
                    },

                    goToFirstPage: async function() {
                        goToPage(0);
                    },

                    goToPage,

                    applyFilters(pageFilters) {
                        executeWithInstantDigest(() => {
                            scope.isLoadingFilters = true;
                        });
                        DashboardFilters.resetGlobalFilters(scope.page.id);
                        return DashboardFilters.initPageFilters(scope.page, { pageFilters: pageFilters }, Object.values(scope.insightsMap))
                            .then(({ filters }) => {
                                scope.handleFiltersChange(filters);
                                scope.isLoadingFilters = false;
                            });
                    }
                };

                /*
                 * If all pages are hidden, the page is undefined, we display a placeholder and this directive is hidden.
                 * It needs to stay in the DOM though cause we need it for export.
                 */
                if (!scope.page) {
                    return;
                }

                if ($stateParams.pageId == null) {
                    $stateParams.pageId = scope.page.id;
                }

                scope.columnNumber = DashboardUtils.getColumnNumber();

                scope.$watchGroup(['page.backgroundColor', 'showGrid', 'editable'], function() {
                    scope.gridColor = DashboardPageUtils.getGridColor(scope.page.backgroundColor);
                    scope.gridContainerBackgroundColor = DashboardPageUtils.computeGridContainerBackgroundColor(scope.showGrid, scope.editable, scope.page.backgroundColor, scope.gridColor);
                });

                scope.FiltersPanelPosition = FiltersPanelPosition;
                scope.FiltersPanelDirection = FiltersPanelDirection;
                scope.isInExport = getCookie('dku_graphics_export') === 'true';
                scope.isFilterPanelDeactivated = !!DashboardFilters.getIsFilterPanelDeactivated(scope.page.id);

                DashboardPageUtils.flushCachedColumnSummaries();
                DashboardFilters.flushDatasetLowerCaseStatus();

                DashboardFilters.initPageFilters(scope.page, $location.search(), Object.values(scope.insightsMap))
                    .then(({ filters, activeFilters }) => {
                        scope.filters = filters;
                        scope.activeFilters = activeFilters;
                        $rootScope.$broadcast('dashboardFiltersChanged', { currentFilters: scope.activeFilters, previousFilters: scope.activeFilters });
                    });

                DashboardFilters.setCrossFilterStatusToPageId(scope.page.id, scope.page.enableCrossFilters);

                scope.tiles = scope.page.grid.tiles ? [...scope.page.grid.tiles] : [];
                scope.$watchCollection('page.grid.tiles', function(newTiles) {
                    scope.tiles = [...newTiles];
                });

                scope.redirectToEdit = () => {
                    WT1.event('dashboard-to-edit-mode-switch');
                    $state.go('projects.project.dashboards.dashboard.edit', {
                        projectKey: $stateParams.projectKey,
                        dashboardId: DashboardUtils.isDashboardEmbeddedInAWorkspace() ? $stateParams.objectId : $stateParams.dashboardId,
                        pageId: $stateParams.pageId
                    });
                };

                scope.$on('dashboardOpenInsightAccessModal', (_, { insight }) => {
                    openInsightAccessModal(insight);
                });

                scope.$on('dashboardOpenMoveCopyTileModal', (_, { tile }) => {
                    openMoveCopyTileModal(tile);
                });

                scope.$on('dashboardExportDatasetTile', (_, { tile }) => {
                    exportDatasetTile(tile);
                });

                scope.$on('dashboardOpenSamplingTab', (_, { tile, section }) => {
                    openSamplingTab(tile, section);
                });

                scope.$on('dashboardOpenMisc', (_, { tile }) => {
                    openMisc(tile);
                });

                scope.$on('dashboardOpenTileInsight', (_, { tile }) => {
                    openTileInsight(tile);
                });

                scope.shouldDisplayDockedFiltersPanel = (page, filters, editable) => {
                    if (scope.isInExport) {
                        return false;
                    }
                    const canFilter = DashboardFilters.canFilter(editable, filters, page.enableCrossFilters);
                    const isDocked = DashboardFilters.isFiltersPanelDocked(page.filtersParams.panelPosition);
                    const isShown = page.showFilterPanel;
                    return canFilter && isDocked && isShown;
                };

                scope.handlePreselectedTileChange = function(preselectedTile) {
                    $timeout(() => {
                        scope.selectedTile = preselectedTile;
                    });
                };

                scope.handleFiltersChange = function(filters) {
                    return $timeout(() => {
                        scope.filters = filters;
                        if (scope.editable) {
                            scope.page.filters = filters;
                        }
                        $rootScope.$broadcast('dashboardFiltersChanged', { currentFilters: filters, previousFilters: scope.activeFilters });
                        const activeFilters = DashboardFilters.getActiveGlobalFilters(scope.page.id);
                        if (!(activeFilters || []).length && !(scope.activeFilters || []).length) {
                            return;
                        } else {
                            scope.activeFilters = activeFilters;
                        }
                    });
                };

                scope.removeTileFromDashboardPage = (tileId) => {
                    return _removeTileFromGrid(scope.page.grid, tileId);
                };

                function _removeTileFromGrid(grid, tileId) {
                    const newTiles = grid.tiles.filter(tile => tile.$tileId !== tileId);

                    if (newTiles.length < grid.tiles.length) {
                        grid.tiles = newTiles;
                        return true;
                    }

                    for (const tile of grid.tiles) {
                        if (tile.tileType === 'GROUP' && tile.grid) {
                            if (_removeTileFromGrid(tile.grid, tileId)) {
                                return true;
                            }
                        }
                    }

                    return false;
                }

                scope.handleTilesChange = function(tiles) {
                    $timeout(() => {
                        scope.page.grid.tiles = tiles;
                    });
                };

                scope.handleGridLoadingStateChange = function(isLoading) {
                    $timeout(() => {
                        scope.isLoading = isLoading;
                    });
                };

                scope.handleIsDeactivatedChange = function(isDeactivated) {
                    $timeout(() => {
                        DashboardFilters.setIsFilterPanelDeactivated(scope.page.id, isDeactivated);
                        scope.activeFilters = isDeactivated ? [] : scope.filters;
                    });
                };

                scope.handleFiltersParamsChange = function(filtersParams) {
                    $timeout(() => {
                        if (filtersParams.panelPosition !== scope.page.filtersParams.panelPosition) {
                            handlePanelPositionChange(filtersParams.panelPosition);
                        }
                        scope.page.filtersParams = filtersParams;
                    });
                };

                scope.handleRaisedError = function(errorData) {
                    const { data, status, headers, config, statusText } = errorData;
                    setErrorInScope.bind(scope)(data, status, headers, config, statusText);
                };

                scope.handleGridInit = function(api) {
                    scope.reloadGrid = api.reload;
                };

                scope.addElements = function() {
                    scope.$emit('dashboardOpenAddInsightModal');
                };

                scope.$on('dashboard-reload-tiles', () => {
                    if (scope.reloadGrid) {
                        scope.reloadGrid();
                    }
                });

                $rootScope.spinnerPosition = 'dashboard-page';

                function openSamplingTab(tile, section) {
                    if (tile.insightType === 'chart' || tile.insightType === 'dataset_table') {
                        const tabName = tile.insightType === 'chart' ? 'sampling-engine' : 'sample-settings';
                        redirectToInsight({ tile, tabName, section });
                    }
                };

                function openMisc(tile) {
                    redirectToInsight({ tile, tabName: null, section: ChartFormattingPaneSections.MISC });
                };

                function openTileInsight(tile) {
                    if (!scope.editable && tile.clickAction !== 'DO_NOTHING' && TileUtils.canLink(tile)) {
                        redirectToInsight({ tile });
                    }
                };

                function exportDatasetTile(tile) {
                    if (tile.insightType === 'dataset_table' && scope.exportDatasetFn) {
                        const insight = scope.insightsMap[tile.insightId];
                        if (insight && insight.params) {
                            const loc = AnyLoc.getLocFromSmart(insight.projectKey, insight.params.datasetSmartName);
                            const hasFilters = insight.$shakerAndDashboardExplorationFilters != null &&
                                        insight.$shakerAndDashboardExplorationFilters.length > 0;
                            scope.exportDatasetFn(loc.projectKey, loc.localId, {
                                exportFiltersText: 'Apply filters from the dashboard',
                                downloadMethod: false,
                                downloadOnly: true,
                                advancedSampling: false,
                                pluginsExport: true,
                                allowFiltering: hasFilters,
                                applyExplorationFilters: hasFilters,
                                explorationFiltersAndSearchQuery: {
                                    explorationFilters: insight.$shakerAndDashboardExplorationFilters
                                }
                            });
                            WT1.event('dashboard-export-dataset-click');
                        }
                    }
                };

                function redirectToInsight({ tile, tabName, section }) {
                    const insight = scope.insightsMap[tile.insightId];
                    const targetInsight = scope.insightsMap[tile.targetInsightId] || { id: tile.targetInsightId };
                    const sref = TileUtils.getTargetUiSref(tile, insight, scope.editable, targetInsight, tabName, section);
                    const [, route, params] = /((?:[a-z]|\.)+)\((.+)\)/i.exec(sref);

                    if (sref) {
                        $state.go(route, JSON.parse(params));
                    }
                };

                function openInsightAccessModal(insight) {
                    CreateModalFromTemplate('/templates/dashboards/insights/insight-access-warning-modal.html', scope, null, function(newScope) {
                        newScope.initForInsights([insight], false);
                    }).then(function(readerAccessChanged) {
                        if (readerAccessChanged) {
                            scope.$emit('dashboardReaderAccessChanged');
                        }
                    });
                };

                function openMoveCopyTileModal(tile) {
                    CreateModalFromTemplate('/templates/dashboards/insights/move-copy-tile-modal.html', scope, 'MoveCopyTileModalController', function(newScope, translate) {
                        newScope.dashboard = scope.dashboard;
                        newScope.page = scope.page;
                        newScope.tile = tile;
                        newScope.uiState = scope.uiState;
                        newScope.insight = tile.tileType == 'INSIGHT' ? scope.insightsMap[tile.insightId] : null;
                        newScope.insightName = tile.tileType == 'INSIGHT' ? newScope.insight.name : tile.titleOptions.title;
                        newScope.insightsMap = scope.insightsMap;
                        newScope.init(newScope.insight, tile.tileParams);
                    });
                };

                function handlePanelPositionChange(panelPosition) {
                    if (panelPosition === FiltersPanelPosition.TILE) {
                        const filterTile = DashboardFilters.findFiltersTileInPage(scope.page);
                        if (filterTile == null) {
                            scope.$emit('dashboardAddFilterTile');
                        }
                    } else {
                        scope.$broadcast('dashboardRemoveFilterTile');
                    }
                };

                /** DASHBOARDS SHORTCUTS */
                const groupingShortcut = DetectUtils.getOS() === 'macos' ? 'command+option+g' : 'ctrl+alt+g';

                Mousetrap.bind(groupingShortcut, function() {
                    $rootScope.$broadcast('dashboardToggleTileGrouping');
                });

                const ctrlOrOption = DetectUtils.getOS() === 'macos' ? 'option' : 'ctrl';
                const nextPageShortcuts = [ctrlOrOption + '+down', ctrlOrOption + '+right'];
                const previousPageShortcuts = [ctrlOrOption + '+up', ctrlOrOption + '+left'];

                Mousetrap.bind(nextPageShortcuts, function() {
                    goToPage(DashboardPageUtils.getNextPageIndex(scope.dashboard, scope.uiState.currentPageIdx));
                });

                Mousetrap.bind(previousPageShortcuts, function() {
                    goToPage(DashboardPageUtils.getPreviousPageIndex(scope.dashboard, scope.uiState.currentPageIdx));
                });

                const unregisterDashboardTitleChangedEvent = $rootScope.$on('dashboardPageTitleChanged', function(_event, newTitle) {
                    scope.page.displayedTitle = newTitle.displayedTitle;
                    scope.pageTitle = newTitle.displayedTitle;
                });

                scope.pageTitle = scope.page && scope.page.displayedTitle ? scope.page.displayedTitle :
                    scope.translate('DASHBOARD.NAVIGATION_DRAWER.PAGE_INDEX', 'Page ' + (scope.pageIdx + 1), { index: scope.pageIdx + 1 });

                scope.$on('$destroy', function() {
                    Mousetrap.unbind(groupingShortcut);
                    Mousetrap.unbind(nextPageShortcuts);
                    Mousetrap.unbind(previousPageShortcuts);
                    unregisterDashboardTitleChangedEvent();
                });
            }
        };
    });
})();
