(function() {
    'use strict';

    const app = angular.module('dataiku.projectdeployer');

    app.service('ProjectDeployerDeploymentService', function($rootScope, CreateModalFromTemplate) {
        this.startCreateDeployment = function(preselectedPublishedProjectKey, preselectedBundleId) {
            return CreateModalFromTemplate("/templates/project-deployer/new-deployment-modal.html", $rootScope, null, function(modalScope) {
                modalScope.deploymentSettings = {
                    publishedProjectKey: preselectedPublishedProjectKey,
                    bundleId: preselectedBundleId
                };
                modalScope.fromBundle = !!(preselectedPublishedProjectKey && preselectedBundleId);
            });
        };

        this.openGovernanceStatusDeploymentId = function(deploymentId, infraId, bundleId) {
            return CreateModalFromTemplate("/templates/deployer/governance-modal.html", $rootScope, "GovernanceAbstractDeployerModalController", function(modalScope) {
                modalScope.deploymentId = deploymentId;
                modalScope.infraId = infraId;
                modalScope.packageId = bundleId;
                modalScope.refreshGovernanceStatuses('projectDeployer');
            });
        };

        this.openGovernanceStatusDeployment = function(deployment) {
            return CreateModalFromTemplate("/templates/deployer/governance-modal.html", $rootScope, "GovernanceAbstractDeployerModalController", function(modalScope) {
                modalScope.deployment = deployment;
                modalScope.refreshGovernanceStatuses('projectDeployer');
            });
        };

        this.openGovernanceStatusNewDeployment = function(publishedProjectKey, infraId, bundleId) {
            return CreateModalFromTemplate("/templates/deployer/governance-modal.html", $rootScope, "GovernanceAbstractDeployerModalController", function(modalScope) {
                modalScope.publishedItemId = publishedProjectKey;
                modalScope.infraId = infraId;
                modalScope.packageId = bundleId;
                modalScope.refreshGovernanceStatuses('projectDeployer');
            });
        };
    });

    app.factory('ProjectDeploymentProgress', function($rootScope) {
        function returnAfterStepUpdate(deployment, report, stepName) {
            if (svc.stepInProgress(deployment, report, stepName)) return true;
            svc.finalizeStepIfNeeded(deployment, report, stepName);
            return svc.stepFailed(deployment, stepName);
        }

        const projectStandardsAllowed = $rootScope.appConfig.licensedFeatures.projectStandardsAllowed;        

        function getDeploymentSteps() {
            const DEPLOYMENT_STEPS = {
                PREPARE_SYNC: { index: 0, name: 'importing bundle', getReport: (report) => report.prepareSyncReport || report.multiNodesSynchronisationPreparationReport },
                PRELOAD_BUNDLE: { index: 1, name: 'preloading bundle', getReport: (report) => report.preloadReport || report.multiNodesBundlePreloadingReport },
                ACTIVATE_CHECK: { index: 2, name: 'running bundle activation checks', getReport: (report) => report.activateCheckReport || report.multiNodesActivationCheckingReport },
                ACTIVATE_BUNDLE: { index: 3, name: 'activating bundle', getReport: (report) => report.activateReport || report.multiNodesBundleActivationReport },
            };

            if (projectStandardsAllowed) {
                Object.values(DEPLOYMENT_STEPS).forEach(deploymentStep => deploymentStep.index += 1);
                DEPLOYMENT_STEPS['PROJECT_STANDARDS_CHECK'] = { index: 0, name: 'verifying Project Standards', getReport: (report) => report.projectStandardsCheckReport };
            }

            return DEPLOYMENT_STEPS;
        }
        

        const svc = {
            STEP_STATUS: {
                SUCCESS: 'SUCCESS',
                ERROR: 'ERROR',
                WARNING: 'WARNING',
                IN_PROGRESS: 'IN_PROGRESS',
                NOT_STARTED: 'NOT_STARTED',
            },
            DEPLOY_STEPS: getDeploymentSteps(),
            DEPLOY_STATUS: {
                DONE: 'DONE',
                DONE_WITH_WARNINGS: 'DONE_WITH_WARNINGS',
                FAILED: 'FAILED',
                INTERRUPTED: 'INTERRUPTED',
                IN_PROGRESS: 'IN_PROGRESS',
            },
            stepStatusDetails: function(step) {
                let details = "";
                if (step.okNodes) {
                    details += `, succeeded on ${step.okNodes} node${step.okNodes <= 1 ? "" : "s"}`;
                }
                if (step.koNodes) {
                    details += `, failed on ${step.koNodes} node${step.koNodes <= 1 ? "" : "s"}`;
                }
                return details;
            },
            initDeployment: function() {
                const DEPLOYMENT_STEPS = Object.values(this.DEPLOY_STEPS).map((step) => ({
                    index: step.index,
                    name: step.name,
                    status: svc.STEP_STATUS.NOT_STARTED,
                    infoMessage: {}
                }));
                return {
                    status: svc.DEPLOY_STATUS.IN_PROGRESS,
                    steps: DEPLOYMENT_STEPS,
                    progress: {
                        current: 0,
                        end: DEPLOYMENT_STEPS.length + 1
                    },
                    error: null,
                    infoMessages: {
                        messages: []
                    }
                };
            },
            updateDeploymentStatus: function(deployment) {
                if (deployment.status === svc.DEPLOY_STATUS.IN_PROGRESS) {
                    deployment.status = deployment.infoMessages.messages.length > 0 ?
                        svc.DEPLOY_STATUS.DONE_WITH_WARNINGS : svc.DEPLOY_STATUS.DONE;
                }
            },
            updateDeploymentSteps: function(report, hooksStatus, deployment) {
                if (hooksStatus) deployment.deploymentHookExecutionStatus = hooksStatus;
                if (report.interrupted) {
                    deployment.status = svc.DEPLOY_STATUS.INTERRUPTED;
                } else if (report.maxSeverity === svc.STEP_STATUS.ERROR) {
                    deployment.status = svc.DEPLOY_STATUS.FAILED;
                }
                deployment.infoMessages.maxSeverity = report.maxSeverity;
                deployment.infoMessages.messages = report.messages;
                if (projectStandardsAllowed && returnAfterStepUpdate(deployment, report, svc.DEPLOY_STEPS.PROJECT_STANDARDS_CHECK)) return;
                if (returnAfterStepUpdate(deployment, report, svc.DEPLOY_STEPS.PREPARE_SYNC)) return;
                if (returnAfterStepUpdate(deployment, report, svc.DEPLOY_STEPS.PRELOAD_BUNDLE)) return;
                if (returnAfterStepUpdate(deployment, report, svc.DEPLOY_STEPS.ACTIVATE_CHECK)) return;
                returnAfterStepUpdate(deployment, report, svc.DEPLOY_STEPS.ACTIVATE_BUNDLE);
            },
            getStepAndReport: function(deployment, deployStep, report) {
                const step = deployment.steps.find(step => step.index === deployStep.index);
                const stepReport = deployStep.getReport(report);
                return { step, stepReport };
            },
            stepFailed: function(deployment, deployStep) {
                const step = deployment.steps.find(step => step.index === deployStep.index);
                return step.status === svc.STEP_STATUS.ERROR;
            },
            finalizeStepIfNeeded: function(deployment, report, deployStep) {
                const { step, stepReport } = svc.getStepAndReport(deployment, deployStep, report);
                if (step.status !== svc.STEP_STATUS.NOT_STARTED && step.status !== svc.STEP_STATUS.IN_PROGRESS) return;
                if (stepReport.maxSeverity === svc.STEP_STATUS.ERROR || stepReport.maxSeverity === svc.STEP_STATUS.WARNING) {
                    step.status = stepReport.maxSeverity;
                    step.infoMessage = stepReport;
                } else {
                    step.status = svc.STEP_STATUS.SUCCESS;
                }
                if (stepReport.okNodes !== undefined) {
                    step.okNodes = stepReport.okNodes;
                }
                if (stepReport.koNodes !== undefined) {
                    step.koNodes = stepReport.koNodes;
                }
                deployment.progress.current = deployStep.index + 2;
            },
            stepInProgress: function(deployment, report, deployStep) {
                const { step, stepReport } = svc.getStepAndReport(deployment, deployStep, report);
                if (stepReport) return false; // the report for this step exists, meaning that the step is done
                step.status = svc.STEP_STATUS.IN_PROGRESS;
                deployment.progress.current = deployStep.index + 1;
                return true;
            }
        }
        return svc;
    });

    app.component('projectDeploymentProjectStandardsModal', {
        bindings: {
            modalControl: '<',
            projectStandardsMessages: '<',
        },
        templateUrl: '/templates/project-deployer/project-standards-modal.html',
        controller: function() {
            const $ctrl = this;

            $ctrl.confirm = function() {
                $ctrl.modalControl.resolve();
            };
            
            $ctrl.cancel = function() {
                $ctrl.modalControl.dismiss();
            };
        }
    });

    app.component('projectDeploymentModal', {
        bindings: {
            modalControl: '<',
            deploymentId: '<',
            jobId: '<',
            lastDeploymentAction: '<',
            insufficientPermissionsMessage: '<',
            peekingUpdateStarted: '<',
            peekingUpdateEnded: '<',
            deployProject: '<',
        },
        templateUrl: '/templates/project-deployer/deploy-modal.html',
        controller: function($scope, $interval, DataikuAPI, ProjectDeploymentProgress) {
            const $ctrl = this;

            $ctrl.STEP_STATUS = ProjectDeploymentProgress.STEP_STATUS;
            $ctrl.DEPLOY_STATUS = ProjectDeploymentProgress.DEPLOY_STATUS;
            $ctrl.stepStatusDetails = ProjectDeploymentProgress.stepStatusDetails;
            $ctrl.MODAL_TITLES = {
                IN_PROGRESS: 'Updating project',
                FAILED: 'Project update failed',
                INTERRUPTED: 'Project update interrupted',
                DONE_WITH_WARNINGS: 'Project successfully updated (with warnings)',
                DONE: 'Project successfully updated',
            };


            $ctrl.modalTitle = $ctrl.MODAL_TITLES.IN_PROGRESS;
            $ctrl.deployment = ProjectDeploymentProgress.initDeployment();
            $ctrl.percentage = 0;
            $ctrl.peekDeploymentActionRepeat = null;

            $ctrl.$onInit = function() {
                if ($ctrl.jobId || $ctrl.lastDeploymentAction && $ctrl.lastDeploymentAction.inProgress) {
                    $ctrl.peekingUpdateStarted();
                    $ctrl.peekDeploymentActionRepeat = $interval(peekDeploymentAction, 1000);
                    peekDeploymentAction();
                } else if ($ctrl.lastDeploymentAction) {
                    onComplete($ctrl.lastDeploymentAction?.report);
                } else {
                    // should not happen, either jobId or lastDeploymentAction should be defined
                    $ctrl.close();
                }
            }

            $ctrl.$onDestroy = function() {
                cancelPeekDeploymentAction();
            }

            $ctrl.abort = function() {
                if ($ctrl.deploymentId) {
                    DataikuAPI.projectdeployer.deployments.abortDeploymentAction($ctrl.deploymentId).error(setErrorInScope.bind($scope));
                }
            }

            $ctrl.retryDeploy = function() {
                $ctrl.deployProject();
                $ctrl.close();
            }

            $ctrl.close = function() {
                $ctrl.modalControl.dismiss();
            }

            function updateTitleAndProgress() {
                $ctrl.percentage = $ctrl.deployment.progress.current / $ctrl.deployment.progress.end * 100;
                $ctrl.modalTitle = $ctrl.MODAL_TITLES[$ctrl.deployment.status];
            }

            function peekDeploymentAction() {
                DataikuAPI.projectdeployer.deployments.peekDeploymentActionProgress($ctrl.jobId, $ctrl.deploymentId)
                    .then(response => {
                        if (response?.data.hasResult || response?.data.unknown) {
                            cancelPeekDeploymentAction();
                            onComplete(response.data.result);
                        } else {
                            onUpdate(response);
                        }
                    })
                    .catch((error) => {
                        cancelPeekDeploymentAction();
                        onError(error);
                    });
            }

            function cancelPeekDeploymentAction() {
                if ($ctrl.peekDeploymentActionRepeat) {
                    $ctrl.peekingUpdateEnded();
                    $interval.cancel($ctrl.peekDeploymentActionRepeat);
                }
                $ctrl.peekDeploymentActionRepeat = null;
            }

            function onComplete(report) {
                $ctrl.deployment.futureResponse = null;
                if (report) {
                    const hooksStatus = report.deploymentHookExecutionStatus;
                    ProjectDeploymentProgress.updateDeploymentSteps(report, hooksStatus, $ctrl.deployment);
                    ProjectDeploymentProgress.updateDeploymentStatus($ctrl.deployment);
                    updateTitleAndProgress();
                } else {
                    $ctrl.modalControl.dismiss();
                }
            }

            function onUpdate(response) {
                $ctrl.deployment.futureResponse = response.data;
                if (response.data.progress) {
                    const report = response.data.progress.report;
                    const deploymentHookExecutionStatus = report ? report.deploymentHookExecutionStatus : null;
                    ProjectDeploymentProgress.updateDeploymentSteps(response.data.progress.report, deploymentHookExecutionStatus, $ctrl.deployment);
                }
                updateTitleAndProgress();
            }

            function onError(error) {
                $ctrl.deployment.status = ProjectDeploymentProgress.DEPLOY_STATUS.FAILED;
                $ctrl.deployment.error = error.data;
                $ctrl.deployment.futureResponse = null;
                updateTitleAndProgress();
                setErrorInScope.bind($scope)(error);
            }
        }
    });
})();

