(function () {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("EdaRecipeCreationController", function (
        $scope, CreateModalFromTemplate,
    ) {
        $scope.showCreateRecipeModal = (cardType) => {
            let controller;
            let template;

            switch (cardType) {
                case "univariate_header":
                    controller = "UnivariateRecipeCreationController";
                    template = "/templates/recipes/single-output-recipe-creation.html";
                    break;

                case "pca":
                    controller = "PcaRecipeCreationController";
                    template = "/templates/recipes/eda_pca-recipe-creation.html";
                    break;

                case "ttest_ztest_1samp":
                case "sign_test_1samp":
                case "shapiro":
                case "ttest_2samp":
                case "mood_test_2samp":
                case "levene_test_2samp":
                case "ks_test_2samp":
                case "oneway_anova":
                case "mood_nsamp":
                case "levene_nsamp":
                case "pairwise_ttest":
                case "pairwise_mood":
                case "pairwise_levene":
                case "chi2_independence_test":
                    controller = "StatsTestRecipeCreationController";
                    template = "/templates/recipes/single-output-recipe-creation.html";
                    break;

                default:
                    throw new Error(`Unsupported analysis type: ${cardType}`);
            }

            CreateModalFromTemplate(template, $scope, controller, (scope) => {
                // preselect the input dataset, if available
                // note: the zone information is carried through the different scopes...
                if ($scope.preselectedInput) {
                    scope.preselectInputDataset($scope.preselectedInput);
                }

                scope.recipeSubType = scope.mapToRecipeSubType ? scope.mapToRecipeSubType(cardType) : undefined;

                scope.recipeCreationSettings = {
                    ...scope.recipeCreationSettings,
                    card: { type: cardType },
                    useDefaultParameters: true,
                };
            });
        }
    });

    app.controller("PcaRecipeCreationController", function (
        $scope, $stateParams, $controller, DataikuAPI,
        RecipeComputablesService, DatasetUtils, PartitionDeps,
    ) {
        $scope.recipeType = "eda_pca";
        $controller("_RecipeCreationControllerBase", { $scope });
        addDatasetUniquenessCheck($scope, DataikuAPI, $stateParams.projectKey);
        fetchManagedDatasetConnections($scope, DataikuAPI);

        // force the main label in the recipe output list (hidden by default)
        $scope.forceMainLabel = true;

        $scope.recipe = {
            projectKey: $stateParams.projectKey,
            type: $scope.recipeType,
            inputs: {
                main: {
                    items: [{
                        ref: '',
                        deps: [],
                    }],
                },
            },
            outputs: {},
            params: {},
        };

        $scope.recipeCreationSettings = {
            card: null,
        };

        $scope.preselectInputDataset = (datasetSmartName) => {
            $scope.recipe.inputs.main.items[0].ref = datasetSmartName;
            $scope.preselectedInputDataset = datasetSmartName;
        };

        $scope.init = (card, dataSpec, rootCard) => {
            const {
                datasetSelection,
                containerSelection,
                inputDatasetSmartName,
            } = dataSpec;

            $scope.recipeCreationSettings = {
                ...$scope.recipeCreationSettings,
                card,
                datasetSelection,
                containerSelection,
                originWorksheetId: $stateParams.worksheetId,
            };

            $scope.preselectInputDataset(inputDatasetSmartName);
        };

        DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, $scope.recipeType)
            .then(data => {
                $scope.availableInputDatasets = data[0];
            });

        RecipeComputablesService.getComputablesMap($scope.recipe, $scope)
            .then(computablesMap => {
                $scope.setComputablesMap(computablesMap);
            });

        const hasMainInput = () => {
            const { inputs } = $scope.recipe;
            const input = inputs["main"];
            return input != null
                && input.items != null
                && input.items.length > 0
                && input.items[0].ref;
        }

        const hasOutput = (roleName) => {
            const { outputs } = $scope.recipe;
            const output = outputs[roleName];
            return output != null
                && output.items != null
                && output.items.length > 0
                && output.items[0].ref;
        }

        const hasAtLeastOneOutput = () => {
            return hasOutput("main")
                || hasOutput("eigenvectors")
                || hasOutput("eigenvalues");
        }

        $scope.hasMainInput = hasMainInput;
        $scope.hasAtLeastOneOutput = hasAtLeastOneOutput;
        $scope.shouldDisplayOutputExplanation = () => !hasAtLeastOneOutput();
        $scope.generateOutputExplanation = () => "This recipe requires at least one output";

        $scope.$watch("recipe.inputs.main.items[0].ref", (inputDsName) => {
            if (!inputDsName) {
                return;
            }

            $scope.recipe.name = "pca_" + inputDsName.replace(/[A-Z]*\./, "");

            if ($scope.preselectedInputDataset && inputDsName != $scope.preselectedInputDataset) {
                $scope.zone = null;
            }
        });

        // body inspiration from creation.js
        $scope.doCreateRecipe = () => {
            const recipe = angular.copy($scope.recipe);
            if ($scope.recipeName) {
                recipe.name = $scope.recipeName.name;
            }

            PartitionDeps.prepareRecipeForSerialize(recipe);
            const settings = { ...$scope.recipeCreationSettings };
            if ($scope.zone) {
                settings.zone = $scope.zone;
            }

            return DataikuAPI.flow.recipes.generic.create(recipe, settings);
        };
    });

    app.controller("_BaseStatsTestRecipeCreationController", function (
        $scope, $stateParams, $controller,
    ) {
        $controller("SingleOutputDatasetRecipeCreationController", { $scope });

        $scope.sanitizeCardType = (cardType) => {
            if (cardType === "univariate_header") {
                return "univariate_analysis"
            }
            return cardType;
        }

        $scope.autosetName = function () {
            if ($scope.io.inputDataset) {
                const niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./, "");
                const cardType = $scope.sanitizeCardType($scope.recipeCreationSettings.card.type);
                $scope.maybeSetNewDatasetName(niceInputName + "_" + cardType);
            }
        };

        $scope.recipeCreationSettings = {};

        $scope.init = (card, dataSpec, rootCard) => {
            const {
                datasetSelection,
                containerSelection,
                inputDatasetSmartName,
            } = dataSpec;

            $scope.recipeSubType = $scope.mapToRecipeSubType ? $scope.mapToRecipeSubType(card.type) : undefined;

            $scope.recipeCreationSettings = {
                ...$scope.recipeCreationSettings,
                card,
                datasetSelection,
                containerSelection,
                originWorksheetId: $stateParams.worksheetId,
                withConfidenceIntervals: rootCard.showConfidenceInterval,
                confidenceLevel: rootCard.confidenceLevel
            };

            $scope.preselectInputDataset(inputDatasetSmartName);
        };

        $scope.getCreationSettings = () => {
            return $scope.recipeCreationSettings;
        };
    });

    app.controller("StatsTestRecipeCreationController", function (
        $scope, $controller,
    ) {
        $scope.recipeType = "eda_stats";

        // init scope with parent controller
        $controller("_BaseStatsTestRecipeCreationController", { $scope });

        $scope.mapToRecipeSubType = (cardType) => {
            switch (cardType) {
                case "ttest_ztest_1samp":
                    return "recipe_one-sample-t-test";
                case "sign_test_1samp":
                    return "recipe_one-sample-sign-test";
                case "shapiro":
                    return "recipe_one-sample-shapiro-test";
                case "ttest_2samp":
                    return "recipe_two-sample-t-test";
                case "mood_test_2samp":
                    return "recipe_two-sample-mood-test";
                case "levene_test_2samp":
                    return "recipe_two-sample-levene-test";
                case "ks_test_2samp":
                    return "recipe_two-sample-ks-test";
                case "oneway_anova":
                    return "recipe_one-way-anova-test";
                case "mood_nsamp":
                    return "recipe_n-sample-mood-test";
                case "levene_nsamp":
                    return "recipe_n-sample-levene-test";
                case "pairwise_ttest":
                    return "recipe_pairwise-t-test";
                case "pairwise_mood":
                    return "recipe_pairwise-mood-test";
                case "pairwise_levene":
                    return "recipe_pairwise-levene-test";
                case "chi2_independence_test":
                    return "recipe_chi2-independence-test";
                default:
                    return undefined;
            }
        }
    });

    app.controller("UnivariateRecipeCreationController", function (
        $scope, $controller,
    ) {
        $scope.recipeType = "eda_univariate";

        // init scope with parent controller
        $controller("_BaseStatsTestRecipeCreationController", { $scope });
    });

    app.controller("_EdaRecipeEditionController", function (
        $scope, $controller, $q, Assert, ComputableSchemaRecipeSave, StateUtils, DataikuAPI
    ) {
        Assert.inScope($scope, "hooks");
        Assert.inScope($scope, "recipe");
        Assert.inScope($scope, "script");
        Assert.inScope($scope, "origRecipe");
        Assert.inScope($scope, "origScript");

        $controller("_RecipeWithEngineBehavior", { $scope });

        $scope.payload = JSON.parse($scope.script.data);
        $scope.origPayload = JSON.parse($scope.origScript.data);
        $scope.originWorksheetId = $scope.recipe.params.originWorksheetId;

        // for propagating errors from downgraded components to angularjs
        $scope.propagateApiError = (apiError) => {
            $scope.fatalAPIError = apiError;
            $scope.$applyAsync(); // trigger angularjs change detection
        }

        $scope.getRecipeSingleInput = () => {
            return $scope.recipe.inputs["main"].items[0];
        };

        $scope.getRecipeMainOutput = () => {
            return $scope.recipe.outputs["main"] ? $scope.recipe.outputs["main"].items[0] : undefined;
        };

        $scope.getRecipeFilter = () => {
            return $scope.recipe.params.datasetSelection.filter;
        };

        $scope.updateRecipePayload = (payload) => {
            $scope.payload = payload;
            $scope.$applyAsync(); // trigger angularjs change detection
        };

        $scope.hooks.getPayloadData = () => {
            return angular.toJson($scope.payload);
        };

        $scope.hooks.recipeIsDirty = () => {
            if ($scope.recipe == null) {
                return false;
            }

            if ($scope.creation) {
                return true;
            }

            // !! WARNING !!
            // when parsing the backend data model, missing fields will be parsed as undefined in $scope.origPayload
            // we must be careful in the frontend to update the model with 'undefined' and not 'null' for these nullable fields,
            // otherwise angular.equals($scope.payload, $scope.origPayload) will evaluate to false
            return !angular.equals($scope.recipe, $scope.origRecipe)
                || !angular.equals($scope.payload, $scope.origPayload);
        };

        $scope.hooks.save = () => {
            const deferred = $q.defer();
            const sr = $scope.hooks.getRecipeSerialized();
            const payload = $scope.hooks.getPayloadData();
            $scope.script.data = payload;
            ComputableSchemaRecipeSave.handleSave($scope, sr, payload, deferred);
            return deferred.promise;
        };

        $scope.$on('recipeSaved', () => {
            // so that the recipe is not dirty anymore.
            $scope.origPayload = JSON.parse($scope.script.data);
        });

        $scope.hooks.updateRecipeStatus = function () {
            var deferred = $q.defer();
            var payload = $scope.hooks.getPayloadData();
            $scope.updateRecipeStatusBase(false, payload).then(function () {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        $scope.hooks.onRecipeLoaded = function () {
            $scope.hooks.updateRecipeStatus();
        };

        $scope.originWorksheetExists = false;
        if ($scope.originWorksheetId) {
            DataikuAPI.statistics.get($scope.projectKey, $scope.originWorksheetId).then(() => {
                $scope.originWorksheetExists = true;
            }).catch(() => {
                $scope.originWorksheetExists = false;
            });
        };

        $scope.goToWorksheet = function () {
            StateUtils.go.statisticsWorksheet($scope.originWorksheetId, $scope.projectKey);
        };
    });

    app.controller("PcaRecipeEditionController", function (
        $scope, $controller,
    ) {
        // init scope with parent controller
        $controller("_EdaRecipeEditionController", { $scope });

        $scope.hasAllRequiredOutputs = () => {
            if (!$scope.recipe || !$scope.recipe.outputs) {
                return false;
            }

            const { main, eigenvectors, eigenvalues } = $scope.recipe.outputs;
            // at least one of the outputs is needed
            return (main && main.items && main.items.length > 0)
                || (eigenvectors && eigenvectors.items && eigenvectors.items.length > 0)
                || (eigenvalues && eigenvalues.items && eigenvalues.items.length > 0);
        };
    });

    app.controller("StatsTestRecipeEditionController", function (
        $scope, $controller,
    ) {
        // init scope with parent controller
        $controller("_EdaRecipeEditionController", { $scope });
    });

    app.controller("UnivariateRecipeEditionController", function (
        $scope, $controller,
    ) {
        // init scope with parent controller
        $controller("_EdaRecipeEditionController", { $scope });
    });
})();
