(function() {
'use strict';

const app = angular.module('dataiku.flow.graph');

/**
 * @name zoneEditor
 * @description
 *      Component to display the zone edition tool. User can edit the name and color of the zone.
 *
 * @param { String }    name                    - name of the zone
 * @param { String }    color                   - color of the zone (if empty, picks a random color in the stock)
 * @param { Boolean }   disableCompactStyling   - if True, controls and control labels don't have the compact styling and take a larger width
 *
 * @example
 *      <zone-editor name="uiState.name" color="uiState.color"></zone-editor>
 */
app.component('zoneEditor', {
    bindings: {
        name: '=',
        color: '=',
        disableCompactStyling: '<?'
    },
    templateUrl: '/templates/zones/edit-zone-box.html',
    controller: function zoneEditorCtrl($timeout) {
        this.$onInit = function () {
            this.stockColors = ["#C82423","#8C2DA7","#31439C","#087ABF","#0F786B","#4B8021","#F9BE40","#C54F00","#D03713","#465A64"];
            this.color = this.color === undefined ? this.stockColors[Math.floor(Math.random() * this.stockColors.length)] : this.color;
            this.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
        }

        this.pickStockColor = color => {
            $timeout(() => {
                this.color = color;
            })
        };
    }
});

/**
 * @name zoneSelector
 * @description
 *      Component to display the zone selector. Provide the option to either create a new zone, or select an already existing one.
 *
 * @param { String }    name                    - name of the zone
 * @param { String }    color                   - color of the zone (if empty, picks a random color in the stock)
 * @param { String }    selectedZone            - id of the selected existing zone when in 'select' mode (preselects the selected or last used zone)
 * @param { Boolean }   creationMode            - 'CREATE' to create a new zone or 'SELECT' to pick an already existing one (by default: 'SELECT' if forceCreation is False)
 * @param { Boolean }   forceCreation           - if True, only displays the creation option
 * @param { Boolean }   disableCompactStyling   - if True, controls and control labels don't have the compact styling and take a larger width
 *
 * @example
 * <zone-selector name="uiState.name"
 *                color="uiState.color"
 *                selected-zone="uiState.selectedZone"
 *                creation-mode="uiState.creationMode"
 *                force-creation="uiState.forceCreation">
 * </zone-selector>
 */
app.component('zoneSelector', {
    bindings: {
        name: '=',
        color: '=',
        creationMode: '=',
        selectedZone: '=',
        forceCreation: '<?',
        projectKey: '<?',
        disableCompactStyling: '<?'
     },
    templateUrl: '/templates/flow-editor/zone-selection-box.html',
    controller:  ['$scope', '$rootScope', '$stateParams', 'DataikuAPI', 'localStorageService', 'FlowGraphSelection', function zoneSelectionCtrl(scope, $rootScope, $stateParams, DataikuAPI, localStorageService, FlowGraphSelection) {
        const ctrl = this;
        
        ctrl.$onInit = function() {
            ctrl.projectKey = ctrl.projectKey === undefined ? $stateParams.projectKey : ctrl.projectKey;
            if (!ctrl.forceCreation) {
                scope.$watch("$ctrl.projectKey", function () {
                    listZones();
                }, true);
                ctrl.selectedZone = preselectZone();
            }  
        };

        ctrl.zoneComparator = (v1, v2) => {
            // If we don't get strings, just compare by index
            if (v1.type !== 'string' || v2.type !== 'string') {
                return (v1.index < v2.index) ? -1 : 1;
            }
            if (ctrl.zones[v1.index].id === 'default') {
                return 1;
            }
            if (ctrl.zones[v2.index].id === 'default') {
                return -1;
            }
            // Compare strings alphabetically, taking locale into account
            return v1.value.localeCompare(v2.value);
        };

        const preselectZone = () => {
            const lastUsedZoneKey = `dku.flow.lastUsedZone.${ctrl.projectKey}`;
            return localStorageService.get(lastUsedZoneKey);
        };

        function listZones() {
            DataikuAPI.flow.zones.list(ctrl.projectKey).success(zones => {
                ctrl.zones = zones;
                ctrl.forceCreation = zones.length === 0;
                ctrl.creationMode = !ctrl.forceCreation ? 'SELECT' : 'CREATE';
            }).error(setErrorInScope.bind(scope));
        };

    }]
});

app.directive('zoneRightColumnSummary', function(
    $controller,
    $rootScope,
    $state,
    $stateParams,
    DataikuAPI,
    CreateModalFromTemplate,
    TaggableObjectsUtils,
    FlowGraph,
    ActivityIndicator,
    Dialogs,
    SubFlowCopyService,
    FlowGraphSelection,
    ZoneService
) {
    return {
        templateUrl: '/templates/zones/right-column-summary.html',

        link: function(scope, element, attrs) {

            $controller('_TaggableObjectsMassActions', {$scope: scope});
            $controller('_TaggableObjectsCapabilities', {$scope: scope});

            scope.$stateParams = $stateParams;

            scope.zoomOnZone = ZoneService.zoomOnZone;

            scope.zoomOutOfZone = ZoneService.zoomOutOfZone;

            scope.$on("objectSummaryEdited", function() {
                const zone = scope.zoneFullInfo.zone;
                const tor = {type: 'FLOW_ZONE', projectKey: $stateParams.projectKey, id: zone.id};
                DataikuAPI.taggableObjects.getMetadata(tor).success(function(metadata) {
                    metadata.tags = zone.tags;
                    DataikuAPI.taggableObjects.setMetaData(tor, metadata).success(function() {
                        ActivityIndicator.success("Saved");
                    });
                }).error(setErrorInScope.bind(scope));
            });

            scope.editZone = () => {
                CreateModalFromTemplate("/templates/zones/edit-zone-modal.html", scope, null, function(newScope){
                    newScope.uiState = {
                        color: scope.selection.selectedObject.color,
                        name: scope.selection.selectedObject.name
                    }
                    newScope.go = function(){
                        DataikuAPI.flow.zones.edit($stateParams.projectKey, scope.selection.selectedObject.id, newScope.uiState.name, newScope.uiState.color).success(function () {
                            scope.$emit('reloadGraph');
                            if ($stateParams.zoneId) {
                                $rootScope.$emit("zonesListChanged", newScope.uiState.name);
                            }
                            newScope.dismiss()
                        }).error(setErrorInScope.bind(newScope));
                    }
                });
            }

            // TODO: code duplication
            function openMultiBuildModal(errorMessage, defaultLimitToZone) {
                return function(data) {
                    if (!data.length) {
                        Dialogs.error(scope, "Nothing to build", errorMessage);
                    } else {
                        CreateModalFromTemplate(
                            "/templates/flow-editor/tools/build-multiple-flow-computables-modal.html",
                            scope,
                            "BuildMultipleComputablesController",
                            function(modalScope) {
                                modalScope.initModal(data, undefined, { origin: "FLOW_ZONE" });
                                modalScope.jobOptions.stopAtFlowZoneBoundary = defaultLimitToZone;
                            }
                        );
                    }
                }
            }

            scope.buildZone = function(zoneId) {
                DataikuAPI.flow.listDownstreamComputables($stateParams.projectKey, {zoneId: scope.selection.selectedObject.id})
                    .success(openMultiBuildModal("This zone has no buildable dataset.", true))
                    .error(setErrorInScope.bind(scope));
            }

            scope.refreshData = function() {
                DataikuAPI.zones.getFullInfo(scope.selection.selectedObject.projectKey, scope.selection.selectedObject.cleanId).success(function(data) {
                    data.zone.cleanId = data.zone.id
                    scope.zoneFullInfo = data;
                    // check that the selection didn't change while getFullInfo was called
                    if (scope.selection.selectedObject && scope.selection.selectedObject.cleanId === data.zone.cleanId) {
                        scope.selection.selectedObject = data.zone;
                        scope.selection.selectedObject.isCollapsed = scope.collapsedZones.find(it => it === data.zone.id) !== undefined;
                    }
                }).error(setErrorInScope.bind(scope));
            };

            scope.deleteZone = () => {
                let items = scope.getSelectedTaggableObjectRefs();
                let success = undefined;
                if ($stateParams.zoneId) {
                    items = [TaggableObjectsUtils.fromNode(scope.nodesGraph.nodes[`zone_${$stateParams.zoneId}`])];
                    success = () => scope.zoomOutOfZone();
                }
                scope.deleteSelected(items, success);
            };

            scope.collapseAllZones = () => {
                const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
                scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'collapseAll');
            }

            scope.expandAllZones = () => {
                const allFlowZones = Object.values(FlowGraph.get().nodes).filter(it => TaggableObjectsUtils.fromNodeType(it.nodeType) === 'FLOW_ZONE');
                scope.toggleZoneCollapse(allFlowZones.map(TaggableObjectsUtils.fromNode), 'expandAll');
            }

            scope.copyZone = () => {
                const startCopyTool = scope.startTool('COPY',  {preselectedNodes: FlowGraphSelection.getSelectedNodes().map(n => n.id)});
                startCopyTool.then(() => {
                    const selectedTaggableObjectRefs = FlowGraphSelection.getSelectedTaggableObjectRefs();
                    const itemsByZones = FlowGraph.nodesByZones((node) => TaggableObjectsUtils.fromNode(node));
                    SubFlowCopyService.start(selectedTaggableObjectRefs, itemsByZones, scope.stopAction);
                });
            };

            scope.$watch("selection.selectedObject",function(nv) {
                if (!scope.selection) scope.selection = {};
                scope.zoneFullInfo = {zone: scope.selection.selectedObject, timeline: {}}; // display temporary (incomplete) data
            });

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) return;
                scope.selection.selectedObject.cleanId = scope.selection.selectedObject.id.split('_')[1];
                scope.refreshData();
            });

            const zonesListChangedListener = $rootScope.$on("zonesListChanged", scope.refreshData);
            scope.$on('$destroy',zonesListChangedListener);
        }
    }
});

app.service('ZoneService', function($stateParams, $state) {
    this.zoomOnZone = zoneId => {
        $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId }));
    };

    this.zoomOutOfZone = (id = null) => {
        $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: null, id}))
    }
})
})();
